﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nifm.h>
#include <nn/os/os_ThreadApi.h>
#include <nn/pctl/pctl_Api.h>
#include <nn/pctl/pctl_ApiSystem.h>
#include <nn/pctl/pctl_ApiForAuthentication.h>
#include <nn/pctl/detail/ipc/pctl_IpcConfig.h>
#include <nn/pctl/detail/service/pctl_IpcServer.h>
#include <nn/pctl/detail/service/pctl_ServiceMain.h>
#include <nn/pctl/detail/service/pctl_ServiceWatcher.h>
#include <nn/pctl/detail/service/common/pctl_FileSystem.h>
#include <nn/pctl/detail/service/common/pctl_SystemInfo.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/time/time_ApiForMenu.h>

using namespace nn::pctl::detail::service;

class WatcherEventTest : public ::testing::Test
{
public:
    static void SetUpTestCase()
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::Initialize());
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::InitializeForMenu());

        nn::nifm::SubmitNetworkRequestAndWait();

        NN_PRAGMA_PUSH_WARNINGS
        NN_DISABLE_WARNING_DEPRECATED_DECLARATIONS // EnsureNetworkClockAvailability deprecated warning 回避
        auto result = nn::time::EnsureNetworkClockAvailability();
        NN_PRAGMA_POP_WARNINGS

        g_IsNetworkClockAvailable = result.IsSuccess();
        if (result.IsFailure())
        {
            NN_LOG("Network clock is not available (0x%08lX). All tests will be skipped...\n",
                result.GetInnerValueForDebug());
        }

        InitializeAllocatorForServer();
        InitializeMain();
        InitializeWatcher();

        // BGスレッド処理のため少し待機する
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));

        // イベント処理を止める
        g_pWatcher->GetWatcherEventManager().DisableEventProcess();

        // 一旦破棄する
        auto& storage = g_pWatcher->GetWatcherEventStorage();
        storage.DiscardAllEvents();
        // イベント保存処理を実行させる
        g_pWatcher->GetWatcherEventManager().ProcessSerializeEventsIfNeededForTest();

        // ダミーで連携済み状態にする
        g_pWatcher->GetNetworkManager().StorePairingInfo(0);
    }

    static void TearDownTestCase()
    {
        g_pWatcher->GetNetworkManager().ClearPairingInfo(false);

        FinalizeWatcher();
        FinalizeMain();

        nn::time::Finalize();
    }

    static bool g_IsNetworkClockAvailable;

protected:
    virtual void SetUp() NN_OVERRIDE
    {
        // テストケース開始時に毎回初期化を行う
        g_pWatcher->GetWatcherEventStorage().ReinitializeForTest();
    }

    virtual void TearDown() NN_OVERRIDE
    {
        // テストケース終了時に毎回イベント保存処理を実行させる
        g_pWatcher->GetWatcherEventManager().ProcessSerializeEventsIfNeededForTest();
    }
};
bool WatcherEventTest::g_IsNetworkClockAvailable = false;
watcher::EventData g_EventDataBuffer[watcher::MaxEventCount];

TEST_F(WatcherEventTest, AddOneEvent)
{
    if (!WatcherEventTest::g_IsNetworkClockAvailable)
    {
        return;
    }

    auto& storage = g_pWatcher->GetWatcherEventStorage();
    storage.AddIdleEvent();
    EXPECT_EQ(1, storage.GetEventCountForTest());

    auto count = storage.GetPartialDataForPost(g_EventDataBuffer, watcher::MaxEventCount);
    EXPECT_EQ(1, static_cast<int>(count));

    auto& e = g_EventDataBuffer[0];
    EXPECT_EQ(watcher::EventType::Idle, e.eventType);

    storage.FinishPostPartialData(count);
    // Post済みの場合その分がリセットされる
    EXPECT_EQ(0, storage.GetEventCountForTest());
}

// AddOneEvent のデータが残っていないかどうかのテスト
TEST_F(WatcherEventTest, RemainNoEvents)
{
    if (!WatcherEventTest::g_IsNetworkClockAvailable)
    {
        return;
    }

    auto& storage = g_pWatcher->GetWatcherEventStorage();
    EXPECT_EQ(0, storage.GetEventCountForTest());

    auto count = storage.GetPartialDataForPost(g_EventDataBuffer, watcher::MaxEventCount);
    EXPECT_EQ(0, static_cast<int>(count));

    storage.FinishPostPartialData(count);
}

// 次のテストのための仕込みを行う
TEST_F(WatcherEventTest, PrepareForNextTest)
{
    if (!WatcherEventTest::g_IsNetworkClockAvailable)
    {
        return;
    }

    auto& storage = g_pWatcher->GetWatcherEventStorage();
    storage.AddIdleEvent();
    EXPECT_EQ(1, storage.GetEventCountForTest());
}

// Discard したら AddOneEvent 時点のデータのみが残っているかどうかのテスト
TEST_F(WatcherEventTest, DiscardEvents)
{
    if (!WatcherEventTest::g_IsNetworkClockAvailable)
    {
        return;
    }

    auto& storage = g_pWatcher->GetWatcherEventStorage();
    EXPECT_EQ(1, storage.GetEventCountForTest());

    storage.AddUnlockEvent();
    storage.AddLockEvent();
    EXPECT_EQ(3, storage.GetEventCountForTest());

    storage.DiscardEventsSinceInitialize();
    EXPECT_EQ(1, storage.GetEventCountForTest());

    auto count = storage.GetPartialDataForPost(g_EventDataBuffer, watcher::MaxEventCount);
    EXPECT_EQ(1, static_cast<int>(count));

    auto& e = g_EventDataBuffer[0];
    EXPECT_EQ(watcher::EventType::Idle, e.eventType);

    storage.FinishPostPartialData(count);
}

// 3000 件以上のデータの追加テスト
TEST_F(WatcherEventTest, WriteOver3000Events)
{
    if (!WatcherEventTest::g_IsNetworkClockAvailable)
    {
        return;
    }

    auto& storage = g_pWatcher->GetWatcherEventStorage();
    static const int OverCount = 5;

    // 毎回Commitするとかなり時間がかかるので一旦無効化する
    // (それでも時間がかかる)
    storage.SetCommitSkipped(true);
    // 目印のため、アプリ起動イベントを用いて
    // applicationId の番号をインクリメントしていく
    nn::ncm::ApplicationId appId;
    appId.value = 0x0100000000010000;
    for (int i = 0; i < watcher::MaxEventCount; ++i)
    {
        storage.AddApplicationPlayedEvent(appId);
        ++appId.value;
    }
    for (int i = 0; i < OverCount; ++i)
    {
        storage.AddApplicationPlayedEvent(appId);
        ++appId.value;
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(common::FileSystem::Commit());
    storage.SetCommitSkipped(false);
    EXPECT_EQ(watcher::MaxEventCount, storage.GetEventCountForTest());

    auto count = storage.GetPartialDataForPost(g_EventDataBuffer, watcher::MaxEventCount);
    EXPECT_EQ(watcher::MaxEventCount, static_cast<int>(count));
    // 最初の OverCount 件分が削られていることの確認
    appId.value = 0x0100000000010000 + static_cast<decltype(appId.value)>(OverCount);
    for (int i = 0; i < count; ++i)
    {
        const auto& data = g_EventDataBuffer[i];
        EXPECT_EQ(watcher::EventType::DidApplicationPlay, data.eventType);
        EXPECT_EQ(appId, data.payload.playedApplication.applicationId);
        ++appId.value;
    }

    storage.FinishPostPartialData(count);
}

// データの部分送信テスト
TEST_F(WatcherEventTest, PartialPostEvents)
{
    if (!WatcherEventTest::g_IsNetworkClockAvailable)
    {
        return;
    }

    auto& storage = g_pWatcher->GetWatcherEventStorage();
    static const int PartialCount = 5;
    static const int TotalCount = 22;
    static const int Divisions = (TotalCount + PartialCount - 1) / PartialCount;

    // 毎回Commitするとかなり時間がかかるので一旦無効化する
    // (それでも時間がかかる)
    storage.SetCommitSkipped(true);
    for (int i = 0; i < TotalCount; ++i)
    {
        storage.AddIdleEvent();
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(common::FileSystem::Commit());
    storage.SetCommitSkipped(false);
    EXPECT_EQ(TotalCount, storage.GetEventCountForTest());

    int remainingCount = TotalCount;
    for (int d = 0; d < Divisions; ++d)
    {
        auto count = storage.GetPartialDataForPost(g_EventDataBuffer, PartialCount);
        int expectCount = (remainingCount >= PartialCount ? PartialCount : remainingCount);
        EXPECT_EQ(expectCount, static_cast<int>(count));

        storage.FinishPostPartialData(count);
        remainingCount -= count;
        EXPECT_EQ(remainingCount, storage.GetEventCountForTest());
    }
    EXPECT_EQ(0, static_cast<int>(remainingCount));

    auto count = storage.GetPartialDataForPost(g_EventDataBuffer, PartialCount);
    EXPECT_EQ(0, static_cast<int>(count));
}

// 送信中に別のイベントが追加されるテスト1(通常)
TEST_F(WatcherEventTest, EventAddedDuringPosting1)
{
    if (!WatcherEventTest::g_IsNetworkClockAvailable)
    {
        return;
    }

    auto& storage = g_pWatcher->GetWatcherEventStorage();
    static const int EventCount = 15;

    // 毎回Commitするとかなり時間がかかるので一旦無効化する
    // (それでも時間がかかる)
    storage.SetCommitSkipped(true);
    for (int i = 0; i < EventCount; ++i)
    {
        storage.AddIdleEvent();
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(common::FileSystem::Commit());
    storage.SetCommitSkipped(false);
    EXPECT_EQ(EventCount, storage.GetEventCountForTest());

    auto count = storage.GetPartialDataForPost(g_EventDataBuffer, watcher::MaxEventCount);
    EXPECT_EQ(EventCount, static_cast<int>(count));

    // Finishの前にイベント追加
    storage.AddUnlockEvent();
    storage.AddLockEvent();

    storage.FinishPostPartialData(count);

    EXPECT_EQ(2, storage.GetEventCountForTest());
    count = storage.GetPartialDataForPost(g_EventDataBuffer, watcher::MaxEventCount);
    EXPECT_EQ(2, static_cast<int>(count));

    storage.FinishPostPartialData(count);
}

// 送信中に別のイベントが追加されるテスト2(部分上書きあり)
TEST_F(WatcherEventTest, EventAddedDuringPosting2)
{
    if (!WatcherEventTest::g_IsNetworkClockAvailable)
    {
        return;
    }

    auto& storage = g_pWatcher->GetWatcherEventStorage();
    static const int EventCount = 15;
    static const int TooManyEventCount = watcher::MaxEventCount - 9; // 全てが上書きされるわけではない数

    // 毎回Commitするとかなり時間がかかるので一旦無効化する
    // (それでも時間がかかる)
    storage.SetCommitSkipped(true);
    for (int i = 0; i < EventCount; ++i)
    {
        storage.AddIdleEvent();
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(common::FileSystem::Commit());
    storage.SetCommitSkipped(false);
    EXPECT_EQ(EventCount, storage.GetEventCountForTest());

    auto count = storage.GetPartialDataForPost(g_EventDataBuffer, watcher::MaxEventCount);
    EXPECT_EQ(EventCount, static_cast<int>(count));

    // Finishの前に大量のイベント追加
    storage.SetCommitSkipped(true);
    // (最初と最後に目印として違うイベントを入れる)
    storage.AddOfflineEvent();
    for (int i = 1; i < TooManyEventCount - 1; ++i)
    {
        storage.AddIdleEvent();
    }
    storage.AddOnlineEvent();
    NN_ABORT_UNLESS_RESULT_SUCCESS(common::FileSystem::Commit());
    storage.SetCommitSkipped(false);
    // この時点では送信中のイベントが残っており
    // EventCount + TooManyEventCount > watcher::MaxEventCount なので
    // 記録済みの件数は watcher::MaxEventCount になるはず
    NN_STATIC_ASSERT(EventCount + TooManyEventCount > watcher::MaxEventCount);
    EXPECT_EQ(watcher::MaxEventCount, storage.GetEventCountForTest());

    storage.FinishPostPartialData(count);

    // 送信済みの処理は上書きされなかった分のみ行われるはず
    EXPECT_EQ(TooManyEventCount, storage.GetEventCountForTest());
    count = storage.GetPartialDataForPost(g_EventDataBuffer, watcher::MaxEventCount);
    EXPECT_EQ(TooManyEventCount, static_cast<int>(count));
    // 目印として付けたイベントが正しく取れているかどうか
    EXPECT_EQ(watcher::EventType::DidGoOffline, g_EventDataBuffer[0].eventType);
    EXPECT_EQ(watcher::EventType::DidComeOnline, g_EventDataBuffer[count - 1].eventType);

    storage.FinishPostPartialData(count);
}

// 送信中に別のイベントが追加されるテスト3(完全上書きあり)
TEST_F(WatcherEventTest, EventAddedDuringPosting3)
{
    if (!WatcherEventTest::g_IsNetworkClockAvailable)
    {
        return;
    }

    auto& storage = g_pWatcher->GetWatcherEventStorage();
    static const int EventCount = 15;

    // 毎回Commitするとかなり時間がかかるので一旦無効化する
    // (それでも時間がかかる)
    storage.SetCommitSkipped(true);
    for (int i = 0; i < EventCount; ++i)
    {
        storage.AddIdleEvent();
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(common::FileSystem::Commit());
    storage.SetCommitSkipped(false);
    EXPECT_EQ(EventCount, storage.GetEventCountForTest());

    auto count = storage.GetPartialDataForPost(g_EventDataBuffer, watcher::MaxEventCount);
    EXPECT_EQ(EventCount, static_cast<int>(count));

    // Finishの前に watcher::MaxEventCount 分のイベント追加
    storage.SetCommitSkipped(true);
    // (最初と最後に目印として違うイベントを入れる)
    storage.AddOfflineEvent();
    for (int i = 1; i < watcher::MaxEventCount - 1; ++i)
    {
        storage.AddIdleEvent();
    }
    storage.AddOnlineEvent();
    NN_ABORT_UNLESS_RESULT_SUCCESS(common::FileSystem::Commit());
    storage.SetCommitSkipped(false);
    // この時点では送信中のイベントが残っており
    // EventCount + watcher::MaxEventCount > watcher::MaxEventCount なので
    // 記録済みの件数は watcher::MaxEventCount になるはず
    NN_STATIC_ASSERT(EventCount + watcher::MaxEventCount > watcher::MaxEventCount);
    EXPECT_EQ(watcher::MaxEventCount, storage.GetEventCountForTest());

    storage.FinishPostPartialData(count);

    // 送信済みの処理は上書きされなかった分のみ行われるはず
    EXPECT_EQ(watcher::MaxEventCount, storage.GetEventCountForTest());
    count = storage.GetPartialDataForPost(g_EventDataBuffer, watcher::MaxEventCount);
    EXPECT_EQ(watcher::MaxEventCount, static_cast<int>(count));
    // 目印として付けたイベントが正しく取れているかどうか
    EXPECT_EQ(watcher::EventType::DidGoOffline, g_EventDataBuffer[0].eventType);
    EXPECT_EQ(watcher::EventType::DidComeOnline, g_EventDataBuffer[count - 1].eventType);

    storage.FinishPostPartialData(count);
}

// 「起動してからそれまでに記録したイベントの総数」がオーバーフローしたときのテスト1(通常Postあり)
TEST_F(WatcherEventTest, OverflowRecordedCount1)
{
    if (!WatcherEventTest::g_IsNetworkClockAvailable)
    {
        return;
    }

    auto& storage = g_pWatcher->GetWatcherEventStorage();
    static const int EventCount = 20;
    static const int CountToOverflow = 15;
    // 意図的にオーバーフローさせるために大きな数を指定
    storage.SetTotalEventCountFromBootForTest((static_cast<uint32_t>(0) - static_cast<uint32_t>(CountToOverflow)));
    storage.SetCommitSkipped(true);
    for (int i = 0; i < EventCount; ++i)
    {
        storage.AddIdleEvent();
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(common::FileSystem::Commit());
    storage.SetCommitSkipped(false);

    auto count = storage.GetPartialDataForPost(g_EventDataBuffer, watcher::MaxEventCount);
    EXPECT_EQ(EventCount, static_cast<int>(count));

    storage.FinishPostPartialData(count);
    EXPECT_EQ(static_cast<uint32_t>(EventCount - CountToOverflow), storage.GetTotalEventCountFromBootForTest());
}

// 「起動してからそれまでに記録したイベントの総数」がオーバーフローしたときのテスト2(EventAddedDuringPosting2 と同じ上書きあり)
TEST_F(WatcherEventTest, OverflowRecordedCount2)
{
    if (!WatcherEventTest::g_IsNetworkClockAvailable)
    {
        return;
    }

    auto& storage = g_pWatcher->GetWatcherEventStorage();
    static const int CountToOverflow = 15;
    // 意図的にオーバーフローさせるために大きな数を指定
    storage.SetTotalEventCountFromBootForTest((static_cast<uint32_t>(0) - static_cast<uint32_t>(CountToOverflow)));

    static const int EventCount = 15;
    static const int TooManyEventCount = watcher::MaxEventCount - 7; // 全てが上書きされるわけではない数

    // 毎回Commitするとかなり時間がかかるので一旦無効化する
    // (それでも時間がかかる)
    storage.SetCommitSkipped(true);
    for (int i = 0; i < EventCount; ++i)
    {
        storage.AddIdleEvent();
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(common::FileSystem::Commit());
    storage.SetCommitSkipped(false);
    EXPECT_EQ(EventCount, storage.GetEventCountForTest());

    auto count = storage.GetPartialDataForPost(g_EventDataBuffer, watcher::MaxEventCount);
    EXPECT_EQ(EventCount, static_cast<int>(count));

    // Finishの前に大量のイベント追加
    storage.SetCommitSkipped(true);
    // (最初と最後に目印として違うイベントを入れる)
    storage.AddOfflineEvent();
    for (int i = 1; i < TooManyEventCount - 1; ++i)
    {
        storage.AddIdleEvent();
    }
    storage.AddOnlineEvent();
    NN_ABORT_UNLESS_RESULT_SUCCESS(common::FileSystem::Commit());
    storage.SetCommitSkipped(false);
    // この時点では送信中のイベントが残っており
    // EventCount + TooManyEventCount > watcher::MaxEventCount なので
    // 記録済みの件数は watcher::MaxEventCount になるはず
    NN_STATIC_ASSERT(EventCount + TooManyEventCount > watcher::MaxEventCount);
    EXPECT_EQ(watcher::MaxEventCount, storage.GetEventCountForTest());

    storage.FinishPostPartialData(count);

    // 送信済みの処理は上書きされなかった分のみ行われるはず
    EXPECT_EQ(TooManyEventCount, storage.GetEventCountForTest());
    count = storage.GetPartialDataForPost(g_EventDataBuffer, watcher::MaxEventCount);
    EXPECT_EQ(TooManyEventCount, static_cast<int>(count));
    // 目印として付けたイベントが正しく取れているかどうか
    EXPECT_EQ(watcher::EventType::DidGoOffline, g_EventDataBuffer[0].eventType);
    EXPECT_EQ(watcher::EventType::DidComeOnline, g_EventDataBuffer[count - 1].eventType);

    storage.FinishPostPartialData(count);

    // カウント自体は単純な合計数になるはず
    EXPECT_EQ(static_cast<uint32_t>(EventCount + TooManyEventCount - CountToOverflow), storage.GetTotalEventCountFromBootForTest());
}

TEST_F(WatcherEventTest, InsertEvent1)
{
    if (!WatcherEventTest::g_IsNetworkClockAvailable)
    {
        return;
    }

    auto& storage = g_pWatcher->GetWatcherEventStorage();
    storage.AddIdleEvent();
    EXPECT_EQ(1, storage.GetEventCountForTest());

    auto count = storage.GetPartialDataForPost(g_EventDataBuffer, watcher::MaxEventCount);
    EXPECT_EQ(1, static_cast<int>(count));

    nn::time::PosixTime t;
    {
        auto& e = g_EventDataBuffer[0];
        EXPECT_EQ(watcher::EventType::Idle, e.eventType);
        t = e.timestamp; // 記録されたイベントの時刻を取得
    }

    storage.FinishPostPartialData(0);
    // Postしていないのでリセットされない
    EXPECT_EQ(1, storage.GetEventCountForTest());

    // 挿入処理
    t.value -= 100; // 以前のイベントより手前の時刻を得る
    nn::ncm::ApplicationId appId;
    appId.value = 0x0100000000010000;
    storage.InsertApplicationLaunchEvent(appId, t);
    t.value += 200; // 以前のイベントより後の時刻を得る
    appId.value = 0x0100000000010000;
    storage.InsertApplicationTerminateEvent(appId, t);
    EXPECT_EQ(3, storage.GetEventCountForTest());

    count = storage.GetPartialDataForPost(g_EventDataBuffer, watcher::MaxEventCount);
    EXPECT_EQ(3, static_cast<int>(count));

    {
        EXPECT_EQ(watcher::EventType::DidApplicationLaunch, g_EventDataBuffer[0].eventType);
        EXPECT_EQ(watcher::EventType::Idle, g_EventDataBuffer[1].eventType);
        EXPECT_EQ(watcher::EventType::DidApplicationTerminate, g_EventDataBuffer[2].eventType);
    }

    storage.FinishPostPartialData(count);
    EXPECT_EQ(0, storage.GetEventCountForTest());
}

TEST_F(WatcherEventTest, InsertEvent_WithOverflow)
{
    if (!WatcherEventTest::g_IsNetworkClockAvailable)
    {
        return;
    }

    auto& storage = g_pWatcher->GetWatcherEventStorage();
    static const int OverCount = 5;
    static const nn::ncm::ApplicationId AppIdForDummyStart = { 0x0100000000010000 };
    static const nn::ncm::ApplicationId AppIdForCheck = { 0x0100000000020000 };

    // 毎回Commitするとかなり時間がかかるので一旦無効化する
    // (それでも時間がかかる)
    storage.SetCommitSkipped(true);
    // 目印のため、アプリ起動イベントを用いて
    // applicationId の番号をインクリメントしていく
    nn::ncm::ApplicationId appId;
    appId.value = AppIdForDummyStart.value;
    for (int i = 0; i < watcher::MaxEventCount; ++i)
    {
        // applicationId はのちにチェックできるように単調増加した値を入れる
        storage.AddApplicationPlayedEvent(appId);
        ++appId.value;
    }
    for (int i = 0; i < OverCount; ++i)
    {
        storage.AddApplicationPlayedEvent(appId);
        ++appId.value;
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(common::FileSystem::Commit());
    storage.SetCommitSkipped(false);
    EXPECT_EQ(watcher::MaxEventCount, storage.GetEventCountForTest());

    // 最初のデータのみ取得
    auto count = storage.GetPartialDataForPost(g_EventDataBuffer, 1);
    EXPECT_EQ(1, static_cast<int>(count));

    nn::time::PosixTime t;
    {
        auto& e = g_EventDataBuffer[0];
        EXPECT_EQ(watcher::EventType::DidApplicationPlay, e.eventType);
        EXPECT_EQ(AppIdForDummyStart.value + OverCount, e.payload.playedApplication.applicationId.value);
        t = e.timestamp; // 記録されたイベントの時刻を取得
    }

    storage.FinishPostPartialData(0);
    // Postしていないのでリセットされない
    EXPECT_EQ(watcher::MaxEventCount, storage.GetEventCountForTest());

    // 挿入処理
    t.value -= 100; // 以前のイベントより手前の時刻を得る
    appId = AppIdForCheck;
    storage.InsertApplicationTerminateEvent(appId, t);

    // 上記で挿入したイベントは古すぎて追加されない
    count = storage.GetPartialDataForPost(g_EventDataBuffer, 1);
    EXPECT_EQ(1, static_cast<int>(count));
    {
        auto& e = g_EventDataBuffer[0];
        EXPECT_EQ(watcher::EventType::DidApplicationPlay, e.eventType);
        EXPECT_EQ(AppIdForDummyStart.value + OverCount, e.payload.playedApplication.applicationId.value);
    }
    storage.FinishPostPartialData(0);
    EXPECT_EQ(watcher::MaxEventCount, storage.GetEventCountForTest());

    t.value += 100; // 以前のイベントと同じ時刻を得る
    appId = AppIdForCheck;
    storage.InsertApplicationLaunchEvent(appId, t);
    EXPECT_EQ(watcher::MaxEventCount, storage.GetEventCountForTest());

    count = storage.GetPartialDataForPost(g_EventDataBuffer, watcher::MaxEventCount);
    EXPECT_EQ(watcher::MaxEventCount, static_cast<int>(count));

    // 挿入によって1つ減っているはずなのでその分を加える
    appId.value = AppIdForDummyStart.value + OverCount + 1;
    for (auto& e : g_EventDataBuffer)
    {
        // DidApplicationLaunch が追加され、それ以外は DidApplicationPlay になるはず
        if (e.eventType == watcher::EventType::DidApplicationLaunch)
        {
            EXPECT_EQ(AppIdForCheck.value, e.payload.applicationLaunch.applicationId.value);
        }
        else
        {
            // DidApplicationResumed 以外のデータは applicationId が単調増加しているはず
            EXPECT_EQ(watcher::EventType::DidApplicationPlay, e.eventType);
            EXPECT_EQ(appId.value, e.payload.playedApplication.applicationId.value);
            ++appId.value;
        }
    }

    storage.FinishPostPartialData(count);
    EXPECT_EQ(0, storage.GetEventCountForTest());
}

TEST_F(WatcherEventTest, InsertEvent_WithOverflow2)
{
    if (!WatcherEventTest::g_IsNetworkClockAvailable)
    {
        return;
    }

    auto& storage = g_pWatcher->GetWatcherEventStorage();
    static const int OverCount = 5;
    static const nn::ncm::ApplicationId AppIdForDummyStart = { 0x0100000000010000 };
    static const nn::ncm::ApplicationId AppIdForCheck = { 0x0100000000020000 };
    nn::time::PosixTime timeBase;

    EXPECT_TRUE(common::GetNetworkTime(&timeBase));

    // 毎回Commitするとかなり時間がかかるので一旦無効化する
    // (それでも時間がかかる)
    storage.SetCommitSkipped(true);
    // 目印のため、アプリ起動イベントを用いて
    // applicationId の番号をインクリメントしていく
    nn::ncm::ApplicationId appId;
    appId.value = AppIdForDummyStart.value;
    for (int i = 0; i < watcher::MaxEventCount; ++i)
    {
        // applicationId はのちにチェックできるように単調増加した値を入れる
        watcher::EventData::Payload payload;
        payload.applicationLaunch.applicationId.value = appId.value;
        storage.InsertEvent(watcher::EventType::DidApplicationLaunch, &payload, timeBase);
        ++appId.value;
        ++timeBase.value;
    }
    for (int i = 0; i < OverCount; ++i)
    {
        watcher::EventData::Payload payload;
        payload.applicationLaunch.applicationId.value = appId.value;
        storage.InsertEvent(watcher::EventType::DidApplicationLaunch, &payload, timeBase);
        ++appId.value;
        ++timeBase.value;
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(common::FileSystem::Commit());
    storage.SetCommitSkipped(false);
    EXPECT_EQ(watcher::MaxEventCount, storage.GetEventCountForTest());

    // 最初のデータのみ取得
    auto count = storage.GetPartialDataForPost(g_EventDataBuffer, 1);
    EXPECT_EQ(1, static_cast<int>(count));

    nn::time::PosixTime t;
    {
        auto& e = g_EventDataBuffer[0];
        EXPECT_EQ(watcher::EventType::DidApplicationLaunch, e.eventType);
        EXPECT_EQ(AppIdForDummyStart.value + OverCount, e.payload.applicationLaunch.applicationId.value);
        t = e.timestamp; // 記録されたイベントの時刻を取得
    }

    storage.FinishPostPartialData(0);
    // Postしていないのでリセットされない
    EXPECT_EQ(watcher::MaxEventCount, storage.GetEventCountForTest());

    // 挿入処理
    t.value -= 100; // 以前のイベントより手前の時刻を得る
    appId = AppIdForCheck;
    storage.InsertApplicationTerminateEvent(appId, t);

    // 上記で挿入したイベントは古すぎて追加されない
    count = storage.GetPartialDataForPost(g_EventDataBuffer, 1);
    EXPECT_EQ(1, static_cast<int>(count));
    {
        auto& e = g_EventDataBuffer[0];
        EXPECT_EQ(watcher::EventType::DidApplicationLaunch, e.eventType);
        EXPECT_EQ(AppIdForDummyStart.value + OverCount, e.payload.applicationLaunch.applicationId.value);
    }
    storage.FinishPostPartialData(0);
    EXPECT_EQ(watcher::MaxEventCount, storage.GetEventCountForTest());

    t.value += 100; // 以前のイベントと同じ時刻を得る
    appId = AppIdForCheck;
    storage.InsertApplicationTerminateEvent(appId, t);
    EXPECT_EQ(watcher::MaxEventCount, storage.GetEventCountForTest());

    // 上記で挿入したイベントはもっとも古い位置に残るはず
    count = storage.GetPartialDataForPost(g_EventDataBuffer, 1);
    EXPECT_EQ(1, static_cast<int>(count));
    {
        auto& e = g_EventDataBuffer[0];
        EXPECT_EQ(watcher::EventType::DidApplicationTerminate, e.eventType);
        EXPECT_EQ(AppIdForCheck.value, e.payload.applicationTerminate.applicationId.value);
    }
    storage.FinishPostPartialData(0);
    EXPECT_EQ(watcher::MaxEventCount, storage.GetEventCountForTest());

    // 最も新しい位置に追加(直前に追加したイベントも押し出される)
    t.value = timeBase.value; // 以前のイベントよりも後の時刻を得る
    appId = AppIdForCheck;
    storage.InsertApplicationTerminateEvent(appId, t);
    EXPECT_EQ(watcher::MaxEventCount, storage.GetEventCountForTest());

    count = storage.GetPartialDataForPost(g_EventDataBuffer, watcher::MaxEventCount);
    EXPECT_EQ(watcher::MaxEventCount, static_cast<int>(count));

    // 挿入によって1つ減っているはずなのでその分を加える
    appId.value = AppIdForDummyStart.value + OverCount + 1;
    for (auto& e : g_EventDataBuffer)
    {
        // DidApplicationTerminate が追加され、それ以外は DidApplicationLaunch になるはず
        if (e.eventType == watcher::EventType::DidApplicationTerminate)
        {
            EXPECT_EQ(AppIdForCheck.value, e.payload.applicationTerminate.applicationId.value);
        }
        else
        {
            // DidApplicationResumed 以外のデータは applicationId が単調増加しているはず
            EXPECT_EQ(watcher::EventType::DidApplicationLaunch, e.eventType);
            EXPECT_EQ(appId.value, e.payload.applicationLaunch.applicationId.value);
            ++appId.value;
        }
    }

    // 最後に挿入したイベントはもっとも新しい位置に残るはず
    {
        auto& e = g_EventDataBuffer[watcher::MaxEventCount - 1];
        EXPECT_EQ(watcher::EventType::DidApplicationTerminate, e.eventType);
        EXPECT_EQ(AppIdForCheck.value, e.payload.applicationTerminate.applicationId.value);
    }

    storage.FinishPostPartialData(count);
    EXPECT_EQ(0, storage.GetEventCountForTest());
}
