﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>
#include <nn/nn_Log.h>

#include <nn/pctl/pctl_Api.h>
#include <nn/pctl/pctl_ApiSystem.h>
#include <nn/pctl/pctl_ApiForAuthentication.h>
#include <nn/pctl/pctl_ApiPairing.h>
#include <nn/pctl/pctl_ApiWatcher.h>
#include <nn/pctl/pctl_ApiForDebug.h>

#if defined(NN_BUILD_CONFIG_OS_WIN)
#include <nn/nsd/nsd_ApiForTest.h>
#else
#include <nn/nsd/nsd_ApiForMenu.h>
#endif

#include <nn/nifm/nifm_Api.h>
#include <nn/nifm/nifm_NetworkConnection.h>
#include <nn/fs.h>
#include <nn/fs/fs_Host.h>

#include "../Common/testPctl_TestUtil.h"


class PairingTest : public ::testing::Test
{
public:
    static nn::pctl::test::PairingInfo PairingCodeInfo;
    static bool PairingEnabled;
    static bool UnlinkPairingEnabled;

protected:
    static void SetUpTestCase()
    {
        nn::nifm::Initialize();
        pNetworkConnection = new nn::nifm::NetworkConnection();
        pNetworkConnection->SubmitRequestAndWait();

        ASSERT_TRUE(pNetworkConnection->IsAvailable());
    }
    static void TearDownTestCase()
    {
        delete pNetworkConnection;
        pNetworkConnection = nullptr;
    }
    static nn::nifm::NetworkConnection* pNetworkConnection;

};

nn::pctl::test::PairingInfo PairingTest::PairingCodeInfo = {};
bool PairingTest::PairingEnabled = true;
bool PairingTest::UnlinkPairingEnabled = true;

nn::nifm::NetworkConnection* PairingTest::pNetworkConnection = nullptr;


TEST_F(PairingTest, Paring)
{
    if (PairingEnabled)
    {
#if defined(NN_BUILD_CONFIG_OS_WIN)
        nn::nsd::SetTd1EnvironmentForTest();
#else
        nn::nsd::EnvironmentIdentifier env;
        nn::nsd::GetEnvironmentIdentifier(&env);
        ASSERT_STREQ("td1", env.value);
#endif

        if (PairingCodeInfo.pairingCode[0] == '\0')
        {
            NNT_ASSERT_RESULT_SUCCESS(nn::pctl::test::GetPairingInfo(&PairingCodeInfo, "ido_junichi+pctl01@exmx.nintendo.co.jp", "Password"));
        }

        nn::pctl::SetPinCode(nullptr);
        nn::pctl::ClearFreeCommunicationApplicationListForDebug();
        nn::pctl::DeletePairing();

        nn::pctl::PairingInfo info;
        NNT_ASSERT_RESULT_SUCCESS(nn::pctl::RequestPairing(&info, PairingCodeInfo.pairingCode));

        nn::pctl::PairingAccountInfo accountInfo;
        char nickname[64];
        size_t size = 0;

        EXPECT_TRUE(info.IsInstanceValid());
        EXPECT_EQ(nn::pctl::PairingState::PairingState_Processing, info.GetState());
        EXPECT_TRUE(info.GetAccountInfo(&accountInfo));
        EXPECT_TRUE(accountInfo.IsInstanceValid());
        EXPECT_GT(accountInfo.GetNickname(nickname, 64), static_cast<size_t>(0));
        NNT_EXPECT_RESULT_SUCCESS(accountInfo.GetMiiImageContentType(&size, nullptr, 0));
        NNT_EXPECT_RESULT_SUCCESS(accountInfo.GetMiiImage(&size, nullptr, 0));

        NNT_ASSERT_RESULT_SUCCESS(nn::pctl::AuthorizePairing(&info));

        EXPECT_TRUE(info.IsInstanceValid());
        EXPECT_EQ(nn::pctl::PairingState::PairingState_Active, info.GetState());
    }
}

TEST_F(PairingTest, Unpairing)
{
    if (UnlinkPairingEnabled)
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::pctl::UnlinkPairing(false));
    }
}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    if (argc > 1 && (std::strcmp(argv[1], "--help") == 0 || std::strcmp(argv[1], "-h") == 0))
    {
        NN_LOG("Usage:\n" " testPctl_Pairing [pairing code] [--without_pairing] [--without_unlinkpairing]\n");
        return;
    }

    ::testing::InitGoogleTest(&argc, argv);

    for (int i = 1; i < argc; ++i)
    {
        if (std::strcmp(argv[i], "--without_pairing") == 0) // Pairing を行わない
        {
            PairingTest::PairingEnabled = false;
        }
        else if (std::strcmp(argv[i], "--without_unlinkpairing") == 0) // UnlinkPairing を行わない
        {
            PairingTest::UnlinkPairingEnabled = false;
        }
        else
        {
            nn::util::Strlcpy(PairingTest::PairingCodeInfo.pairingCode, argv[i], sizeof(PairingTest::PairingCodeInfo.pairingCode));
        }
    }

    auto ret = RUN_ALL_TESTS();

    nnt::Exit(ret);
}
