﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/pctl/detail/service/json/pctl_JsonStructuredWriter.h>
#include <nn/util/util_FormatString.h>

using namespace nn::pctl::detail::service;

static char JsonDataBuffer[256];

#define EMPTY_SPECIFIER

TEST(JsonOutput, WriteUnsignedInt)
{
    bool (* functionOutputValue)(nn::util::optional<uint64_t>*, void*, int) = [](nn::util::optional<uint64_t>* outValue, void*, int) -> bool
    {
        *outValue = 123;
        return true;
    };

    NN_DETAIL_PCTL_JSON_BEGIN_WRITE(EMPTY_SPECIFIER, JsonDataWrite)
        NN_DETAIL_PCTL_JSON_WRITE_VALUE_UINT64(functionOutputValue)
    NN_DETAIL_PCTL_JSON_END_WRITE()

    std::memset(JsonDataBuffer, 0, sizeof(JsonDataBuffer));

    json::JsonStructuredWriter writer(nullptr, JsonDataWrite);
    size_t len;
    auto result = writer.FillData(&len, JsonDataBuffer, sizeof(JsonDataBuffer));
    EXPECT_EQ(json::JsonStructuredWriter::WriteResult::Succeeded, result);
    EXPECT_EQ(3, len);
    EXPECT_EQ(0, std::strncmp(JsonDataBuffer, "123", 3));
}

TEST(JsonOutput, WriteUnsignedInt_Null)
{
    bool (* functionOutputValue)(nn::util::optional<uint64_t>*, void*, int) = [](nn::util::optional<uint64_t>* outValue, void*, int) -> bool
    {
        *outValue = nn::util::nullopt;
        return true;
    };

    NN_DETAIL_PCTL_JSON_BEGIN_WRITE(EMPTY_SPECIFIER, JsonDataWrite)
        NN_DETAIL_PCTL_JSON_WRITE_VALUE_UINT64(functionOutputValue)
    NN_DETAIL_PCTL_JSON_END_WRITE()

    std::memset(JsonDataBuffer, 0, sizeof(JsonDataBuffer));

    json::JsonStructuredWriter writer(nullptr, JsonDataWrite);
    size_t len;
    auto result = writer.FillData(&len, JsonDataBuffer, sizeof(JsonDataBuffer));
    EXPECT_EQ(json::JsonStructuredWriter::WriteResult::Succeeded, result);
    EXPECT_EQ(4, len);
    EXPECT_EQ(0, std::strncmp(JsonDataBuffer, "null", 3));
}

TEST(JsonOutput, WriteSignedInt)
{
    bool (* functionOutputValue)(nn::util::optional<int64_t>*, void*, int) = [](nn::util::optional<int64_t>* outValue, void*, int) -> bool
    {
        *outValue = -456;
        return true;
    };

    NN_DETAIL_PCTL_JSON_BEGIN_WRITE(EMPTY_SPECIFIER, JsonDataWrite)
        NN_DETAIL_PCTL_JSON_WRITE_VALUE_INT64(functionOutputValue)
    NN_DETAIL_PCTL_JSON_END_WRITE()

    std::memset(JsonDataBuffer, 0, sizeof(JsonDataBuffer));

    json::JsonStructuredWriter writer(nullptr, JsonDataWrite);
    size_t len;
    auto result = writer.FillData(&len, JsonDataBuffer, sizeof(JsonDataBuffer));
    EXPECT_EQ(json::JsonStructuredWriter::WriteResult::Succeeded, result);
    EXPECT_EQ(4, len);
    EXPECT_EQ(0, std::strncmp(JsonDataBuffer, "-456", 4));
}

TEST(JsonOutput, WriteSignedInt_Null)
{
    bool (* functionOutputValue)(nn::util::optional<int64_t>*, void*, int) = [](nn::util::optional<int64_t>* outValue, void*, int) -> bool
    {
        *outValue = nn::util::nullopt;
        return true;
    };

    NN_DETAIL_PCTL_JSON_BEGIN_WRITE(EMPTY_SPECIFIER, JsonDataWrite)
        NN_DETAIL_PCTL_JSON_WRITE_VALUE_INT64(functionOutputValue)
    NN_DETAIL_PCTL_JSON_END_WRITE()

    std::memset(JsonDataBuffer, 0, sizeof(JsonDataBuffer));

    json::JsonStructuredWriter writer(nullptr, JsonDataWrite);
    size_t len;
    auto result = writer.FillData(&len, JsonDataBuffer, sizeof(JsonDataBuffer));
    EXPECT_EQ(json::JsonStructuredWriter::WriteResult::Succeeded, result);
    EXPECT_EQ(4, len);
    EXPECT_EQ(0, std::strncmp(JsonDataBuffer, "null", 4));
}

TEST(JsonOutput, WriteDouble)
{
    bool (* functionOutputValue)(nn::util::optional<double>*, void*, int) = [](nn::util::optional<double>* outValue, void*, int) -> bool
    {
        *outValue = 123.25;
        return true;
    };

    NN_DETAIL_PCTL_JSON_BEGIN_WRITE(EMPTY_SPECIFIER, JsonDataWrite)
        NN_DETAIL_PCTL_JSON_WRITE_VALUE_DOUBLE(functionOutputValue)
    NN_DETAIL_PCTL_JSON_END_WRITE()

    std::memset(JsonDataBuffer, 0, sizeof(JsonDataBuffer));

    json::JsonStructuredWriter writer(nullptr, JsonDataWrite);
    size_t len;
    auto result = writer.FillData(&len, JsonDataBuffer, sizeof(JsonDataBuffer));
    EXPECT_EQ(json::JsonStructuredWriter::WriteResult::Succeeded, result);
    EXPECT_EQ(6, len);
    EXPECT_EQ(0, std::strncmp(JsonDataBuffer, "123.25", 6));
}

TEST(JsonOutput, WriteDouble_Null)
{
    bool (* functionOutputValue)(nn::util::optional<double>*, void*, int) = [](nn::util::optional<double>* outValue, void*, int) -> bool
    {
        *outValue = nn::util::nullopt;
        return true;
    };

    NN_DETAIL_PCTL_JSON_BEGIN_WRITE(EMPTY_SPECIFIER, JsonDataWrite)
        NN_DETAIL_PCTL_JSON_WRITE_VALUE_DOUBLE(functionOutputValue)
    NN_DETAIL_PCTL_JSON_END_WRITE()

    std::memset(JsonDataBuffer, 0, sizeof(JsonDataBuffer));

    json::JsonStructuredWriter writer(nullptr, JsonDataWrite);
    size_t len;
    auto result = writer.FillData(&len, JsonDataBuffer, sizeof(JsonDataBuffer));
    EXPECT_EQ(json::JsonStructuredWriter::WriteResult::Succeeded, result);
    EXPECT_EQ(4, len);
    EXPECT_EQ(0, std::strncmp(JsonDataBuffer, "null", 4));
}

TEST(JsonOutput, WriteBoolean_True)
{
    bool (* functionOutputValue)(nn::util::optional<bool>*, void*, int) = [](nn::util::optional<bool>* outValue, void*, int) -> bool
    {
        *outValue = true;
        return true;
    };

    NN_DETAIL_PCTL_JSON_BEGIN_WRITE(EMPTY_SPECIFIER, JsonDataWrite)
        NN_DETAIL_PCTL_JSON_WRITE_VALUE_BOOLEAN(functionOutputValue)
    NN_DETAIL_PCTL_JSON_END_WRITE()

    std::memset(JsonDataBuffer, 0, sizeof(JsonDataBuffer));

    json::JsonStructuredWriter writer(nullptr, JsonDataWrite);
    size_t len;
    auto result = writer.FillData(&len, JsonDataBuffer, sizeof(JsonDataBuffer));
    EXPECT_EQ(json::JsonStructuredWriter::WriteResult::Succeeded, result);
    EXPECT_EQ(4, len);
    EXPECT_EQ(0, std::strncmp(JsonDataBuffer, "true", 4));
}

TEST(JsonOutput, WriteBoolean_False)
{
    bool (* functionOutputValue)(nn::util::optional<bool>*, void*, int) = [](nn::util::optional<bool>* outValue, void*, int) -> bool
    {
        *outValue = false;
        return true;
    };

    NN_DETAIL_PCTL_JSON_BEGIN_WRITE(EMPTY_SPECIFIER, JsonDataWrite)
        NN_DETAIL_PCTL_JSON_WRITE_VALUE_BOOLEAN(functionOutputValue)
    NN_DETAIL_PCTL_JSON_END_WRITE()

    std::memset(JsonDataBuffer, 0, sizeof(JsonDataBuffer));

    json::JsonStructuredWriter writer(nullptr, JsonDataWrite);
    size_t len;
    auto result = writer.FillData(&len, JsonDataBuffer, sizeof(JsonDataBuffer));
    EXPECT_EQ(json::JsonStructuredWriter::WriteResult::Succeeded, result);
    EXPECT_EQ(5, len);
    EXPECT_EQ(0, std::strncmp(JsonDataBuffer, "false", 5));
}

TEST(JsonOutput, WriteBoolean_Null)
{
    bool (* functionOutputValue)(nn::util::optional<bool>*, void*, int) = [](nn::util::optional<bool>* outValue, void*, int) -> bool
    {
        *outValue = nn::util::nullopt;
        return true;
    };

    NN_DETAIL_PCTL_JSON_BEGIN_WRITE(EMPTY_SPECIFIER, JsonDataWrite)
        NN_DETAIL_PCTL_JSON_WRITE_VALUE_BOOLEAN(functionOutputValue)
    NN_DETAIL_PCTL_JSON_END_WRITE()

    std::memset(JsonDataBuffer, 0, sizeof(JsonDataBuffer));

    json::JsonStructuredWriter writer(nullptr, JsonDataWrite);
    size_t len;
    auto result = writer.FillData(&len, JsonDataBuffer, sizeof(JsonDataBuffer));
    EXPECT_EQ(json::JsonStructuredWriter::WriteResult::Succeeded, result);
    EXPECT_EQ(4, len);
    EXPECT_EQ(0, std::strncmp(JsonDataBuffer, "null", 4));
}

TEST(JsonOutput, WriteString)
{
    static const char StringData[] = "Test String";
    static const char ExpectJsonStringData[] = "\"Test String\""; // JSONなので " " で括られる
    static const size_t StringDataLen = std::extent<decltype(StringData)>::value - 1;
    static const size_t ExpectJsonStringDataLen = std::extent<decltype(ExpectJsonStringData)>::value - 1;
    bool (* functionOutputValue)(nn::util::optional<size_t>*, const char**, void*, int) = [](nn::util::optional<size_t>* outValueLength, const char** outValue, void*, int) -> bool
    {
        *outValueLength = static_cast<size_t>(StringDataLen);
        *outValue = StringData;
        return true;
    };

    NN_DETAIL_PCTL_JSON_BEGIN_WRITE(EMPTY_SPECIFIER, JsonDataWrite)
        NN_DETAIL_PCTL_JSON_WRITE_VALUE_STRING(functionOutputValue)
    NN_DETAIL_PCTL_JSON_END_WRITE()

    std::memset(JsonDataBuffer, 0, sizeof(JsonDataBuffer));

    json::JsonStructuredWriter writer(nullptr, JsonDataWrite);
    size_t len;
    auto result = writer.FillData(&len, JsonDataBuffer, sizeof(JsonDataBuffer));
    EXPECT_EQ(json::JsonStructuredWriter::WriteResult::Succeeded, result);
    EXPECT_EQ(ExpectJsonStringDataLen, len);
    EXPECT_EQ(0, std::strncmp(JsonDataBuffer, ExpectJsonStringData, ExpectJsonStringDataLen));
}

TEST(JsonOutput, WriteString_Null)
{
    bool (* functionOutputValue)(nn::util::optional<size_t>*, const char**, void*, int) = [](nn::util::optional<size_t>* outValueLength, const char**, void*, int) -> bool
    {
        *outValueLength = nn::util::nullopt;
        return true;
    };

    NN_DETAIL_PCTL_JSON_BEGIN_WRITE(EMPTY_SPECIFIER, JsonDataWrite)
        NN_DETAIL_PCTL_JSON_WRITE_VALUE_STRING(functionOutputValue)
    NN_DETAIL_PCTL_JSON_END_WRITE()

    std::memset(JsonDataBuffer, 0, sizeof(JsonDataBuffer));

    json::JsonStructuredWriter writer(nullptr, JsonDataWrite);
    size_t len;
    auto result = writer.FillData(&len, JsonDataBuffer, sizeof(JsonDataBuffer));
    EXPECT_EQ(json::JsonStructuredWriter::WriteResult::Succeeded, result);
    EXPECT_EQ(4, len);
    EXPECT_EQ(0, std::strncmp(JsonDataBuffer, "null", 4));
}

TEST(JsonOutput, WriteString_NotEnoughSize)
{
    static const char VeryBigData[512] = { 0 };
    NN_STATIC_ASSERT(sizeof(JsonDataBuffer) < sizeof(VeryBigData));
    bool (* functionOutputValue)(nn::util::optional<size_t>*, const char**, void*, int) = [](nn::util::optional<size_t>* outValueLength, const char** outValue, void*, int) -> bool
    {
        *outValueLength = static_cast<size_t>(std::extent<decltype(VeryBigData)>::value);
        *outValue = VeryBigData;
        return true;
    };

    NN_DETAIL_PCTL_JSON_BEGIN_WRITE(EMPTY_SPECIFIER, JsonDataWrite)
        NN_DETAIL_PCTL_JSON_WRITE_VALUE_STRING(functionOutputValue)
    NN_DETAIL_PCTL_JSON_END_WRITE()

    std::memset(JsonDataBuffer, 0, sizeof(JsonDataBuffer));

    json::JsonStructuredWriter writer(nullptr, JsonDataWrite);
    size_t len;
    auto result = writer.FillData(&len, JsonDataBuffer, sizeof(JsonDataBuffer));
    EXPECT_EQ(json::JsonStructuredWriter::WriteResult::NeedMoreBuffer, result);
}

TEST(JsonOutput, WriteObject_OneValue)
{
    static const char ExpectJsonData[] = "{\"hoge\":-456}";
    static const size_t ExpectJsonDataLength = std::extent<decltype(ExpectJsonData)>::value - 1;
    bool (* functionOutputObject)(bool*, void*, int) = [](bool* outWriteNull, void*, int) -> bool
    {
        *outWriteNull = false;
        return true;
    };
    bool (* functionOutputValue)(nn::util::optional<int64_t>*, void*, int) = [](nn::util::optional<int64_t>* outValue, void*, int) -> bool
    {
        *outValue = -456;
        return true;
    };

    NN_DETAIL_PCTL_JSON_BEGIN_WRITE(EMPTY_SPECIFIER, JsonDataWrite)
        NN_DETAIL_PCTL_JSON_WRITE_OBJECT_BEGIN(functionOutputObject)
            NN_DETAIL_PCTL_JSON_WRITE_KEY("hoge") NN_DETAIL_PCTL_JSON_WRITE_VALUE_INT64(functionOutputValue)
        NN_DETAIL_PCTL_JSON_WRITE_OBJECT_END()
    NN_DETAIL_PCTL_JSON_END_WRITE()

    std::memset(JsonDataBuffer, 0, sizeof(JsonDataBuffer));

    json::JsonStructuredWriter writer(nullptr, JsonDataWrite);
    size_t len;
    auto result = writer.FillData(&len, JsonDataBuffer, sizeof(JsonDataBuffer));
    EXPECT_EQ(json::JsonStructuredWriter::WriteResult::Succeeded, result);
    EXPECT_EQ(ExpectJsonDataLength, len);
    EXPECT_EQ(0, std::strncmp(JsonDataBuffer, ExpectJsonData, ExpectJsonDataLength));
}

TEST(JsonOutput, WriteObject_ThreeValues)
{
    static const char ExpectJsonData[] = "{\"number1\":2,\"number2\":4.5,\"number3\":-20}";
    static const size_t ExpectJsonDataLength = std::extent<decltype(ExpectJsonData)>::value - 1;
    bool (* functionOutputObject)(bool*, void*, int) = [](bool* outWriteNull, void*, int) -> bool
    {
        *outWriteNull = false;
        return true;
    };
    bool (* functionOutputValue1)(nn::util::optional<uint64_t>*, void*, int) = [](nn::util::optional<uint64_t>* outValue, void*, int) -> bool
    {
        *outValue = 2;
        return true;
    };
    bool (* functionOutputValue2)(nn::util::optional<double>*, void*, int) = [](nn::util::optional<double>* outValue, void*, int) -> bool
    {
        *outValue = 4.5;
        return true;
    };
    bool (* functionOutputValue3)(nn::util::optional<int64_t>*, void*, int) = [](nn::util::optional<int64_t>* outValue, void*, int) -> bool
    {
        *outValue = -20;
        return true;
    };

    NN_DETAIL_PCTL_JSON_BEGIN_WRITE(EMPTY_SPECIFIER, JsonDataWrite)
        NN_DETAIL_PCTL_JSON_WRITE_OBJECT_BEGIN(functionOutputObject)
            NN_DETAIL_PCTL_JSON_WRITE_KEY("number1") NN_DETAIL_PCTL_JSON_WRITE_VALUE_UINT64(functionOutputValue1)
            NN_DETAIL_PCTL_JSON_WRITE_KEY("number2") NN_DETAIL_PCTL_JSON_WRITE_VALUE_DOUBLE(functionOutputValue2)
            NN_DETAIL_PCTL_JSON_WRITE_KEY("number3") NN_DETAIL_PCTL_JSON_WRITE_VALUE_INT64(functionOutputValue3)
        NN_DETAIL_PCTL_JSON_WRITE_OBJECT_END()
    NN_DETAIL_PCTL_JSON_END_WRITE()

    std::memset(JsonDataBuffer, 0, sizeof(JsonDataBuffer));

    json::JsonStructuredWriter writer(nullptr, JsonDataWrite);
    size_t len;
    auto result = writer.FillData(&len, JsonDataBuffer, sizeof(JsonDataBuffer));
    EXPECT_EQ(json::JsonStructuredWriter::WriteResult::Succeeded, result);
    EXPECT_EQ(ExpectJsonDataLength, len);
    EXPECT_EQ(0, std::strncmp(JsonDataBuffer, ExpectJsonData, ExpectJsonDataLength));
}

TEST(JsonOutput, WriteObject_ThreeValues_OneOmitted)
{
    static const char ExpectJsonData[] = "{\"number1\":2,\"number3\":-20}";
    static const size_t ExpectJsonDataLength = std::extent<decltype(ExpectJsonData)>::value - 1;
    bool (* functionOutputObject)(bool*, void*, int) = [](bool* outWriteNull, void*, int) -> bool
    {
        *outWriteNull = false;
        return true;
    };
    bool (* functionOutputKeyName1)(const char**, void*, int) = [](const char** outKeyName, void*, int) -> bool
    {
        *outKeyName = "number1";
        return true;
    };
    bool (* functionOutputValue1)(nn::util::optional<uint64_t>*, void*, int) = [](nn::util::optional<uint64_t>* outValue, void*, int) -> bool
    {
        *outValue = 2;
        return true;
    };
    bool (* functionOutputKeyName2)(const char**, void*, int) = [](const char** outKeyName, void*, int) -> bool
    {
        *outKeyName = nullptr; // omit
        return true;
    };
    bool (* functionOutputValue2)(nn::util::optional<double>*, void*, int) = [](nn::util::optional<double>* outValue, void*, int) -> bool
    {
        *outValue = 4.5;
        return true;
    };
    bool (* functionOutputValue3)(nn::util::optional<int64_t>*, void*, int) = [](nn::util::optional<int64_t>* outValue, void*, int) -> bool
    {
        *outValue = -20;
        return true;
    };

    NN_DETAIL_PCTL_JSON_BEGIN_WRITE(EMPTY_SPECIFIER, JsonDataWrite)
        NN_DETAIL_PCTL_JSON_WRITE_OBJECT_BEGIN(functionOutputObject)
            NN_DETAIL_PCTL_JSON_WRITE_KEY_OPTIONAL(functionOutputKeyName1) NN_DETAIL_PCTL_JSON_WRITE_VALUE_UINT64(functionOutputValue1)
            NN_DETAIL_PCTL_JSON_WRITE_KEY_OPTIONAL(functionOutputKeyName2) NN_DETAIL_PCTL_JSON_WRITE_VALUE_DOUBLE(functionOutputValue2)
            NN_DETAIL_PCTL_JSON_WRITE_KEY("number3") NN_DETAIL_PCTL_JSON_WRITE_VALUE_INT64(functionOutputValue3)
        NN_DETAIL_PCTL_JSON_WRITE_OBJECT_END()
    NN_DETAIL_PCTL_JSON_END_WRITE()

    std::memset(JsonDataBuffer, 0, sizeof(JsonDataBuffer));

    json::JsonStructuredWriter writer(nullptr, JsonDataWrite);
    size_t len;
    auto result = writer.FillData(&len, JsonDataBuffer, sizeof(JsonDataBuffer));
    EXPECT_EQ(json::JsonStructuredWriter::WriteResult::Succeeded, result);
    EXPECT_EQ(ExpectJsonDataLength, len);
    EXPECT_EQ(0, std::strncmp(JsonDataBuffer, ExpectJsonData, ExpectJsonDataLength));
}

TEST(JsonOutput, WriteObject_NestDefinition)
{
    static const char ExpectJsonData[] = "{\"hoge\":\"piyo\",\"some\":-20}";
    static const size_t ExpectJsonDataLength = std::extent<decltype(ExpectJsonData)>::value - 1;

    static bool (*functionOutputValue1)(nn::util::optional<size_t>*, const char**, void*, int) = [](nn::util::optional<size_t>* outValueLength, const char** outValue, void*, int) -> bool
    {
        *outValueLength = 4;
        *outValue = "piyo";
        return true;
    };
    static bool (* functionOutputValue2)(nn::util::optional<int64_t>*, void*, int) = [](nn::util::optional<int64_t>* outValue, void*, int) -> bool
    {
        *outValue = -20;
        return true;
    };

    NN_DETAIL_PCTL_JSON_BEGIN_WRITE(static, JsonDataWriteInner)
        NN_DETAIL_PCTL_JSON_WRITE_KEY("hoge") NN_DETAIL_PCTL_JSON_WRITE_VALUE_STRING(functionOutputValue1)
        NN_DETAIL_PCTL_JSON_WRITE_KEY("some") NN_DETAIL_PCTL_JSON_WRITE_VALUE_INT64(functionOutputValue2)
    NN_DETAIL_PCTL_JSON_END_WRITE()

    bool (* functionOutputObject)(nn::util::optional<int>*, const json::WriteDataDefinition**, void*, int) = [](nn::util::optional<int>* outDefinitionLength, const json::WriteDataDefinition** outDefinitions, void*, int) -> bool
    {
        *outDefinitionLength = static_cast<int>(std::extent<decltype(JsonDataWriteInner)>::value);
        *outDefinitions = JsonDataWriteInner;
        return true;
    };

    NN_DETAIL_PCTL_JSON_BEGIN_WRITE(EMPTY_SPECIFIER, JsonDataWrite)
        NN_DETAIL_PCTL_JSON_WRITE_OBJECT_ANY(functionOutputObject)
    NN_DETAIL_PCTL_JSON_END_WRITE()

    std::memset(JsonDataBuffer, 0, sizeof(JsonDataBuffer));

    json::JsonStructuredWriter writer(nullptr, JsonDataWrite);
    size_t len;
    auto result = writer.FillData(&len, JsonDataBuffer, sizeof(JsonDataBuffer));
    EXPECT_EQ(json::JsonStructuredWriter::WriteResult::Succeeded, result);
    EXPECT_EQ(ExpectJsonDataLength, len);
    EXPECT_EQ(0, std::strncmp(JsonDataBuffer, ExpectJsonData, ExpectJsonDataLength));
}

TEST(JsonOutput, WriteObject_Repeat)
{
    static const char ExpectJsonData[] = "{\"item1\":-456,\"item2\":4.5,\"item3\":\"hello\"}";
    static const size_t ExpectJsonDataLength = std::extent<decltype(ExpectJsonData)>::value - 1;

    static bool (* functionOutputValue1)(nn::util::optional<int64_t>*, void*, int) = [](nn::util::optional<int64_t>* outValue, void*, int index) -> bool
    {
        EXPECT_EQ(0, index);
        *outValue = -456;
        return true;
    };
    static bool (* functionOutputValue2)(nn::util::optional<double>*, void*, int) = [](nn::util::optional<double>* outValue, void*, int index) -> bool
    {
        EXPECT_EQ(1, index);
        *outValue = 4.5;
        return true;
    };
    static bool (* functionOutputValue3)(nn::util::optional<size_t>*, const char**, void*, int) = [](nn::util::optional<size_t>* outValueLength, const char** outValue, void*, int index) -> bool
    {
        EXPECT_EQ(2, index);
        *outValueLength = 5;
        *outValue = "hello";
        return true;
    };
    NN_DETAIL_PCTL_JSON_BEGIN_WRITE(static, JsonDataWriteValues)
        NN_DETAIL_PCTL_JSON_WRITE_VALUE_INT64(functionOutputValue1)
        NN_DETAIL_PCTL_JSON_WRITE_VALUE_DOUBLE(functionOutputValue2)
        NN_DETAIL_PCTL_JSON_WRITE_VALUE_STRING(functionOutputValue3)
    NN_DETAIL_PCTL_JSON_END_WRITE()

    static bool (* functionOutputRepeatObject)(nn::util::optional<bool>*, const char**, int*, const json::WriteDataDefinition**, void*, int, int) =
    [](nn::util::optional<bool>* outHasItem, const char** outKeyName, int* outDefinitionLength, const json::WriteDataDefinition** outDefinition,
        void*, int index, int itemIndex) -> bool
    {
        static char keyNameBuffer[8];

        EXPECT_EQ(0, index);
        if (itemIndex >= 3)
        {
            *outHasItem = false;
        }
        else
        {
            *outHasItem = true;
            nn::util::SNPrintf(keyNameBuffer, 7, "item%d", itemIndex + 1);
            *outKeyName = keyNameBuffer;
            *outDefinitionLength = 1;
            *outDefinition = &JsonDataWriteValues[itemIndex];
        }
        return true;
    };
    NN_DETAIL_PCTL_JSON_BEGIN_WRITE(static, JsonDataWrite)
        NN_DETAIL_PCTL_JSON_WRITE_OBJECT_REPEAT(functionOutputRepeatObject)
    NN_DETAIL_PCTL_JSON_END_WRITE()

    std::memset(JsonDataBuffer, 0, sizeof(JsonDataBuffer));

    json::JsonStructuredWriter writer(nullptr, JsonDataWrite);
    size_t len;
    auto result = writer.FillData(&len, JsonDataBuffer, sizeof(JsonDataBuffer));
    EXPECT_EQ(json::JsonStructuredWriter::WriteResult::Succeeded, result);
    EXPECT_EQ(ExpectJsonDataLength, len);
    EXPECT_EQ(0, std::strncmp(JsonDataBuffer, ExpectJsonData, ExpectJsonDataLength));
}

TEST(JsonOutput, WriteArray_OneValue)
{
    static const char ExpectJsonData[] = "[-14.25]";
    static const size_t ExpectJsonDataLength = std::extent<decltype(ExpectJsonData)>::value - 1;
    bool (* functionOutputArray)(bool*, void*, int) = [](bool* outWriteNull, void*, int) -> bool
    {
        *outWriteNull = false;
        return true;
    };
    bool (* functionOutputValue)(nn::util::optional<double>*, void*, int) = [](nn::util::optional<double>* outValue, void*, int) -> bool
    {
        *outValue = -14.25;
        return true;
    };

    NN_DETAIL_PCTL_JSON_BEGIN_WRITE(EMPTY_SPECIFIER, JsonDataWrite)
        NN_DETAIL_PCTL_JSON_WRITE_ARRAY_BEGIN_FIXED(functionOutputArray)
            NN_DETAIL_PCTL_JSON_WRITE_VALUE_DOUBLE(functionOutputValue)
        NN_DETAIL_PCTL_JSON_WRITE_ARRAY_END()
    NN_DETAIL_PCTL_JSON_END_WRITE()

    std::memset(JsonDataBuffer, 0, sizeof(JsonDataBuffer));

    json::JsonStructuredWriter writer(nullptr, JsonDataWrite);
    size_t len;
    auto result = writer.FillData(&len, JsonDataBuffer, sizeof(JsonDataBuffer));
    EXPECT_EQ(json::JsonStructuredWriter::WriteResult::Succeeded, result);
    EXPECT_EQ(ExpectJsonDataLength, len);
    EXPECT_EQ(0, std::strncmp(JsonDataBuffer, ExpectJsonData, ExpectJsonDataLength));
}

TEST(JsonOutput, WriteArray_ThreeValues)
{
    static const char ExpectJsonData[] = "[3.25,\"bar\",true]";
    static const size_t ExpectJsonDataLength = std::extent<decltype(ExpectJsonData)>::value - 1;
    bool (* functionOutputArray)(bool*, void*, int) = [](bool* outWriteNull, void*, int) -> bool
    {
        *outWriteNull = false;
        return true;
    };
    bool (* functionOutputValue1)(nn::util::optional<double>*, void*, int) = [](nn::util::optional<double>* outValue, void*, int) -> bool
    {
        *outValue = 3.25;
        return true;
    };
    bool (* functionOutputValue2)(nn::util::optional<size_t>*, const char**, void*, int) = [](nn::util::optional<size_t>* outValueLength, const char** outValue, void*, int) -> bool
    {
        *outValueLength = 3;
        *outValue = "bar";
        return true;
    };
    bool (* functionOutputValue3)(nn::util::optional<bool>*, void*, int) = [](nn::util::optional<bool>* outValue, void*, int) -> bool
    {
        *outValue = true;
        return true;
    };

    NN_DETAIL_PCTL_JSON_BEGIN_WRITE(EMPTY_SPECIFIER, JsonDataWrite)
        NN_DETAIL_PCTL_JSON_WRITE_ARRAY_BEGIN_FIXED(functionOutputArray)
            NN_DETAIL_PCTL_JSON_WRITE_VALUE_DOUBLE(functionOutputValue1)
            NN_DETAIL_PCTL_JSON_WRITE_VALUE_STRING(functionOutputValue2)
            NN_DETAIL_PCTL_JSON_WRITE_VALUE_BOOLEAN(functionOutputValue3)
        NN_DETAIL_PCTL_JSON_WRITE_ARRAY_END()
    NN_DETAIL_PCTL_JSON_END_WRITE()

    std::memset(JsonDataBuffer, 0, sizeof(JsonDataBuffer));

    json::JsonStructuredWriter writer(nullptr, JsonDataWrite);
    size_t len;
    auto result = writer.FillData(&len, JsonDataBuffer, sizeof(JsonDataBuffer));
    EXPECT_EQ(json::JsonStructuredWriter::WriteResult::Succeeded, result);
    EXPECT_EQ(ExpectJsonDataLength, len);
    EXPECT_EQ(0, std::strncmp(JsonDataBuffer, ExpectJsonData, ExpectJsonDataLength));
}

TEST(JsonOutput, WriteArray_Repeated)
{
    static const int WriteValues[] = { 1, 1, 2, 3, 5, 8 };
    static const int WriteValuesLength = static_cast<int>(std::extent<decltype(WriteValues)>::value);
    static const char ExpectJsonData[] = "[1,1,2,3,5,8]";
    static const size_t ExpectJsonDataLength = std::extent<decltype(ExpectJsonData)>::value - 1;
    bool (* functionOutputArray)(nn::util::optional<bool>*, void*, int, int) = [](nn::util::optional<bool>* outHasItem, void*, int, int itemIndex) -> bool
    {
        *outHasItem = (itemIndex >= 0 && itemIndex < WriteValuesLength);
        return true;
    };
    bool (* functionOutputValue)(nn::util::optional<int64_t>*, void*, int) = [](nn::util::optional<int64_t>* outValue, void*, int index) -> bool
    {
        EXPECT_GE(index, 0);
        EXPECT_LT(index, WriteValuesLength);
        *outValue = static_cast<int64_t>(WriteValues[index]);
        return true;
    };

    NN_DETAIL_PCTL_JSON_BEGIN_WRITE(EMPTY_SPECIFIER, JsonDataWrite)
        NN_DETAIL_PCTL_JSON_WRITE_ARRAY_BEGIN(functionOutputArray)
            NN_DETAIL_PCTL_JSON_WRITE_VALUE_INT64(functionOutputValue)
        NN_DETAIL_PCTL_JSON_WRITE_ARRAY_END()
    NN_DETAIL_PCTL_JSON_END_WRITE()

    std::memset(JsonDataBuffer, 0, sizeof(JsonDataBuffer));

    json::JsonStructuredWriter writer(nullptr, JsonDataWrite);
    size_t len;
    auto result = writer.FillData(&len, JsonDataBuffer, sizeof(JsonDataBuffer));
    EXPECT_EQ(json::JsonStructuredWriter::WriteResult::Succeeded, result);
    EXPECT_EQ(ExpectJsonDataLength, len);
    EXPECT_EQ(0, std::strncmp(JsonDataBuffer, ExpectJsonData, ExpectJsonDataLength));
}

// いろいろ複合(オブジェクト内にオブジェクトと配列と通常値、null可能bool値、null可能オブジェクト、小さいバッファーでループ)
TEST(JsonOutput, Write_Complex)
{
    static const char ExpectJsonData[] = "{\"hoge\":\"piyo\",\"nullableBool\":null,\"some\":{\"one\":1,\"true\":true},\"a\":[1,2,3],\"b\":null}";
    static const size_t ExpectJsonDataLength = std::extent<decltype(ExpectJsonData)>::value - 1;

    static bool (*functionOutputValuePiyo)(nn::util::optional<size_t>*, const char**, void*, int) = [](nn::util::optional<size_t>* outValueLength, const char** outValue, void*, int index) -> bool
    {
        EXPECT_EQ(0, index);
        *outValueLength = 4;
        *outValue = "piyo";
        return true;
    };
    static bool (* functionOutputValueNullableBool)(nn::util::optional<bool>*, void*, int) = [](nn::util::optional<bool>* outValue, void*, int index) -> bool
    {
        EXPECT_EQ(1, index);
        *outValue = nn::util::nullopt;
        return true;
    };

    static bool (* functionOutputValueOne)(nn::util::optional<int64_t>*, void*, int) = [](nn::util::optional<int64_t>* outValue, void*, int) -> bool
    {
        *outValue = 1;
        return true;
    };
    static bool (* functionOutputValueTrue)(nn::util::optional<bool>*, void*, int) = [](nn::util::optional<bool>* outValue, void*, int) -> bool
    {
        *outValue = true;
        return true;
    };
    NN_DETAIL_PCTL_JSON_BEGIN_WRITE(static, JsonDataWriteInner)
        NN_DETAIL_PCTL_JSON_WRITE_KEY("one") NN_DETAIL_PCTL_JSON_WRITE_VALUE_INT64(functionOutputValueOne)
        NN_DETAIL_PCTL_JSON_WRITE_KEY("true") NN_DETAIL_PCTL_JSON_WRITE_VALUE_BOOLEAN(functionOutputValueTrue)
    NN_DETAIL_PCTL_JSON_END_WRITE()

    bool (* functionOutputObjectInner)(nn::util::optional<int>*, const json::WriteDataDefinition**, void*, int) = [](nn::util::optional<int>* outDefinitionLength, const json::WriteDataDefinition** outDefinitions, void*, int index) -> bool
    {
        EXPECT_EQ(2, index);
        *outDefinitionLength = static_cast<int>(std::extent<decltype(JsonDataWriteInner)>::value);
        *outDefinitions = JsonDataWriteInner;
        return true;
    };

    bool (* functionOutputArray)(nn::util::optional<bool>*, void*, int, int) = [](nn::util::optional<bool>* outHasItem, void*, int index, int itemIndex) -> bool
    {
        EXPECT_EQ(3, index);
        *outHasItem = (itemIndex >= 0 && itemIndex < 3);
        return true;
    };
    bool (* functionOutputArrayValue)(nn::util::optional<int64_t>*, void*, int) = [](nn::util::optional<int64_t>* outValue, void*, int index) -> bool
    {
        EXPECT_GE(index, 0);
        EXPECT_LT(index, 3);
        *outValue = static_cast<int64_t>(index + 1);
        return true;
    };
    bool (* functionOutputArray2)(nn::util::optional<bool>*, void*, int, int) = [](nn::util::optional<bool>* outHasItem, void*, int index, int) -> bool
    {
        EXPECT_EQ(4, index);
        *outHasItem = nn::util::nullopt;
        return true;
    };

    bool (* functionOutputObject)(bool*, void*, int) = [](bool* outWriteNull, void*, int) -> bool
    {
        *outWriteNull = false;
        return true;
    };

    NN_DETAIL_PCTL_JSON_BEGIN_WRITE(EMPTY_SPECIFIER, JsonDataWrite)
        NN_DETAIL_PCTL_JSON_WRITE_OBJECT_BEGIN(functionOutputObject)
            NN_DETAIL_PCTL_JSON_WRITE_KEY("hoge")          NN_DETAIL_PCTL_JSON_WRITE_VALUE_STRING(functionOutputValuePiyo)
            NN_DETAIL_PCTL_JSON_WRITE_KEY("nullableBool")  NN_DETAIL_PCTL_JSON_WRITE_VALUE_BOOLEAN(functionOutputValueNullableBool)
            NN_DETAIL_PCTL_JSON_WRITE_KEY("some")          NN_DETAIL_PCTL_JSON_WRITE_OBJECT_ANY(functionOutputObjectInner)
            NN_DETAIL_PCTL_JSON_WRITE_KEY("a")             NN_DETAIL_PCTL_JSON_WRITE_ARRAY_BEGIN(functionOutputArray)
                NN_DETAIL_PCTL_JSON_WRITE_VALUE_INT64(functionOutputArrayValue)
            NN_DETAIL_PCTL_JSON_WRITE_ARRAY_END()
            NN_DETAIL_PCTL_JSON_WRITE_KEY("b")             NN_DETAIL_PCTL_JSON_WRITE_ARRAY_BEGIN(functionOutputArray2)
                NN_DETAIL_PCTL_JSON_WRITE_VALUE_INT64(functionOutputArrayValue)
            NN_DETAIL_PCTL_JSON_WRITE_ARRAY_END()
        NN_DETAIL_PCTL_JSON_WRITE_OBJECT_END()
    NN_DETAIL_PCTL_JSON_END_WRITE()

    std::memset(JsonDataBuffer, 0, sizeof(JsonDataBuffer));
    char smallBuffer[32];
    std::memset(smallBuffer, 0, sizeof(smallBuffer));

    json::JsonStructuredWriter writer(nullptr, JsonDataWrite);
    size_t totalLen = 0;
    size_t pos = 0;
    while (NN_STATIC_CONDITION(true))
    {
        size_t len;
        auto result = writer.FillData(&len, smallBuffer, sizeof(smallBuffer));
        if (result == json::JsonStructuredWriter::WriteResult::NoMoreData)
            break;
        EXPECT_EQ(json::JsonStructuredWriter::WriteResult::Succeeded, result);
        std::memcpy(&JsonDataBuffer[pos], smallBuffer, len);
        pos += len;
        totalLen += len;
    }
    EXPECT_EQ(ExpectJsonDataLength, totalLen);
    EXPECT_EQ(0, std::strncmp(JsonDataBuffer, ExpectJsonData, ExpectJsonDataLength));
}

// キャッシュを利用した出力
TEST(JsonOutput, Write_WithCache)
{
    static const size_t CacheSize = 10240;
    static const size_t WriteStringLength = 8191;
    static const size_t OutputBufferSize = 2048;

    // 仕込み
    void* cacheBuffer = malloc(CacheSize);
    ASSERT_NE(cacheBuffer, nullptr);
    static char* stringData = static_cast<char*>(malloc(WriteStringLength + 1));
    ASSERT_NE(stringData, nullptr);
    char* outputBuffer = static_cast<char*>(malloc(OutputBufferSize));
    ASSERT_NE(outputBuffer, nullptr);
    std::memset(stringData, 'X', WriteStringLength);
    stringData[WriteStringLength] = 0;

    // 11 == strlen("{\"hoge\":") + strlen("\"\"") + strlen("}")
    static const size_t ExpectJsonStringDataLen = WriteStringLength + 11;
    bool (* functionOutputValue)(nn::util::optional<size_t>*, const char**, void*, int) = [](nn::util::optional<size_t>* outValueLength, const char** outValue, void*, int) -> bool
    {
        *outValueLength = static_cast<size_t>(WriteStringLength);
        *outValue = stringData;
        return true;
    };

    NN_DETAIL_PCTL_JSON_BEGIN_WRITE(EMPTY_SPECIFIER, JsonDataWrite)
        NN_DETAIL_PCTL_JSON_WRITE_OBJECT_BEGIN(nullptr)
            NN_DETAIL_PCTL_JSON_WRITE_KEY("hoge")  NN_DETAIL_PCTL_JSON_WRITE_VALUE_STRING(functionOutputValue)
        NN_DETAIL_PCTL_JSON_WRITE_OBJECT_END()
    NN_DETAIL_PCTL_JSON_END_WRITE()

    std::memset(outputBuffer, 0, OutputBufferSize);

    json::JsonStructuredWriter writer(nullptr, JsonDataWrite);
    writer.SetCacheBuffer(cacheBuffer, CacheSize);
    size_t totalLen = 0;
    char lastTwoChars[3] = {};
    while (NN_STATIC_CONDITION(true))
    {
        size_t len;
        auto result = writer.FillData(&len, outputBuffer, OutputBufferSize);
        if (result == json::JsonStructuredWriter::WriteResult::NoMoreData)
        {
            break;
        }
        // (書き込み時点での)最後の2文字を取得
        if (len == 1)
        {
            // 「前回の最後の2文字」から1文字だけ増えたので、
            // 「前回の最後の2文字」の2文字目を「今回の最後の2文字」の1文字目にする
            lastTwoChars[0] = lastTwoChars[1];
            lastTwoChars[1] = outputBuffer[0];
        }
        else if (len > 0)
        {
            std::memcpy(lastTwoChars, outputBuffer + len - 2, 2);
        }
        if (totalLen == 0)
        {
            // 先頭文字の比較
            EXPECT_EQ(0, std::strncmp("{\"hoge\":\"", outputBuffer, 9));
        }
        totalLen += len;
        EXPECT_EQ(json::JsonStructuredWriter::WriteResult::Succeeded, result);
    }
    EXPECT_EQ(ExpectJsonStringDataLen, totalLen);
    EXPECT_EQ(0, std::strncmp("\"}", lastTwoChars, 2));

    free(outputBuffer);
    free(stringData);
    free(cacheBuffer);
}
