﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/pctl/pctl_Api.h>
#include <nn/pctl/pctl_ApiForAuthentication.h>
#include <nn/pctl/pctl_ApiForRecovery.h>
#include <nn/pctl/pctl_ApiSystem.h>
#include <nn/pctl/pctl_ApiWatcher.h>
#include <nn/pctl/detail/ipc/pctl_IpcConfig.h>

#if !defined(NN_DETAIL_PCTL_CONFIG_FULL_CAPABILITY_FOR_RECOVERY)
TEST(DisableFeaturesTest, Dummy)
{
}
#else

class DisableFeaturesTest : public ::testing::Test
{
public:
    static void SetUpTestCase()
    {
        nn::pctl::SafetyLevelSettings customSettings;

        customSettings.ratingAge = 15;
        customSettings.snsPostRestriction = true;
        customSettings.freeCommunicationRestriction = true;

        nn::pctl::SetPinCode("0123");
        nn::pctl::SetCustomSafetyLevelSettings(customSettings);

        EXPECT_TRUE(nn::pctl::IsRestrictionEnabled());
        EXPECT_TRUE(nn::pctl::DisableAllFeatures());
    }

    static void TearDownTestCase()
    {
        nn::pctl::PostEnableAllFeatures();
        nn::pctl::DeleteSettings();
    }
};

TEST_F(DisableFeaturesTest, TestNotRestricted_ForLaunchCheckWithRatingRestriction)
{
    nn::ns::ApplicationControlProperty prop;
    nn::ncm::ApplicationId appId = { 0x0100000010001000 }; // 適当なApplicationId

    prop.parentalControlFlag = static_cast<nn::Bit32>(nn::ns::ParentalControlFlag::FreeCommunication);
    prop.ratingAge[static_cast<int>(nn::ns::RatingOrganization::CERO)] = 18;

    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::ConfirmLaunchApplicationPermission(appId, prop));
}

TEST_F(DisableFeaturesTest, TestNotRestricted_ForOtherRestrictions)
{
    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::ConfirmSnsPostPermission());
    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::ConfirmSystemSettingsPermission());
}

TEST_F(DisableFeaturesTest, TestNotRestricted_ForRestrictionStatus)
{
    EXPECT_TRUE(nn::pctl::IsRestrictionTemporaryUnlocked());
    // SystemSettingsEntered は影響を受けない
    EXPECT_FALSE(nn::pctl::IsRestrictedSystemSettingsEntered());
}

// 設定がない扱いのチェック
TEST_F(DisableFeaturesTest, TestNoSettings)
{
    EXPECT_FALSE(nn::pctl::IsRestrictionEnabled());
    EXPECT_EQ(nn::pctl::Feature_None, nn::pctl::GetRestrictedFeatures());
}

TEST_F(DisableFeaturesTest, TestDisabled_PlayTimer)
{
    nn::pctl::PlayTimerSettings settings = {};
    settings.isEnabled = true;
    settings.isWeekSettingsUsed = false;
    settings.playTimerMode = nn::pctl::PlayTimerMode::PlayTimerMode_Suspend;
    settings.dailySettings.isLimitTimeEnabled = false;
    settings.dailySettings.isBedtimeEnabled = true;
    // 適当な値
    settings.dailySettings.bedtimeHour = 13;
    settings.dailySettings.bedtimeMinute = 0;
    nn::pctl::SetPlayTimerSettingsForDebug(settings);

    nn::pctl::StartPlayTimer();
    EXPECT_FALSE(nn::pctl::IsPlayTimerEnabled());
    EXPECT_FALSE(nn::pctl::IsRestrictedByPlayTimer());
    nn::pctl::StopPlayTimer();
}

#endif // defined(NN_DETAIL_PCTL_CONFIG_FULL_CAPABILITY_FOR_RECOVERY)
