﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nnt/nntest.h>

#include <nn/pcm/pcm.h>
#include <nn/pcm/pcm_Api.private.h>

struct PcmTestParam
{
    nn::pcm::MeasuringPoint point;
    const char*             pointName;
};

class PcmTest : public ::testing::TestWithParam<PcmTestParam>
{};

namespace {

#if 0
    const int DefaultAllowableErrorPercent = 10;
    const int DefaultAllowableErrorMicroWatt = 1000;
    bool IsNearEnough(long long expectedMicroWatt, long long actualMicroWatt,
                int allowableErrorPercent = DefaultAllowableErrorPercent,
                int allowableErrorMicroWatt = DefaultAllowableErrorMicroWatt)
    {
        if (actualMicroWatt < expectedMicroWatt * (100 - allowableErrorPercent) / 100 &&
            actualMicroWatt < expectedMicroWatt - allowableErrorMicroWatt)
        {
            return false;
        }
        else if (expectedMicroWatt * (100 + allowableErrorPercent) / 100 < actualMicroWatt &&
                expectedMicroWatt + allowableErrorMicroWatt < actualMicroWatt)
        {
            return false;
        }
        else
        {
            return true;
        }
    }
#endif

}

TEST_P(PcmTest, ReadCurrentPower)
{
    nn::pcm::Initialize();

    // TODO: EDEV / SDEV 1.8 以前 / SDEV preMP 以降のそれぞれで正しい計測ポイントのセットがサポートされているかを自動判定できるテスト
    // 同一コードでやるにはボードの見分けが必要
    if (nn::pcm::IsSupported(GetParam().point))
    {
        int power = nn::pcm::ReadCurrentPower(GetParam().point);
        NN_LOG("PCM Test: power = %d mW (@ %s)\n", power, GetParam().pointName);
    }
    else
    {
        ASSERT_EQ(0, nn::pcm::ReadCurrentPower(GetParam().point));
        NN_LOG("PCM Test: %s is not implemented on this board.\n", GetParam().pointName);
    }

    nn::pcm::Finalize();
}

TEST_P(PcmTest, ReadCurrentVoltage)
{
    nn::pcm::Initialize();

    // TODO: EDEV / SDEV 1.8 以前 / SDEV preMP 以降のそれぞれで正しい計測ポイントのセットがサポートされているかを自動判定できるテスト
    // 同一コードでやるにはボードの見分けが必要
    if (nn::pcm::IsSupported(GetParam().point))
    {
        int voltage = nn::pcm::ReadCurrentVoltage(GetParam().point);
        NN_LOG("PCM Test: voltage = %d mV (@ %s)\n", voltage, GetParam().pointName);
    }
    else
    {
        ASSERT_EQ(0, nn::pcm::ReadCurrentVoltage(GetParam().point));
        NN_LOG("PCM Test: %s is not implemented on this board.\n", GetParam().pointName);
    }

    nn::pcm::Finalize();
}

// Test patterns to be instantiated on NX boards only
const PcmTestParam PcmTestNxParamSet[] =
{
    { nn::pcm::MeasuringPoint_Gpu, "GPU" },
    { nn::pcm::MeasuringPoint_Cpu, "CPU" },
    { nn::pcm::MeasuringPoint_Ddr, "DDR" },
    { nn::pcm::MeasuringPoint_Backlight, "Backlight" },
    { nn::pcm::MeasuringPoint_VsysAp, "VsysAp" },
    { nn::pcm::MeasuringPoint_Vdd, "Vdd" },
    { nn::pcm::MeasuringPoint_VsysCore, "VsysCore" },
    { nn::pcm::MeasuringPoint_Soc1V8, "Soc1V8" },
    { nn::pcm::MeasuringPoint_Lpddr1V8, "Lpddr1V8" },
    { nn::pcm::MeasuringPoint_Reg1V32, "Reg1V32" },
    { nn::pcm::MeasuringPoint_Vdd3V3Sys, "Vdd3V3Sys" },
    { nn::pcm::MeasuringPoint_VddDdr0V6, "VddDdr0V6" },
};

INSTANTIATE_TEST_CASE_P(
    PcmTestHardwareNx,
    PcmTest,
    ::testing::ValuesIn(PcmTestNxParamSet)
);
