﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//---------------------------------------------------------------------------
//  Mutex 関連機能のテスト
//---------------------------------------------------------------------------

#include "../Common/test_Pragma.h"

#include <nn/os/os_Config.h>
#include <nn/nn_SdkText.h>
#if defined(NN_BUILD_CONFIG_OS_WIN32)
#include <nn/nn_Windows.h>
#endif

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Log.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os.h>
#include "../Common/test_Helper.h"
#include "../Common/test_Calibration.h"

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>

namespace nnt { namespace os { namespace performance {

void PrintTimeNano(const char* label, nn::os::Tick time, int numIterations)
{
    auto timeNano = time.ToTimeSpan().GetNanoSeconds();
    NN_LOG("##teamcity[buildStatisticValue key='%s(nsec)' value='%lld']\n", label, timeNano / numIterations);
}

TEST(MutexPerformance, test_MutexPerformance)
{
    CLEAR_GOOGLE_TEST();

    // ミューテックスの構築（引数２つのコンストラクタで初期化）
    static const int NumIterations = 1000000;
    nn::os::Mutex   mutex( false, 0 );
    nn::os::Mutex   recursiveMutex( true, 0 );
    {
        auto time = nn::os::GetSystemTick();
        for(int i = 0; i < NumIterations; ++i)
        {
            mutex.Lock();
            mutex.Unlock();
        }
        time = nn::os::GetSystemTick() - time;
        PrintTimeNano("MutexAvgLockUnlockTime", time, NumIterations);
    }

    {
        auto time = nn::os::GetSystemTick();
        for(int i = 0; i < NumIterations; ++i)
        {
            mutex.Lock();
            mutex.Unlock();
        }
        time = nn::os::GetSystemTick() - time;
        PrintTimeNano("RecursiveMutexAvgLockUnlockTime", time, NumIterations);
    }

    JUDGE_GOOGLE_TEST();
}

TEST(Semaphore, test_SemaphorePerformance)
{
    CLEAR_GOOGLE_TEST();

    // ミューテックスの構築（引数２つのコンストラクタで初期化）
    static const int NumIterations = 1000000;

    {
        nn::os::Semaphore   semaphore(1,1);
        auto time = nn::os::GetSystemTick();
        for(int i = 0; i < NumIterations; ++i)
        {
            semaphore.Acquire();
            semaphore.Release();
        }
        time = nn::os::GetSystemTick() - time;
        PrintTimeNano("SemaphoreAcquireReleaseAvgTime", time, NumIterations);
    }

    {
        nn::os::Semaphore   semaphore(2,2);
        auto time = nn::os::GetSystemTick();
        for(int i = 0; i < NumIterations; ++i)
        {
            semaphore.Acquire();
            semaphore.Acquire();
            semaphore.Release(2);
        }
        time = nn::os::GetSystemTick() - time;
        PrintTimeNano("SemaphoreAcquireReleaseMultipleAvgTime", time, NumIterations);
    }

    JUDGE_GOOGLE_TEST();
}

}}} // namespace nnt::os::performance

//---------------------------------------------------------------------------
//  Test Main 関数
//---------------------------------------------------------------------------

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    int result;

    NNT_CALIBRATE_INITIALIZE();
    SEQ_INITIALIZE();
    INITIALIZE_TEST_COUNT();

    // テスト開始
    SEQ_CHECK(0);
    NN_LOG("=== Start Performance Tests\n");

    // GoogleTest おまじない
    ::testing::InitGoogleTest(&argc, argv);
    result = RUN_ALL_TESTS();

    // テスト終了
    NN_LOG("\n=== End Performance Tests\n");

    // 集計結果の表示
    g_Result.Show();

    nnt::Exit(result);
}
