﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//---------------------------------------------------------------------------
//  MultipleWait 専用のテストヘルパー
//---------------------------------------------------------------------------

#include "../Common/test_Pragma.h"

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os.h>
#include "../Common/test_Helper.h"
#include "../Common/test_Calibration.h"
#include "test_MultipleWaitHelper.h"

//---------------------------------------------------------------------------

namespace nnt { namespace os { namespace multiWait {

int         g_seq;
const int   messageQueueSize = 4;
uintptr_t   g_MQBuffer1[ messageQueueSize ];
uintptr_t   g_MQBuffer2[ messageQueueSize ];

MultipleWaitHelper  g_MWHelper;

NN_ALIGNAS( 4096 ) char g_StackOfThread1[ 16384 ];
NN_ALIGNAS( 4096 ) char g_StackOfThread2[ 16384 ];

//---------------------------------------------------------------------------
// スレッド１の生成
void MultipleWaitHelper::CreateThread1( TestThreadAction state )
{
    if (state == THR_NOTUSE)
    {
        return;
    }

    SEQ_NONE();
    NNT_OS_LOG("Create Thread1:");
    nn::Result err = nn::os::CreateThread(
                        &m_Thread1,                     // Thread オブジェクト
                        &ThreadFunc1,                   // エントリ関数
                        reinterpret_cast<void*>(0x1111),// 引数
                        g_StackOfThread1,               // スタック
                        16384,                          // スタックサイズ
                        priThread1);                    // 優先度

    CheckBool( err.IsSuccess() );
}

// スレッド２の生成
void MultipleWaitHelper::CreateThread2( TestThreadAction state )
{
    if (state == THR_NOTUSE)
    {
        return;
    }

    SEQ_NONE();
    NNT_OS_LOG("Create Thread2:");
    nn::Result err = nn::os::CreateThread(
                      &m_Thread2,                     // Thread オブジェクト
                      &ThreadFunc2,                   // エントリ関数
                      reinterpret_cast<void*>(0x2222),// 引数
                      g_StackOfThread2,               // スタック
                      16384,                          // スタックサイズ
                      priThread2);                    // 優先度

    CheckBool( err.IsSuccess() );
}

//---------------------------------------------------------------------------
// スレッド１の破棄
void MultipleWaitHelper::DestroyThread1( TestThreadAction state )
{
    if (state == THR_NOTUSE)
    {
        return;
    }

    SEQ_NONE();
    NNT_OS_LOG("Destroy Thread1:");
    nn::os::DestroyThread( &m_Thread1 );

    CheckBool( true );
}

// スレッド２の破棄
void MultipleWaitHelper::DestroyThread2( TestThreadAction state )
{
    if (state == THR_NOTUSE)
    {
        return;
    }

    SEQ_NONE();
    NNT_OS_LOG("Destroy Thread2:");
    nn::os::DestroyThread( &m_Thread2 );

    CheckBool( true );
}


//---------------------------------------------------------------------------
// MultipleWait 対象オブジェクトの Initialize
void MultipleWaitHelper::InitializeMultipleWait()
{
    SEQ_NONE();
    NNT_OS_LOG("Initialize MultipleWait:");

    // Semaphore
    nn::os::InitializeSemaphore(&m_Semaphore, 0, 1);

    // Event
    nn::os::InitializeEvent(&m_Event, false, nn::os::EventClearMode_AutoClear);

    // MessageQueue1
    nn::os::InitializeMessageQueue(&m_MessageQueue1,
                                    g_MQBuffer1, messageQueueSize);

    // MessageQueue2
    nn::os::InitializeMessageQueue(&m_MessageQueue2,
                                    g_MQBuffer2, messageQueueSize);
    for (int i=0; i<messageQueueSize; i++)
    {   // メッセージキューバッファを全て埋める
        TrySendMessageQueue( &m_MessageQueue2, 0x55555555 );
    }

    CheckBool( true );
}

//---------------------------------------------------------------------------
// MultipleWait オブジェクトのクリーンナップ（待ちスレッドを全解除）
void MultipleWaitHelper::Cleanup()
{
    // Cleanup 後の状態へ
    m_isCleanuped = true;

    // セマフォをリリース
    int count = m_Semaphore._count;
    while (count < m_Semaphore._maxCount)
    {
        nn::os::ReleaseSemaphore( &m_Semaphore );
        count = nn::os::GetCurrentSemaphoreCount( &m_Semaphore ) + 1;
    }

    // イベントへ通知
    nn::os::ClearEvent( &m_Event );
    nn::os::SignalEvent( &m_Event );
    nn::os::ClearEvent( &m_Event );
    nn::os::SignalEvent( &m_Event );

    // ダミーでメッセージキュー１に送信
    nn::os::TrySendMessageQueue( &m_MessageQueue1, 0 );
    nn::os::TrySendMessageQueue( &m_MessageQueue1, 0 );

    // ダミーでメッセージキュー２から受信
    uintptr_t   data;
    nn::os::TryReceiveMessageQueue( &data, &m_MessageQueue2 );
    nn::os::TryReceiveMessageQueue( &data, &m_MessageQueue2 );

    // タイミング合わせ
    testSleep( NNT_CALIBRATE_RATE() );

    // ２つのスレッドが終了するまで待つ
    if (( g_MWHelper.GetThreadExpectState1() != THR_STATE_NONE ) &&
        ( m_Thread1._state == nn::os::ThreadType::State_Started ))
    {
        nn::os::WaitThread( &m_Thread1 );
    }
    if (( g_MWHelper.GetThreadExpectState2() != THR_STATE_NONE ) &&
        ( m_Thread2._state == nn::os::ThreadType::State_Started ))
    {
        nn::os::WaitThread( &m_Thread2 );
    }
}

//---------------------------------------------------------------------------
// MultipleWait オブジェクトの Finalize
void MultipleWaitHelper::FinalizeMultipleWait()
{
    SEQ_NONE();
    NNT_OS_LOG("Finalize MultipleWait:");

    nn::os::FinalizeSemaphore( &m_Semaphore );
    nn::os::FinalizeEvent( &m_Event );
    nn::os::FinalizeMessageQueue( &m_MessageQueue1 );
    nn::os::FinalizeMessageQueue( &m_MessageQueue2 );

    CheckBool( true );
}

//---------------------------------------------------------------------------
//  スレッド実行の開始
void MultipleWaitHelper::StartThread1( TestThreadAction state )
{
    SEQ_NONE();
    if (state != THR_NOTUSE)
    {
        NNT_OS_LOG("Start Thread1:");
        nn::os::StartThread( &m_Thread1 );
        CheckBool( true );
    }
    else
    {
        NNT_OS_LOG("Thread1 is not used\n");
    }
}

void MultipleWaitHelper::StartThread2( TestThreadAction state )
{
    SEQ_NONE();
    if (state != THR_NOTUSE)
    {
        NNT_OS_LOG("Start Thread2:");
        nn::os::StartThread( &m_Thread2 );
        CheckBool( true );
    }
    else
    {
        NNT_OS_LOG("Thread2 is not used\n");
    }
}

//---------------------------------------------------------------------------
// 多重待ちの内部状態のチェック（非シグナル状態にしてリターン）
void    MultipleWaitHelper::MultipleWaitCheck( TestThreadAction th1,
                                               TestThreadAction th2)
{
    uint8_t expectFlag = 0;

    // expectFlag は各ビットがオブジェクトの状態変更を意味する。
    //  0x01: Semaphore カウンタ +1
    //  0x02: Event がシグナル状態化
    //  0x04: MessageQueue1 にデータを１つ送信
    //  0x08: MessageQueue2 からデータを１つ受信
    //  0x10: Thread2 が終了

    switch (th1)
    {
      case THR_CALL_REL_SEM:              expectFlag |= 0x01;    break;
      case THR_CALL_SIG_EVENT:            expectFlag |= 0x02;    break;
      case THR_CALL_SEND_MSGQUE:          expectFlag |= 0x04;    break;
      case THR_CALL_TRY_SEND_MSGQUE:      expectFlag |= 0x04;    break;
      case THR_CALL_TIMED_SEND_MSGQUE:    expectFlag |= 0x04;    break;
      case THR_CALL_JAM_MSGQUE:           expectFlag |= 0x04;    break;
      case THR_CALL_TRY_JAM_MSGQUE:       expectFlag |= 0x04;    break;
      case THR_CALL_TIMED_JAM_MSGQUE:     expectFlag |= 0x04;    break;
      case THR_CALL_RECV_MSGQUE:          expectFlag |= 0x08;    break;
      case THR_CALL_TRY_RECV_MSGQUE:      expectFlag |= 0x08;    break;
      case THR_CALL_TIMED_RECV_MSGQUE:    expectFlag |= 0x08;    break;

      case THR_CALL_SIGNAL_ALL:           expectFlag |= 0x1F;    break;
      default:                            break;
    }

    switch (th2)
    {
      case THR_CALL_REL_SEM:              expectFlag |= 0x01;    break;
      case THR_CALL_SIG_EVENT:            expectFlag |= 0x02;    break;
      case THR_CALL_SEND_MSGQUE:          expectFlag |= 0x04;    break;
      case THR_CALL_TRY_SEND_MSGQUE:      expectFlag |= 0x04;    break;
      case THR_CALL_TIMED_SEND_MSGQUE:    expectFlag |= 0x04;    break;
      case THR_CALL_JAM_MSGQUE:           expectFlag |= 0x04;    break;
      case THR_CALL_TRY_JAM_MSGQUE:       expectFlag |= 0x04;    break;
      case THR_CALL_TIMED_JAM_MSGQUE:     expectFlag |= 0x04;    break;
      case THR_CALL_RECV_MSGQUE:          expectFlag |= 0x08;    break;
      case THR_CALL_TRY_RECV_MSGQUE:      expectFlag |= 0x08;    break;
      case THR_CALL_TIMED_RECV_MSGQUE:    expectFlag |= 0x08;    break;

      case THR_CALL_SIGNAL_ALL:           expectFlag |= 0x1F;    break;
      default:                            break;
    }

    NNT_OS_LOG("expectFlag=%x\n", expectFlag);

    // Semaphore オブジェクトの内部状態を参照
    NNT_OS_LOG("(-----): Semaphore count:");
    CheckParam( m_Semaphore._count, (expectFlag & 0x1) ? 1 : 0 );

    // Event オブジェクトの内部状態を参照
    NNT_OS_LOG("(-----): Event state:");
    CheckParam( m_Event._signalState, (expectFlag & 0x2) ? true : false );

    // MessageQueue1 オブジェクトの内部状態を参照
    NNT_OS_LOG("(-----): MessageQueue1 count:");
    CheckParam( m_MessageQueue1._count, (expectFlag & 0x4) ? 1 : 0 );

    // MessageQueue2 オブジェクトの内部状態を参照
    NNT_OS_LOG("(-----): MessageQueue2 count:");
    CheckParam( m_MessageQueue2._count, (expectFlag & 0x8) ? 3 : 4 );
}

//---------------------------------------------------------------------------
// スレッド状態のチェック
void MultipleWaitHelper::ThreadStateCheck( const nn::os::ThreadType& pThread,
                                              TestThreadAction       action,
                                              TestThreadLastState    expect,
                                              bool                   apiResult,
                                              bool                   apiExpect )
{
    // 現在のスレッド状態を表示
    switch (pThread._state)
    {
        case nn::os::ThreadType::State_Initialized:
            {
                NNT_OS_LOG(" INITIALIZED");
            }
            break;

        case nn::os::ThreadType::State_Started:
            {
                NNT_OS_LOG(" STARTED");
            }
            break;

        case nn::os::ThreadType::State_Exited:
            {
                if ( action == THR_WAIT_TIMED_ANY )
                {
                    NNT_OS_LOG(" TIMEDOUT->EXITED");
                }
                else
                {
                    NNT_OS_LOG(" EXITED");
                }
            }
            break;

        case nn::os::ThreadType::State_NotInitialized:
            {
                NNT_OS_LOG(" NotInitialized");
            }
            break;

        default:
            {
                NNT_OS_LOG(" UNKNOWN");
            }
            break;
    }


    // スレッドの事後期待値を検査
    switch (expect)
    {
        // スレッド未使用
        case THR_STATE_NONE:
            {
                CheckBool( true );
            }
            break;

        // EXITED
        case THR_STATE_EXITED:
            {
                CheckParam( pThread._state, nn::os::ThreadType::State_Exited );
                NNT_OS_LOG("(-----): API return value:");
                CheckParam( apiResult, apiExpect );
            }
            break;

        // WAITING（待ちを継続）
        case THR_STATE_WAITING:
            {
                CheckParam( pThread._state, nn::os::ThreadType::State_Started );
            }
            break;

        // タイムアウトを検知して終了
        case THR_STATE_TIMEDOUT:
            {
                CheckParam( pThread._state, nn::os::ThreadType::State_Exited );
                NNT_OS_LOG("(-----): API return value:");
                CheckParam( apiResult, apiExpect );
            }
            break;

        // 想定外のエラー（テスト不具合）
        default:
            {
                NN_ASSERT( false );
            }
            break;
    }
}

//---------------------------------------------------------------------------
//  OS-API 発行
//---------------------------------------------------------------------------

bool MultipleWaitHelper::CallMultipleWaitApi( TestThreadAction    action,
                                              int                 seq,
                                              TestThreadLastState expectState,
                                              bool                expectResult)
{
    const int   maxNumObjects = 5;
    int         numObjects    = maxNumObjects;
    auto        timeout = nn::TimeSpan::FromNanoSeconds( NNT_CALIBRATE_RATE() * (expectState == THR_STATE_TIMEDOUT ? 10 : 10000 ) );

    NN_UNUSED( expectResult );

    nn::os::MultiWaitType           table;
    nn::os::MultiWaitHolderType     holder[maxNumObjects];
    nn::os::MultiWaitHolderType*    signaledHolder;

    // 多重待ちオブジェクトの構築
    InitializeMultiWait( &table );

    // セマフォ
    nn::os::InitializeMultiWaitHolder( &(holder[0]), &m_Semaphore );
    nn::os::LinkMultiWaitHolder( &table, &(holder[0]) );

    // イベント
    nn::os::InitializeMultiWaitHolder( &(holder[1]), &m_Event );
    nn::os::LinkMultiWaitHolder( &table, &(holder[1]) );

    // メッセージキュー（NotEmpty）
    nn::os::InitializeMultiWaitHolder( &(holder[2]), &m_MessageQueue1,
                                 nn::os::MessageQueueWaitType_WaitForNotEmpty);
    nn::os::LinkMultiWaitHolder( &table, &(holder[2]) );

    // メッセージキュー（NotFull）
    nn::os::InitializeMultiWaitHolder( &(holder[3]), &m_MessageQueue2,
                                 nn::os::MessageQueueWaitType_WaitForNotFull);
    nn::os::LinkMultiWaitHolder( &table, &(holder[3]) );

    if (( nn::os::GetCurrentThread() == &m_Thread1 ) &&
        ( g_MWHelper.GetThreadAction2() != THR_NOTUSE ))
    {
        // スレッド
        nn::os::InitializeMultiWaitHolder( &(holder[4]), &m_Thread2 );
        nn::os::LinkMultiWaitHolder( &table, &(holder[4]) );
    }
    else
    {
        --numObjects;
    }

    bool result = true;

    SEQ_CHECK( seq );
    NNT_OS_LOG("Call ");

    // スレッド事前条件の検査
    switch (action)
    {
        // NOTUSE 系（スレッドを使用しない）
        case THR_NOTUSE:
            {
                NN_ASSERT( false );
            }
            break;

        // WaitAny() 待ち状態（無期限）
        case THR_WAIT_ANY:
            {
                NNT_OS_LOG("WaitAny(num=%d) goes WAIT\n", numObjects);
                signaledHolder = nn::os::WaitAny( &table );
                result = (signaledHolder == NULL ? false : true);
            }
            break;

        // TimedWaitAny() 待ち状態（時限あり）
        case THR_WAIT_TIMED_ANY:
            {
                NNT_OS_LOG("TimedWaitAny(num=%d) goes WAIT\n", numObjects);
                signaledHolder = nn::os::TimedWaitAny( &table, timeout );
                result = (signaledHolder == NULL ? false : true);

                // タイムアウトによる待ち解除なら、その旨を返す
                if ( result == false )
                {
                    return result;
                }
            }
            break;

        // TimedWaitAny() 待ち状態（負のタイムアウト値）
        case THR_WAIT_TIMED_ANY_MINUS:
            {
                NNT_OS_LOG("TimedWaitAny(num=%d) goes WAIT (timeout=-1nsec)\n", numObjects);
                signaledHolder = nn::os::TimedWaitAny( &table, nn::TimeSpan::FromNanoSeconds( -1 ) );
                result = (signaledHolder == NULL ? false : true);

                // タイムアウトによる待ち解除なら、その旨を返す
                if ( result == false )
                {
                    return result;
                }
            }
            break;

        // WaitAny() の発行
        case THR_CALL_ANY:
            {
                NNT_OS_LOG("WaitAny(num=%d)\n", numObjects);
                signaledHolder = nn::os::WaitAny( &table );
                result = (signaledHolder == NULL ? false : true);
            }
            break;

        // TryWaitAny() の発行
        case THR_CALL_TRY_ANY:
            {
                NNT_OS_LOG("TryWaitAny(num=%d)\n", numObjects);
                signaledHolder = nn::os::TryWaitAny( &table );
                result = (signaledHolder == NULL ? false : true);
            }
            break;

        // TimedWaitAny() の発行
        case THR_CALL_TIMED_ANY:
            {
                NNT_OS_LOG("TimedWaitAny(num=%d)\n", numObjects);
                signaledHolder = nn::os::TimedWaitAny( &table, timeout );
                result = (signaledHolder == NULL ? false : true);

                // タイムアウトによる待ち解除なら、その旨を返す
                if ( result == false )
                {
                    return result;
                }
             }
            break;

        // TimedWaitAny() の発行（負のタイムアウト値）
        case THR_CALL_TIMED_ANY_MINUS:
            {
                NNT_OS_LOG("TimedWaitAny(num=%d) (timeout=-1nsec)\n", numObjects);
                signaledHolder = nn::os::TimedWaitAny( &table, nn::TimeSpan::FromNanoSeconds( -1 ) );
                result = (signaledHolder == NULL ? false : true);

                // タイムアウトによる待ち解除なら、その旨を返す
                if ( result == false )
                {
                    return result;
                }
             }
            break;


        // ReleaseSemaphore() の発行
        case THR_CALL_REL_SEM:
            {
                NNT_OS_LOG("ReleaseSemaphore()\n");
                nn::os::ReleaseSemaphore( &m_Semaphore );
                int semCount = nn::os::GetCurrentSemaphoreCount( &m_Semaphore );
                NN_UNUSED( semCount );
            }
            break;

        // SignalEvent() の発行
        case THR_CALL_SIG_EVENT:
            {
                NNT_OS_LOG("SignalEvent()\n");
                nn::os::SignalEvent( &m_Event );
            }
            break;



        // SendMessageQueue() の発行
        case THR_CALL_SEND_MSGQUE:
            {
                NNT_OS_LOG("SendMessageQueue()\n");
                nn::os::SendMessageQueue( &m_MessageQueue1, 0x1234 );
            }
            break;

        // TrySendMessageQueue() の発行
        case THR_CALL_TRY_SEND_MSGQUE:
            {
                NNT_OS_LOG("TrySendMessageQueue()\n");
                result = nn::os::TrySendMessageQueue(&m_MessageQueue1, 0x1234);
            }
            break;

        // TimedSendMessageQueue() の発行
        case THR_CALL_TIMED_SEND_MSGQUE:
            {
                NNT_OS_LOG("TimedSendMessageQueue()\n");
                result = nn::os::TimedSendMessageQueue(&m_MessageQueue1, 0x1234, timeout );

                // タイムアウトによる待ち解除なら、その旨を返す
                if ( result == false )
                {
                    return result;
                }
            }
            break;


        // JamMessageQueue() の発行
        case THR_CALL_JAM_MSGQUE:
            {
                NNT_OS_LOG("JamMessageQueue()\n");
                nn::os::JamMessageQueue( &m_MessageQueue1, 0x5678 );
            }
            break;

        // TryJamMessageQueue() の発行
        case THR_CALL_TRY_JAM_MSGQUE:
            {
                NNT_OS_LOG("TryJamMessageQueue()\n");
                result = nn::os::TryJamMessageQueue( &m_MessageQueue1, 0x5678 );
            }
            break;

        // TimedJamMessageQueue() の発行
        case THR_CALL_TIMED_JAM_MSGQUE:
            {
                NNT_OS_LOG("TimedJamMessageQueue()\n");
                result = nn::os::TimedJamMessageQueue( &m_MessageQueue1, 0x5678, timeout );

                // タイムアウトによる待ち解除なら、その旨を返す
                if ( result == false )
                {
                    return result;
                }
            }
            break;


        // ReceiveMessageQueue() の発行
        case THR_CALL_RECV_MSGQUE:
            {
                uintptr_t   data;
                NNT_OS_LOG("ReceiveMessageQueue()\n");
                nn::os::ReceiveMessageQueue( &data, &m_MessageQueue2 );
            }
            break;

        // TryReceiveMessageQueue() の発行
        case THR_CALL_TRY_RECV_MSGQUE:
            {
                uintptr_t   data;
                NNT_OS_LOG("TryReceiveMessageQueue()\n");
                result = nn::os::TryReceiveMessageQueue( &data, &m_MessageQueue2 );
            }
            break;

        // TimedReceiveMessageQueue() の発行
        case THR_CALL_TIMED_RECV_MSGQUE:
            {
                uintptr_t   data;
                NNT_OS_LOG("TimedReceiveMessageQueue()\n");
                result = nn::os::TimedReceiveMessageQueue( &data, &m_MessageQueue2, timeout );

                // タイムアウトによる待ち解除なら、その旨を返す
                if ( result == false )
                {
                    return result;
                }
            }
            break;


        // 全オブジェクトをシグナル化
        case THR_CALL_SIGNAL_ALL:
            {
                NNT_OS_LOG("ReleaseSemaphore()\n");
                nn::os::ReleaseSemaphore( &m_Semaphore );
                int semCount = nn::os::GetCurrentSemaphoreCount( &m_Semaphore );
                NN_UNUSED( semCount );

                NNT_OS_LOG("SignalEvent()\n");
                nn::os::SignalEvent( &m_Event );

                NNT_OS_LOG("TrySendMessageQueue()\n");
                nn::os::TrySendMessageQueue( &m_MessageQueue1, 0xABCD );

                uintptr_t   data;
                NNT_OS_LOG("TryReceiveMessageQueue()\n");
                nn::os::TryReceiveMessageQueue( &data, &m_MessageQueue2 );
            }
            break;


        // 想定外のエラー（テスト不具合）
        default:
            {
                NN_ASSERT( false );
            }
            break;
    }

    for (int i = 0; i < numObjects; ++i)
    {
        nn::os::UnlinkMultiWaitHolder( &holder[i] );
        nn::os::FinalizeMultiWaitHolder( &holder[i] );
    }
    nn::os::FinalizeMultiWait( &table );

    return result;
}   // NOLINT(readability/fn_size)

//---------------------------------------------------------------------------
//  テスト用スレッド関数の本体
//---------------------------------------------------------------------------
void    ThreadFunc1(void* exinf)
{
    TestThreadAction    action      = g_MWHelper.GetThreadAction1();
    TestThreadLastState expectState = g_MWHelper.GetThreadExpectState1();
    bool                expect      = g_MWHelper.GetThreadExpect1();
    bool                result;

    NN_UNUSED( exinf );

    SEQ_SYNC_WAIT( NNT_POINT_THREAD1_START );

    // スレッド起動
    SEQ_CHECK( g_seq + 1 );
    NNT_OS_LOG("Begin of thread1:");
    CheckBool( true );

    // Thread1 から API を発行
    g_MWHelper.SetApiResult1( false );
    result = g_MWHelper.CallMultipleWaitApi( action, g_seq + 2, expectState, expect );
    g_MWHelper.SetApiResult1( result );

    // Cleanuped ならスレッド終了（THR_STATE_WAITING だった場合など）
    if ( g_MWHelper.IsCleanuped() )
    {
        NNT_OS_LOG("(-----): Thread1: Canceled\n");
        return;
    }

    SEQ_SYNC_ACK( NNT_POINT_THREAD1_CALLED );
    SEQ_SYNC_WAIT( NNT_POINT_THREAD1_CALLED );

    // スレッドの事後状態別の後処理
    switch (expectState)
    {
        // API から普通に戻ってきた場合
        case THR_STATE_EXITED:
            {
                SEQ_CHECK( g_seq + 5 );
                NNT_OS_LOG("Thread1: return from API:");
                CheckParam( result, expect );
            }
            break;

        case THR_STATE_TIMEDOUT:
            break;

        // 想定外のエラー（テスト不具合）
        case THR_STATE_WAITING:
        default:
            CheckBool( false );
            break;
    }

    // スレッド終了
    SEQ_CHECK( expectState == THR_STATE_TIMEDOUT ? g_seq + 9 : g_seq + 6 );
    NNT_OS_LOG("End of thread1:");
    CheckBool( true );

    SEQ_SYNC_ACK( NNT_POINT_THREAD1_END );
}


void    ThreadFunc2(void* exinf)
{
    TestThreadAction    action      = g_MWHelper.GetThreadAction2();
    TestThreadLastState expectState = g_MWHelper.GetThreadExpectState2();
    bool                expect      = g_MWHelper.GetThreadExpect2();
    bool                result;

    NN_UNUSED( exinf );

    SEQ_SYNC_WAIT( NNT_POINT_THREAD2_START );

    // スレッド起動
    SEQ_CHECK( g_seq + 3 );
    NNT_OS_LOG("Begin of thread2:");
    CheckBool( true );

    // Thread2 から API を発行
    g_MWHelper.SetApiResult2( false );
    result = g_MWHelper.CallMultipleWaitApi( action, g_seq + 4, expectState, expect );
    g_MWHelper.SetApiResult2( result );

    // Cleanuped ならスレッド終了（THR_STATE_WAITING だった場合など）
    if ( g_MWHelper.IsCleanuped() )
    {
        NNT_OS_LOG("(-----): Thread2: Canceled\n");
        return;
    }

    SEQ_SYNC_ACK( NNT_POINT_THREAD2_CALLED );
    SEQ_SYNC_WAIT( NNT_POINT_THREAD2_CALLED );

    // スレッドの事後状態別の後処理
    switch (expectState)
    {
        // API から普通に戻ってきた場合
        case THR_STATE_EXITED:
            {
                SEQ_CHECK( g_seq + 7 );
                NNT_OS_LOG("Thread2: return from API:");
                CheckParam( result, expect );
            }
            break;

        case THR_STATE_TIMEDOUT:
            break;

        // 想定外のエラー（テスト不具合）
        case THR_STATE_WAITING:
        default:
            CheckBool( false );
            break;
    }

    // スレッド終了
    SEQ_CHECK( expectState == THR_STATE_TIMEDOUT? g_seq + 10 : g_seq + 8 );
    NNT_OS_LOG("End of thread2:");
    CheckBool( true );

    SEQ_SYNC_ACK( NNT_POINT_THREAD2_END );
}

//---------------------------------------------------------------------------

int    MultipleWaitHelper::GetLastSeq( TestThreadLastState th1ex,
                                       TestThreadLastState th2ex )
{
    static const int lastSeqTable[THR_STATE_MAX][THR_STATE_MAX] =
    {
    //  th2: NONE,WAITING, EXITED, TIMEDOUT
        {       0,      0,      0,      0,      },  // NONE
        {       2,      4,      8,      10,     },  // WAITING
        {       6,      6,      8,      10,     },  // EXITED
        {       9,      9,      9,      10,     },  // TIMEDOUT
    };                                              // th1

    return g_seq + lastSeqTable[th1ex][th2ex];
}

//---------------------------------------------------------------------------
//  doTest() 関数
//---------------------------------------------------------------------------

void    MultipleWaitHelper::doTest( TestThreadAction    th1,
                                    TestThreadAction    th2,
                                    bool                rslt1,
                                    bool                rslt2,
                                    TestThreadLastState th1ex,
                                    TestThreadLastState th2ex,
                                    const char*         functionName,
                                    int                 line)
{
    // シーケンス番号を次の 100番台 へ進める
    g_seq = g_Sequence.Get();
    g_seq = ((g_seq + 100) / 100) * 100;

    // 個別のテスト結果をクリア
    CLEAR_GOOGLE_TEST();

    // テスト用 API の発行元情報を退避
    g_Sequence.SetFunctionName( functionName );
    g_Sequence.SetLineNumber( line );

    // Cleanup 前の状態へ
    m_isCleanuped = false;
    g_MWHelper.SetApiResult1( true );
    g_MWHelper.SetApiResult2( true );

    // スレッドの生成
    g_seq = (g_seq / 100) * 100;
    SEQ_SET( g_seq );
    NNT_OS_LOG("Start Test\n");
    CreateThread1( th1 );
    CreateThread2( th2 );

    // 多重待ちオブジェクトの Initialize
    InitializeMultipleWait();

    // スレッドの状態を構築
    SetThreadInfo1( th1, th1ex, rslt1 );
    SetThreadInfo2( th2, th2ex, rslt2 );

    // 順番にテストシーケンスを指示する
    SEQ_SYNC_INIT();

    StartThread1( th1 );
    StartThread2( th2 );
    SEQ_SYNC_SIGNAL( NNT_POINT_THREAD1_START );
    SEQ_SYNC_DELAY( 20, NNT_POINT_THREAD1_CALLED );

    if (th2ex != THR_STATE_NONE)
    {
        SEQ_SYNC_SIGNAL( NNT_POINT_THREAD2_START );
        SEQ_SYNC_DELAY( 20, NNT_POINT_THREAD2_CALLED );
    }

    SEQ_SYNC_SIGNAL( NNT_POINT_THREAD1_CALLED );
    SEQ_SYNC_DELAY( 20, NNT_POINT_THREAD1_END );

    if (th2ex != THR_STATE_NONE)
    {
        SEQ_SYNC_SIGNAL( NNT_POINT_THREAD2_CALLED );
        SEQ_SYNC_DELAY( 20, NNT_POINT_THREAD2_END );
    }

    // タイムアウト系の API は、もう片方のスレッドで同期が成立しない場合、
    // この時点で既にタイムアウトが成立し、対象スレッドは
    // THR_TIMEDOUT_EXITED 状態という形で既に Exited になっている。

    // スレッド１の状態を検査
    SEQ_WAIT_UNTIL( GetLastSeq(th1ex, th2ex), g_seq + 11 );
    NNT_OS_LOG("Thread1 state check:");
    ThreadStateCheck( m_Thread1, th1, th1ex, GetApiResult1(), rslt1 );

    // スレッド２の状態を検査
    SEQ_NONE();
    NNT_OS_LOG("Thread2 state check:");
    ThreadStateCheck( m_Thread2, th2, th2ex, GetApiResult2(), rslt2 );

    // 多重待ちの状態を検査
    SEQ_NONE();
    NNT_OS_LOG("MultipleWait state check:");
    MultipleWaitCheck( th1, th2 );

    // 多重待ち状態を初期状態へ
    SEQ_NONE();
    NNT_OS_LOG("Cleanup MultipleWait and Threads\n");
    Cleanup();

    // 多重待ちオブジェクトの Finalize
    FinalizeMultipleWait();

    // スレッドオブジェクトの破棄
    DestroyThread1( th1 );
    DestroyThread2( th2 );

    // 個別のテスト結果で GoogleTest に判定を通知
    JUDGE_GOOGLE_TEST();
}

}}} // namespace nnt::os::multiWait

