﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//---------------------------------------------------------------------------
//  MultipleWait 関連機能のテスト
//---------------------------------------------------------------------------

#include "../Common/test_Pragma.h"

#include <nn/os/os_Config.h>
#include <nn/nn_SdkText.h>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/os.h>
#include "../Common/test_Helper.h"
#include "../Common/test_Calibration.h"
#include "test_MultipleWaitHelper.h"

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <thread>

//---------------------------------------------------------------------------

namespace nnt { namespace os { namespace multiWait {

NN_ALIGNAS(4096) char g_ThreadStack[ 0x4000 ];

//---------------------------------------------------------------------------
// InitializeMultipleWait 直後の初期状態チェック

TEST(InitializeMultipleWait, test_MultipleWaitInitialState)
{
    // InitializeMultipleWait 初期状態チェック
    doTest1(THR_NOTUSE,             // Thread1 は未使用
            true,                   // Thread1 での API 返値
            THR_STATE_NONE);        // Thread1 は未使用
}

//---------------------------------------------------------------------------
// 単独の WaitAny() テスト

TEST(SimpleWaitAny, test_WaitAny)
{
    // WaitAny 待ち（１スレッド）
    doTest1(THR_WAIT_ANY,           // Thread1 は WaitAny()
            true,                   // Thread1 での API 返値
            THR_STATE_WAITING);     // Thread1 は待ち
}

TEST(SimpleWaitAny, test_ReleaseSemaphoreAndWaitAny)
{
    // Semaphore 条件を成立させてから、WaitAny 発行（すぐに復帰）
    doTest2(THR_CALL_REL_SEM,       // Thread1 は ReleaseSemaphore()
            THR_CALL_ANY,           // Thread2 は WaitAny()
            true,                   // Thread1 での API 返値
            true,                   // Thread2 での API 返値
            THR_STATE_EXITED,       // Thread1 は終了
            THR_STATE_EXITED);      // Thread2 は終了
}

TEST(SimpleWaitAny, test_SignalEventAndWaitAny)
{
    // Event 条件を成立させてから、WaitAny 発行（すぐに復帰）
    doTest2(THR_CALL_SIG_EVENT,     // Thread1 は SignalEvent()
            THR_CALL_ANY,           // Thread2 は WaitAny()
            true,                   // Thread1 での API 返値
            true,                   // Thread2 での API 返値
            THR_STATE_EXITED,       // Thread1 は終了
            THR_STATE_EXITED);      // Thread2 は終了
}

TEST(SimpleWaitAny, test_SendMessageAndWaitAny)
{
    // MessageQueue1 条件を成立させてから、WaitAny 発行（すぐに復帰）
    doTest2(THR_CALL_SEND_MSGQUE,   // Thread1 は SendMessageQueue()
            THR_CALL_ANY,           // Thread2 は WaitAny()
            true,                   // Thread1 での API 返値
            true,                   // Thread2 での API 返値
            THR_STATE_EXITED,       // Thread1 は終了
            THR_STATE_EXITED);      // Thread2 は終了
}

TEST(SimpleWaitAny, test_ReceiveMessageAndWaitAny)
{
    // MessageQueue2 条件を成立させてから、WaitAny 発行（すぐに復帰）
    doTest2(THR_CALL_RECV_MSGQUE,   // Thread1 は ReceiveMessageQueue()
            THR_CALL_ANY,           // Thread2 は WaitAny()
            true,                   // Thread1 での API 返値
            true,                   // Thread2 での API 返値
            THR_STATE_EXITED,       // Thread1 は終了
            THR_STATE_EXITED);      // Thread2 は終了
}

//---------------------------------------------------------------------------
// 単独の TryWaitAny() テスト

#if 0
TEST(SimpleTryWaitAny, test_TryWaitAny)
{
    // TryWaitAny 発行（１スレッド）
    doTest1(THR_CALL_TRY_ANY,       // Thread1 は TryWaitAny()
            false,                  // Thread1 での API 返値
            THR_STATE_EXITED);      // Thread1 は待ち
}
#endif

TEST(SimpleTryWaitAny, test_ReleaseSemaphoreAndTryWaitAny)
{
    // Semaphore 条件を成立させてから、TryWaitAny 発行（すぐに復帰）
    doTest2(THR_CALL_REL_SEM,       // Thread1 は ReleaseSemaphore()
            THR_CALL_TRY_ANY,       // Thread2 は TryWaitAny()
            true,                   // Thread1 での API 返値
            true,                   // Thread2 での API 返値
            THR_STATE_EXITED,       // Thread1 は終了
            THR_STATE_EXITED);      // Thread2 は終了
}

TEST(SimpleTryWaitAny, test_SignalEventAndTryWaitAny)
{
    // Event 条件を成立させてから、TryWaitAny 発行（すぐに復帰）
    doTest2(THR_CALL_SIG_EVENT,     // Thread1 は SignalEvent()
            THR_CALL_TRY_ANY,       // Thread2 は TryWaitAny()
            true,                   // Thread1 での API 返値
            true,                   // Thread2 での API 返値
            THR_STATE_EXITED,       // Thread1 は終了
            THR_STATE_EXITED);      // Thread2 は終了
}

TEST(SimpleTryWaitAny, test_SendMessageAndTryWaitAny)
{
    // MessageQueue1 条件を成立させてから、TryWaitAny 発行（すぐに復帰）
    doTest2(THR_CALL_SEND_MSGQUE,   // Thread1 は SendMessageQueue()
            THR_CALL_TRY_ANY,       // Thread2 は TryWaitAny()
            true,                   // Thread1 での API 返値
            true,                   // Thread2 での API 返値
            THR_STATE_EXITED,       // Thread1 は終了
            THR_STATE_EXITED);      // Thread2 は終了
}

TEST(SimpleTryWaitAny, test_ReceiveMessageAndTryWaitAny)
{
    // MessageQueue2 条件を成立させてから、TryWaitAny 発行（すぐに復帰）
    doTest2(THR_CALL_RECV_MSGQUE,   // Thread1 は ReceiveMessageQueue()
            THR_CALL_TRY_ANY,       // Thread2 は TryWaitAny()
            true,                   // Thread1 での API 返値
            true,                   // Thread2 での API 返値
            THR_STATE_EXITED,       // Thread1 は終了
            THR_STATE_EXITED);      // Thread2 は終了
}


//---------------------------------------------------------------------------
// 単独の TimedWaitAny() テスト

TEST(SimpleTimedWaitAny, test_TimedWaitAny)
{
    // TimedWaitAny 発行（１スレッド）
    doTest1(THR_WAIT_TIMED_ANY,     // Thread1 は TimedWaitAny()
            false,                  // Thread1 での API 返値
            THR_STATE_TIMEDOUT);    // Thread1 は待ち
}

TEST(SimpleTimedWaitAny, test_ReleaseSemaphoreAndTimedWaitAny)
{
    // Semaphore 条件を成立させてから、TimedWaitAny 発行（すぐに復帰）
    doTest2(THR_CALL_REL_SEM,       // Thread1 は ReleaseSemaphore()
            THR_CALL_TIMED_ANY,     // Thread2 は TimedWaitAny()
            true,                   // Thread1 での API 返値
            true,                   // Thread2 での API 返値
            THR_STATE_EXITED,       // Thread1 は終了
            THR_STATE_EXITED);      // Thread2 は終了
}

TEST(SimpleTimedWaitAny, test_SignalEventAndTimedWaitAny)
{
    // Event 条件を成立させてから、TimedWaitAny 発行（すぐに復帰）
    doTest2(THR_CALL_SIG_EVENT,     // Thread1 は SignalEvent()
            THR_CALL_TIMED_ANY,     // Thread2 は TimedWaitAny()
            true,                   // Thread1 での API 返値
            true,                   // Thread2 での API 返値
            THR_STATE_EXITED,       // Thread1 は終了
            THR_STATE_EXITED);      // Thread2 は終了
}

TEST(SimpleTimedWaitAny, test_SendMessageAndTimedWaitAny)
{
    // MessageQueue1 条件を成立させてから、TimedWaitAny 発行（すぐに復帰）
    doTest2(THR_CALL_SEND_MSGQUE,   // Thread1 は SendMessageQueue()
            THR_CALL_TIMED_ANY,     // Thread2 は TimedWaitAny()
            true,                   // Thread1 での API 返値
            true,                   // Thread2 での API 返値
            THR_STATE_EXITED,       // Thread1 は終了
            THR_STATE_EXITED);      // Thread2 は終了
}

TEST(SimpleTimedWaitAny, test_ReceiveMessageAndTimedWaitAny)
{
    // MessageQueue2 条件を成立させてから、TimedWaitAny 発行（すぐに復帰）
    doTest2(THR_CALL_RECV_MSGQUE,   // Thread1 は ReceiveMessageQueue()
            THR_CALL_TIMED_ANY,     // Thread2 は TimedWaitAny()
            true,                   // Thread1 での API 返値
            true,                   // Thread2 での API 返値
            THR_STATE_EXITED,       // Thread1 は終了
            THR_STATE_EXITED);      // Thread2 は終了
}


// 負のタイムアウト値を許容することになったときのためにテストを残しておく
#if 0
//---------------------------------------------------------------------------
// 単独の TimedWaitAny() テスト（負のタイムアウト値）

TEST(SimpleTimedWaitAnyWithMinusTimeout, test_TimedWaitAnyWithMinusTimeout)
{
    // TimedWaitAny 発行（１スレッド）
    doTest1(THR_WAIT_TIMED_ANY_MINUS, // Thread1 は TimedWaitAny()
            false,                    // Thread1 での API 返値
            THR_STATE_TIMEDOUT);      // Thread1 は待ち
}

TEST(SimpleTimedWaitAnyWithMinusTimeout, test_SignalEventAndTimedWaitAnyWithMinusTimeout)
{
    // Event 条件を成立させてから、TimedWaitAny 発行（すぐに復帰）
    doTest2(THR_CALL_SIG_EVENT,       // Thread1 は SignalEvent()
            THR_CALL_TIMED_ANY_MINUS, // Thread2 は TimedWaitAny()
            true,                     // Thread1 での API 返値
            true,                     // Thread2 での API 返値
            THR_STATE_EXITED,         // Thread1 は終了
            THR_STATE_EXITED);        // Thread2 は終了
}

#else
// 負のタイムアウト値を許容しない場合の DEATH TEST

TEST(SimpleTimedWaitAnyWithMinusTimeoutDeathTest, test_TimedWaitAnyWithMinusTimeoutDeathTest)
{
    // TimedWaitAny 発行（１スレッド）
    EXPECT_DEATH_IF_SUPPORTED(
        doTest1(THR_WAIT_TIMED_ANY_MINUS, // Thread1 は TimedWaitAny()
                false,                    // Thread1 での API 返値
                THR_STATE_TIMEDOUT)       // Thread1 は待ち
    , "");
}


#endif


//---------------------------------------------------------------------------
// WaitAny 待機中スレッドを起床する

TEST(CombinedWaitAny, test_WaitAnyAndReleaseSemaphore)
{
    // WaitAny 発行後に Semaphore 条件を成立させて多重待ちを起床
    doTest2(THR_WAIT_ANY,           // Thread1 は WaitAny()
            THR_CALL_REL_SEM,       // Thread2 は ReleaseSemaphore()
            true,                   // Thread1 での API 返値
            true,                   // Thread2 での API 返値
            THR_STATE_EXITED,       // Thread1 は終了
            THR_STATE_EXITED);      // Thread2 は終了
}

TEST(CombinedWaitAny, test_WaitAnyAndSignalEvent)
{
    // WaitAny 発行後に Event 条件を成立させて多重待ちを起床
    doTest2(THR_WAIT_ANY,           // Thread1 は WaitAny()
            THR_CALL_SIG_EVENT,     // Thread2 は SignalEvent()
            true,                   // Thread1 での API 返値
            true,                   // Thread2 での API 返値
            THR_STATE_EXITED,       // Thread1 は終了
            THR_STATE_EXITED);      // Thread2 は終了
}

TEST(CombinedWaitAny, test_WaitAnyAndSendMessage)
{
    // WaitAny 発行後に MessageQueue1 条件を成立させて多重待ちを起床
    doTest2(THR_WAIT_ANY,           // Thread1 は WaitAny()
            THR_CALL_SEND_MSGQUE,   // Thread2 は SendMessageQueue()
            true,                   // Thread1 での API 返値
            true,                   // Thread2 での API 返値
            THR_STATE_EXITED,       // Thread1 は終了
            THR_STATE_EXITED);      // Thread2 は終了
}

TEST(CombinedWaitAny, test_WaitAnyAndReceiveMessage)
{
    // WaitAny 発行後に MessageQueue2 条件を成立させて多重待ちを起床
    doTest2(THR_WAIT_ANY,           // Thread1 は WaitAny()
            THR_CALL_RECV_MSGQUE,   // Thread2 は ReceiveMessageQueue()
            true,                   // Thread1 での API 返値
            true,                   // Thread2 での API 返値
            THR_STATE_EXITED,       // Thread1 は終了
            THR_STATE_EXITED);      // Thread2 は終了
}

//---------------------------------------------------------------------------
// TimedWaitAny 待機中スレッドを起床する

TEST(CombinedTimedWaitAny, test_TimedWaitAnyAndReleaseSemaphore)
{
    // TimedWaitAny 発行後に Semaphore 条件を成立させて多重待ちを起床
    doTest2(THR_WAIT_TIMED_ANY,     // Thread1 は TimedWaitAny()
            THR_CALL_REL_SEM,       // Thread2 は ReleaseSemaphore()
            true,                   // Thread1 での API 返値
            true,                   // Thread2 での API 返値
            THR_STATE_EXITED,       // Thread1 は終了
            THR_STATE_EXITED);      // Thread2 は終了
}

TEST(CombinedTimedWaitAny, test_TimedWaitAnyAndSignalEvent)
{
    // TimedWaitAny 発行後に Event 条件を成立させて多重待ちを起床
    doTest2(THR_WAIT_TIMED_ANY,     // Thread1 は TimedWaitAny()
            THR_CALL_SIG_EVENT,     // Thread2 は SignalEvent()
            true,                   // Thread1 での API 返値
            true,                   // Thread2 での API 返値
            THR_STATE_EXITED,       // Thread1 は終了
            THR_STATE_EXITED);      // Thread2 は終了
}

TEST(CombinedTimedWaitAny, test_TimedWaitAnyAndSendMessage)
{
    // TimedWaitAny 発行後に MessageQueue1 条件を成立させて多重待ちを起床
    doTest2(THR_WAIT_TIMED_ANY,     // Thread1 は TimedWaitAny()
            THR_CALL_SEND_MSGQUE,   // Thread2 は SendMessageQueue()
            true,                   // Thread1 での API 返値
            true,                   // Thread2 での API 返値
            THR_STATE_EXITED,       // Thread1 は終了
            THR_STATE_EXITED);      // Thread2 は終了
}

TEST(CombinedTimedWaitAny, test_TimedWaitAnyAndReceiveMessage)
{
    // TimedWaitAny 発行後に MessageQueue2 条件を成立させて多重待ちを起床
    doTest2(THR_WAIT_TIMED_ANY,     // Thread1 は TimedWaitAny()
            THR_CALL_RECV_MSGQUE,   // Thread2 は ReceiveMessageQueue()
            true,                   // Thread1 での API 返値
            true,                   // Thread2 での API 返値
            THR_STATE_EXITED,       // Thread1 は終了
            THR_STATE_EXITED);      // Thread2 は終了
}


//---------------------------------------------------------------------------
//  Initialize -> Link -> Unlink -> Finalize の確認
//  割込みイベントを含まないテストケース。
TEST(MultiWaitSequentialTest, test_SequentialTest)
{
    nn::os::MultiWaitType           multiWait;
    nn::os::MultiWaitHolderType     holder1;
    nn::os::MultiWaitHolderType     holder2;
    nn::os::MultiWaitHolderType*    signaledHolder;
    nn::os::EventType               event1;
    nn::os::EventType               event2;

    // 個別のテスト結果をクリア
    CLEAR_GOOGLE_TEST();

    // 最初からシグナル状態のイベントを用意
    nn::os::InitializeEvent(&event1, true,  nn::os::EventClearMode_ManualClear);

    // 最初は非シグナル状態のイベントを用意
    nn::os::InitializeEvent(&event2, false, nn::os::EventClearMode_ManualClear);

    // 多重待ちオブジェクトリストを構築
    nn::os::InitializeMultiWait( &multiWait );

    nn::os::InitializeMultiWaitHolder( &holder1, &event1 );
    nn::os::LinkMultiWaitHolder( &multiWait, &holder1 );
    nn::os::SetMultiWaitHolderUserData( &holder1, 0x11111111 );

    nn::os::InitializeMultiWaitHolder( &holder2, &event2 );
    nn::os::LinkMultiWaitHolder( &multiWait, &holder2 );
    nn::os::SetMultiWaitHolderUserData( &holder2, 0x22222222 );

    // テストその１
    // event1, event2 のうち event1 で多重待ち解除するケース
    {
        SEQ_NONE();
        NNT_OS_LOG("WaitAny( {event1, event2} ) expect=true");

        // WaitAny を発行（すぐに戻ってくる）
        signaledHolder = nn::os::WaitAny( &multiWait );

        // シグナルが成立している Holder オブジェクトが正しいか？
        CheckBool( signaledHolder == &holder1 );

        // ユーザデータ値が正しいか？
        auto value = nn::os::GetMultiWaitHolderUserData(signaledHolder);
        SEQ_NONE();
        NNT_OS_LOG("    SignaledHolder's UserData=0x%08x", value);
        CheckBool( value == 0x11111111 );
    }

    // リストから外す
    nn::os::UnlinkMultiWaitHolder( &holder1 );
    nn::os::UnlinkMultiWaitHolder( &holder2 );

    // ファイナライズ
    nn::os::FinalizeMultiWaitHolder( &holder1 );
    nn::os::FinalizeMultiWaitHolder( &holder2 );
    nn::os::FinalizeMultiWait( &multiWait );

    // 個別のテスト結果で GoogleTest に判定を通知
    JUDGE_GOOGLE_TEST();
}


#if defined( NN_BUILD_CONFIG_OS_WIN32 ) || \
    (defined( NN_BUILD_CONFIG_OS_HORIZON ) && defined( NN_BUILD_CONFIG_HARDWARE_BDSLIMX6 )) || \
    (defined( NN_BUILD_CONFIG_OS_HORIZON ) && defined( NN_BUILD_CONFIG_HARDWARE_JETSONTK1 )) || \
    (defined( NN_BUILD_CONFIG_OS_HORIZON ) && defined( NN_BUILD_CONFIG_HARDWARE_JETSONTK2 )) || \
    (defined( NN_BUILD_CONFIG_OS_HORIZON ) && defined( NN_BUILD_CONFIG_SOC_TEGRA_X1 )) || \
    (defined( NN_BUILD_CONFIG_OS_HORIZON ) && defined( NN_BUILD_CONFIG_HARDWARE_JUNO )) || \
    (defined( NN_BUILD_CONFIG_OS_HORIZON ) && defined( NN_BUILD_CONFIG_HARDWARE_SMMA53 ))
//---------------------------------------------------------------------------
//  Initialize -> Link -> Unlink -> Finalize の確認
//  割込みイベントを含むテストケース。
void test_SequentialTestWithKernelObject_HelperThread(void* p)
{
    auto* sem = reinterpret_cast<nn::os::SemaphoreType*>( p );

    nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(50) );
    nn::os::ReleaseSemaphore( sem );
}

TEST(MultiWaitSequentialTest, test_SequentialTestWithKernelObject)
{
    nn::os::MultiWaitType           multiWait;
    nn::os::MultiWaitHolderType     holder1;
    nn::os::MultiWaitHolderType     holder2;
    nn::os::MultiWaitHolderType     holder3;
    nn::os::MultiWaitHolderType*    signaledHolder = NULL;
    nn::os::EventType               event;
    nn::os::SemaphoreType           semaphore;
    nn::os::TimerEventType          timerEvent;

    // 個別のテスト結果をクリア
    CLEAR_GOOGLE_TEST();

    // 最初は非シグナル状態のイベントを用意
    nn::os::InitializeEvent(&event, false, nn::os::EventClearMode_ManualClear);

    // 初期カウント 0 のセマフォを用意
    nn::os::InitializeSemaphore(&semaphore, 0, 1);

    // 100msec おきにシグナル化される ManualReset の割込みイベント
    nn::os::InitializeTimerEvent(&timerEvent, nn::os::EventClearMode_ManualClear);
    nn::os::StartPeriodicTimerEvent(&timerEvent, nn::TimeSpan::FromMilliSeconds(100), nn::TimeSpan::FromMilliSeconds(100));

    // 多重待ちオブジェクトリストを構築
    nn::os::InitializeMultiWait( &multiWait );

    nn::os::InitializeMultiWaitHolder( &holder1, &event );
    nn::os::LinkMultiWaitHolder( &multiWait, &holder1 );
    nn::os::SetMultiWaitHolderUserData( &holder1, 0x11111111 );

    nn::os::InitializeMultiWaitHolder( &holder2, &semaphore );
    nn::os::LinkMultiWaitHolder( &multiWait, &holder2 );
    nn::os::SetMultiWaitHolderUserData( &holder2, 0x22222222 );

    nn::os::InitializeMultiWaitHolder( &holder3, &timerEvent );
    nn::os::LinkMultiWaitHolder( &multiWait, &holder3 );
    nn::os::SetMultiWaitHolderUserData( &holder3, 0x33333333 );

    // テストその１
    // event, sem, timerEvent のうち、timerEvent で多重待ち解除するケース
    // timerEvent は多重待ちに入る前に一旦クリアされる。
    // WaitAny, TryWaitAny, TimedWaitAny 全てをテスト
    NNT_OS_LOG(NN_TEXT("テストその１： 割込みイベントで待ち解除するケース\n"));
    for (int i = 0; i < 3; ++i)
    {
        SEQ_NONE();

        // WaitAny を発行（timerEvent がシグナル化されるまで待機）
        nn::os::ClearTimerEvent( &timerEvent );
        switch (i)
        {
        case 0:
                NNT_OS_LOG("WaitAny( {event, semaphore, timerEvent} )");
                signaledHolder = nn::os::WaitAny( &multiWait );
                break;
        case 1:
                NNT_OS_LOG("TryWaitAny( {event, semaphore, timerEvent} )");
                signaledHolder = nn::os::TryWaitAny( &multiWait );
                break;
        case 2:
                NNT_OS_LOG("TimedWaitAny( {event, semaphore, timerEvent} )");
                signaledHolder = nn::os::TimedWaitAny( &multiWait,
                                        nn::TimeSpan::FromMilliSeconds(500));
                break;
        default:
                signaledHolder = NULL;
                break;
        }

        if (i == 1)
        {
            NNT_OS_LOG(" expect=false");
            CheckBool( signaledHolder == NULL );
        }
        else
        {
            // シグナルが成立している Holder オブジェクトが正しいか？
            NNT_OS_LOG(" expect=true");
            CheckBool( signaledHolder == &holder3 );

            // ユーザデータ値が正しいか？
            auto value = nn::os::GetMultiWaitHolderUserData(signaledHolder);
            SEQ_NONE();
            NNT_OS_LOG("    SignaledHolder's UserData=0x%08x", value);
            CheckBool( value == 0x33333333 );
        }
    }


    // テストその２
    // event, sem, timerEvent のうち、timerEvent で多重待ち解除するケース
    // timerEvent は多重待ちに入る前から既にシグナル状態。
    // WaitAny, TryWaitAny, TimedWaitAny 全てをテスト
    NNT_OS_LOG(NN_TEXT("テストその２： 割込みイベントで待ち解除するケース（事前シグナル化）\n"));
    for (int i = 0; i < 3; ++i)
    {
        SEQ_NONE();

        // WaitAny を発行（timerEvent は既にシグナル状態なので即リターン）
        switch (i)
        {
        case 0:
                NNT_OS_LOG("WaitAny( {event, semaphore, timerEvent} )");
                signaledHolder = nn::os::WaitAny( &multiWait );
                break;
        case 1:
                NNT_OS_LOG("TryWaitAny( {event, semaphore, timerEvent} )");
                signaledHolder = nn::os::TryWaitAny( &multiWait );
                break;
        case 2:
                NNT_OS_LOG("TimedWaitAny( {event, semaphore, timerEvent} )");
                signaledHolder = nn::os::TimedWaitAny( &multiWait,
                                        nn::TimeSpan::FromMilliSeconds(500));
                break;
        default:
                signaledHolder = NULL;
                break;
        }

        // シグナルが成立している Holder オブジェクトが正しいか？
        NNT_OS_LOG(" expect=true");
        CheckBool( signaledHolder == &holder3 );

        // ユーザデータ値が正しいか？
        auto value = nn::os::GetMultiWaitHolderUserData(signaledHolder);
        SEQ_NONE();
        NNT_OS_LOG("    SignaledHolder's UserData=0x%08x", value);
        CheckBool( value == 0x33333333 );
    }

    // テストその３
    // event, sem, timerEvent のうち、semaphore で多重待ち解除するケース
    // semaphore は多重待ちに入ってから 50msec 後ぐらいにシグナル化。
    // WaitAny, TryWaitAny, TimedWaitAny 全てをテスト
    NNT_OS_LOG(NN_TEXT("テストその３： セマフォで待ち解除するケース\n"));
    for (int i = 0; i < 3; ++i)
    {
        nn::os::ThreadType  thread;
        nn::os::CreateThread( &thread, test_SequentialTestWithKernelObject_HelperThread, &semaphore, g_ThreadStack, sizeof(g_ThreadStack), nn::os::DefaultThreadPriority);

        SEQ_NONE();

        // WaitAny を発行（semaphore がシグナル状態なのですぐにリターン）
        nn::os::ClearTimerEvent( &timerEvent );

        signaledHolder = nn::os::WaitAny( &multiWait );

        nn::os::ClearTimerEvent( &timerEvent );

        nn::os::StartThread( &thread );

        // WaitAny を発行（timerEvent がシグナル化されるまで待機）
        nn::os::ClearTimerEvent( &timerEvent );
        switch (i)
        {
        case 0:
                NNT_OS_LOG("WaitAny( {event, semaphore, timerEvent} )");
                signaledHolder = nn::os::WaitAny( &multiWait );
                break;
        case 1:
                NNT_OS_LOG("TryWaitAny( {event, semaphore, timerEvent} )");
                signaledHolder = nn::os::TryWaitAny( &multiWait );
                break;
        case 2:
                NNT_OS_LOG("TimedWaitAny( {event, semaphore, timerEvent} )");
                signaledHolder = nn::os::TimedWaitAny( &multiWait,
                                        nn::TimeSpan::FromMilliSeconds(500));
                break;
        default:
                signaledHolder = NULL;
                break;
        }

        if (i == 1)
        {
            NNT_OS_LOG(" expect=false");
            CheckBool( signaledHolder == NULL );
        }
        else
        {
            // シグナルが成立している Holder オブジェクトが正しいか？
            NNT_OS_LOG(" expect=true");
            CheckBool( signaledHolder == &holder2 );

            // ユーザデータ値が正しいか？
            auto value = nn::os::GetMultiWaitHolderUserData(signaledHolder);
            SEQ_NONE();
            NNT_OS_LOG("    SignaledHolder's UserData=0x%08x", value);
            CheckBool( value == 0x22222222 );
        }

        nn::os::AcquireSemaphore( &semaphore );

        nn::os::WaitThread( &thread );
        nn::os::DestroyThread( &thread );
    }


    // テストその４
    // event, sem, timerEvent のうち、semaphore で多重待ち解除するケース
    // semaphore は多重待ちに入る前から既にシグナル状態。
    // WaitAny, TryWaitAny, TimedWaitAny 全てをテスト
    NNT_OS_LOG(NN_TEXT("テストその４： セマフォで待ち解除するケース（事前シグナル化）\n"));
    for (int i = 0; i < 3; ++i)
    {
        SEQ_NONE();

        // WaitAny を発行（semaphore がシグナル状態なのですぐにリターン）
        nn::os::ClearTimerEvent( &timerEvent );

        signaledHolder = nn::os::WaitAny( &multiWait );

        nn::os::ClearTimerEvent( &timerEvent );
        nn::os::ReleaseSemaphore( &semaphore );

        // WaitAny を発行（timerEvent がシグナル化されるまで待機）
        nn::os::ClearTimerEvent( &timerEvent );
        switch (i)
        {
        case 0:
                NNT_OS_LOG("WaitAny( {event, semaphore, timerEvent} )");
                signaledHolder = nn::os::WaitAny( &multiWait );
                break;
        case 1:
                NNT_OS_LOG("TryWaitAny( {event, semaphore, timerEvent} )");
                signaledHolder = nn::os::TryWaitAny( &multiWait );
                break;
        case 2:
                NNT_OS_LOG("TimedWaitAny( {event, semaphore, timerEvent} )");
                signaledHolder = nn::os::TimedWaitAny( &multiWait,
                                        nn::TimeSpan::FromMilliSeconds(500));
                break;
        default:
                signaledHolder = NULL;
                break;
        }

        // シグナルが成立している Holder オブジェクトが正しいか？
        NNT_OS_LOG(" expect=true");
        CheckBool( signaledHolder == &holder2 );

        // ユーザデータ値が正しいか？
        auto value = nn::os::GetMultiWaitHolderUserData(signaledHolder);
        SEQ_NONE();
        NNT_OS_LOG("    SignaledHolder's UserData=0x%08x", value);
        CheckBool( value == 0x22222222 );

        nn::os::AcquireSemaphore( &semaphore );
    }


    // テストその５
    // event, sem, timerEvent の多重待ちでタイムアウトするケース
    NNT_OS_LOG(NN_TEXT("テストその５： 多重待ちでタイムアウトするケース\n"));
    {
        SEQ_NONE();
        NNT_OS_LOG("TimedWaitAny( {event, semaphore, timerEvent} )");

        // TimedWaitAny を発行
        nn::os::ClearTimerEvent( &timerEvent );

        signaledHolder = nn::os::WaitAny( &multiWait );

        nn::os::ClearTimerEvent( &timerEvent );

        signaledHolder = nn::os::TimedWaitAny( &multiWait,
                                        nn::TimeSpan::FromMilliSeconds(20));

        // シグナルが成立している Holder オブジェクトが正しいか？
        NNT_OS_LOG(" expect=false");
        CheckBool( signaledHolder == NULL );
    }

    // リストから外す
    nn::os::UnlinkMultiWaitHolder( &holder1 );
    nn::os::UnlinkMultiWaitHolder( &holder2 );
    nn::os::UnlinkMultiWaitHolder( &holder3 );

    // ファイナライズ
    nn::os::FinalizeMultiWaitHolder( &holder1 );
    nn::os::FinalizeMultiWaitHolder( &holder2 );
    nn::os::FinalizeMultiWaitHolder( &holder3 );
    nn::os::FinalizeMultiWait( &multiWait );

    nn::os::FinalizeEvent( &event );
    nn::os::FinalizeSemaphore( &semaphore );
    nn::os::FinalizeTimerEvent( &timerEvent );

    // 個別のテスト結果で GoogleTest に判定を通知
    JUDGE_GOOGLE_TEST();
}   // NOLINT(readability/fn_size)
#endif

TEST(MultiWaitUtility, test_Syntax)
{
    if (NN_STATIC_CONDITION(0))
    {
        using namespace nn::os;
        EventType* e1 = 0;
        SemaphoreType* e2 = 0;
        ThreadType* e3 = 0;
        MessageQueueType* q = 0;
        MultiWaitType m;

        int n;
        MultiWaitHolderType* h;
        std::pair<MultiWaitHolderType*, int> p;

        n = WaitAny();
        n = WaitAny(e1);
        n = WaitAny(e1, e2);
        n = WaitAny(e1, e2, e3);
        n = WaitAny(e1, e2, e3, q & MessageQueueWaitType_WaitForNotEmpty);
        h = WaitAny(&m);
        p = WaitAny(&m, e1);
        p = WaitAny(&m, e1, e2);
        p = WaitAny(&m, e1, e2, e3);
        p = WaitAny(&m, e1, e2, e3, q & MessageQueueWaitType_WaitForNotEmpty);

        n = TryWaitAny();
        n = TryWaitAny(e1);
        n = TryWaitAny(e1, e2);
        n = TryWaitAny(e1, e2, e3);
        n = TryWaitAny(e1, e2, e3, q & MessageQueueWaitType_WaitForNotEmpty);
        h = TryWaitAny(&m);
        p = TryWaitAny(&m, e1);
        p = TryWaitAny(&m, e1, e2);
        p = TryWaitAny(&m, e1, e2, e3);
        p = TryWaitAny(&m, e1, e2, e3, q & MessageQueueWaitType_WaitForNotEmpty);

        nn::TimeSpan timeout;
        n = TimedWaitAny(timeout);
        n = TimedWaitAny(timeout, e1);
        n = TimedWaitAny(timeout, e1, e2);
        n = TimedWaitAny(timeout, e1, e2, e3);
        n = TimedWaitAny(timeout, e1, e2, e3, q & MessageQueueWaitType_WaitForNotEmpty);
        h = TimedWaitAny(&m, timeout);
        p = TimedWaitAny(&m, timeout, e1);
        p = TimedWaitAny(&m, timeout, e1, e2);
        p = TimedWaitAny(&m, timeout, e1, e2, e3);
        p = TimedWaitAny(&m, timeout, e1, e2, e3, q & MessageQueueWaitType_WaitForNotEmpty);

        NN_UNUSED(n);
        NN_UNUSED(h);
        NN_UNUSED(p);

        // TryWaitAny を bool で受けることはできない
        //if (TryWaitAny(e1)) {}

        // TryWaitAny を int で受けることはできる
        switch (TryWaitAny(e1, e2))
        {
            case 0:
            case 1:
            default:
            {
            }
        }
    }
}

TEST(MultiWaitUtility, test_WaitAnyMulti)
{
    nn::os::MultiWaitType multiWait;
    nn::os::InitializeMultiWait(&multiWait);
    nn::os::Event e0(nn::os::EventClearMode_ManualClear);
    nn::os::MultiWaitHolderType holder;
    nn::os::InitializeMultiWaitHolder(&holder, e0.GetBase());
    nn::os::LinkMultiWaitHolder(&multiWait, &holder);

    nn::os::EventType es[16];
    for (auto&& e: es)
    {
        nn::os::InitializeEvent(&e, false, nn::os::EventClearMode_ManualClear);
    }

    #define TEST_WAIT_ANY_MULTI(...) \
    { \
        const nn::TimeSpan timeout = 0; \
        nn::os::EventType* targets[] = { __VA_ARGS__ }; \
        EXPECT_TRUE(nn::os::TryWaitAny(__VA_ARGS__) == -1); \
        { \
            auto p = nn::os::TryWaitAny(&multiWait, __VA_ARGS__); \
            EXPECT_TRUE(p.first == nullptr); \
            EXPECT_TRUE(p.second == -1); \
        } \
        { \
            auto p = nn::os::TimedWaitAny(&multiWait, timeout, __VA_ARGS__); \
            EXPECT_TRUE(p.first == nullptr); \
            EXPECT_TRUE(p.second == -1); \
        } \
        for (auto&& e : targets) \
        { \
            nn::os::SignalEvent(e); \
            auto i = &e - targets; \
            EXPECT_TRUE(nn::os::WaitAny(__VA_ARGS__) == i); \
            EXPECT_TRUE(nn::os::TimedWaitAny(timeout, __VA_ARGS__) == i); \
            { \
                auto p = nn::os::TryWaitAny(&multiWait, __VA_ARGS__); \
                EXPECT_TRUE(p.first == nullptr); \
                EXPECT_TRUE(p.second == i); \
            } \
            { \
                auto p = nn::os::TimedWaitAny(&multiWait, timeout, __VA_ARGS__); \
                EXPECT_TRUE(p.first == nullptr); \
                EXPECT_TRUE(p.second == i); \
            } \
            nn::os::ClearEvent(e); \
        } \
        { \
            e0.Signal(); \
            EXPECT_TRUE(nn::os::TryWaitAny(__VA_ARGS__) == -1); \
            EXPECT_TRUE(nn::os::TimedWaitAny(timeout, __VA_ARGS__) == -1); \
            { \
                auto p = nn::os::TryWaitAny(&multiWait, __VA_ARGS__); \
                EXPECT_TRUE(p.first == &holder); \
                EXPECT_TRUE(p.second == -1); \
            } \
            { \
                auto p = nn::os::TimedWaitAny(&multiWait, timeout, __VA_ARGS__); \
                EXPECT_TRUE(p.first == &holder); \
                EXPECT_TRUE(p.second == -1); \
            } \
            e0.Clear(); \
        } \
    }

    TEST_WAIT_ANY_MULTI(es + 0);
    TEST_WAIT_ANY_MULTI(es + 0, es + 1);
    TEST_WAIT_ANY_MULTI(es + 0, es + 1, es + 2);
    TEST_WAIT_ANY_MULTI(es + 0, es + 1, es + 2, es + 3);
    TEST_WAIT_ANY_MULTI(es + 0, es + 1, es + 2, es + 3, es + 4);
    TEST_WAIT_ANY_MULTI(es + 0, es + 1, es + 2, es + 3, es + 4, es + 5);
    TEST_WAIT_ANY_MULTI(es + 0, es + 1, es + 2, es + 3, es + 4, es + 5, es + 6);
    TEST_WAIT_ANY_MULTI(es + 0, es + 1, es + 2, es + 3, es + 4, es + 5, es + 6, es + 7);
    TEST_WAIT_ANY_MULTI(es + 0, es + 1, es + 2, es + 3, es + 4, es + 5, es + 6, es + 7, es + 8);
    TEST_WAIT_ANY_MULTI(es + 0, es + 1, es + 2, es + 3, es + 4, es + 5, es + 6, es + 7, es + 8, es + 9);
    TEST_WAIT_ANY_MULTI(es + 0, es + 1, es + 2, es + 3, es + 4, es + 5, es + 6, es + 7, es + 8, es + 9, es + 10);
    TEST_WAIT_ANY_MULTI(es + 0, es + 1, es + 2, es + 3, es + 4, es + 5, es + 6, es + 7, es + 8, es + 9, es + 10, es + 11);
    TEST_WAIT_ANY_MULTI(es + 0, es + 1, es + 2, es + 3, es + 4, es + 5, es + 6, es + 7, es + 8, es + 9, es + 10, es + 11, es + 12);
    TEST_WAIT_ANY_MULTI(es + 0, es + 1, es + 2, es + 3, es + 4, es + 5, es + 6, es + 7, es + 8, es + 9, es + 10, es + 11, es + 12, es + 13);
    TEST_WAIT_ANY_MULTI(es + 0, es + 1, es + 2, es + 3, es + 4, es + 5, es + 6, es + 7, es + 8, es + 9, es + 10, es + 11, es + 12, es + 13, es + 14);
    TEST_WAIT_ANY_MULTI(es + 0, es + 1, es + 2, es + 3, es + 4, es + 5, es + 6, es + 7, es + 8, es + 9, es + 10, es + 11, es + 12, es + 13, es + 14, es + 15);

    #undef TEST_WAIT_ANY_MULTI

    for (auto&& e: es)
    {
        nn::os::FinalizeEvent(&e);
    }

    nn::os::UnlinkMultiWaitHolder(&holder);
    nn::os::FinalizeMultiWaitHolder(&holder);
    nn::os::FinalizeMultiWait(&multiWait);
}

#if !defined(NN_BUILD_CONFIG_TOOLCHAIN_GCC)
TEST(MultiWaitUtility, test_WaitAnyVariation1)
{
    const nn::TimeSpan timeout = 0;
    {
        nn::os::Event w(nn::os::EventClearMode_ManualClear);
        EXPECT_TRUE(nn::os::TryWaitAny(w.GetBase()) == -1);
        EXPECT_TRUE(nn::os::TimedWaitAny(timeout, w.GetBase()) == -1);
        std::thread t([&]
        {
            w.Signal();
        });
        EXPECT_TRUE(nn::os::WaitAny(w.GetBase()) == 0);
        t.join();
        EXPECT_TRUE(nn::os::TryWaitAny(w.GetBase()) == 0);
    }
    {
        nn::os::Semaphore w(0, 10);
        EXPECT_TRUE(nn::os::TryWaitAny(w.GetBase()) == -1);
        EXPECT_TRUE(nn::os::TimedWaitAny(timeout, w.GetBase()) == -1);
        std::thread t([&]
        {
            w.Release();
        });
        EXPECT_TRUE(nn::os::WaitAny(w.GetBase()) == 0);
        t.join();
        EXPECT_TRUE(nn::os::TryWaitAny(w.GetBase()) == 0);
    }
    {
        nn::os::TimerEvent w(nn::os::EventClearMode_ManualClear);
        EXPECT_TRUE(nn::os::TryWaitAny(w.GetBase()) == -1);
        EXPECT_TRUE(nn::os::TimedWaitAny(timeout, w.GetBase()) == -1);
        std::thread t([&]
        {
            w.StartOneShot(nn::TimeSpan::FromMilliSeconds(1));
        });
        EXPECT_TRUE(nn::os::WaitAny(w.GetBase()) == 0);
        t.join();
        EXPECT_TRUE(nn::os::TryWaitAny(w.GetBase()) == 0);
    }
    {
        nn::os::ThreadType w;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&w, [] (void*) {}, nullptr, g_ThreadStack, sizeof(g_ThreadStack), nn::os::DefaultThreadPriority));
        EXPECT_TRUE(nn::os::TryWaitAny(&w) == -1);
        EXPECT_TRUE(nn::os::TimedWaitAny(timeout, &w) == -1);
        std::thread t([&]
        {
            nn::os::StartThread(&w);
        });
        EXPECT_TRUE(nn::os::WaitAny(&w) == 0);
        t.join();
        EXPECT_TRUE(nn::os::TryWaitAny(&w) == 0);
        nn::os::WaitThread(&w);
        nn::os::DestroyThread(&w);
    }
    {
        uintptr_t queueBuffer[1];
        nn::os::MessageQueue w(queueBuffer, 1);
        EXPECT_TRUE(nn::os::TryWaitAny(w.GetBase() & nn::os::MessageQueueWaitType_WaitForNotEmpty) == -1);
        EXPECT_TRUE(nn::os::TimedWaitAny(timeout, w.GetBase() & nn::os::MessageQueueWaitType_WaitForNotEmpty) == -1);
        EXPECT_TRUE(nn::os::TryWaitAny(w.GetBase() & nn::os::MessageQueueWaitType_WaitForNotFull) == 0);
        {
            std::thread t([&]
            {
                w.Send(0);
            });
            EXPECT_TRUE(nn::os::WaitAny(w.GetBase() & nn::os::MessageQueueWaitType_WaitForNotEmpty) == 0);
            t.join();
        }
        EXPECT_TRUE(nn::os::TryWaitAny(w.GetBase() & nn::os::MessageQueueWaitType_WaitForNotEmpty) == 0);
        EXPECT_TRUE(nn::os::TryWaitAny(w.GetBase() & nn::os::MessageQueueWaitType_WaitForNotFull) == -1);
        {
            std::thread t([&]
            {
                uintptr_t x;
                w.Receive(&x);
            });
            EXPECT_TRUE(nn::os::WaitAny(w.GetBase() & nn::os::MessageQueueWaitType_WaitForNotFull) == 0);
            t.join();
        }
        EXPECT_TRUE(nn::os::TryWaitAny(w.GetBase() & nn::os::MessageQueueWaitType_WaitForNotEmpty) == -1);
        EXPECT_TRUE(nn::os::TryWaitAny(w.GetBase() & nn::os::MessageQueueWaitType_WaitForNotFull) == 0);
    }
}
#endif // !defined(NN_BUILD_CONFIG_TOOLCHAIN_GCC)

}}} // namespace nnt::os::multiWait

//---------------------------------------------------------------------------
//  Test Main 関数
//---------------------------------------------------------------------------

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    NNT_CALIBRATE_INITIALIZE();
    SEQ_INITIALIZE();
    INITIALIZE_TEST_COUNT();

    // テスト開始
    SEQ_CHECK(0);
    NNT_OS_LOG("=== Start Test of MultipleWait APIs\n");

    // GoogleTest おまじない
    ::testing::InitGoogleTest(&argc, argv);
    int result = RUN_ALL_TESTS();

    // テスト終了
    NNT_OS_LOG("\n=== End Test of MultipleWait APIs\n");

    // 集計結果の表示
    g_Result.Show();

    nnt::Exit(result);
}
