﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//---------------------------------------------------------------------------
//  MessageQueue 専用のテストヘルパー
//---------------------------------------------------------------------------

#pragma once

#include <nn/os/os_Config.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include "../Common/test_Helper.h"
#include "../Common/test_Calibration.h"

#if defined(NN_BUILD_CONFIG_OS_WIN32)
#include <nnt/nntest.h>
#endif

//---------------------------------------------------------------------------

namespace nnt { namespace os { namespace messageQueue {

const int   priThread1  = 24;
const int   priThread2  = 24;

const int   messageQueueSize = 8;

const uintptr_t valueIndex0 = 0x01010101;
const uintptr_t valueIndex1 = 0x02020202;
const uintptr_t valueAdd    = 0x12345678;

//---------------------------------------------------------------------------
//  単体テストフレームワーク
//---------------------------------------------------------------------------
//【機能概要】
//  ２つのスレッドと、１つのメッセージキューを使用し、それぞれ指定された
//  前提条件となるリソース状態を整えてから、指定された API を発行し、
//  その返値および、スレッドやメッセージキューオブジェクトの事後状態を
//  期待値と比較することで、あらゆる条件下での単体テストを実施する。
//
//  指定した前提条件となるオブジェクト状態への遷移は公開 API のみで行ない、
//  オブジェクトの中身を直接書き換えたりはしない。ただし、事後状態を
//  参照する場合には、オブジェクトの中身を直接参照する場合もある。
//
//  全てのオブジェクトは、１回のテスト毎に Initialize/Finalize が行なわれる。
//  スレッド１が必ず最初に動作を開始し、スレッド２が続いて動作を開始する。
//
//  メッセージキュー内に蓄積されるデータは、初期状態構築時は、先頭から、
//  0x01010101, 0x02020202, 0x03030303, .... という値が、テスト毎に
//  決められた数だけ詰められる。また、Send/Jam 系の API テストでは、
//  送信データとして 0x12345678 が使用される。
//  受信時は、これらのデータの内、想定された値が取得できているかを確認する。
//
//【使用方法】
//  以下の関数１回につき、１つの機能テストを実施する。
//  １回の関数実行により、シーケンス番号が +100 ずつ進む。
//  先に setExpectValue() でデータの期待値をセットし、doTest() を発行する。
//  setExpectValue() は一度セットすると、その後に使い回し可能。
//  第１引数がスレッド１での受信用、第２引数がスレッド２での受信用。
//
//    1) １スレッド用
//        void  setExpectValue( value1, value2 );
//        void  doTest1(th1, mq, rslt1, th1ex, mqex);
//
//    2) ２スレッド用
//        void  setExpectValue( value1, value2 );
//        void  doTest2(th1, th2, mq, rslt1, rslt2, th1ex, th2ex, mqex);
//
//  各引数は以下のような意味を持つ。
//
//      th1:    スレッド１のアクション
//      th2:    スレッド２のアクション
//      mq:     メッセージキューの事前状態
//      rslt1:  スレッド１でのテスト対象 API の返値（期待値）
//      rslt2:  スレッド２でのテスト対象 API の返値（期待値）
//      th1ex:  スレッド１の事後状態（期待値）
//      th2ex:  スレッド２の事後状態（期待値）
//      mqex:   メッセージキューの事後状態（期待値）
//
//  スレッドのアクションを示す th1/th2 には以下の状態を指定する。
//
//      THR_NOTUSE                  そのスレッドは使用しない
//
//      THR_WAIT_SEND               SendMessage() 待ち状態（無期限）
//      THR_WAIT_TIMED_SEND         TimedSendMessage() 待ち状態（時限あり）
//      THR_WAIT_TIMED_SEND_MINUS   TimedSendMessage() 待ち状態（負のタイムアウト値）
//      THR_WAIT_JAM                JamMessage() 待ち状態（無期限）
//      THR_WAIT_TIMED_JAM          TimedJamMessage() 待ち状態（時限あり）
//      THR_WAIT_TIMED_JAM_MINUS    TimedJamMessage() 待ち状態（負のタイムアウト値）
//      THR_WAIT_RECV               ReceiveMessage() 待ち状態（無期限）
//      THR_WAIT_TIMED_RECV         TimedReceiveMessage() 待ち状態（時限あり）
//      THR_WAIT_TIMED_RECV_MINUS   TimedReceiveMessage() 待ち状態（負のタイムアウト値）
//      THR_WAIT_PEEK               PeekMessage() 待ち状態（無期限）
//      THR_WAIT_TIMED_PEEK         TimedPeekMessage() 待ち状態（時限あり）
//      THR_WAIT_TIMED_PEEK_MINUS   TimedPeekMessage() 待ち状態（負のタイムアウト値）
//
//      THR_CALL_SEND               SendMessageQueue() の発行
//      THR_CALL_TRY_SEND           TrySendMessageQueue() の発行
//      THR_CALL_TIMED_SEND         TimedSendMessageQueue() の発行
//      THR_CALL_TIMED_SEND_MINUS   TimedSendMessageQueue() の発行（負のタイムアウト値）
//      THR_CALL_JAM                JamMessageQueue() の発行
//      THR_CALL_TRY_JAM            TryJamMessageQueue() の発行
//      THR_CALL_TIMED_JAM          JamMessageQueue() の発行
//      THR_CALL_TIMED_JAM_MINUS    JamMessageQueue() の発行（負のタイムアウト値）
//      THR_CALL_RECV               TimedReceiveMessageQueue() の発行
//      THR_CALL_TRY_RECV           TryReceiveMessageQueue() の発行
//      THR_CALL_TIMED_RECV         TimedReceiveMessageQueue() の発行
//      THR_CALL_TIMED_RECV_MINUS   TimedReceiveMessageQueue() の発行（負のタイムアウト値）
//      THR_CALL_PEEK               PeekMessageQueue() の発行
//      THR_CALL_TRY_PEEK           TryPeekMessageQueue() の発行
//      THR_CALL_TIMED_PEEK         TimedPeekMessageQueue() の発行
//      THR_CALL_TIMED_PEEK_MINUS   TimedPeekMessageQueue() の発行（負のタイムアウト値）
//
//  スレッドの最終状態を示す th1ex/th2ex には以下の状態を指定する。
//
//      THR_STATE_NONE              スレッド未使用
//      THR_STATE_EXITED            終了状態
//      THR_STATE_WAITING           待ち状態を継続中
//      THR_STATE_TIMEDOUT_EXITED   タイムアウトを検知して終了
//
//  メッセージキュー状態を示す mq/mqex は以下の状態を指定する。
//
//      MQ_NUM(n)                   データが n 個入っている状態
//
//  rslt1/rslt2 には bool を指定する。
//---------------------------------------------------------------------------

enum TestThreadAction
{
    THR_NOTUSE          = 0,    // スレッドを使用しない

    THR_WAIT_SEND,              // SendMessage() 待ち状態（無期限）
    THR_WAIT_TIMED_SEND,        // TimedSendMessage() 待ち状態（時限あり）
    THR_WAIT_TIMED_SEND_MINUS,  // TimedSendMessage() 待ち状態（負のタイムアウト値）
    THR_WAIT_JAM,               // JamMessage() 待ち状態（無期限）
    THR_WAIT_TIMED_JAM,         // TimedJamMessage() 待ち状態（時限あり）
    THR_WAIT_TIMED_JAM_MINUS,   // TimedJamMessage() 待ち状態（負のタイムアウト値）
    THR_WAIT_RECV,              // ReceiveMessage() 待ち状態（無期限）
    THR_WAIT_TIMED_RECV,        // TimedReceiveMessage() 待ち状態（時限あり）
    THR_WAIT_TIMED_RECV_MINUS,  // TimedReceiveMessage() 待ち状態（負のタイムアウト値）
    THR_WAIT_PEEK,              // PeekMessage() 待ち状態（無期限）
    THR_WAIT_TIMED_PEEK,        // TimedPeekMessage() 待ち状態（時限あり）
    THR_WAIT_TIMED_PEEK_MINUS,  // TimedPeekMessage() 待ち状態（負のタイムアウト値）

    THR_CALL_SEND,              // SendMessageQueue() の発行
    THR_CALL_TRY_SEND,          // TrySendMessageQueue() の発行
    THR_CALL_TIMED_SEND,        // TimedSendMessageQueue() の発行
    THR_CALL_TIMED_SEND_MINUS,  // TimedSendMessageQueue() の発行（負のタイムアウト値）
    THR_CALL_JAM,               // JamMessageQueue() の発行
    THR_CALL_TRY_JAM,           // TryJamMessageQueue() の発行
    THR_CALL_TIMED_JAM,         // JamMessageQueue() の発行
    THR_CALL_TIMED_JAM_MINUS,   // JamMessageQueue() の発行（負のタイムアウト値）
    THR_CALL_RECV,              // TimedReceiveMessageQueue() の発行
    THR_CALL_TRY_RECV,          // TryReceiveMessageQueue() の発行
    THR_CALL_TIMED_RECV,        // TimedReceiveMessageQueue() の発行
    THR_CALL_TIMED_RECV_MINUS,  // TimedReceiveMessageQueue() の発行（負のタイムアウト値）
    THR_CALL_PEEK,              // PeekMessageQueue() の発行
    THR_CALL_TRY_PEEK,          // TryPeekMessageQueue() の発行
    THR_CALL_TIMED_PEEK,        // TimedPeekMessageQueue() の発行
    THR_CALL_TIMED_PEEK_MINUS,  // TimedPeekMessageQueue() の発行（負のタイムアウト値）
};

enum TestThreadLastState
{
    THR_STATE_NONE      = 0,    // スレッド未使用
    THR_STATE_WAITING   = 1,    // 待ち状態を継続中
    THR_STATE_EXITED    = 2,    // 終了状態
    THR_STATE_TIMEDOUT  = 3,    // タイムアウトを検知して終了
    THR_STATE_MAX,
};

#define MQ_NUM(n)   (n)         // データが n 個入っている状態

enum TestSyncPoint
{
    NNT_POINT_THREAD1_START     = 1,
    NNT_POINT_THREAD2_START,
    NNT_POINT_THREAD1_CALLED,
    NNT_POINT_THREAD2_CALLED,
    NNT_POINT_THREAD1_END,
    NNT_POINT_THREAD2_END,
};

//---------------------------------------------------------------------------

class   MessageQueueHelper
{
public:
    MessageQueueHelper()    {}
    ~MessageQueueHelper()   {}

    // テスト実行
    void    doTest( int                 mq,
                    TestThreadAction    th1,
                    TestThreadAction    th2,
                    bool                rslt1,
                    bool                rslt2,
                    int                 mqex,
                    TestThreadLastState th1ex,
                    TestThreadLastState th2ex,
                    const char*         functionName,
                    int                 line);

    // Receive/Get 系で受信時の期待値を設定
    void        SetExpectValue( uintptr_t value1, uintptr_t value2 );
    uintptr_t   GetExpectValue1()
    {
        return m_expectValue1;
    }
    uintptr_t   GetExpectValue2()
    {
        return m_expectValue2;
    }

    // スレッドの実行開始
    void    StartThread1( TestThreadAction );
    void    StartThread2( TestThreadAction );

    // スレッド１情報セット ＆ 取得
    void    SetThreadInfo1( TestThreadAction    action,
                            TestThreadLastState expectState,
                            bool                expect)
    {
        m_TestThreadAction1         = action;
        m_TestThreadExpectState1    = expectState;
        m_expect1                   = expect;
    }
    TestThreadAction    GetThreadAction1()
    {
        return  m_TestThreadAction1;
    }
    TestThreadLastState GetThreadExpectState1()
    {
        return  m_TestThreadExpectState1;
    }
    bool        GetThreadExpect1()
    {
        return  m_expect1;
    }

    // スレッド２情報セット ＆ 取得
    void    SetThreadInfo2( TestThreadAction    action,
                            TestThreadLastState expectState,
                            bool                expect)
    {
        m_TestThreadAction2         = action;
        m_TestThreadExpectState2    = expectState;
        m_expect2                   = expect;
    }
    TestThreadAction    GetThreadAction2()
    {
        return  m_TestThreadAction2;
    }
    TestThreadLastState GetThreadExpectState2()
    {
        return  m_TestThreadExpectState2;
    }
    bool        GetThreadExpect2()
    {
        return  m_expect2;
    }

    // メッセージキューとスレッドをクリーンアップ（待ち解除）
    void    Cleanup();

    // スレッドの状態チェック（Thread の内部実装に依存）
    void    ThreadStateCheck( const nn::os::ThreadType&,
                              TestThreadAction,
                              TestThreadLastState,
                              bool,
                              bool );

    // メッセージキューの状態チェック（MessageQueue の内部実装に依存）
    void    MessageQueueCheck( int mqex );

    // MessageQueue オブジェクト状態の構築
    void    ConstructMessageQueue( int mq );

    // MessageQueue 関連 API を発行
    bool    CallMessageQueueApi( TestThreadAction    action,
                                 int                 seq,
                                 TestThreadLastState expectState,
                                 uintptr_t           expectValue);

    // TesThreadState によるシーケンスカウンタの進み量を返す
    int     GetLastSeq( TestThreadLastState th1ex, TestThreadLastState th2ex );

    // MessageQueue オブジェクト内部のデータ個数を返す
    int     GetMessageQueueCount()
    {
            // MessageQueue オブジェクトの内部実装に依存
            return m_MessageQueue._count;
    }

    // OS-API 発行後の result の格納と取得
    bool GetApiResult1()
    {
        return m_resultOfApi1;
    }
    bool GetApiResult2()
    {
        return m_resultOfApi2;
    }
    void SetApiResult1(bool result)
    {
        m_resultOfApi1 = result;
    }
    void SetApiResult2(bool result)
    {
        m_resultOfApi2 = result;
    }

    bool IsCleanuped()
    {
        return m_isCleanuped;
    }

private:
    void CreateThread1( TestThreadAction action );
    void CreateThread2( TestThreadAction action );
    void DestroyThread1( TestThreadAction action );
    void DestroyThread2( TestThreadAction action );
    void InitializeMessage();
    void FinalizeMessage();

    uintptr_t               m_expectValue1;
    uintptr_t               m_expectValue2;

    nn::os::ThreadType          m_Thread1;
    nn::os::ThreadType          m_Thread2;
    nn::os::MessageQueueType    m_MessageQueue;

    TestThreadAction        m_TestThreadAction1;
    TestThreadAction        m_TestThreadAction2;
    TestThreadLastState     m_TestThreadExpectState1;
    TestThreadLastState     m_TestThreadExpectState2;

    bool                    m_expect1;
    bool                    m_expect2;
    bool                    m_resultOfApi1;
    bool                    m_resultOfApi2;
    bool                    m_isCleanuped;
};

//---------------------------------------------------------------------------

extern void ThreadFunc1(void* exinf);
extern void ThreadFunc2(void* exinf);

extern  MessageQueueHelper  g_MqHelper;

#define doTest1(mq, th1, rslt1, mqex, th1ex)         \
        g_MqHelper.doTest((mq), (th1), (THR_NOTUSE), (rslt1), (true), (mqex), (th1ex), (THR_STATE_NONE), __FUNCTION__, __LINE__)

#define doTest2(mq, th1, th2, rslt1, rslt2, mqex, th1ex, th2ex)         \
        g_MqHelper.doTest((mq), (th1), (th2), (rslt1), (rslt2), (mqex), (th1ex), (th2ex), __FUNCTION__, __LINE__)

#define setExpectValue(v1,v2)   g_MqHelper.SetExpectValue( (v1), (v2) );

}}} // namespace nnt::os::messageQueue


