﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//---------------------------------------------------------------------------
//  LightEvent 関連機能のテスト
//---------------------------------------------------------------------------

#include "../Common/test_Pragma.h"

#include <nn/nn_Common.h>
#include <nn/nn_SdkText.h>
#include <nn/os.h>
#include "../Common/test_Helper.h"
#include "../Common/test_Calibration.h"
#include "test_LightEventHelper.h"

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>

#define EVENT_AUTO_CLEAR    nn::os::EventClearMode_AutoClear
#define EVENT_MANUAL_CLEAR  nn::os::EventClearMode_ManualClear

namespace nnt { namespace os { namespace lightEvent {

//---------------------------------------------------------------------------
// InitializeLightEvent 直後の初期状態チェック

TEST(InitializeLightEvent, test_initiallySignaled1)
{
    // InitializeLightEvent 初期状態チェック
    doTest1(false,                      // LightEvent は Not Signaled
            EVENT_AUTO_CLEAR,           // LightEvent は自動クリアモード
            THR_NOTUSE,                 // Thread1 も未使用
            true,                       // Thread1 での API 返値
            false,                      // LightEvent は Not Signaled
            THR_STATE_NONE);            // Thread1 は終了
}

TEST(InitializeSendLightEvent, test_initiallySignaled2)
{
    // InitializeLightEvent 初期状態チェック
    doTest1(false,                      // LightEvent は Not Signaled
            EVENT_MANUAL_CLEAR,         // LightEvent は手動クリアモード
            THR_NOTUSE,                 // Thread1 も未使用
            true,                       // Thread1 での API 返値
            false,                      // LightEvent は Not Signaled
            THR_STATE_NONE);            // Thread1 は終了
}

TEST(InitializeSendLightEvent, test_initiallyNotSignaled1)
{
    // InitializeLightEvent 初期状態チェック
    doTest1(true,                       // LightEvent は Signaled
            EVENT_AUTO_CLEAR,           // LightEvent は自動クリアモード
            THR_NOTUSE,                 // Thread1 も未使用
            true,                       // Thread1 での API 返値
            true,                       // LightEvent は Signaled
            THR_STATE_NONE);            // Thread1 は終了
}

TEST(InitializeSendLightEvent, test_initiallyNotSignaled2)
{
    // InitializeLightEvent 初期状態チェック
    doTest1(true,                       // LightEvent は Signaled
            EVENT_MANUAL_CLEAR,         // LightEvent は手動クリアモード
            THR_NOTUSE,                 // Thread1 も未使用
            true,                       // Thread1 での API 返値
            true,                       // LightEvent は Signaled
            THR_STATE_NONE);            // Thread1 は終了
}

//---------------------------------------------------------------------------
// 単独の SignalLightEvent() テスト

TEST(SignalLightEvent, test_NotSignalToSignal)
{
    // SignalLightEvent 非シグナルからシグナルへ
    doTest1(false,                      // LightEvent は Not Signaled
            EVENT_AUTO_CLEAR,           // LightEvent は自動クリアモード
            THR_CALL_SIGNAL,            // Thread1 は SignalLightEvent()
            true,                       // Thread1 での API 返値
            true,                       // LightEvent は Signaled
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(SignalLightEvent, test_SignalToSignal)
{
    // SignalLightEvent シグナルからシグナルへ
    doTest1(true,                       // LightEvent は Signaled
            EVENT_AUTO_CLEAR,           // LightEvent は自動クリアモード
            THR_CALL_SIGNAL,            // Thread1 は SignalLightEvent()
            true,                       // Thread1 での API 返値
            true,                       // LightEvent は Signaled
            THR_STATE_EXITED);          // Thread1 は終了
}

//---------------------------------------------------------------------------

// 単独の ClearLightEvent() テスト

TEST(ClearLightEvent, test_NotSignalToNotSignal1)
{
    // SignalLightEvent 非シグナルからシグナルへ
    doTest1(false,                      // LightEvent は Not Signaled
            EVENT_AUTO_CLEAR,           // LightEvent は自動クリアモード
            THR_CALL_CLEAR,             // Thread1 は ClearLightEvent()
            true,                       // Thread1 での API 返値
            false,                      // LightEvent は Not Signaled
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(ClearLightEvent, test_NotSignalToNotSignal2)
{
    // SignalLightEvent 非シグナルからシグナルへ
    doTest1(false,                      // LightEvent は Not Signaled
            EVENT_MANUAL_CLEAR,         // LightEvent は手動クリアモード
            THR_CALL_CLEAR,             // Thread1 は ClearLightEvent()
            true,                       // Thread1 での API 返値
            false,                      // LightEvent は Not Signaled
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(ClearLightEvent, test_SignalToNotSignal1)
{
    // SignalLightEvent シグナルからシグナルへ
    doTest1(true,                       // LightEvent は Signaled
            EVENT_AUTO_CLEAR,           // LightEvent は自動クリアモード
            THR_CALL_CLEAR,             // Thread1 は ClearLightEvent()
            true,                       // Thread1 での API 返値
            false,                      // LightEvent は Not Signaled
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(ClearLightEvent, test_SignalToNotSignal2)
{
    // SignalLightEvent シグナルからシグナルへ
    doTest1(true,                       // LightEvent は Signaled
            EVENT_MANUAL_CLEAR,         // LightEvent は手動クリアモード
            THR_CALL_CLEAR,             // Thread1 は ClearLightEvent()
            true,                       // Thread1 での API 返値
            false,                      // LightEvent は Not Signaled
            THR_STATE_EXITED);          // Thread1 は終了
}

//---------------------------------------------------------------------------
// 単独の WaitLightEvent() AutoClear テスト

TEST(WaitLightEventAutoClear, test_WaitLightEventAutoClear1)
{
    // AUTO_CLEAR/Signaled に対して WaitLightEvent 発行
    doTest1(true,                       // LightEvent は Signaled
            EVENT_AUTO_CLEAR,           // LightEvent は自動クリアモード
            THR_CALL_WAIT,              // Thread1 は WaitLightEvent()
            true,                       // Thread1 での API 返値
            false,                      // LightEvent は Not Signaled
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(WaitLightEventAutoClear, test_WaitLightEventAutoClear2)
{
    // AUTO_CLEAR/NotSignaled に対して WaitLightEvent 発行
    doTest1(false,                      // LightEvent は Not Signaled
            EVENT_AUTO_CLEAR,           // LightEvent は自動クリアモード
            THR_WAIT_WAIT,              // Thread1 は WaitLightEvent()
            true,                       // Thread1 での API 返値
            false,                      // LightEvent は Not Signaled
            THR_STATE_WAITING);         // Thread1 はシグナル待ち
}

TEST(WaitLightEventAutoClear, test_WaitLightEventAutoClear3)
{
    // AUTO_CLEAR/Signaled に対して WaitLightEvent 発行×２スレッド
    doTest2(true,                       // LightEvent は Signaled
            EVENT_AUTO_CLEAR,           // LightEvent は自動クリアモード
            THR_CALL_WAIT,              // Thread1 は WaitLightEvent()
            THR_WAIT_WAIT,              // Thread1 は WaitLightEvent()
            true,                       // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            false,                      // LightEvent は Not Signaled
            THR_STATE_EXITED,           // Thread1 は終了
            THR_STATE_WAITING);         // Thread2 はシグナル待ち
}

TEST(WaitLightEventAutoClear, test_WaitLightEventAutoClear4)
{
    // AUTO_CLEAR/NotSignaled に対して WaitLightEvent 発行×２スレッド
    doTest2(false,                      // LightEvent は Not Signaled
            EVENT_AUTO_CLEAR,           // LightEvent は自動クリアモード
            THR_WAIT_WAIT,              // Thread1 は WaitLightEvent()
            THR_WAIT_WAIT,              // Thread1 は WaitLightEvent()
            true,                       // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            false,                      // LightEvent は Not Signaled
            THR_STATE_WAITING,          // Thread1 はシグナル待ち
            THR_STATE_WAITING);         // Thread2 はシグナル待ち
}

//---------------------------------------------------------------------------
// 単独の WaitLightEvent() ManualClear テスト

TEST(WaitLightEventManualClear, test_WaitLightEventManualClear1)
{
    // MANUAL_CLEAR/Signaled に対して WaitLightEvent 発行
    doTest1(true,                       // LightEvent は Signaled
            EVENT_MANUAL_CLEAR,         // LightEvent は手動クリアモード
            THR_CALL_WAIT,              // Thread1 は WaitLightEvent()
            true,                       // Thread1 での API 返値
            true,                       // LightEvent は Signaled
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(WaitLightEventManualClear, test_WaitLightEventManualClear2)
{
    // MANUAL_CLEAR/NotSignaled に対して WaitLightEvent 発行
    doTest1(false,                      // LightEvent は Not Signaled
            EVENT_MANUAL_CLEAR,         // LightEvent は手動クリアモード
            THR_WAIT_WAIT,              // Thread1 は WaitLightEvent()
            true,                       // Thread1 での API 返値
            false,                      // LightEvent は Not Signaled
            THR_STATE_WAITING);         // Thread1 はシグナル待ち
}

TEST(WaitLightEventManualClear, test_WaitLightEventManualClear3)
{
    // MANUAL_CLEAR/Signaled に対して WaitLightEvent() ×２スレッド
    doTest2(true,                       // LightEvent は Signaled
            EVENT_MANUAL_CLEAR,         // LightEvent は手動クリアモード
            THR_CALL_WAIT,              // Thread1 は WaitLightEvent()
            THR_CALL_WAIT,              // Thread1 は WaitLightEvent()
            true,                       // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            true,                       // LightEvent は Signaled
            THR_STATE_EXITED,           // Thread1 は終了
            THR_STATE_EXITED);          // Thread2 は終了
}

TEST(WaitLightEventManualClear, test_WaitLightEventManualClear4)
{
    // MANUAL_CLEAR/NotSignaled に対して WaitLightEvent() ×２スレッド
    doTest2(false,                      // LightEvent は Not Signaled
            EVENT_MANUAL_CLEAR,         // LightEvent は手動クリアモード
            THR_WAIT_WAIT,              // Thread1 は WaitLightEvent()
            THR_WAIT_WAIT,              // Thread1 は WaitLightEvent()
            true,                       // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            false,                      // LightEvent は Not Signaled
            THR_STATE_WAITING,          // Thread1 はシグナル待ち
            THR_STATE_WAITING);         // Thread2 はシグナル待ち
}

//---------------------------------------------------------------------------
// 単独の TryWaitLightEvent() AutoClear テスト

TEST(TryWaitLightEventAutoClear, test_TryWaitLightEventAutoClear1)
{
    // AUTO_CLEAR/Signaled に対して TryWaitLightEvent 発行
    doTest1(true,                       // LightEvent は Signaled
            EVENT_AUTO_CLEAR,           // LightEvent は自動クリアモード
            THR_CALL_TRY_WAIT,          // Thread1 は TryWaitLightEvent()
            true,                       // Thread1 での API 返値
            false,                      // LightEvent は Not Signaled
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(TryWaitLightEventAutoClear, test_TryWaitLightEventAutoClear2)
{
    // AUTO_CLEAR/NotSignaled に対して TryWaitLightEvent 発行
    doTest1(false,                      // LightEvent は Not Signaled
            EVENT_AUTO_CLEAR,           // LightEvent は自動クリアモード
            THR_CALL_TRY_WAIT,          // Thread1 は TryWaitLightEvent()
            false,                      // Thread1 での API 返値
            false,                      // LightEvent は Not Signaled
            THR_STATE_EXITED);          // Thread1 はシグナル待ち
}

TEST(TryWaitLightEventAutoClear, test_TryWaitLightEventAutoClear3)
{
    // AUTO_CLEAR/Signaled に対して TryWaitLightEvent 発行×２スレッド
    doTest2(true,                       // LightEvent は Signaled
            EVENT_AUTO_CLEAR,           // LightEvent は自動クリアモード
            THR_CALL_TRY_WAIT,          // Thread1 は TryWaitLightEvent()
            THR_CALL_TRY_WAIT,          // Thread1 は TryWaitLightEvent()
            true,                       // Thread1 での API 返値
            false,                      // Thread2 での API 返値
            false,                      // LightEvent は Not Signaled
            THR_STATE_EXITED,           // Thread1 は終了
            THR_STATE_EXITED);          // Thread2 はシグナル待ち
}

TEST(TryWaitLightEventAutoClear, test_TryWaitLightEventAutoClear4)
{
    // AUTO_CLEAR/NotSignaled に対して TryWaitLightEvent() ×２スレッド
    doTest2(false,                      // LightEvent は Not Signaled
            EVENT_AUTO_CLEAR,           // LightEvent は自動クリアモード
            THR_CALL_TRY_WAIT,          // Thread1 は TryWaitLightEvent()
            THR_CALL_TRY_WAIT,          // Thread1 は TryWaitLightEvent()
            false,                      // Thread1 での API 返値
            false,                      // Thread2 での API 返値
            false,                      // LightEvent は Not Signaled
            THR_STATE_EXITED,           // Thread1 はシグナル待ち
            THR_STATE_EXITED);          // Thread2 はシグナル待ち
}

//---------------------------------------------------------------------------
// 単独の TryWaitLightEvent() ManualClear テスト

TEST(TryWaitLightEventManualClear, test_TryWaitLightEventManualClear1)
{
    // MANUAL_CLEAR/Signaled に対して TryWaitLightEvent 発行
    doTest1(true,                       // LightEvent は Signaled
            EVENT_MANUAL_CLEAR,         // LightEvent は手動クリアモード
            THR_CALL_TRY_WAIT,          // Thread1 は TryWaitLightEvent()
            true,                       // Thread1 での API 返値
            true,                       // LightEvent は Signaled
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(TryWaitLightEventManualClear, test_TryWaitLightEventManualClear2)
{
    // MANUAL_CLEAR/NotSignaled に対して TryWaitLightEvent 発行
    doTest1(false,                      // LightEvent は Not Signaled
            EVENT_MANUAL_CLEAR,         // LightEvent は手動クリアモード
            THR_CALL_TRY_WAIT,          // Thread1 は TryWaitLightEvent()
            false,                      // Thread1 での API 返値
            false,                      // LightEvent は Not Signaled
            THR_STATE_EXITED);          // Thread1 はシグナル待ち
}

TEST(TryWaitLightEventManualClear, test_TryWaitLightEventManualClear3)
{
    // MANUAL_CLEAR/Signaled に対して TryWaitLightEvent 発行×２スレッド
    doTest2(true,                       // LightEvent は Signaled
            EVENT_MANUAL_CLEAR,         // LightEvent は手動クリアモード
            THR_CALL_TRY_WAIT,          // Thread1 は TryWaitLightEvent()
            THR_CALL_TRY_WAIT,          // Thread1 は TryWaitLightEvent()
            true,                       // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            true,                       // LightEvent は Signaled
            THR_STATE_EXITED,           // Thread1 は終了
            THR_STATE_EXITED);          // Thread2 は終了
}

TEST(TryWaitLightEventManualClear, test_TryWaitLightEventManualClear4)
{
    // MANUAL_CLEAR/NotSignaled に対して TryWaitLightEvent 発行×２スレッド
    doTest2(false,                      // LightEvent は Not Signaled
            EVENT_MANUAL_CLEAR,         // LightEvent は手動クリアモード
            THR_CALL_TRY_WAIT,          // Thread1 は TryWaitLightEvent()
            THR_CALL_TRY_WAIT,          // Thread1 は TryWaitLightEvent()
            false,                      // Thread1 での API 返値
            false,                      // Thread2 での API 返値
            false,                      // LightEvent は Not Signaled
            THR_STATE_EXITED,           // Thread1 はシグナル待ち
            THR_STATE_EXITED);          // Thread2 はシグナル待ち
}

//---------------------------------------------------------------------------
// 単独の TimedWaitLightEvent() AutoClear テスト

TEST(TimedWaitLightEventAutoClear, test_TimedWaitLightEventAutoClear1)
{
    // AUTO_CLEAR/Signaled のイベントに TimedWaitLightEvent 発行
    doTest1(true,                       // LightEvent は Signaled
            EVENT_AUTO_CLEAR,           // LightEvent は自動クリアモード
            THR_CALL_TIMED_WAIT,        // Thread1 は TimedWaitLightEvent()
            true,                       // Thread1 での API 返値
            false,                      // LightEvent は Not Signaled
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(TimedWaitLightEventAutoClear, test_TimedWaitLightEventAutoClear2)
{
    // AUTO_CLEAR/NotSignaled のイベントに TimedWaitLightEvent 発行
    doTest1(false,                      // LightEvent は Not Signaled
            EVENT_AUTO_CLEAR,           // LightEvent は自動クリアモード
            THR_WAIT_TIMED_WAIT,        // Thread1 は TimedWaitLightEvent()
            false,                      // Thread1 での API 返値
            false,                      // LightEvent は Not Signaled
            THR_STATE_TIMEDOUT);        // Thread1 はタイムアウト
}

TEST(TimedWaitLightEventAutoClear, test_TimedWaitLightEventAutoClear3)
{
    // AUTO_CLEAR/Signaled のイベントに TimedWaitLightEvent 発行×２スレッド
    doTest2(true,                       // LightEvent は Signaled
            EVENT_AUTO_CLEAR,           // LightEvent は自動クリアモード
            THR_CALL_TIMED_WAIT,        // Thread1 は TimedWaitLightEvent()
            THR_WAIT_TIMED_WAIT,        // Thread1 は TimedWaitLightEvent()
            true,                       // Thread1 での API 返値
            false,                      // Thread2 での API 返値
            false,                      // LightEvent は Not Signaled
            THR_STATE_EXITED,           // Thread1 は終了
            THR_STATE_TIMEDOUT);        // Thread2 はタイムアウト
}

TEST(TimedWaitLightEventAutoClear, test_TimedWaitLightEventAutoClear4)
{
    // AUTO_CLEAR/NotSignaled のイベントに TimedWaitLightEvent 発行×２スレッド
    doTest2(false,                      // LightEvent は Not Signaled
            EVENT_AUTO_CLEAR,           // LightEvent は自動クリアモード
            THR_WAIT_TIMED_WAIT,        // Thread1 は TimedWaitLightEvent()
            THR_WAIT_TIMED_WAIT,        // Thread1 は TimedWaitLightEvent()
            false,                      // Thread1 での API 返値
            false,                      // Thread2 での API 返値
            false,                      // LightEvent は Not Signaled
            THR_STATE_TIMEDOUT,         // Thread1 はタイムアウト
            THR_STATE_TIMEDOUT);        // Thread2 はタイムアウト
}

//---------------------------------------------------------------------------
// 単独の TimedWaitLightEvent() ManualClear テスト

TEST(TimedWaitLightEventManualClear, test_TimedWaitLightEventManualClear1)
{
    // MANUAL_CLEAR/Signaled のイベントに TimedWaitLightEvent() 発行
    doTest1(true,                       // LightEvent は Signaled
            EVENT_MANUAL_CLEAR,         // LightEvent は手動クリアモード
            THR_CALL_TIMED_WAIT,        // Thread1 は TimedWaitLightEvent()
            true,                       // Thread1 での API 返値
            true,                       // LightEvent は Signaled
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(TimedWaitLightEventManualClear, test_TimedWaitLightEventManualClear2)
{
    // MANUAL_CLEAR/NotSignaled のイベントに TimedWaitLightEvent() 発行
    doTest1(false,                      // LightEvent は Not Signaled
            EVENT_MANUAL_CLEAR,         // LightEvent は手動クリアモード
            THR_WAIT_TIMED_WAIT,        // Thread1 は TimedWaitLightEvent()
            false,                      // Thread1 での API 返値
            false,                      // LightEvent は Not Signaled
            THR_STATE_TIMEDOUT);        // Thread1 はタイムアウト
}

TEST(TimedWaitLightEventManualClear, test_TimedWaitLightEventManualClear3)
{
    // MANUAL_CLEAR/Signaled のイベントに TimedWaitLightEvent() ×２スレッド
    doTest2(true,                       // LightEvent は Signaled
            EVENT_MANUAL_CLEAR,         // LightEvent は手動クリアモード
            THR_CALL_TIMED_WAIT,        // Thread1 は TimedWaitLightEvent()
            THR_CALL_TIMED_WAIT,        // Thread1 は TimedWaitLightEvent()
            true,                       // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            true,                       // LightEvent は Signaled
            THR_STATE_EXITED,           // Thread1 は終了
            THR_STATE_EXITED);          // Thread2 は終了
}

TEST(TimedWaitLightEventManualClear, test_TimedWaitLightEventManualClear4)
{
    // MANUAL_CLEAR/NotSignaled のイベントに TimedWaitLightEvent() ×２スレッド
    doTest2(false,                      // LightEvent は Not Signaled
            EVENT_MANUAL_CLEAR,         // LightEvent は手動クリアモード
            THR_WAIT_TIMED_WAIT,        // Thread1 は TimedWaitLightEvent()
            THR_WAIT_TIMED_WAIT,        // Thread1 は TimedWaitLightEvent()
            false,                      // Thread1 での API 返値
            false,                      // Thread2 での API 返値
            false,                      // LightEvent は Not Signaled
            THR_STATE_TIMEDOUT,         // Thread1 はタイムアウト
            THR_STATE_TIMEDOUT);        // Thread2 はタイムアウト
}


// 負のタイムアウト値を許容することになったときのためにテストを残しておく
#if 0
//---------------------------------------------------------------------------
// 単独の TimedWaitLightEvent() テスト（負のタイムアウト値）

TEST(TimedWaitLightEventWithMinusTimeout, test_TimedWaitLightEventWithMinusTimeout1)
{
    // AUTO_CLEAR/Signaled のイベントに対して TimedWaitLightEvent 発行
    doTest1(true,                       // LightEvent は Signaled
            EVENT_AUTO_CLEAR,           // LightEvent は自動クリアモード
            THR_CALL_TIMED_WAIT_MINUS,  // Thread1 は TimedWaitLightEvent()
            true,                       // Thread1 での API 返値
            false,                      // LightEvent は Not Signaled
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(TimedWaitLightEventWithMinusTimeout, test_TimedWaitLightEventWithMinusTimeout2)
{
    // AUTO_CLEAR/NotSignaled のイベントに対して TimedWaitLightEvent 発行
    doTest1(false,                      // LightEvent は Not Signaled
            EVENT_AUTO_CLEAR,           // LightEvent は自動クリアモード
            THR_WAIT_TIMED_WAIT_MINUS,  // Thread1 は TimedWaitLightEvent()
            false,                      // Thread1 での API 返値
            false,                      // LightEvent は Not Signaled
            THR_STATE_TIMEDOUT);        // Thread1 はタイムアウト
}

#else
// 負のタイムアウト値を許容しない場合の DEATH TEST

TEST(TimedWaitLightEventWithMinusTimeoutDeathTest, test_TimedWaitLightEventWithMinusTimeoutDeathTest)
{
    // AUTO_CLEAR/Signaled のイベントに対して TimedWaitLightEvent 発行
    EXPECT_DEATH_IF_SUPPORTED(
        doTest1(true,                       // LightEvent は Signaled
                EVENT_AUTO_CLEAR,           // LightEvent は自動クリアモード
                THR_CALL_TIMED_WAIT_MINUS,  // Thread1 は TimedWaitLightEvent()
                true,                       // Thread1 での API 返値
                false,                      // LightEvent は Not Signaled
                THR_STATE_EXITED)           // Thread1 は終了
    , "");
}
#endif

//---------------------------------------------------------------------------
// AUTO_CLEAR/Wait 待機中に Signal する（TimedWait 含む）

TEST(CombinedLightEventTestAutoClear, test_WaitAndSignalAutoClear)
{
    // Thread1 で WaitLightEvent()、Thread2 で SignalLightEvent()
    doTest2(false,                      // LightEvent は Not Signaled
            EVENT_AUTO_CLEAR,           // LightEvent は自動クリアモード
            THR_WAIT_WAIT,              // Thread1 は WaitLightEvent()
            THR_CALL_SIGNAL,            // Thread2 は SignalLightEvent()
            true,                       // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            false,                      // LightEvent は Not Signaled
            THR_STATE_EXITED,           // Thread1 は終了
            THR_STATE_EXITED);          // Thread2 も終了
}

TEST(CombinedLightEventTestAutoClear, test_TimdeWaitAndSignalAutoClear)
{
    // Thread1 で WaitLightEvent()、Thread2 で SignalLightEvent()
    doTest2(false,                      // LightEvent は Not Signaled
            EVENT_AUTO_CLEAR,           // LightEvent は自動クリアモード
            THR_WAIT_TIMED_WAIT,        // Thread1 は TimedWaitLightEvent()
            THR_CALL_SIGNAL,            // Thread2 は SignalLightEvent()
            true,                       // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            false,                      // LightEvent は Not Signaled
            THR_STATE_EXITED,           // Thread1 は終了
            THR_STATE_EXITED);          // Thread2 も終了
}

//---------------------------------------------------------------------------
// MANUAL_CLEAR/Wait 待機中に Signal する（TimedWait 含む）

TEST(CombinedLightEventTestManualClear, test_WaitAndSignalManualClear)
{
    // Thread1 で WaitLightEvent()、Thread2 で SignalLightEvent()
    doTest2(false,                      // LightEvent は Not Signaled
            EVENT_MANUAL_CLEAR,         // LightEvent は手動クリアモード
            THR_WAIT_WAIT,              // Thread1 は WaitLightEvent()
            THR_CALL_SIGNAL,            // Thread2 は SignalLightEvent()
            true,                       // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            true,                       // LightEvent は Not Signaled
            THR_STATE_EXITED,           // Thread1 は終了
            THR_STATE_EXITED);          // Thread2 も終了
}

TEST(CombinedLightEventTestManualClear, test_TimdeWaitAndSignalManualClear)
{
    // Thread1 で WaitLightEvent()、Thread2 で SignalLightEvent()
    doTest2(false,                      // LightEvent は Not Signaled
            EVENT_MANUAL_CLEAR,         // LightEvent は手動クリアモード
            THR_WAIT_TIMED_WAIT,        // Thread1 は TimedWaitLightEvent()
            THR_CALL_SIGNAL,            // Thread2 は SignalLightEvent()
            true,                       // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            true,                       // LightEvent は Not Signaled
            THR_STATE_EXITED,           // Thread1 は終了
            THR_STATE_EXITED);          // Thread2 も終了
}

//---------------------------------------------------------------------------
//  LightEvent クラスのテストその１
//  ここでは、LightEvent クラスが OS-API をラッピングしたものであるという
//  前提で各メソッドの簡単な動作確認のみを行なうテストである。
//  test_LightEventClassTestAutoClear ではブロックしない AutoClear イベントの
//  テストを実施する。
//
TEST(LightEventClass, test_LightEventClassTestAutoClear)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    {
        const nn::TimeSpan timeout = nn::TimeSpan::FromMicroSeconds(1);

        SEQ_SET(4000);
        NNT_OS_LOG(NN_TEXT("main: LightEvent インスタンスの生成（AutoClear）\n"));
        nn::os::LightEvent lightEvent(nn::os::EventClearMode_AutoClear);

        // ここからがテスト
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: LightEvent::TryWait() が false を返す"));
        CheckBool( lightEvent.TryWait() == false );

        // Wait のテスト
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: LightEvent::Signal() に成功する\n"));
        lightEvent.Signal();

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: LightEvent::Wait() に成功する\n"));
        lightEvent.Wait();

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: LightEvent::TryWait() が false を返す"));
        CheckBool( lightEvent.TryWait() == false );

        // TryWait のテスト
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: LightEvent::Signal() に成功する\n"));
        lightEvent.Signal();

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: LightEvent::TryWait() が true を返す"));
        CheckBool( lightEvent.TryWait() == true );

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: LightEvent::TryWait() が false を返す"));
        CheckBool( lightEvent.TryWait() == false );

        // TimedWait のテスト
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: LightEvent::Signal() に成功する\n"));
        lightEvent.Signal();

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: LightEvent::TimedWait() が true を返す"));
        CheckBool( lightEvent.TimedWait(timeout) == true );

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: LightEvent::TimedWait(0) が false を返す"));
        CheckBool( lightEvent.TimedWait(nn::TimeSpan(0)) == false );
    }

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}

//---------------------------------------------------------------------------
//  LightEvent クラスのテストその２
//  ここでは、LightEvent クラスが OS-API をラッピングしたものであるという
//  前提で各メソッドの簡単な動作確認のみを行なうテストである。
//  test_LightEventClassTestManualClear ではブロックしない ManualClear イベントの
//  テストを実施する。
//
TEST(LightEventClass, test_LightEventClassTestManualClear)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    {
        const nn::TimeSpan timeout = nn::TimeSpan::FromMicroSeconds(1);

        SEQ_SET(4100);
        NNT_OS_LOG(NN_TEXT("main: LightEvent インスタンスの生成（ManualClear）\n"));
        nn::os::LightEvent lightEvent(nn::os::EventClearMode_ManualClear);

        // ここからがテスト
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: LightEvent::TryWait() が false を返す"));
        CheckBool( lightEvent.TryWait() == false );

        // Wait のテスト
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: LightEvent::Signal() に成功する\n"));
        lightEvent.Signal();

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: LightEvent::Wait() に成功する\n"));
        lightEvent.Wait();

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: LightEvent::TryWait() が true を返す"));
        CheckBool( lightEvent.TryWait() == true );

        // TryWait のテスト
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: LightEvent::Signal() に成功する\n"));
        lightEvent.Signal();

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: LightEvent::TryWait() が true を返す"));
        CheckBool( lightEvent.TryWait() == true );

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: LightEvent::TryWait() が true を返す"));
        CheckBool( lightEvent.TryWait() == true );

        // TimedWait のテスト
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: LightEvent::Signal() に成功する\n"));
        lightEvent.Signal();

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: LightEvent::TimedWait() が true を返す"));
        CheckBool( lightEvent.TimedWait(timeout) == true );

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: LightEvent::TimedWait(0) が true を返す"));
        CheckBool( lightEvent.TimedWait(nn::TimeSpan(0)) == true );

        // Clear の確認
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: LightEvent::Clear() に成功する\n"));
        lightEvent.Clear();

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: LightEvent::TryWait() が false を返す"));
        CheckBool( lightEvent.TryWait() == false );

    }

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}

//---------------------------------------------------------------------------
//  LightEvent クラスのテストその３
//  ここでは、LightEvent クラスが OS-API をラッピングしたものであるという
//  前提で各メソッドの簡単な動作確認のみを行なうテストである。
//  test_LightEventClassTestBlock ではブロック系のテストを２スレッド使って実施する。
//
NN_ALIGNAS( 4096 ) char g_StackOfThread[ 8192 ];

void LightEventClassTestThread(void* arg)
{
    auto& lightEvent = *static_cast<nn::os::LightEvent*>(arg);
    auto  interval   = nn::TimeSpan::FromMilliSeconds(100);

    for (int i = 0; i < 1; ++i)
    {
        nn::os::SleepThread( interval );

        SEQ_CHECK(4202 + i * 3);
        NNT_OS_LOG(NN_TEXT("sub:  LightEvent::Signal() でシグナル化\n"));
        lightEvent.Signal();
    }
}

TEST(LightEventClass, test_LightEventClassTestBlock)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    {
        const nn::TimeSpan timeout = nn::TimeSpan::FromSeconds(1);

        SEQ_SET(4200);
        NNT_OS_LOG(NN_TEXT("main: LightEvent インスタンスの生成"));
        nn::os::LightEvent lightEvent(nn::os::EventClearMode_AutoClear);

        // テスト用の子スレッドを作成
        nn::os::ThreadType  thread;
        {
            auto pri    = nn::os::GetThreadPriority(nn::os::GetCurrentThread());
            auto result = nn::os::CreateThread( &thread,
                                                &LightEventClassTestThread,
                                                &lightEvent,
                                                g_StackOfThread,
                                                sizeof(g_StackOfThread),
                                                pri + 1);
            CheckBool( result.IsSuccess() );
        }

        // 子スレッドを起動
        nn::os::StartThread( &thread );

        // Step1: Wait で待機して子スレッドから起床
        SEQ_CHECK(4201);
        NNT_OS_LOG(NN_TEXT("main: LightEvent::Wait() で待機\n"));
        lightEvent.Wait();
        SEQ_CHECK(4203);
        NNT_OS_LOG(NN_TEXT("main: -> 起床\n"));

        // Step2: TimedWait で待機し、子スレッドからシグナルしてもらう
        SEQ_CHECK(4204);
        NNT_OS_LOG(NN_TEXT("main: LightEvent::TimedWait() で待機\n"));
        lightEvent.TimedWait(timeout);
        SEQ_CHECK(4206);
        NNT_OS_LOG(NN_TEXT("main: -> 起床\n"));

        // 子スレッドの終了待ち
        nn::os::WaitThread( &thread );
        nn::os::DestroyThread( &thread );
    }

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}

//---------------------------------------------------------------------------
//  LightEvent クラスの型変換関数のテスト
//  ここでは、LightEvent クラスが OS-API をラッピングしたものであるという
//  前提で各メソッドの簡単な動作確認のみを行なうテストである。
//  ここでは、GetBase() および operator LightEventType&() の動作テストを行なう。
//
TEST(LightEventClass, test_LightEventClassTestTypeExchange)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    {
        SEQ_SET(4300);
        NNT_OS_LOG(NN_TEXT("LightEvent インスタンスの生成（AutoClear）\n"));
        nn::os::LightEvent lightEvent(nn::os::EventClearMode_AutoClear);

        // ここからがテスト
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("LightEvent::GetBase() で LightEventType オブジェクトを取得"));
        nn::os::LightEventType*  lightEventType = lightEvent.GetBase();
        CheckBool( lightEventType != NULL );

        // 初期シグナル状態が false であることを確認
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("LightEvent::TryWait() で非シグナル状態であることを確認"));
        CheckBool( lightEvent.TryWait() == false );

        // LightEventType にてセットし、TryWait で true を確認
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("lightEventType を使ってシグナル状態にセット"));
        nn::os::SignalLightEvent(lightEventType);
        CheckBool( lightEvent.TryWait() == true );

        // LightEventType にてクリアし、TryWait で false を確認
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("lightEventType を使って非シグナル状態にクリア"));
        nn::os::ClearLightEvent(lightEventType);
        CheckBool( lightEvent.TryWait() == false );

        // operator LightEventType&() の確認
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("operator LightEventType&() を使ってオブジェクトの参照を取得"));
        nn::os::LightEventType& lightEventTypeRefer = lightEvent;
        CheckBool( &lightEventTypeRefer == lightEventType );

        // operator const LightEventType&() の確認
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("operator const LightEventType&() を使ってオブジェクトの参照を取得"));
        const nn::os::LightEventType& lightEventTypeReferConst = lightEvent;
        CheckBool( &lightEventTypeReferConst == lightEventType );
    }

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}


}}} // namespace nnt::os::lightEvent

//---------------------------------------------------------------------------
//  Test Main 関数
//---------------------------------------------------------------------------

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    int result;

    NNT_CALIBRATE_INITIALIZE();
    SEQ_INITIALIZE();
    INITIALIZE_TEST_COUNT();

    // テスト開始
    SEQ_CHECK(0);
    NNT_OS_LOG("=== Start Test of LightEvent APIs\n");

    // GoogleTest おまじない
    ::testing::InitGoogleTest(&argc, argv);
    result = RUN_ALL_TESTS();

    // テスト終了
    NNT_OS_LOG("\n=== End Test of LightEvent APIs\n");

    // 集計結果の表示
    g_Result.Show();

    nnt::Exit(result);
}
