﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//---------------------------------------------------------------------------
//  InterruptEvent 関連機能のテスト
//---------------------------------------------------------------------------

#include "../Common/test_Pragma.h"

#include <nn/os/os_Config.h>
#include <nn/nn_SdkText.h>
#include <nn/nn_Common.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os.h>
#include <nn/os/os_InterruptEvent.h>
#include "../Common/test_Helper.h"
#include "../Common/test_Calibration.h"

#if defined( NN_BUILD_CONFIG_OS_WIN32 )
    #include "test_InterruptEvent-os.win32.h"
#elif defined( NN_BUILD_CONFIG_OS_HORIZON )
    #include "test_InterruptEvent-os.horizon.h"
#endif

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>

#define AUTO_CLEAR    nn::os::EventClearMode_AutoClear
#define MANUAL_CLEAR  nn::os::EventClearMode_ManualClear

namespace nnt { namespace os { namespace interruptEvent {

#if defined( NN_BUILD_CONFIG_OS_WIN32 ) || \
    (defined( NN_BUILD_CONFIG_OS_HORIZON ) && defined( NN_BUILD_CONFIG_HARDWARE_BDSLIMX6 )) || \
    (defined( NN_BUILD_CONFIG_OS_HORIZON ) && defined( NN_BUILD_CONFIG_HARDWARE_JETSONTK1 )) || \
    (defined( NN_BUILD_CONFIG_OS_HORIZON ) && defined( NN_BUILD_CONFIG_HARDWARE_JETSONTK2 )) || \
    (defined( NN_BUILD_CONFIG_OS_HORIZON ) && defined( NN_BUILD_CONFIG_SOC_TEGRA_X1 )) || \
    (defined( NN_BUILD_CONFIG_OS_HORIZON ) && defined( NN_BUILD_CONFIG_HARDWARE_JUNO )) || \
    (defined( NN_BUILD_CONFIG_OS_HORIZON ) && defined( NN_BUILD_CONFIG_HARDWARE_SMMA53 ))
//---------------------------------------------------------------------------
// 周期的な InterruptEvent 受理（AutoReset）
//
//  100msec 周期で発生＆自動クリアされる InterruptNameTest1" という名前を持つ
//  割込みイベントを WaitInterruptEvent で受理するプログラム。
//  Tick を使って時間を計測しながら、起床を 10 回繰り返す。
//
TEST(InterruptEvent, test_PeriodicWaitInterruptEventAutoReset)
{
    nn::os::InterruptEventType  irqEvent;

    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();
    NNT_OS_LOG(NN_TEXT("<< AutoReset の InterruptEvent に対する Wait のテスト >>\n"));
    NNT_OS_LOG(NN_TEXT("<< 100msec 毎にシグナル化される割込みイベントを待機できているか >>\n"));

    // 割込みイベントの初期化
    nn::os::InitializeInterruptEvent( &irqEvent,
                                      InterruptNameTest1,
                                      AUTO_CLEAR );

    // 前準備
    ClearTimerInterruptStatus();
    nn::os::ClearInterruptEvent( &irqEvent );

    nn::os::WaitInterruptEvent( &irqEvent );
    ClearTimerInterruptStatus();
    nn::os::ClearInterruptEvent( &irqEvent );

    // 割込みイベント待機を 10 回繰り返す
    nn::os::Tick    prevTick = nn::os::GetSystemTick();

    for  (int i = 0; i < 10; ++i)
    {
        // 割込みイベント待ち
        nn::os::WaitInterruptEvent( &irqEvent );
        ClearTimerInterruptStatus();

        // 前回起床した時からの時間を出力
        nn::os::Tick currTick = nn::os::GetSystemTick();
        nn::TimeSpan span     = nn::os::ConvertToTimeSpan(currTick - prevTick);
        NNT_OS_LOG("Span = %d (usec)", span.GetMicroSeconds() );

        // およそ 100msec 周期か？（誤差 80% まで許容）
        CheckBool( (span.GetMicroSeconds() >=  20 * 1000) &&
                   (span.GetMicroSeconds() <= 180 * 1000) );

        prevTick = currTick;
    }

    // 割込みイベントの削除
    nn::os::FinalizeInterruptEvent( &irqEvent );

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}

//---------------------------------------------------------------------------
// 周期的な InterruptEvent 受理（ManualReset）
//
//  100msec 周期で発生＆自動クリアされない InterruptNameTest2 という名前を持つ
//  割込みイベントを WaitInterruptEvent で受理するプログラム。
//  Tick を使って時間を計測しながら、起床を 10 回繰り返す。
//  また、割込みクリアを行なわなかった場合はすぐに起床できることも 10 回確認。
//
TEST(InterruptEvent, test_PeriodicWaitInterruptEventManualReset)
{
    nn::TimeSpan    spanSequence1( 0 );
    nn::TimeSpan    spanSequence2( 0 );
    nn::os::InterruptEventType  irqEvent;

    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();
    NNT_OS_LOG(NN_TEXT("<< ManualReset の InterruptEvent に対する Wait と Clear のテスト >>\n"));
    NNT_OS_LOG(NN_TEXT("<< 100msec 毎にシグナル化される割込みイベントを待機できているか >>\n"));

    // 割込みイベントの初期化
    nn::os::InitializeInterruptEvent( &irqEvent,
                                      InterruptNameTest2,
                                      MANUAL_CLEAR );

    // 前準備
    ClearTimerInterruptStatus();
    nn::os::ClearInterruptEvent( &irqEvent );

    nn::os::WaitInterruptEvent( &irqEvent );
    ClearTimerInterruptStatus();
    nn::os::ClearInterruptEvent( &irqEvent );

    // 割込みイベント待機を 10 回繰り返す
    nn::os::Tick    prevTick = nn::os::GetSystemTick();

    // 正しくクリアを行なう場合は、約 100msec 毎に出力される。
    for  (int i = 0; i < 10; ++i)
    {
        // 割込みイベント待ち と クリア
        nn::os::WaitInterruptEvent( &irqEvent );
        ClearTimerInterruptStatus();
        nn::os::ClearInterruptEvent( &irqEvent );

        // 前回起床した時からの時間を出力
        nn::os::Tick currTick = nn::os::GetSystemTick();
        nn::TimeSpan span     = nn::os::ConvertToTimeSpan(currTick - prevTick);
        NNT_OS_LOG("Span = %d (usec)", span.GetMicroSeconds() );

        // およそ 100msec 周期か？（誤差 80% まで許容）
        CheckBool( (span.GetMicroSeconds() >=  20 * 1000) &&
                   (span.GetMicroSeconds() <= 180 * 1000) );

        spanSequence1 += span;
        prevTick = currTick;
    }

    // クリアを行なわない場合は、セットされたままなので直ぐに返ってくる
    nn::os::WaitInterruptEvent( &irqEvent );
    prevTick = nn::os::GetSystemTick();

    for  (int i = 0; i < 10; ++i)
    {
        // 割込みイベント待ち と クリア
        nn::os::WaitInterruptEvent( &irqEvent );
        ClearTimerInterruptStatus();

        // 前回起床した時からの時間を出力
        nn::os::Tick currTick = nn::os::GetSystemTick();
        nn::TimeSpan span     = nn::os::ConvertToTimeSpan(currTick - prevTick);
        NNT_OS_LOG("Span = %d (usec)\n", span.GetMicroSeconds() );

        spanSequence2 += span;
        prevTick = currTick;
    }

    // 計測結果を表示
    NNT_OS_LOG(NN_TEXT("- Wait と Clear を行なう場合の実行時間：        %7d (usec)\n"), spanSequence1.GetMicroSeconds());
    NNT_OS_LOG(NN_TEXT("- Wait だけで Clear を行なわない場合の実行時間：%7d (usec)"), spanSequence2.GetMicroSeconds());
    CheckBool( spanSequence1.GetMilliSeconds() > spanSequence2.GetMilliSeconds() );

    // 割込みイベントの削除
    ClearTimerInterruptStatus();
    nn::os::ClearInterruptEvent( &irqEvent );
    nn::os::FinalizeInterruptEvent( &irqEvent );

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}


//---------------------------------------------------------------------------
// TryWaitInterruptEvent 動作テスト
//
//  100msec 周期で発生＆自動クリアされない InterruptNameTest2 という名前を持つ
//  割込みイベントを TryWaitInterruptEvent で受理するプログラム。
//  シグナル状態の割込みイベントに対して TryWait して true を確認した後、
//  非シグナル状態の割込みイベントに対しても TryWait して false を確認。
//
TEST(InterruptEvent, test_TryWaitInterruptEvent)
{
    nn::os::InterruptEventType  irqEvent;

    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();
    NNT_OS_LOG(NN_TEXT("<< ManualReset の InterruptEvent に対する TryWait のテスト >>\n"));

    // 割込みイベントの初期化
    nn::os::InitializeInterruptEvent( &irqEvent,
                                      InterruptNameTest2,
                                      MANUAL_CLEAR );

    // TryWait で true になるケースの確認（ManualReset イベントを利用）
    //  - まず WaitInterruptEvent() でシグナル化するまで待つ
    //  - 次に TryWaitInterruptEvent() で true になるのを確認

    NNT_OS_LOG(NN_TEXT("- Wait 直後の TryWait で true になるか"));
    nn::os::WaitInterruptEvent( &irqEvent );

    bool ret = nn::os::TryWaitInterruptEvent( &irqEvent );
    CheckBool( ret == true );

    // TryWait で false になるケースの確認（ManualReset イベントを利用）
    //  - まず WaitInterruptEvent() でシグナル化するまで待つ
    //  - 次に ClearInterruptEvent() でクリアする
    //  - 直ぐに TryWaitInterruptEvent() して false になるのを確認

    NNT_OS_LOG(NN_TEXT("- Wait 直後に Clear し、すぐに TryWait で false になるか"));

    ClearTimerInterruptStatus();
    nn::os::ClearInterruptEvent( &irqEvent );

    nn::os::WaitInterruptEvent( &irqEvent );
    ClearTimerInterruptStatus();
    nn::os::ClearInterruptEvent( &irqEvent );

    ret = nn::os::TryWaitInterruptEvent( &irqEvent );
    CheckBool( ret == false );

    // 割込みイベントの削除
    nn::os::FinalizeInterruptEvent( &irqEvent );

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}


//---------------------------------------------------------------------------
// TimedWaitInterruptEvent 動作テスト
//
//  100msec 周期で発生＆自動クリアされない InterruptNameTest2 と、
//  発生すらしない InterruptNameTest3 という名前を持つ割込みイベントを
//  WaitInterruptEvent で受理するプログラム。
//  Tick を使って時間を計測しながら、起床を 10 回繰り返す。
//
TEST(InterruptEvent, test_TimedWaitInterruptEvent)
{
    nn::os::InterruptEventType  irqEvent;
    nn::os::InterruptEventType  irqEventNotSignaled;

    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();
    NNT_OS_LOG(NN_TEXT("<< ManualReset の InterruptEvent に対する TimedWait のテスト >>\n"));

    // 割込みイベントの初期化
    nn::os::InitializeInterruptEvent( &irqEvent,
                                      InterruptNameTest2,
                                      MANUAL_CLEAR );
    nn::os::InitializeInterruptEvent( &irqEventNotSignaled,
                                      InterruptNameTest3,
                                      MANUAL_CLEAR );

    // TimedWait で true になるケースの確認（ManualReset イベントを利用）
    //  - まず WaitInterruptEvent() でシグナル化するまで待つ
    //  - 次に TimedWaitInterruptEvent() で true になるのを確認

    NNT_OS_LOG(NN_TEXT("- Wait 直後の TimedWait で true になるか"));
    nn::os::WaitInterruptEvent( &irqEvent );

    bool ret = nn::os::TimedWaitInterruptEvent( &irqEvent,
                                nn::TimeSpan::FromMilliSeconds(50) );
    CheckBool( ret == true );

    // TimedWait で false になるケースの確認（ManualReset イベントを利用）
    //  - まず WaitInterruptEvent() でシグナル化するまで待つ
    //  - 次に ClearInterruptEvent() でクリアする
    //  - 直ぐに TimedWaitInterruptEvent() して false になるのを確認

    NNT_OS_LOG(NN_TEXT("- Wait 直後に Clear し、すぐに TimedWait で false になるか"));

    ClearTimerInterruptStatus();
    nn::os::ClearInterruptEvent( &irqEvent );

    nn::os::WaitInterruptEvent( &irqEvent );
    ClearTimerInterruptStatus();
    nn::os::ClearInterruptEvent( &irqEvent );
    ret = nn::os::TimedWaitInterruptEvent( &irqEvent,
                                nn::TimeSpan::FromMilliSeconds(50) );
    CheckBool( ret == false );

    // TimedWait で false になるケースの確認（NotSignaled イベントを利用）
    //  - TimedWaitInterruptEvent() してタイムアウトするのを確認

    NNT_OS_LOG(NN_TEXT("- シグナル化されない割込みイベントに対して TimedWait で false になるか"));
    nn::os::Tick startTick = nn::os::GetSystemTick();
    ret = nn::os::TimedWaitInterruptEvent( &irqEventNotSignaled,
                                nn::TimeSpan::FromMilliSeconds(50) );
    nn::os::Tick endTick   = nn::os::GetSystemTick();
    CheckBool( ret == false );

    int64_t elapse = nn::os::ConvertToTimeSpan(endTick - startTick).GetMicroSeconds();
    NNT_OS_LOG(NN_TEXT("- タイウアウト計測時間は 50000usec 以上か（result: %d usec）"), elapse);
    CheckBool( elapse >= 50000 );

    // 割込みイベントの削除
    nn::os::FinalizeInterruptEvent( &irqEvent );
    nn::os::FinalizeInterruptEvent( &irqEventNotSignaled );

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}


// 負のタイムアウト値を許容することになったときのためにテストを残しておく
#if 0
//---------------------------------------------------------------------------
// TimedWaitInterruptEvent 負のタイムアウト値の動作テスト
//
//  100msec 周期で発生＆自動クリアされない InterruptNameTest2 と、
//  発生すらしない InterruptNameTest3 という名前を持つ割込みイベントを
//  WaitInterruptEvent で受理するプログラム。
//  それぞれ負のタイムアウト値を指定して確認する。
//
TEST(InterruptEvent, test_TimedWaitInterruptEventWithMinusTimeout)
{
    nn::os::InterruptEventType  irqEvent;
    nn::os::InterruptEventType  irqEventNotSignaled;

    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();
    NNT_OS_LOG(NN_TEXT("<< ManualReset の InterruptEvent に対する TimedWait のテスト >>\n"));

    // 割込みイベントの初期化
    nn::os::InitializeInterruptEvent( &irqEvent,
                                      InterruptNameTest2,
                                      MANUAL_CLEAR );
    nn::os::InitializeInterruptEvent( &irqEventNotSignaled,
                                      InterruptNameTest3,
                                      MANUAL_CLEAR );

    // TimedWait で true になるケースの確認（ManualReset イベントを利用）
    //  - まず WaitInterruptEvent() でシグナル化するまで待つ
    //  - 次に TimedWaitInterruptEvent() で true になるのを確認

    NNT_OS_LOG(NN_TEXT("- タイムアウト値が負でも Wait 直後の TimedWait で true になるか"));
    nn::os::WaitInterruptEvent( &irqEvent );

    bool ret = nn::os::TimedWaitInterruptEvent( &irqEvent,
                                nn::TimeSpan::FromNanoSeconds(-1) );
    CheckBool( ret == true );

    // TimedWait で false になるケースの確認（NotSignaled イベントを利用）
    //  - TimedWaitInterruptEvent() してタイムアウトするのを確認

    NNT_OS_LOG(NN_TEXT("- タイムアウト値が負でもシグナル化されない割込みイベントに対して TimedWait で false になるか"));
    nn::os::Tick startTick = nn::os::GetSystemTick();
    ret = nn::os::TimedWaitInterruptEvent( &irqEventNotSignaled,
                                nn::TimeSpan::FromNanoSeconds(-1) );
    nn::os::Tick endTick   = nn::os::GetSystemTick();
    CheckBool( ret == false );

    int64_t elapse = nn::os::ConvertToTimeSpan(endTick - startTick).GetMicroSeconds();
    NNT_OS_LOG(NN_TEXT("- タイウアウト計測時間は即時（50000usec 以下）か（result: %d usec）"), elapse);
    CheckBool( elapse <= 50000 );

    // 割込みイベントの削除
    nn::os::FinalizeInterruptEvent( &irqEvent );
    nn::os::FinalizeInterruptEvent( &irqEventNotSignaled );

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}

#else
// 負のタイムアウト値を許容しない場合の DEATH TEST

TEST(InterruptEventDeathTest, test_TimedWaitInterruptEventWithMinusTimeoutDeathTest)
{
    nn::os::InterruptEventType  irqEvent;

    // 割込みイベントの初期化
    nn::os::InitializeInterruptEvent( &irqEvent,
                                      InterruptNameTest2,
                                      MANUAL_CLEAR );

    EXPECT_DEATH_IF_SUPPORTED(
        nn::os::TimedWaitInterruptEvent( &irqEvent, nn::TimeSpan::FromNanoSeconds(-1) )
    , "");

    // 割込みイベントの削除
    nn::os::FinalizeInterruptEvent( &irqEvent );
}

#endif

//---------------------------------------------------------------------------
//  InterruptEvent クラスのテスト
//  ここでは、InterruptEvent クラスが OS-API をラッピングしたものであるという
//  前提で各メソッドの動作確認テストを行なう。
//  100msec 周期で発生＆自動クリアされない InterruptNameTest2 という名前を持つ
//  割込みイベントを InterruptEvent クラスを用いて受理するテスト。
//
TEST(InterruptEventClass, test_InterruptEventClassTestManualClear)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    {
        const nn::TimeSpan  zeroTimeout  = 0;
        const nn::TimeSpan  shortTimeout = nn::TimeSpan::FromMicroSeconds(1);
        const nn::TimeSpan  longTimeout  = nn::TimeSpan::FromSeconds(10);

        NNT_OS_LOG(NN_TEXT("<< ManualReset の周期割込みに対する InterruptEvent クラスのテスト >>\n"));

        // 割込みイベントクラスのインスタンスを生成
        NNT_OS_LOG(NN_TEXT("main: interruptEvent インスタンスの生成（ManualClear）\n"));
        nn::os::InterruptEvent interruptEvent(InterruptNameTest2, MANUAL_CLEAR);

        // ここからがテスト

        // Wait のテスト
        NNT_OS_LOG(NN_TEXT("main: interruptEvent.Clear() に成功する\n"));
        ClearTimerInterruptStatus();
        interruptEvent.Clear();

        NNT_OS_LOG(NN_TEXT("main: interruptEvent.Wait() に成功する\n"));
        interruptEvent.Wait();

        NNT_OS_LOG(NN_TEXT("main: interruptEvent.Wait() に成功する（すぐに返る）\n"));
        interruptEvent.Wait();

        // TryWait のテスト
        NNT_OS_LOG(NN_TEXT("main: interruptEvent.TryWait() が true を返す"));
        CheckBool( interruptEvent.TryWait() == true );

        NNT_OS_LOG(NN_TEXT("main: interruptEvent.Clear() に成功する\n"));
        ClearTimerInterruptStatus();
        interruptEvent.Clear();

        // 割込み周期のタイミング調整のため、一度 Wait->Clear を行なう
        interruptEvent.Wait();
        ClearTimerInterruptStatus();
        interruptEvent.Clear();

        NNT_OS_LOG(NN_TEXT("main: interruptEvent.TryWait() が false を返す"));
        CheckBool( interruptEvent.TryWait() == false );

        // TimedWait のテスト
        NNT_OS_LOG(NN_TEXT("main: interruptEvent.TimedWait(10sec) が true を返す"));
        CheckBool( interruptEvent.TimedWait(longTimeout) == true );

        NNT_OS_LOG(NN_TEXT("main: interruptEvent.TimedWait(1usec) が true を返す"));
        CheckBool( interruptEvent.TimedWait(shortTimeout) == true );

        NNT_OS_LOG(NN_TEXT("main: interruptEvent.TimedWait(0) が true を返す"));
        CheckBool( interruptEvent.TimedWait(zeroTimeout) == true );

        // 割込み周期のタイミング調整のため、一度 Clear->Wait->Clear を行なう
        ClearTimerInterruptStatus();
        interruptEvent.Clear();
        interruptEvent.Wait();
        ClearTimerInterruptStatus();
        interruptEvent.Clear();

        NNT_OS_LOG(NN_TEXT("main: interruptEvent.TimedWait(0) が false を返す"));
        CheckBool( interruptEvent.TimedWait(zeroTimeout) == false );

        NNT_OS_LOG(NN_TEXT("main: interruptEvent.TimedWait(1usec) が false を返す"));
        CheckBool( interruptEvent.TimedWait(shortTimeout) == false );
    }

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}

//---------------------------------------------------------------------------
//  InterruptEvent クラスの型変換関数のテスト
//  ここでは、InterruptEvent クラスが OS-API をラッピングしたものであるという
//  前提で各メソッドの簡単な動作確認のみを行なうテストである。
//  ここでは、GetBase() および operator InterruptEventType&() の動作テストを行なう。
//
TEST(InterruptEventClass, test_InterruptEventClassTestTypeExchange)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    {
        NNT_OS_LOG(NN_TEXT("InterruptEvent インスタンスの生成（ManualClear）\n"));
        nn::os::InterruptEvent interruptEvent(InterruptNameTest2, MANUAL_CLEAR);

        // ここからがテスト
        NNT_OS_LOG(NN_TEXT("interruptEvent.GetBase() で InterruptEventType オブジェクトを取得"));
        nn::os::InterruptEventType*  interruptEventType = interruptEvent.GetBase();
        CheckBool( interruptEventType != NULL );

        ClearTimerInterruptStatus();
        interruptEvent.Clear();
        interruptEvent.Wait();
        ClearTimerInterruptStatus();
        interruptEvent.Clear();

        // 初期シグナル状態が false であることを確認
        NNT_OS_LOG(NN_TEXT("interruptEvent.TryWait() で非シグナル状態であることを確認"));
        CheckBool( interruptEvent.TryWait() == false );

        // 割込み発生を待って、TryWait で true を確認
        NNT_OS_LOG(NN_TEXT("interruptEventType の割込み発生を待つ"));
        nn::os::WaitInterruptEvent(interruptEventType);
        CheckBool( interruptEvent.TryWait() == true );

        // InterruptEventType にてクリアし、TryWait で false を確認
        NNT_OS_LOG(NN_TEXT("interruptEventType を使って非シグナル状態にクリア"));
        ClearTimerInterruptStatus();
        nn::os::ClearInterruptEvent(interruptEventType);
        CheckBool( interruptEvent.TryWait() == false );

        // operator InterruptEventType&() の確認
        NNT_OS_LOG(NN_TEXT("operator InterruptEventType&() を使ってオブジェクトの参照を取得"));
        nn::os::InterruptEventType& interruptEventTypeRefer = interruptEvent;
        CheckBool( &interruptEventTypeRefer == interruptEventType );

        // operator const InterruptEventType&() の確認
        NNT_OS_LOG(NN_TEXT("operator const InterruptEventType&() を使ってオブジェクトの参照を取得"));
        const nn::os::InterruptEventType& interruptEventTypeReferConst = interruptEvent;
        CheckBool( &interruptEventTypeReferConst == interruptEventType );
    }

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}

#endif


//---------------------------------------------------------------------------
//  SIGLO-31003 複数のスレッドで WaitInterruptEvent を行なうテスト。
//  AutoClear な場合の OS ライブラリ内部での ResetSignal() 実装修正用の
//  リグレッション用のテスト。
//
//  テストロジック：
//  1) Core0 で AutoClear な InterruptEvent を作成
//  2) Core1 で 1) の InterruptEvent を Wait()
//  3) Core2 で 1) の InterruptEvent を Wait()
//  4) Core0 で 1) の InterruptEvent を Signal() して、2) と 3) のスレッドの
//     片方（どちらかは問わない）だけが起床していることを確認する。
//
NN_OS_ALIGNAS_THREAD_STACK  char g_ThreadStack1[0x4000];
NN_OS_ALIGNAS_THREAD_STACK  char g_ThreadStack2[0x4000];
nn::os::Semaphore                g_Semaphore(0, 10);
nn::os::Barrier                  g_Barrier(3);

void InterruptEventSIGLO31003_Thread1(void* arg)
{
    auto& event = *reinterpret_cast<nn::os::InterruptEvent*>(arg);

    event.Wait();
    ClearTimerInterruptStatus();
    g_Semaphore.Release();
    g_Barrier.Await();

    event.TimedWait( nn::TimeSpan::FromHours(1) );
    ClearTimerInterruptStatus();
    g_Semaphore.Release();
}

void InterruptEventSIGLO31003_Thread2(void* arg)
{
    auto& event = *reinterpret_cast<nn::os::InterruptEvent*>(arg);

    event.Wait();
    ClearTimerInterruptStatus();
    g_Semaphore.Release();
    g_Barrier.Await();

    event.TimedWait( nn::TimeSpan::FromHours(1) );
    ClearTimerInterruptStatus();
    g_Semaphore.Release();
}

TEST(InterruptEventSIGLO31003, test_InterruptEventSIGLO31003_Main)
{
    // 搭載コア数を算出（最下位の 1 のビット数をカウントしてコア数とする）
    nn::Bit64 coreMask = nn::os::GetThreadAvailableCoreMask();
              coreMask = nn::util::maskt1<nn::Bit64>(coreMask);
    int       coreNums = nn::util::cntt0<nn::Bit64>(~coreMask);
    NNT_OS_LOG("Core nums = %d\n", coreNums);
    if (coreNums < 3)
    {
        // CPU コア数が 3 未満ならテストをスキップ
        NNT_OS_LOG("Skipped this test because core nums are less than 3.\n");
        SUCCEED();
        return;
    }

    nn::os::InterruptEvent event(InterruptNameTest2, AUTO_CLEAR);

    int pri = nn::os::GetThreadCurrentPriority( nn::os::GetCurrentThread() );

    nn::os::ThreadType  thread1;
    nn::os::ThreadType  thread2;
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::CreateThread(
                                        &thread1,
                                        InterruptEventSIGLO31003_Thread1,
                                        &event,
                                        g_ThreadStack1,
                                        sizeof(g_ThreadStack1),
                                        pri,
                                        1) );
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::CreateThread(
                                        &thread2,
                                        InterruptEventSIGLO31003_Thread2,
                                        &event,
                                        g_ThreadStack2,
                                        sizeof(g_ThreadStack2),
                                        pri,
                                        2) );
    // 前準備
    ClearTimerInterruptStatus();
    event.Clear();
    event.Wait();
    ClearTimerInterruptStatus();

    // スレッド動作開始
    nn::os::StartThread( &thread1 );
    nn::os::StartThread( &thread2 );

    // thread1, thread2 が InterruptEvent を Wait() してから Signal() する
    nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(100) );
    // この辺のタイミングで Timer2 によって event がシグナルされる
    nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(20) );

    // Wait が解除されるのに十分な時間を経てから
    // g_Semaphore のカウンタを見て、解除された Wait() の数を検査する
    EXPECT_EQ(1, g_Semaphore.GetCurrentCount());

    // 次のテストに移行するための準備
    ClearTimerInterruptStatus();
    event.Clear();
    event.Wait();
    ClearTimerInterruptStatus();

    // g_Semaphore カウンタも 0 にしておく。
    g_Semaphore.Acquire();
    g_Semaphore.Acquire();
    EXPECT_EQ(0, g_Semaphore.GetCurrentCount());
    g_Barrier.Await();

    // thread1, thread2 が InterruptEvent を TimedWait() してから Signal() する
    nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(100) );
    // この辺のタイミングで Timer2 によって event がシグナルされる
    nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(20) );

    // Wait が解除されるのに十分な時間を経てから
    // g_Semaphore のカウンタを見て、解除された Wait() の数を検査する
    EXPECT_EQ(1, g_Semaphore.GetCurrentCount());

    // テストを終了するための後処理
    nn::os::WaitThread( &thread1 );
    nn::os::WaitThread( &thread2 );
    nn::os::DestroyThread( &thread1 );
    nn::os::DestroyThread( &thread2 );
}

}}} // namespace nnt::os::interruptEvent

//---------------------------------------------------------------------------
//  Test Main 関数
//---------------------------------------------------------------------------

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

#if defined( NN_BUILD_CONFIG_OS_WIN32 ) || \
    (defined( NN_BUILD_CONFIG_OS_HORIZON ) && defined( NN_BUILD_CONFIG_HARDWARE_BDSLIMX6 )) || \
    (defined( NN_BUILD_CONFIG_OS_HORIZON ) && defined( NN_BUILD_CONFIG_HARDWARE_JETSONTK1 )) || \
    (defined( NN_BUILD_CONFIG_OS_HORIZON ) && defined( NN_BUILD_CONFIG_HARDWARE_JETSONTK2 )) || \
    (defined( NN_BUILD_CONFIG_OS_HORIZON ) && defined( NN_BUILD_CONFIG_SOC_TEGRA_X1 )) || \
    (defined( NN_BUILD_CONFIG_OS_HORIZON ) && defined( NN_BUILD_CONFIG_HARDWARE_JUNO )) || \
    (defined( NN_BUILD_CONFIG_OS_HORIZON ) && defined( NN_BUILD_CONFIG_HARDWARE_SMMA53 ))
//---------------------------------------------------------------------------
    nnt::os::interruptEvent::InterruptGenerator interruptGenerator;

    NNT_CALIBRATE_INITIALIZE();
    SEQ_INITIALIZE();
    INITIALIZE_TEST_COUNT();

    // テスト開始
    SEQ_CHECK(0);
    NNT_OS_LOG("=== Start Test of Event APIs\n");

    // GoogleTest おまじない
    ::testing::InitGoogleTest(&argc, argv);
    int result = RUN_ALL_TESTS();

    // テスト終了
    NNT_OS_LOG("\n=== End Test of Event APIs\n");

    // 集計結果の表示
    g_Result.Show();
#else
    int result = 0;
#endif

    nnt::Exit(result);
}
