﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../Common/test_Pragma.h"

#include <nn/os/os_Config.h>
#include <nn/nn_SdkText.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_Abort.h>
#include <nn/os.h>
#include <nn/dd.h>
#include <nn/os/os_InterruptEvent.h>
#include "../Common/test_Helper.h"

#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    #include "test_InterruptEvent-os.horizon.h"
#else
    #error "未サポートの OS 種別が指定されています。"
#endif

namespace nnt { namespace os { namespace interruptEvent {

uintptr_t   TimerRegisterVirtualAddress;

#if defined( NN_BUILD_CONFIG_HARDWARE_BDSLIMX6 )
#define NNT_REG_GPT_CR      (*reinterpret_cast<volatile uint32_t*>(TimerRegisterVirtualAddress + 0x00))
#define NNT_REG_GPT_PR      (*reinterpret_cast<volatile uint32_t*>(TimerRegisterVirtualAddress + 0x04))
#define NNT_REG_GPT_SR      (*reinterpret_cast<volatile uint32_t*>(TimerRegisterVirtualAddress + 0x08))
#define NNT_REG_GPT_IR      (*reinterpret_cast<volatile uint32_t*>(TimerRegisterVirtualAddress + 0x0c))
#define NNT_REG_GPT_OCR1    (*reinterpret_cast<volatile uint32_t*>(TimerRegisterVirtualAddress + 0x10))
#define NNT_REG_GPT_OCR2    (*reinterpret_cast<volatile uint32_t*>(TimerRegisterVirtualAddress + 0x14))
#define NNT_REG_GPT_OCR3    (*reinterpret_cast<volatile uint32_t*>(TimerRegisterVirtualAddress + 0x18))
#define NNT_REG_GPT_ICR1    (*reinterpret_cast<volatile uint32_t*>(TimerRegisterVirtualAddress + 0x1c))
#define NNT_REG_GPT_ICR2    (*reinterpret_cast<volatile uint32_t*>(TimerRegisterVirtualAddress + 0x20))
#define NNT_REG_GPT_CNT     (*reinterpret_cast<volatile uint32_t*>(TimerRegisterVirtualAddress + 0x24))
#elif defined( NN_BUILD_CONFIG_HARDWARE_JETSONTK1 ) || \
      defined( NN_BUILD_CONFIG_HARDWARE_JETSONTK2 )
#define NNT_REG_TMR1_PTV            (*reinterpret_cast<volatile uint32_t*>(TimerRegisterVirtualAddress + 0x00))
#define NNT_REG_TMR1_PCR            (*reinterpret_cast<volatile uint32_t*>(TimerRegisterVirtualAddress + 0x04))
#define NNT_REG_TIMERUS_USEC_CFG    (*reinterpret_cast<volatile uint32_t*>(TimerRegisterVirtualAddress + 0x14))
#elif defined( NN_BUILD_CONFIG_SOC_TEGRA_X1 )
#define NNT_REG_TMR1_PTV            (*reinterpret_cast<volatile uint32_t*>(TimerRegisterVirtualAddress + 0x00))
#define NNT_REG_TMR1_PCR            (*reinterpret_cast<volatile uint32_t*>(TimerRegisterVirtualAddress + 0x04))
#define NNT_REG_TIMERUS_USEC_CFG    (*reinterpret_cast<volatile uint32_t*>(TimerRegisterVirtualAddress + 0x14))
#elif defined( NN_BUILD_CONFIG_HARDWARE_SMMA53 ) || \
      defined( NN_BUILD_CONFIG_HARDWARE_JUNO)
#define NNT_REG_TIMER1LOAD      (*reinterpret_cast<volatile uint32_t*>(TimerRegisterVirtualAddress + 0x00))
#define NNT_REG_TIMER1VALUE     (*reinterpret_cast<volatile uint32_t*>(TimerRegisterVirtualAddress + 0x04))
#define NNT_REG_TIMER1CONTROL   (*reinterpret_cast<volatile uint32_t*>(TimerRegisterVirtualAddress + 0x08))
#define NNT_REG_TIMER1INTCLR    (*reinterpret_cast<volatile uint32_t*>(TimerRegisterVirtualAddress + 0x0c))
#define NNT_REG_TIMER1RIS       (*reinterpret_cast<volatile uint32_t*>(TimerRegisterVirtualAddress + 0x10))
#define NNT_REG_TIMER1MIS       (*reinterpret_cast<volatile uint32_t*>(TimerRegisterVirtualAddress + 0x14))
#define NNT_REG_TIMER1BGLOAD    (*reinterpret_cast<volatile uint32_t*>(TimerRegisterVirtualAddress + 0x18))
#endif


//---------------------------------------------------------------------------
//  タイマーステータスのクリア
void ClearTimerInterruptStatus()
NN_NOEXCEPT
{
#if defined( NN_BUILD_CONFIG_HARDWARE_BDSLIMX6 )
    NNT_REG_GPT_SR = 0x1;
#elif defined( NN_BUILD_CONFIG_HARDWARE_JETSONTK1 ) || \
      defined( NN_BUILD_CONFIG_HARDWARE_JETSONTK2 ) || \
      defined( NN_BUILD_CONFIG_SOC_TEGRA_X1 )
    // 割り込みをクリア
    NNT_REG_TMR1_PCR |= 0x1 << 30;
#elif defined( NN_BUILD_CONFIG_HARDWARE_SMMA53 ) || \
      defined( NN_BUILD_CONFIG_HARDWARE_JUNO )
    NNT_REG_TIMER1INTCLR = 0x1;
#endif

    nn::dd::EnsureMemoryAccess();
}

//---------------------------------------------------------------------------
//  割込みイベント用の Horizon ターゲット環境の準備
//  100msec 毎にタイマー割込みを発生させる
//
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
//  コンストラクタ
//
//  - タイマーの設定（100msec 周期）と割込み準備
//
InterruptGeneratorForHorizon::InterruptGeneratorForHorizon()
NN_NOEXCEPT
{
    if (TimerRegisterRegionSize > 0)
    {
        // タイマー I/O がマッピングされている仮想アドレスを取得
        uintptr_t vadrs = nn::dd::QueryIoMappingAddress(TimerRegisterPhysicalAddress, TimerRegisterRegionSize);
        if (!vadrs)
        {
            NN_ABORT(NN_TEXT("*** OS テストで利用するタイマーレジスタがマッピングされていません。"));
        }
        TimerRegisterVirtualAddress = vadrs;

        NNT_OS_LOG("Timer: padrs=0x%016llx\n", TimerRegisterPhysicalAddress);
        NNT_OS_LOG("Timer: vadrs=0x%08x\n",    TimerRegisterVirtualAddress);
        NNT_OS_LOG("Timer: size =0x%08x\n",    TimerRegisterRegionSize);
    }

#if defined( NN_BUILD_CONFIG_HARDWARE_BDSLIMX6 )
    // GPT の無効化
    NNT_REG_GPT_CR = 0;

    // GPT の割込みを全て禁止
    NNT_REG_GPT_IR = 0x00;

    // GPT の SW Reset をアサート
    NNT_REG_GPT_CR |= 0x1 << 15;

    // GPT の SW Reset が自動的にデアサートするのを待つ
    while ( NNT_REG_GPT_CR & (0x1 << 15) )
    {
        ;
    }

    ClearTimerInterruptStatus();

    // GPT_SR のクリア
    NNT_REG_GPT_SR = 0x3f;

    // GPT のクロックソースの設定（Peripheral Clock）
    NNT_REG_GPT_CR |= 0x1 << 6;

    // ENMOD=1 によりカウンタをクリア
    NNT_REG_GPT_CR |= 0x1 << 1;

    // GPT のタイマー周期を設定
    NNT_REG_GPT_OCR1 = GptPeripheralClock / 10;

    // Enable GPT
    NNT_REG_GPT_CR |= 0x1 << 0;

    // GPT の割込み許可の設定（Output Compare1 のみ許可）
    NNT_REG_GPT_IR = 0x01;
#elif defined( NN_BUILD_CONFIG_HARDWARE_JETSONTK1 ) || \
      defined( NN_BUILD_CONFIG_HARDWARE_JETSONTK2 ) || \
      defined( NN_BUILD_CONFIG_SOC_TEGRA_X1 )
    // clk_m 周波数を 12MHz に設定（p.55: Only 12 MHz is currently supported.)
    NNT_REG_TIMERUS_USEC_CFG = 0xb;

    ClearTimerInterruptStatus();

    // 100msec 周期に設定（p.256: If you program the value n, the counter trigger will actually be n+1.）
    NNT_REG_TMR1_PTV = 100 * 1000 - 1;

    // Periodic に設定
    NNT_REG_TMR1_PTV |= 0x1 << 30;

    // タイマーを開始
    NNT_REG_TMR1_PTV |= 0x1 << 31;
#elif defined( NN_BUILD_CONFIG_HARDWARE_SMMA53 ) || \
      defined( NN_BUILD_CONFIG_HARDWARE_JUNO )
    // タイマー動作を停止させた状態で、他の設定を行なう
    NNT_REG_TIMER1CONTROL   = (0u << 7) |   // 0: Disable timer module
                              (1u << 6) |   // 1: periodic mode
                              (0u << 5) |   // 0: Disable timer interrupt
                              (1u << 1);    // 1: 32bit counter

    // タイマー周期（100msec 周期に設定）
    const uint32_t baseCount = Sp804Timer1ClockFrequency / (10 /* Hz */);
    NNT_REG_TIMER1LOAD       = baseCount;
    NNT_REG_TIMER1BGLOAD     = baseCount;

    ClearTimerInterruptStatus();

    // 割込みを許可
    NNT_REG_TIMER1CONTROL |= 0x1 << 5;

    // タイマーの動作を開始
    NNT_REG_TIMER1CONTROL |= 0x1 << 7;
#endif
}

//---------------------------------------------------------------------------
//  デストラクタ
//
//  - タイマーの停止
//
InterruptGeneratorForHorizon::~InterruptGeneratorForHorizon()
NN_NOEXCEPT
{
#if defined( NN_BUILD_CONFIG_HARDWARE_BDSLIMX6 )
    // GPT の無効化
    NNT_REG_GPT_CR = 0;

    // GPT の割込みを全て禁止
    NNT_REG_GPT_IR = 0x00;
#elif defined( NN_BUILD_CONFIG_HARDWARE_JETSONTK1 ) || \
      defined( NN_BUILD_CONFIG_HARDWARE_JETSONTK2 ) || \
      defined( NN_BUILD_CONFIG_SOC_TEGRA_X1 )
    // タイマーを停止
    NNT_REG_TMR1_PTV &= ~(0x1 << 31);
#elif defined( NN_BUILD_CONFIG_HARDWARE_SMMA53 ) || \
      defined( NN_BUILD_CONFIG_HARDWARE_JUNO )
    // Timer1 の割込み禁止と動作の停止
    NNT_REG_TIMER1CONTROL &= ~((0x1 << 7) | (0x1 << 5));
#endif

    ClearTimerInterruptStatus();
}


}}} // namespace nnt::os::interruptEvent

