﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../Common/test_Pragma.h"

#include <cstring>
#include <nn/os/os_Config.h>
#include <nn/nn_SdkText.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Result.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os.h>

#include <nn/svc/svc_Base.h>
#include <nn/svc/svc_Dd.h>
#include <nn/svc/svc_HardwareParamsSelect.h>
#include <nn/svc/svc_Synchronization.h>
#include <nn/svc/svc_Server.h>
#include <nn/svc/svc_Tcb.h>
#include <nn/svc/svc_Result.h>

#include "test_Helper.h"
#include "test_NamedPipe.h"

namespace nnt { namespace os { namespace detail {

namespace {

    NN_ALIGNAS(4096) uint32_t   serverBuffer[4096 / sizeof(uint32_t)];
    NN_ALIGNAS(4096) uint32_t   clientBuffer[4096 / sizeof(uint32_t)];

}


//  Horizon 環境の場合は、固定のポート名を設定する。
//  OS テストが途中で Terminate された場合でも、NamedPipeServerImplByHorizon()
//  にて最初にポート名を開く前に、同名のポート名が存在している場合には、
//  そのポートを削除して、開き直すようにする。

#define NNT_OS_PORT_NAME    "OSTEST_PORT"

void CreateUniquePortName(char* portName, size_t size)
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(size >= PortNameLengthMax, "");
    static_assert(sizeof(NNT_OS_PORT_NAME) <= PortNameLengthMax, "");

    // 最後の NUL 文字を含めて memcpy でポート名を設定する
    std::memcpy(portName, NNT_OS_PORT_NAME, sizeof(NNT_OS_PORT_NAME));
}

//=============================================================================
// NamedPipeServerImplByHorizon
//=============================================================================

//-----------------------------------------------------------------------------
// コンストラクタ
//
//  指定の名前で Horizon 名前付きポートを開き、以後 ReplyValue() と RecvHandle()
//  が利用可能となる。これらは、プロセスを超えて通信を行なうことが出来る。
//
NamedPipeServerImplByHorizon::NamedPipeServerImplByHorizon(const char* portName)
NN_NOEXCEPT
{
    nn::svc::Handle  handle;

    // 名前付きポートを削除してみる（残っていた場合に事前に削除する）
    auto result = nn::svc::ManageNamedPort(
                        &handle,                        // ハンドル格納先
                        portName,                       // ポートの名前
                        0);                             // 最大セッション数
    if (result.IsSuccess())
    {
        NNT_OS_LOG(NN_TEXT("Server: NamedPort が残っていたので削除しました。\n"));
    }

    // 名前付きポートを作成する
    result = nn::svc::ManageNamedPort(
                        &handle,                        // ハンドル格納先
                        portName,                       // ポートの名前
                        2);                             // 最大セッション数

    if (!result.IsSuccess())
    {
        NN_ABORT(NN_TEXT("Server: NamedPort を作成できませんでした。"));
    }

    // 作成されたポートのハンドルを設定
    std::memcpy(m_PortName, portName, PortNameLengthMax);
    m_Port    = handle;
    m_Session = INVALID_HANDLE;
}

//-----------------------------------------------------------------------------
// デストラクタ
//
//  既定の名前で作成した Horizon 名前付きポートを閉じる。
//
NamedPipeServerImplByHorizon::~NamedPipeServerImplByHorizon()
NN_NOEXCEPT
{
    // セッションを閉じる
    if (m_Session != INVALID_HANDLE)
    {
        nn::svc::CloseHandle( m_Session );
        m_Session = INVALID_HANDLE;
    }

    // ポートを閉じる
    if (m_Port != INVALID_HANDLE)
    {
        // 先にポートのハンドルを閉じる
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::svc::CloseHandle( m_Port ));
        m_Port = INVALID_HANDLE;

        /// 名前付きポートを削除する
        nn::svc::Handle handle;
        nn::Result ret = nn::svc::ManageNamedPort(
                                    &handle,            // ハンドル格納先
                                    m_PortName,         // ポートの名前
                                    0);                 // 最大セッション数

        if (!ret.IsSuccess())
        {
            NN_ABORT(NN_TEXT("Server: NamedPort を削除できませんでした。"));
        }
    }
}


//-----------------------------------------------------------------------------
// 接続
//
//  サーバ側でポートに接続する
//
void NamedPipeServerImplByHorizon::Connect()
NN_NOEXCEPT
{
    if (m_Port == INVALID_HANDLE)
    {
        NN_ABORT(NN_TEXT("Server: NamedPort が作成されていません。"));
    }

    // いわゆる Listen 待ち
    for (;;)
    {
        int32_t index;
        nn::Result ret = nn::svc::WaitSynchronization(&index, &m_Port, 1, nn::svc::WAIT_INFINITE);
        if (ret.IsSuccess())
        {
            break;
        }
    }

    // クライアントからの要求を確認したので Accept する
    nn::svc::Handle session;
    nn::Result ret = nn::svc::AcceptSession( &session, m_Port );
    if (!ret.IsSuccess())
    {
        NN_ABORT(NN_TEXT("Server: NamedPort への接続に失敗しました。"));
    }

    // 接続に成功ならリターン
    m_Session = session;
}

//-----------------------------------------------------------------------------
// 接続解除
//
void NamedPipeServerImplByHorizon::Disconnect()
NN_NOEXCEPT
{
    if (m_Session == INVALID_HANDLE)
    {
        return;
    }

    nn::svc::CloseHandle( m_Session );
    m_Session = INVALID_HANDLE;
}

//-----------------------------------------------------------------------------
// 32bit 符号なし整数値の返信
//
void NamedPipeServerImplByHorizon::ReplyValue(uint32_t value)
NN_NOEXCEPT
{
    if (m_Session == INVALID_HANDLE)
    {
        NN_ABORT(NN_TEXT("Server: セッションが作成されていません。"));
    }

    // サーバ用のユーザ領域にメッセージを構築
    uint32_t* adrs = serverBuffer;

    // メッセージヘッダ
    adrs[0] = 0;
    adrs[1] = 1;

    // 任意データ転送
    adrs[2] = value;

    int32_t index;
    nn::svc::Handle array[1];
    nn::Result ret = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(adrs), 4096, array, 0, m_Session, 0);
    if (!(ret <= nn::svc::ResultTimeout()))
    {
        NN_ABORT(NN_TEXT("Server: データ送信に失敗しました。"));
    }
}

//-----------------------------------------------------------------------------
// ハンドルの受信
//
//  nn::os::NativeHandle 型の値を受信して返す。失敗時には 0 を返す。
//  受信待ち状態で Client 側が切断した場合、一旦 Disconnect() した上で、
//  再度 Connect() を行ない、セッションを張り替える。
//  値はカーネルが管理するハンドル値である。
//
nn::os::NativeHandle NamedPipeServerImplByHorizon::RecvHandle()
NN_NOEXCEPT
{
    if (m_Session == INVALID_HANDLE)
    {
        NN_ABORT(NN_TEXT("Server: セッションが作成されていません。"));
    }

    for (;;)
    {
        // 受信
        int32_t index;
        nn::Result ret = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(serverBuffer), 4096, &m_Session, 1, INVALID_HANDLE, -1);
        if ((!ret.IsSuccess()) || (index != 0))
        {
            NN_ABORT(NN_TEXT("Server: データ受信に失敗しました。"));
        }
        auto value = static_cast<nn::os::NativeHandle>( serverBuffer[3] );
        auto cmd   = static_cast<uintptr_t>( serverBuffer[4] );

        NNT_OS_LOG("Server: received cmd  = %d\n", cmd);
        NNT_OS_LOG("Server: received value= 0x%08x\n", value);

        if (cmd == RequireCloseSession)
        {
            ReplyValue(1);

            Disconnect();
            Connect();
            continue;
        }

        ReplyValue(0);
        return value;
    }
}

//-----------------------------------------------------------------------------
// ハンドルの破棄
//
void NamedPipeServerImplByHorizon::DestroyHandle(nn::os::NativeHandle handle)
{
    nn::svc::CloseHandle( nn::svc::Handle(handle) );
}


//=============================================================================
// NamedPipeClientImplByHorizon
//=============================================================================

//-----------------------------------------------------------------------------
// コンストラクタ
//
//  指定の名前で Horizon 名前付きポートを開き、以後 SendHandle() と RecvValue()
//  が利用可能となる。これらは、プロセスを超えて通信を行なうことが出来る。
//
NamedPipeClientImplByHorizon::NamedPipeClientImplByHorizon(const char* portName)
NN_NOEXCEPT
{
    // ハンドル値を一旦無効化
    m_Session = INVALID_HANDLE;

    // ポートを開く
    nn::svc::Handle handle;
    nn::Result ret = nn::svc::ConnectToNamedPort(
                    &handle,                        // ハンドル格納先
                    portName);                      // ポートの名前

    if (!ret.IsSuccess())
    {
        NN_ABORT( NN_TEXT("Client: Session の獲得に失敗しました。") );
    }

    // 作成されたポートのハンドルを設定してリターン
    m_Session = handle;
}

//-----------------------------------------------------------------------------
// デストラクタ
//
//  既定の名前で接続した Horizon 名前付きポートのセッションを閉じる。
//
NamedPipeClientImplByHorizon::~NamedPipeClientImplByHorizon()
NN_NOEXCEPT
{
    if (m_Session == INVALID_HANDLE)
    {
        return;
    }

    SendHandle( RequireCloseSession, nn::os::NativeHandle(0) );

    nn::svc::CloseHandle( m_Session );
    m_Session = INVALID_HANDLE;
}


//-----------------------------------------------------------------------------
// ハンドルの送信
//
void NamedPipeClientImplByHorizon::SendHandle(int cmd, nn::os::NativeHandle handle)
NN_NOEXCEPT
{
    if (m_Session == INVALID_HANDLE)
    {
        NN_ABORT(NN_TEXT("Client: セッションがオープンされていません。"));
    }

    // カーネル管理 TLS の IPC 領域にメッセージを構築
    uint32_t* adrs = reinterpret_cast<uint32_t*>( clientBuffer );

    // メッセージヘッダ
    adrs[0] = 0;
    adrs[1] = (1 << 31) | 1;

    // 特殊データ転送
    adrs[2] = 1 << 1;           // 特殊データの Copy データ数 1
    adrs[3] = static_cast<uint32_t>( handle );

    // 任意データ（１個）
    adrs[4] = cmd;

    nn::Result ret = nn::svc::SendSyncRequestWithUserBuffer( reinterpret_cast<uintptr_t>(clientBuffer), 4096, m_Session );
    if (cmd == RequireEndOfTest)
    {
        nn::svc::CloseHandle( m_Session );
        m_Session = INVALID_HANDLE;
    }
    else if (!ret.IsSuccess())
    {
        NN_ABORT(NN_TEXT("Client: データ送信に失敗しました。"));
    }

    NNT_OS_LOG("Client: send: handle=%d\n", adrs[2]);
}

//-----------------------------------------------------------------------------
// 32bit 符号なし整数値の受信
//
uint32_t NamedPipeClientImplByHorizon::RecvValue()
NN_NOEXCEPT
{
    if (m_Session == INVALID_HANDLE)
    {
        NN_ABORT(NN_TEXT("Client: セッションがオープンされていません。"));
    }

    int32_t index;
    nn::Result ret = nn::svc::ReplyAndReceiveWithUserBuffer(&index, reinterpret_cast<uintptr_t>(clientBuffer), 4096, &m_Session, 1, INVALID_HANDLE, -1);
    if ((!ret.IsSuccess()) || (index != 0))
    {
        NN_ABORT(NN_TEXT("Client: データ受信に失敗しました。"));
    }

    return clientBuffer[2];
}


//-----------------------------------------------------------------------------

}}} // namespace nnt::os::detail


