﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//---------------------------------------------------------------------------
//  Test Helper 機能
//
//  テストを実装する上で必要となる以下の機能を提供する。
//  ・シーケンスチェッカー
//  ・各個別テストにおける合否判定および集計
//  ・時間待ち機能（Win32 で実装）
//---------------------------------------------------------------------------

#pragma once

#include <nn/os/os_Config.h>
#include <nn/nn_SdkText.h>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os.h>
#include <nn/os/os_UserExceptionHandler.h>
#include <nnt/nnt_Argument.h>

#include "test_Calibration.h"

#if defined(NN_BUILD_CONFIG_OS_WIN32)
#include <nnt/nntest.h>
#endif

//---------------------------------------------------------------------------
// ミリ秒単位の待ち
inline void testSleep(int64_t timespan)
{
    nn::os::SleepThread( nn::TimeSpan::FromNanoSeconds( timespan ) );
}

//---------------------------------------------------------------------------
// ログを出力する場合は切り替えてビルドして下さい。
//
#if 1
#define NNT_OS_LOG(...) do { } while(NN_STATIC_CONDITION(false))
#else
#define NNT_OS_LOG(...) NN_LOG(__VA_ARGS__)
#endif

//---------------------------------------------------------------------------
// メモリバリア
#if defined(NN_BUILD_CONFIG_OS_WIN32)
#include <nn/nn_Windows.h>
#define NNT_MEMORY_BARRIER()    MemoryBarrier()
#elif defined(NN_BUILD_CONFIG_COMPILER_GCC) || defined(NN_BUILD_CONFIG_COMPILER_CLANG)
#define NNT_MEMORY_BARRIER()    __sync_synchronize()
#endif

//---------------------------------------------------------------------------
//  テスト用シーケンスチェッククラス
//---------------------------------------------------------------------------

class   Sequence
{
public:
    Sequence()  NN_NOEXCEPT : m_functionName(NULL), m_lineNumber(0)
    {
        Initialize();
    }

    ~Sequence() NN_NOEXCEPT {}

    void    Initialize()
    {
        m_next = 0;
    }

    // シーケンス番号を表示
    void    Print()
    {
        NNT_OS_LOG("(%5d): ", m_next);
    }

    // シーケンス番号を表示
    void    None()
    {
        NNT_OS_LOG("(-----): ");
    }

    // シーケンスエラー
    void    Error(const char* filename, int line);

    // シーケンス番号をチェックし、インクリメント
    void    Check(int num, const char* filename, int line);

    // シーケンス番号をスキップし、インクリメント
    void    Skip(int num, const char* filename, int line);

    // 次のシーケンス番号を取得
    int     Get()
    {
        return m_next;
    }

    // シーケンス番号を設定
    void    Set(int num, const char* filename, int line)
    {
        m_next = num;
        Check(num, filename, line);
    }

    // シーケンスの待機
    void    WaitUntil(int num, int next, const char* filename, int line);

    // シーケンスを制御する待機とシグナル
    void    SyncInit()
    {
        m_signal = 0;
        m_ack = 0;
    }

    void    SyncWaitUnsafe(int bitpos, const char* filename, int line);
    void    SyncSignalUnsafe(int bitpos);
    void    SyncAckUnsafe(int bitpos);
    void    SyncDelay(int count, int bitpos);

    // 関数名と行番号の設定と取得
    void    SetFunctionName(const char* name)
    {
        m_functionName = name;
    }
    void    SetLineNumber(int line)
    {
        m_lineNumber   = line;
    }
    const char*   GetFunctionName()
    {
        return m_functionName;
    }
    int     GetLineNumber()
    {
        return m_lineNumber;
    }

private:
    volatile int    m_next;
    volatile uint32_t   m_signal;
    volatile uint32_t   m_ack;

    const char* m_functionName;
    int         m_lineNumber;
};

//---------------------------------------------------------------------------

extern Sequence g_Sequence;

#define SEQ_INITIALIZE()    g_Sequence.Initialize()
#define SEQ_ERROR()         g_Sequence.Error(__FUNCTION__, __LINE__)
#define SEQ_CHECK(num)      g_Sequence.Check((num), __FUNCTION__, __LINE__)
#define SEQ_SKIP(num)       g_Sequence.Skip((num), __FUNCTION__, __LINE__)
#define SEQ_NONE()          g_Sequence.None()
#define SEQ_SET(num)        g_Sequence.Set((num), __FUNCTION__, __LINE__)
#define SEQ_WAIT_UNTIL(num, next) g_Sequence.WaitUntil((num), (next), __FUNCTION__, __LINE__)

#define SEQ_SYNC_INIT()             g_Sequence.SyncInit()
#define SEQ_SYNC_WAIT(bitpos)       g_Sequence.SyncWaitUnsafe((bitpos), __FUNCTION__, __LINE__)
#define SEQ_SYNC_SIGNAL(bitpos)     g_Sequence.SyncSignalUnsafe(bitpos)

#define SEQ_SYNC_ACK(bitpos)        g_Sequence.SyncAckUnsafe(bitpos)
#define SEQ_SYNC_DELAY(num, bitpos) g_Sequence.SyncDelay((num), (bitpos))

//---------------------------------------------------------------------------
//  テスト用結果判定クラス
//---------------------------------------------------------------------------

class   CountResult
{
public:
    CountResult()
    {
        m_gross_retry   = 0;
        m_gross_ok      = 0;
        m_gross_ng      = 0;

        Initialize();
    }

    void    Initialize()
    {
        ClearPartResult();
        ClearTotalResult();
    }

    void    ClearTotalResult()
    {
        m_total_retry   = 0;
        m_total_ok      = 0;
        m_total_ng      = 0;
    }

    void    ClearPartRetry()
    {
        m_num_retry = 0;
    }

    void    ClearPartResult()
    {
        m_num_ok    = 0;
        m_num_ng    = 0;
    }

    bool    IsPartTestPassed()
    {
        return m_num_ng == 0;
    }
    int     IncrementRetryCount()
    {
        ++m_gross_retry;
        ++m_total_retry;
        return ++m_num_retry;
    }

    bool    JudgeGoogleTest(const char* functionName, int line);

    void    Check(bool state);
    void    Show();
    void    ShowGross();

private:
    int     m_gross_ok;
    int     m_gross_ng;
    int     m_total_ok;
    int     m_total_ng;
    int     m_num_ok;
    int     m_num_ng;
    int     m_gross_retry;
    int     m_total_retry;
    int     m_num_retry;
};

inline bool CountResult::JudgeGoogleTest(const char* functionName, int line)
{
    // 個別のテスト結果で GoogleTest に判定を通知
    if ( IsPartTestPassed() )
    {
        return true;
    }
    else
    {
        NN_UNUSED( functionName );
        NN_UNUSED( line );
        return false;
    }
}

#define INITIALIZE_TEST_COUNT() g_Result.Initialize()
#define CLEAR_GOOGLE_TEST()     g_Result.ClearPartRetry();                  \
                                gtest_loop: g_Result.ClearPartResult()
#define JUDGE_GOOGLE_TEST()     do                                          \
            {                                                               \
                if (!g_Result.JudgeGoogleTest(__FUNCTION__, __LINE__))      \
                {                                                           \
                    int retryCount = g_Result.IncrementRetryCount();        \
                    NN_LOG("=====[ TEST RETRY (%d) ]=====\n", retryCount);  \
                    if (retryCount >= 1)                                    \
                    {                                                       \
                        NN_ASSERT( false, NN_TEXT("リトライを断念しました。") );     \
                    }                                                       \
                    goto gtest_loop;                                        \
                }                                                           \
            } while (NN_STATIC_CONDITION(0))

extern CountResult  g_Result;

//---------------------------------------------------------------------------
//  合否判定および集計

inline void CheckBool(bool state)
{
    g_Result.Check(state);
}

template<class T>
inline void CheckParam(T result, T expect)
{
    NNT_OS_LOG(" result=%d expect=%d", result, expect);
    g_Result.Check( result == expect );
};

inline void CheckParam(int result, int expect)
{
    NNT_OS_LOG(" value=%d expect=%d", result, expect);
    g_Result.Check( result == expect );
}

inline void CheckParam(uintptr_t result, uintptr_t expect)
{
    NNT_OS_LOG(" value=0x%p expect=0x%p", result, expect);
    g_Result.Check( result == expect );
}

inline void CheckParam(void* result, void* expect)
{
    NNT_OS_LOG(" value=0x%p expect=0x%p", result, expect);
    g_Result.Check( result == expect );
}

#if 0
#define     IS_RESULT_SUCCESS(result)   g_Result.Check((result).IsSuccess())
#endif

#if 0
#define     IS_RESULT_ERROR(result, expect) do                          \
            {                                                           \
                NNT_OS_LOG(" result=%d expect=%d",                      \
                                result.GetInnerValueForDebug(),         \
                                expect.GetInnerValueForDebug() );       \
                g_Result.Check( result <= expect );                     \
            } while (NN_STATIC_CONDITION(0))
#else
#define     IS_RESULT_ERROR(result, expect) do                          \
            {                                                           \
                NNT_OS_LOG(" result=%d expect=%d",                      \
                                result.GetInnerValueForDebug(),         \
                                expect.GetInnerValueForDebug() );       \
                g_Result.Check( result.GetInnerValueForDebug() ==       \
                                expect.GetInnerValueForDebug() );       \
            } while (NN_STATIC_CONDITION(0))
#endif

//---------------------------------------------------------------------------
//  ユーザー例外ハンドラ関連
//---------------------------------------------------------------------------

inline void RestoreUserExceptionHandler()
{
    // テストのために設定したユーザー例外ハンドラを解除
    // (もしかしたらデフォルトユーザー例外ハンドラを登録し直すようになるかも)
    nn::os::SetUserExceptionHandler(NULL, NULL, 0, NULL);
}

//---------------------------------------------------------------------------
//  その他のユーティリティ
//---------------------------------------------------------------------------

#if defined(NN_BUILD_CONFIG_OS_WIN32)
inline bool IsDllEnvironment()
{
    return false;
}
#elif defined(NN_BUILD_CONFIG_OS_HORIZON)
extern "C" int _start;

inline bool IsDllEnvironment()
{
    return (&_start) != reinterpret_cast<int*>(0x00200000);
}
#endif

//---------------------------------------------------------------------------

