﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/result/testResult_Assert.h>

#include <nn/account/account_ApiForAdministrators.h>
#include <nn/account/account_ApiForSystemServices.h>
#include <nn/fs.h>
#include <nn/fs/fs_Utility.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/fs_SystemSaveDataPrivate.h>
#include <nn/fs/fs_SaveDataTypes.h>
#include <nn/nifm/nifm_ApiForSystem.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/olsc/olsc_Result.h>
#include <nn/olsc/sfdl/olsc_IOlscService.sfdl.h>
#include <nn/olsc/srv/olsc_SystemEventManager.h>
#include <nn/olsc/srv/util/olsc_MountManager.h>
#include <nn/olsc/srv/util/olsc_File.h>
#include <nn/olsc/srv/util/olsc_SaveDataAccessLockManager.h>
#include <nn/olsc/srv/util/olsc_SeriesPosition.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/time.h>
#include <nn/util/util_FormatString.h>

#include "testOlsc_SaveDataUtil.h"

#include <vector>



using namespace nn;
using namespace nn::olsc;
using namespace nn::olsc::srv;
using namespace nn::olsc::srv::database;

namespace {
    class OlscSaveDataAccessLockManagerTest : public testing::Test
    {
    protected:
        virtual void SetUp()
        {
            NNT_ASSERT_RESULT_SUCCESS(InitializeSaveData({}));
        }

        virtual void TearDown()
        {
        }

        static void SetUpTestCase()
        {
            nnt::olsc::Initialize();
        }

        static void TearDownTestCase()
        {
            nnt::olsc::Finalize();
        }

    protected:
        static Result InitializeSaveData(fs::UserId userId)
        {
            fs::SystemSaveDataId toDelete[] = {
                nnt::olsc::MountInfoForTestDeviceSave.systemSaveDataId,
                nnt::olsc::MountInfoForTestUserSettingSave.systemSaveDataId,
                nnt::olsc::MountInfoForTestUserSeriesInfoSave.systemSaveDataId,
            };
            for (auto& id : toDelete)
            {
                NN_RESULT_TRY(fs::DeleteSystemSaveData(fs::SaveDataSpaceId::System, id, userId))
                    NN_RESULT_CATCH(fs::ResultTargetNotFound)
                    {
                    }
                NN_RESULT_END_TRY;
            }

            NN_RESULT_SUCCESS;
        }

    };

    std::array<char, 4 * 1024 * 1024> g_WorkBuffer;

    nn::util::optional<fs::SaveDataInfo> GetSaveDataInfo(const account::Uid& uid, ApplicationId appId)
    {
        std::unique_ptr<nn::fs::SaveDataIterator> iter;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenSaveDataIterator(&iter, fs::SaveDataSpaceId::User));

        int64_t count;
        nn::fs::SaveDataInfo info;
        while (NN_STATIC_CONDITION(true))
        {
            count = 0;
            NN_ABORT_UNLESS_RESULT_SUCCESS(iter->ReadSaveDataInfo(&count, &info, 1));
            if (count == 0)
            {
                return nn::util::nullopt;
            }
            if (info.applicationId.value == appId.value
                && info.saveDataUserId == fs::ConvertAccountUidToFsUserId(uid))
            {
                return info;
            }
        }

    }

    void TestOpenExporter(const account::Uid& uid, const ApplicationId apps[], int appCount, bool expected)
    {
        for (int i = 0; i < appCount; ++i)
        {
            auto& appId = apps[i];
            auto saveDataInfo = GetSaveDataInfo(uid, appId);
            fs::SaveDataTransferManagerForCloudBackUp manager;
            std::unique_ptr<fs::ISaveDataDivisionExporter> exporter;

            NN_ABORT_UNLESS(saveDataInfo);
            auto result = manager.OpenSaveDataExporter(&exporter, fs::SaveDataSpaceId::User, saveDataInfo->saveDataId);

            EXPECT_EQ(expected, result.IsSuccess());
        }
    }
}


TEST_F(OlscSaveDataAccessLockManagerTest, Basic)
{
    auto uid = nnt::olsc::GetFirstUserId();

    NNT_ASSERT_RESULT_SUCCESS(InitializeSaveData(fs::ConvertAccountUidToFsUserId(uid)));

    const int MaxAppCount = 32;
    ApplicationId apps[MaxAppCount];
    for (int i = 0; i < MaxAppCount; ++i)
    {
        apps[i] = { static_cast<Bit64>(i + 1) };
    }

    // ダミーのセーブデータ作成
    const size_t SaveDataSize = 1024 * 1024;
    const size_t JournalSize = 1024 * 1024;
    for (auto& app : apps)
    {
        NNT_ASSERT_RESULT_SUCCESS(nnt::olsc::CreateTestSaveData("dummy", uid, app, SaveDataSize, JournalSize, g_WorkBuffer.data(), g_WorkBuffer.size()));
    }

    // エクスポート可能なことの確認
    TestOpenExporter(uid, apps, MaxAppCount, true);

    auto mainApp = apps[0];
    olsc::srv::util::SaveDataAccessLockManager m;
    m.Enable(mainApp);

    ApplicationId listBuffer[MaxAppCount];
    // Enable 直後に mainApp はエクスポート不可能になりリストに載ってくる
    TestOpenExporter(uid, apps, 1, false);
    auto listedCount = m.ListLockedApplicationIds(listBuffer, MaxAppCount);
    EXPECT_EQ(1, listedCount);
    EXPECT_EQ(apps[0], listBuffer[0]);

    // Update 後は全てのセーブがエクスポート不可能
    m.Update(mainApp, apps, MaxAppCount);
    TestOpenExporter(uid, apps, MaxAppCount, false);

    // Suppress 後はエクスポート可能
    m.Suppress();
    TestOpenExporter(uid, apps, MaxAppCount, true);

    // Unsuppress 後はエクスポート不可能
    m.Unsuppress();
    TestOpenExporter(uid, apps, MaxAppCount, false);

    // Disable 前はリスティング可能
    listedCount = m.ListLockedApplicationIds(listBuffer, MaxAppCount);
    EXPECT_EQ(MaxAppCount, listedCount);
    for (int i = 0; i < MaxAppCount; ++i)
    {
        EXPECT_EQ(apps[i], listBuffer[i]);
    }

    // Disable 後は全てのセーブがエクスポート可能
    m.Disable(mainApp);
    TestOpenExporter(uid, apps, MaxAppCount, true);

    // Disable 後はリストが消える
    listedCount = m.ListLockedApplicationIds(listBuffer, MaxAppCount);
    EXPECT_EQ(0, listedCount);
}

