﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/result/testResult_Assert.h>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_StringUtil.h>

#include <nn/http/json/http_RapidJsonApi.h>
#include <nn/http/json/http_JsonErrorMap.h>
#include <nn/olsc/olsc_Result.h>
#include <nn/olsc/srv/olsc_InternalTypes.h>

#include <olsc_MixedInfoAdaptor.h>
#include <olsc_OutputStream.h>

#include "testOlsc_InputStream.h"
#include "testOlsc_SaveDataUtil.h"

using namespace nn;
using namespace nn::olsc;

namespace mixed_Info_success {

const char MixedInfoJson[] =
"{"
    "\"save_data_archive\" : "
    "{"
        "\"id\"                      : 1234,"
        "\"nsa_id\"                  : \"aaaaaaaaaaaaaaaa\","
        "\"application_id\"          : \"bbbbbbbbbbbbbbbb\","
        "\"ua_id\"                   : \"cccccccccccccccc\","
        "\"device_id\"               : \"dddddddddddddddd\","
        "\"data_size\"               : 123456789,"
        "\"auto_backup\"             : false,"
        "\"encoded_digest\"          : \"bD1gyr6Kck4t7RExh_MPDPfVK-3aoCYb7Cbp0FuBrX8=\","
        "\"status\"                  : \"fixed\","
        "\"has_thumbnail\"           : false,"
        "\"launch_required_version\" : 5678,"
        "\"num_of_partitions\"   : 16,"
        "\"series_id\"               : 100,"
        "\"saved_at_as_unixtime\"    : 1514732400,"
        "\"timeout_at_as_unixtime\"  : 1514732401,"
        "\"finished_at_as_unixtime\" : 1514732402,"
        "\"component_files\": ["
        "{"
            "\"id\" : 101,"
            "\"index\" : 1,"
            "\"datatype\" : \"meta\","
            "\"status\" : \"hand_over\","
            "\"raw_size\" : 1024,"
            "\"archive_size\": 32,"
            "\"encoded_raw_digest\" : \"0q4lCQl9QyGz9WlE2ToFk-dG14WETO-AkBEfFA1WlAg=\","
            "\"encoded_archive_digest\" : \"hzZnQoSxtyJF6Qr72cSQww==\""
        "},"
        "{"
            "\"id\" : 102,"
            "\"index\" : 2,"
            "\"datatype\" : \"save\","
            "\"status\" : \"hand_over\","
            "\"raw_size\" : 1024,"
            "\"archive_size\" : 128,"
            "\"encoded_raw_digest\" : \"zKVIfdp0SC7vsx82sbC27rZpX9ZZISfnKTotLjotwag=\","
            "\"encoded_archive_digest\" : \"ILadtj0OkJMPxwbWDzhwuQ==\""
        "}"
        "]"
    "}"
"}";


struct Cancelable
{
    bool IsCancelled() const { return false; }
};

static const fs::SaveDataCommitId TestSaveDataCommitId = 0x4e728abeca603d6c;

const Bit8 digest2[] = {
    0xd2, 0xae, 0x25, 0x09, 0x09, 0x7d, 0x43, 0x21,
    0xb3, 0xf5, 0x69, 0x44, 0xd9, 0x3a, 0x05, 0x93,
    0xe7, 0x46, 0xd7, 0x85, 0x84, 0x4c, 0xef, 0x80,
    0x90, 0x11, 0x1f, 0x14, 0x0d, 0x56, 0x94, 0x08
};

const Bit8 digest3[] = {
    0x87, 0x36, 0x67, 0x42, 0x84, 0xb1, 0xb7, 0x22,
    0x45, 0xe9, 0x0a, 0xfb, 0xd9, 0xc4, 0x90, 0xc3
};

const Bit8 digest4[] = {
    0xcc, 0xa5, 0x48, 0x7d, 0xda, 0x74, 0x48, 0x2e,
    0xef, 0xb3, 0x1f, 0x36, 0xb1, 0xb0, 0xb6, 0xee,
    0xb6, 0x69, 0x5f, 0xd6, 0x59, 0x21, 0x27, 0xe7,
    0x29, 0x3a, 0x2d, 0x2e, 0x3a, 0x2d, 0xc1, 0xa8
};

const Bit8 digest5[] = {
    0x20, 0xb6, 0x9d, 0xb6, 0x3d, 0x0e, 0x90, 0x93,
    0x0f, 0xc7, 0x06, 0xd6, 0x0f, 0x38, 0x70, 0xb9
};

TEST(OlscJsonAdaptorTest, MixedInfo)
{
    auto uid = nnt::olsc::GetFirstUserId();
    nn::fs::DeleteSystemSaveData(nnt::olsc::SystemSaveDataIdForTest, uid);
    srv::util::DefaultMountManager mountManager(nnt::olsc::MountInfoForTest, nnt::olsc::MountInfoForTest, nnt::olsc::MountInfoForTest);
    srv::database::SaveDataArchiveInfoCache sdaInfoCache(uid, mountManager);

    olsc::srv::adaptor::SaveDataArchiveInfoStorageOutputStream sdaOutputStream(sdaInfoCache);
    const int CfCount = 2;
    olsc::srv::ComponentFileInfo cfList[CfCount];
    olsc::srv::adaptor::ComponentFileInfoMemoryOutputStream cfOutputStream(cfList, CfCount);
    olsc::srv::adaptor::MixedInfoAdaptor adaptor(&sdaOutputStream, &cfOutputStream);

    nnt::olsc::MemoryInputStreamForRapidJson inputStream;
    inputStream.Set(MixedInfoJson, sizeof(MixedInfoJson));

    const Cancelable c;
    NNT_EXPECT_RESULT_SUCCESS(http::json::ImportJsonByRapidJson<http::json::DefaultJsonErrorMap>(adaptor, inputStream, &c));
    NNT_EXPECT_RESULT_SUCCESS(adaptor.GetResult());

    EXPECT_TRUE(sdaInfoCache.GetCount() == 1);
    EXPECT_TRUE(cfOutputStream.GetCount() == 2);

    srv::SaveDataArchiveInfo sda;

    sdaInfoCache.List(&sda, 1, 0);

    // appId でソートされるため順番は逆になる
    {
        EXPECT_EQ(1234, sda.id);
        EXPECT_EQ(0xaaaaaaaaaaaaaaaaull, sda.nsaId.id);
        EXPECT_EQ(0xbbbbbbbbbbbbbbbbull, sda.applicationId.value);
        account::Uid expUid = { { 0, 0xccccccccccccccccull } };
        EXPECT_TRUE(expUid == sda.userId);
        EXPECT_EQ(123456789, sda.dataSize);
        EXPECT_EQ(0xddddddddddddddddull, sda.deviceId);
        EXPECT_EQ(TestSaveDataCommitId, sda.seriesInfo.commitId);
        EXPECT_EQ(olsc::srv::SaveDataArchiveStatus::Fixed, sda.status);
        EXPECT_EQ(false, sda.autoBackup);
        EXPECT_EQ(false, sda.hasThumbnail);
        EXPECT_EQ(5678, sda.launchRequiredVersion);
        EXPECT_EQ(16, sda.numOfPartitions);
        EXPECT_EQ(100, sda.seriesInfo.seriesId);
        EXPECT_EQ(1514732400, sda.savedAt.value);
        EXPECT_EQ(1514732401, sda.timeoutAt.value);
        EXPECT_EQ(1514732402, sda.finishedAt.value);
    }
    {
        EXPECT_EQ(101, cfList[0].id);
        EXPECT_EQ(1, cfList[0].clientArgument.chunkId);
        EXPECT_EQ(olsc::srv::ComponentFileType::Meta, cfList[0].type);
        EXPECT_EQ(olsc::srv::ComponentFileStatus::HandOver, cfList[0].status);
        EXPECT_EQ(1024, cfList[0].componentFileSize);
        EXPECT_EQ(0, std::memcmp(digest2, cfList[0].componentFileDigest.data, sizeof(cfList[1].componentFileDigest.data)));
        EXPECT_EQ(32, cfList[0].saveDataChunkSize);
        EXPECT_EQ(0, std::memcmp(digest3, cfList[0].saveDataChunkDigest.data, sizeof(cfList[1].saveDataChunkDigest.data)));
    }
    {
        EXPECT_EQ(102, cfList[1].id);
        EXPECT_EQ(2, cfList[1].clientArgument.chunkId);
        EXPECT_EQ(olsc::srv::ComponentFileType::Save, cfList[1].type);
        EXPECT_EQ(olsc::srv::ComponentFileStatus::HandOver, cfList[1].status);
        EXPECT_EQ(1024, cfList[1].componentFileSize);
        EXPECT_EQ(0, std::memcmp(digest4, cfList[1].componentFileDigest.data, sizeof(cfList[1].componentFileDigest.data)));
        EXPECT_EQ(128, cfList[1].saveDataChunkSize);
        EXPECT_EQ(0, std::memcmp(digest5, cfList[1].saveDataChunkDigest.data, sizeof(cfList[1].saveDataChunkDigest.data)));
    }
}

} // namespace mixed_info_success

namespace mixed_info_fail {
const int JsonLen = 2048;

const char SaveDataArchiveParseError[][JsonLen] =
{
    // ダイジェスト異常
    "{ \"save_data_archive\": {"
        "\"id\"                      : 1234,"
        "\"nsa_id\"                  : \"aaaaaaaaaaaaaaaa\","
        "\"application_id\"          : \"bbbbbbbbbbbbbbbb\","
        "\"ua_id\"                   : \"cccccccccccccccc\","
        "\"device_id\"               : \"dddddddddddddddd\","
        "\"data_size\"               : 123456789,"
        "\"auto_backup\"             : false,"
        "\"encoded_digest\"          : \"cbD1gyr6Kck4t7RExh_MPDPfVK-3aoCYb7Cbp0FuBrX8=\","
        "\"status\"                  : \"fixed\","
        "\"has_thumbnail\"           : false,"
        "\"launch_required_version\" : 5678,"
        "\"num_of_partitions\"   : 16,"
        "\"series_id\"               : 100,"
        "\"saved_at_as_unixtime\"    : 1514732400,"
        "\"timeout_at_as_unixtime\"  : 1514732401,"
        "\"finished_at_as_unixtime\" : 1514732402,"
        "\"component_files\": ["
        "{"
            "\"id\" : 101,"
            "\"index\" : 1,"
            "\"datatype\" : \"meta\","
            "\"status\" : \"hand_over\","
            "\"raw_size\" : 1024,"
            "\"archive_size\": 32,"
            "\"encoded_raw_digest\" : \"0q4lCQl9QyGz9WlE2ToFk-dG14WETO-AkBEfFA1WlAg=\","
            "\"encoded_archive_digest\" : \"hzZnQoSxtyJF6Qr72cSQww==\""
        "},"
        "{"
            "\"id\" : 102,"
            "\"index\" : 2,"
            "\"datatype\" : \"save\","
            "\"status\" : \"hand_over\","
            "\"raw_size\" : 1024,"
            "\"archive_size\" : 128,"
            "\"encoded_raw_digest\" : \"zKVIfdp0SC7vsx82sbC27rZpX9ZZISfnKTotLjotwag=\","
            "\"encoded_archive_digest\" : \"ILadtj0OkJMPxwbWDzhwuQ==\""
        "}"
    "]}}",
    // ダイジェスト異常
    "{ \"save_data_archive\": {"
        "\"id\"                      : 1234,"
        "\"nsa_id\"                  : \"aaaaaaaaaaaaaaaa\","
        "\"application_id\"          : \"bbbbbbbbbbbbbbbb\","
        "\"ua_id\"                   : \"cccccccccccccccc\","
        "\"device_id\"               : \"dddddddddddddddd\","
        "\"data_size\"               : 123456789,"
        "\"auto_backup\"             : false,"
        "\"encoded_digest\"          : \"D1gyr6Kck4t7RExh_MPDPfVK-3aoCYb7Cbp0FuBrX8=\","
        "\"status\"                  : \"fixed\","
        "\"has_thumbnail\"           : false,"
        "\"launch_required_version\" : 5678,"
        "\"num_of_partitions\"   : 16,"
        "\"series_id\"               : 100,"
        "\"saved_at_as_unixtime\"    : 1514732400,"
        "\"timeout_at_as_unixtime\"  : 1514732401,"
        "\"finished_at_as_unixtime\" : 1514732402,"
        "\"component_files\": ["
        "{"
            "\"id\" : 101,"
            "\"index\" : 1,"
            "\"datatype\" : \"meta\","
            "\"status\" : \"hand_over\","
            "\"raw_size\" : 1024,"
            "\"archive_size\": 32,"
            "\"encoded_raw_digest\" : \"0q4lCQl9QyGz9WlE2ToFk-dG14WETO-AkBEfFA1WlAg=\","
            "\"encoded_archive_digest\" : \"hzZnQoSxtyJF6Qr72cSQww==\""
        "},"
        "{"
            "\"id\" : 102,"
            "\"index\" : 2,"
            "\"datatype\" : \"save\","
            "\"status\" : \"hand_over\","
            "\"raw_size\" : 1024,"
            "\"archive_size\" : 128,"
            "\"encoded_raw_digest\" : \"zKVIfdp0SC7vsx82sbC27rZpX9ZZISfnKTotLjotwag=\","
            "\"encoded_archive_digest\" : \"ILadtj0OkJMPxwbWDzhwuQ==\""
        "}"
    "]}}",
    // num_of_partitions 大きすぎ
    "{ \"save_data_archive\": {"
        "\"id\"                      : 1234,"
        "\"nsa_id\"                  : \"aaaaaaaaaaaaaaaa\","
        "\"application_id\"          : \"bbbbbbbbbbbbbbbb\","
        "\"ua_id\"                   : \"cccccccccccccccc\","
        "\"device_id\"               : \"dddddddddddddddd\","
        "\"data_size\"               : 123456789,"
        "\"auto_backup\"             : false,"
        "\"encoded_digest\"          : \"bD1gyr6Kck4t7RExh_MPDPfVK-3aoCYb7Cbp0FuBrX8=\","
        "\"status\"                  : \"fixed\","
        "\"has_thumbnail\"           : false,"
        "\"launch_required_version\" : 5678,"
        "\"num_of_partitions\"   : 4294967296,"
        "\"series_id\"               : 100,"
        "\"saved_at_as_unixtime\"    : 1514732400,"
        "\"timeout_at_as_unixtime\"  : 1514732401,"
        "\"finished_at_as_unixtime\" : 1514732402,"
        "\"component_files\": ["
        "{"
            "\"id\" : 101,"
            "\"index\" : 1,"
            "\"datatype\" : \"meta\","
            "\"status\" : \"hand_over\","
            "\"raw_size\" : 1024,"
            "\"archive_size\": 32,"
            "\"encoded_raw_digest\" : \"0q4lCQl9QyGz9WlE2ToFk-dG14WETO-AkBEfFA1WlAg=\","
            "\"encoded_archive_digest\" : \"hzZnQoSxtyJF6Qr72cSQww==\""
        "},"
        "{"
            "\"id\" : 102,"
            "\"index\" : 2,"
            "\"datatype\" : \"save\","
            "\"status\" : \"hand_over\","
            "\"raw_size\" : 1024,"
            "\"archive_size\" : 128,"
            "\"encoded_raw_digest\" : \"zKVIfdp0SC7vsx82sbC27rZpX9ZZISfnKTotLjotwag=\","
            "\"encoded_archive_digest\" : \"ILadtj0OkJMPxwbWDzhwuQ==\""
        "}"
    "]}}",
    // num_of_partitions 小さすぎ
    "{ \"save_data_archive\": {"
        "\"id\"                      : 1234,"
        "\"nsa_id\"                  : \"aaaaaaaaaaaaaaaa\","
        "\"application_id\"          : \"bbbbbbbbbbbbbbbb\","
        "\"ua_id\"                   : \"cccccccccccccccc\","
        "\"device_id\"               : \"dddddddddddddddd\","
        "\"data_size\"               : 123456789,"
        "\"auto_backup\"             : false,"
        "\"encoded_digest\"          : \"bD1gyr6Kck4t7RExh_MPDPfVK-3aoCYb7Cbp0FuBrX8=\","
        "\"status\"                  : \"fixed\","
        "\"has_thumbnail\"           : false,"
        "\"launch_required_version\" : 5678,"
        "\"num_of_partitions\"   : 0,"
        "\"series_id\"               : 100,"
        "\"saved_at_as_unixtime\"    : 1514732400,"
        "\"timeout_at_as_unixtime\"  : 1514732401,"
        "\"finished_at_as_unixtime\" : 1514732402,"
        "\"component_files\": ["
        "{"
            "\"id\" : 101,"
            "\"index\" : 1,"
            "\"datatype\" : \"meta\","
            "\"status\" : \"hand_over\","
            "\"raw_size\" : 1024,"
            "\"archive_size\": 32,"
            "\"encoded_raw_digest\" : \"0q4lCQl9QyGz9WlE2ToFk-dG14WETO-AkBEfFA1WlAg=\","
            "\"encoded_archive_digest\" : \"hzZnQoSxtyJF6Qr72cSQww==\""
        "},"
        "{"
            "\"id\" : 102,"
            "\"index\" : 2,"
            "\"datatype\" : \"save\","
            "\"status\" : \"hand_over\","
            "\"raw_size\" : 1024,"
            "\"archive_size\" : 128,"
            "\"encoded_raw_digest\" : \"zKVIfdp0SC7vsx82sbC27rZpX9ZZISfnKTotLjotwag=\","
            "\"encoded_archive_digest\" : \"ILadtj0OkJMPxwbWDzhwuQ==\""
        "}"
    "]}}",
    // dataSize 大きすぎ
    "{ \"save_data_archive\": {"
        "\"id\"                      : 1234,"
        "\"nsa_id\"                  : \"aaaaaaaaaaaaaaaa\","
        "\"application_id\"          : \"bbbbbbbbbbbbbbbb\","
        "\"ua_id\"                   : \"cccccccccccccccc\","
        "\"device_id\"               : \"dddddddddddddddd\","
        "\"data_size\"               : 4294967296,"
        "\"auto_backup\"             : false,"
        "\"encoded_digest\"          : \"bD1gyr6Kck4t7RExh_MPDPfVK-3aoCYb7Cbp0FuBrX8=\","
        "\"status\"                  : \"fixed\","
        "\"has_thumbnail\"           : false,"
        "\"launch_required_version\" : 5678,"
        "\"num_of_partitions\"   : 16,"
        "\"series_id\"               : 100,"
        "\"saved_at_as_unixtime\"    : 1514732400,"
        "\"timeout_at_as_unixtime\"  : 1514732401,"
        "\"finished_at_as_unixtime\" : 1514732402,"
        "\"component_files\": ["
        "{"
            "\"id\" : 101,"
            "\"index\" : 1,"
            "\"datatype\" : \"meta\","
            "\"status\" : \"hand_over\","
            "\"raw_size\" : 1024,"
            "\"archive_size\": 32,"
            "\"encoded_raw_digest\" : \"0q4lCQl9QyGz9WlE2ToFk-dG14WETO-AkBEfFA1WlAg=\","
            "\"encoded_archive_digest\" : \"hzZnQoSxtyJF6Qr72cSQww==\""
        "},"
        "{"
            "\"id\" : 102,"
            "\"index\" : 2,"
            "\"datatype\" : \"save\","
            "\"status\" : \"hand_over\","
            "\"raw_size\" : 1024,"
            "\"archive_size\" : 128,"
            "\"encoded_raw_digest\" : \"zKVIfdp0SC7vsx82sbC27rZpX9ZZISfnKTotLjotwag=\","
            "\"encoded_archive_digest\" : \"ILadtj0OkJMPxwbWDzhwuQ==\""
        "}"
    "]}}",
    // dataSize 小さすぎ
    "{ \"save_data_archive\": {"
        "\"id\"                      : 1234,"
        "\"nsa_id\"                  : \"aaaaaaaaaaaaaaaa\","
        "\"application_id\"          : \"bbbbbbbbbbbbbbbb\","
        "\"ua_id\"                   : \"cccccccccccccccc\","
        "\"device_id\"               : \"dddddddddddddddd\","
        "\"data_size\"               : 0,"
        "\"auto_backup\"             : false,"
        "\"encoded_digest\"          : \"bD1gyr6Kck4t7RExh_MPDPfVK-3aoCYb7Cbp0FuBrX8=\","
        "\"status\"                  : \"fixed\","
        "\"has_thumbnail\"           : false,"
        "\"launch_required_version\" : 5678,"
        "\"num_of_partitions\"   : 16,"
        "\"series_id\"               : 100,"
        "\"saved_at_as_unixtime\"    : 1514732400,"
        "\"timeout_at_as_unixtime\"  : 1514732401,"
        "\"finished_at_as_unixtime\" : 1514732402,"
        "\"component_files\": ["
        "{"
            "\"id\" : 101,"
            "\"index\" : 1,"
            "\"datatype\" : \"meta\","
            "\"status\" : \"hand_over\","
            "\"raw_size\" : 1024,"
            "\"archive_size\": 32,"
            "\"encoded_raw_digest\" : \"0q4lCQl9QyGz9WlE2ToFk-dG14WETO-AkBEfFA1WlAg=\","
            "\"encoded_archive_digest\" : \"hzZnQoSxtyJF6Qr72cSQww==\""
        "},"
        "{"
            "\"id\" : 102,"
            "\"index\" : 2,"
            "\"datatype\" : \"save\","
            "\"status\" : \"hand_over\","
            "\"raw_size\" : 1024,"
            "\"archive_size\" : 128,"
            "\"encoded_raw_digest\" : \"zKVIfdp0SC7vsx82sbC27rZpX9ZZISfnKTotLjotwag=\","
            "\"encoded_archive_digest\" : \"ILadtj0OkJMPxwbWDzhwuQ==\""
        "}"
    "]}}",
    // ua_id 異常
    "{ \"save_data_archive\": {"
        "\"id\"                      : 1234,"
        "\"nsa_id\"                  : \"aaaaaaaaaaaaaaaa\","
        "\"application_id\"          : \"bbbbbbbbbbbbbbbb\","
        "\"ua_id\"                   : \"0\","
        "\"device_id\"               : \"dddddddddddddddd\","
        "\"data_size\"               : 123456789,"
        "\"auto_backup\"             : false,"
        "\"encoded_digest\"          : \"bD1gyr6Kck4t7RExh_MPDPfVK-3aoCYb7Cbp0FuBrX8=\","
        "\"status\"                  : \"fixed\","
        "\"has_thumbnail\"           : false,"
        "\"launch_required_version\" : 5678,"
        "\"num_of_partitions\"   : 16,"
        "\"series_id\"               : 100,"
        "\"saved_at_as_unixtime\"    : 1514732400,"
        "\"timeout_at_as_unixtime\"  : 1514732401,"
        "\"finished_at_as_unixtime\" : 1514732402,"
        "\"component_files\": ["
        "{"
            "\"id\" : 101,"
            "\"index\" : 1,"
            "\"datatype\" : \"meta\","
            "\"status\" : \"hand_over\","
            "\"raw_size\" : 1024,"
            "\"archive_size\": 32,"
            "\"encoded_raw_digest\" : \"0q4lCQl9QyGz9WlE2ToFk-dG14WETO-AkBEfFA1WlAg=\","
            "\"encoded_archive_digest\" : \"hzZnQoSxtyJF6Qr72cSQww==\""
        "},"
        "{"
            "\"id\" : 102,"
            "\"index\" : 2,"
            "\"datatype\" : \"save\","
            "\"status\" : \"hand_over\","
            "\"raw_size\" : 1024,"
            "\"archive_size\" : 128,"
            "\"encoded_raw_digest\" : \"zKVIfdp0SC7vsx82sbC27rZpX9ZZISfnKTotLjotwag=\","
            "\"encoded_archive_digest\" : \"ILadtj0OkJMPxwbWDzhwuQ==\""
        "}"
    "]}}",
    // app_id 異常
    "{ \"save_data_archive\": {"
        "\"id\"                      : 1234,"
        "\"nsa_id\"                  : \"aaaaaaaaaaaaaaaa\","
        "\"application_id\"          : \"0\","
        "\"ua_id\"                   : \"cccccccccccccccc\","
        "\"device_id\"               : \"dddddddddddddddd\","
        "\"data_size\"               : 123456789,"
        "\"auto_backup\"             : false,"
        "\"encoded_digest\"          : \"bD1gyr6Kck4t7RExh_MPDPfVK-3aoCYb7Cbp0FuBrX8=\","
        "\"status\"                  : \"fixed\","
        "\"has_thumbnail\"           : false,"
        "\"launch_required_version\" : 5678,"
        "\"num_of_partitions\"   : 16,"
        "\"series_id\"               : 100,"
        "\"saved_at_as_unixtime\"    : 1514732400,"
        "\"timeout_at_as_unixtime\"  : 1514732401,"
        "\"finished_at_as_unixtime\" : 1514732402,"
        "\"component_files\": ["
        "{"
            "\"id\" : 101,"
            "\"index\" : 1,"
            "\"datatype\" : \"meta\","
            "\"status\" : \"hand_over\","
            "\"raw_size\" : 1024,"
            "\"archive_size\": 32,"
            "\"encoded_raw_digest\" : \"0q4lCQl9QyGz9WlE2ToFk-dG14WETO-AkBEfFA1WlAg=\","
            "\"encoded_archive_digest\" : \"hzZnQoSxtyJF6Qr72cSQww==\""
        "},"
        "{"
            "\"id\" : 102,"
            "\"index\" : 2,"
            "\"datatype\" : \"save\","
            "\"status\" : \"hand_over\","
            "\"raw_size\" : 1024,"
            "\"archive_size\" : 128,"
            "\"encoded_raw_digest\" : \"zKVIfdp0SC7vsx82sbC27rZpX9ZZISfnKTotLjotwag=\","
            "\"encoded_archive_digest\" : \"ILadtj0OkJMPxwbWDzhwuQ==\""
        "}"
    "]}}",
    // status 異常
    "{ \"save_data_archive\": {"
        "\"id\"                      : 1234,"
        "\"nsa_id\"                  : \"aaaaaaaaaaaaaaaa\","
        "\"application_id\"          : \"bbbbbbbbbbbbbbbb\","
        "\"ua_id\"                   : \"cccccccccccccccc\","
        "\"device_id\"               : \"dddddddddddddddd\","
        "\"data_size\"               : 123456789,"
        "\"auto_backup\"             : false,"
        "\"encoded_digest\"          : \"bD1gyr6Kck4t7RExh_MPDPfVK-3aoCYb7Cbp0FuBrX8=\","
        "\"status\"                  : \"fixedd\","
        "\"has_thumbnail\"           : false,"
        "\"launch_required_version\" : 5678,"
        "\"num_of_partitions\"   : 16,"
        "\"series_id\"               : 100,"
        "\"saved_at_as_unixtime\"    : 1514732400,"
        "\"timeout_at_as_unixtime\"  : 1514732401,"
        "\"finished_at_as_unixtime\" : 1514732402,"
        "\"component_files\": ["
        "{"
            "\"id\" : 101,"
            "\"index\" : 1,"
            "\"datatype\" : \"meta\","
            "\"status\" : \"hand_over\","
            "\"raw_size\" : 1024,"
            "\"archive_size\": 32,"
            "\"encoded_raw_digest\" : \"0q4lCQl9QyGz9WlE2ToFk-dG14WETO-AkBEfFA1WlAg=\","
            "\"encoded_archive_digest\" : \"hzZnQoSxtyJF6Qr72cSQww==\""
        "},"
        "{"
            "\"id\" : 102,"
            "\"index\" : 2,"
            "\"datatype\" : \"save\","
            "\"status\" : \"hand_over\","
            "\"raw_size\" : 1024,"
            "\"archive_size\" : 128,"
            "\"encoded_raw_digest\" : \"zKVIfdp0SC7vsx82sbC27rZpX9ZZISfnKTotLjotwag=\","
            "\"encoded_archive_digest\" : \"ILadtj0OkJMPxwbWDzhwuQ==\""
        "}"
    "]}}",
    // Validate エラー
        "{ \"save_data_archive\": {"
        "\"id\"                      : 1234,"
        "\"nsa_id\"                  : \"aaaaaaaaaaaaaaaa\","
        "\"application_id\"          : \"bbbbbbbbbbbbbbbb\","
        "\"ua_id\"                   : \"cccccccccccccccc\","
        "\"device_id\"               : \"dddddddddddddddd\","
        "\"data_size\"               : 123456789,"
        "\"auto_backup\"             : false,"
        "\"encoded_digest\"          : \"bD1gyr6Kck4t7RExh_MPDPfVK-3aoCYb7Cbp0FuBrX8=\","
        "\"has_thumbnail\"           : false,"
        "\"launch_required_version\" : 5678,"
        "\"num_of_partitions\"   : 16,"
        "\"series_id\"               : 100,"
        "\"saved_at_as_unixtime\"    : 1514732400,"
        "\"timeout_at_as_unixtime\"  : 1514732401,"
        "\"finished_at_as_unixtime\" : 1514732402,"
        "\"component_files\": ["
        "{"
            "\"id\" : 101,"
            "\"index\" : 1,"
            "\"datatype\" : \"meta\","
            "\"status\" : \"hand_over\","
            "\"raw_size\" : 1024,"
            "\"archive_size\": 32,"
            "\"encoded_raw_digest\" : \"0q4lCQl9QyGz9WlE2ToFk-dG14WETO-AkBEfFA1WlAg=\","
            "\"encoded_archive_digest\" : \"hzZnQoSxtyJF6Qr72cSQww==\""
        "},"
        "{"
            "\"id\" : 102,"
            "\"index\" : 2,"
            "\"datatype\" : \"save\","
            "\"status\" : \"hand_over\","
            "\"raw_size\" : 1024,"
            "\"archive_size\" : 128,"
            "\"encoded_raw_digest\" : \"zKVIfdp0SC7vsx82sbC27rZpX9ZZISfnKTotLjotwag=\","
            "\"encoded_archive_digest\" : \"ILadtj0OkJMPxwbWDzhwuQ==\""
        "}"
    "]}}",
};

const char CompomentFileParseError[][JsonLen] =
{
    // ダイジェスト異常
    "{ \"save_data_archive\": {"
        "\"id\"                      : 1234,"
        "\"nsa_id\"                  : \"aaaaaaaaaaaaaaaa\","
        "\"application_id\"          : \"bbbbbbbbbbbbbbbb\","
        "\"ua_id\"                   : \"cccccccccccccccc\","
        "\"device_id\"               : \"dddddddddddddddd\","
        "\"data_size\"               : 123456789,"
        "\"auto_backup\"             : false,"
        "\"encoded_digest\"          : \"bD1gyr6Kck4t7RExh_MPDPfVK-3aoCYb7Cbp0FuBrX8=\","
        "\"status\"                  : \"fixed\","
        "\"has_thumbnail\"           : false,"
        "\"launch_required_version\" : 5678,"
        "\"num_of_partitions\"   : 16,"
        "\"series_id\"               : 100,"
        "\"saved_at_as_unixtime\"    : 1514732400,"
        "\"timeout_at_as_unixtime\"  : 1514732401,"
        "\"finished_at_as_unixtime\" : 1514732402,"
        "\"component_files\": ["
        "{"
            "\"id\" : 101,"
            "\"index\" : 1,"
            "\"datatype\" : \"meta\","
            "\"status\" : \"hand_over\","
            "\"archive_size\": 32,"
            "\"encoded_archive_digest\" : \"hzZnQoSxtyJF6Qr72cSQww==\""
        "},"
        "{"
            "\"id\" : 102,"
            "\"index\" : 2,"
            "\"datatype\" : \"save\","
            "\"status\" : \"hand_over\","
            "\"archive_size\" : 128,"
            "\"encoded_archive_digest\" : \"aILadtj0OkJMPxwbWDzhwuQ==\""
        "}"
    "]}}",
    "{ \"save_data_archive\": {"
        "\"id\"                      : 1234,"
        "\"nsa_id\"                  : \"aaaaaaaaaaaaaaaa\","
        "\"application_id\"          : \"bbbbbbbbbbbbbbbb\","
        "\"ua_id\"                   : \"cccccccccccccccc\","
        "\"device_id\"               : \"dddddddddddddddd\","
        "\"data_size\"               : 123456789,"
        "\"auto_backup\"             : false,"
        "\"encoded_digest\"          : \"bD1gyr6Kck4t7RExh_MPDPfVK-3aoCYb7Cbp0FuBrX8=\","
        "\"status\"                  : \"fixed\","
        "\"has_thumbnail\"           : false,"
        "\"launch_required_version\" : 5678,"
        "\"num_of_partitions\"   : 16,"
        "\"series_id\"               : 100,"
        "\"saved_at_as_unixtime\"    : 1514732400,"
        "\"timeout_at_as_unixtime\"  : 1514732401,"
        "\"finished_at_as_unixtime\" : 1514732402,"
        "\"component_files\": ["
        "{"
            "\"id\" : 101,"
            "\"index\" : 1,"
            "\"datatype\" : \"meta\","
            "\"status\" : \"hand_over\","
            "\"archive_size\": 32,"
            "\"encoded_archive_digest\" : \"hzZnQoSxtyJF6Qr72cSQww==\""
        "},"
        "{"
            "\"id\" : 102,"
            "\"index\" : 2,"
            "\"datatype\" : \"save\","
            "\"status\" : \"hand_over\","
            "\"archive_size\" : 128,"
            "\"encoded_archive_digest\" : \"aILadtj0OkJMPxwbWDzhwuQ==\""
        "}"
    "]}}",
    // status 異常
    "{ \"save_data_archive\": {"
        "\"id\"                      : 1234,"
        "\"nsa_id\"                  : \"aaaaaaaaaaaaaaaa\","
        "\"application_id\"          : \"bbbbbbbbbbbbbbbb\","
        "\"ua_id\"                   : \"cccccccccccccccc\","
        "\"device_id\"               : \"dddddddddddddddd\","
        "\"data_size\"               : 123456789,"
        "\"auto_backup\"             : false,"
        "\"encoded_digest\"          : \"bD1gyr6Kck4t7RExh_MPDPfVK-3aoCYb7Cbp0FuBrX8=\","
        "\"status\"                  : \"fixed\","
        "\"has_thumbnail\"           : false,"
        "\"launch_required_version\" : 5678,"
        "\"num_of_partitions\"   : 16,"
        "\"series_id\"               : 100,"
        "\"saved_at_as_unixtime\"    : 1514732400,"
        "\"timeout_at_as_unixtime\"  : 1514732401,"
        "\"finished_at_as_unixtime\" : 1514732402,"
        "\"component_files\": ["
        "{"
            "\"id\" : 101,"
            "\"index\" : 1,"
            "\"datatype\" : \"meta\","
            "\"status\" : \"uploadinga\","
            "\"archive_size\": 32,"
            "\"encoded_archive_digest\" : \"hzZnQoSxtyJF6Qr72cSQww==\""
        "},"
        "{"
            "\"id\" : 102,"
            "\"index\" : 2,"
            "\"datatype\" : \"save\","
            "\"status\" : \"hand_over\","
            "\"archive_size\" : 128,"
            "\"encoded_archive_digest\" : \"ILadtj0OkJMPxwbWDzhwuQ==\""
        "}"
    "]}}",
    "{ \"save_data_archive\": {"
        "\"id\"                      : 1234,"
        "\"nsa_id\"                  : \"aaaaaaaaaaaaaaaa\","
        "\"application_id\"          : \"bbbbbbbbbbbbbbbb\","
        "\"ua_id\"                   : \"cccccccccccccccc\","
        "\"device_id\"               : \"dddddddddddddddd\","
        "\"data_size\"               : 123456789,"
        "\"auto_backup\"             : false,"
        "\"encoded_digest\"          : \"bD1gyr6Kck4t7RExh_MPDPfVK-3aoCYb7Cbp0FuBrX8=\","
        "\"status\"                  : \"fixed\","
        "\"has_thumbnail\"           : false,"
        "\"launch_required_version\" : 5678,"
        "\"num_of_partitions\"   : 16,"
        "\"series_id\"               : 100,"
        "\"saved_at_as_unixtime\"    : 1514732400,"
        "\"timeout_at_as_unixtime\"  : 1514732401,"
        "\"finished_at_as_unixtime\" : 1514732402,"
        "\"component_files\": ["
        "{"
            "\"id\" : 101,"
            "\"index\" : 1,"
            "\"datatype\" : \"meta\","
            "\"status\" : \"hand_over\","
            "\"archive_size\": 32,"
            "\"encoded_archive_digest\" : \"hzZnQoSxtyJF6Qr72cSQww==\""
        "},"
        "{"
            "\"id\" : 102,"
            "\"index\" : 2,"
            "\"datatype\" : \"save\","
            "\"status\" : \"uploadinga\","
            "\"archive_size\" : 128,"
            "\"encoded_archive_digest\" : \"ILadtj0OkJMPxwbWDzhwuQ==\""
        "}"
    "]}}",
    // type 異常
    "{ \"save_data_archive\": {"
        "\"id\"                      : 1234,"
        "\"nsa_id\"                  : \"aaaaaaaaaaaaaaaa\","
        "\"application_id\"          : \"bbbbbbbbbbbbbbbb\","
        "\"ua_id\"                   : \"cccccccccccccccc\","
        "\"device_id\"               : \"dddddddddddddddd\","
        "\"data_size\"               : 123456789,"
        "\"auto_backup\"             : false,"
        "\"encoded_digest\"          : \"bD1gyr6Kck4t7RExh_MPDPfVK-3aoCYb7Cbp0FuBrX8=\","
        "\"status\"                  : \"fixed\","
        "\"has_thumbnail\"           : false,"
        "\"launch_required_version\" : 5678,"
        "\"num_of_partitions\"   : 16,"
        "\"series_id\"               : 100,"
        "\"saved_at_as_unixtime\"    : 1514732400,"
        "\"timeout_at_as_unixtime\"  : 1514732401,"
        "\"finished_at_as_unixtime\" : 1514732402,"
        "\"component_files\": ["
        "{"
            "\"id\" : 101,"
            "\"index\" : 1,"
            "\"datatype\" : \"metaa\","
            "\"status\" : \"hand_over\","
            "\"archive_size\": 32,"
            "\"encoded_archive_digest\" : \"hzZnQoSxtyJF6Qr72cSQww==\""
        "},"
        "{"
            "\"id\" : 102,"
            "\"index\" : 2,"
            "\"datatype\" : \"save\","
            "\"status\" : \"hand_over\","
            "\"archive_size\" : 128,"
            "\"encoded_archive_digest\" : \"ILadtj0OkJMPxwbWDzhwuQ==\""
        "}"
    "]}}",
    "{ \"save_data_archive\": {"
        "\"id\"                      : 1234,"
        "\"nsa_id\"                  : \"aaaaaaaaaaaaaaaa\","
        "\"application_id\"          : \"bbbbbbbbbbbbbbbb\","
        "\"ua_id\"                   : \"cccccccccccccccc\","
        "\"device_id\"               : \"dddddddddddddddd\","
        "\"data_size\"               : 123456789,"
        "\"auto_backup\"             : false,"
        "\"encoded_digest\"          : \"bD1gyr6Kck4t7RExh_MPDPfVK-3aoCYb7Cbp0FuBrX8=\","
        "\"status\"                  : \"fixed\","
        "\"has_thumbnail\"           : false,"
        "\"launch_required_version\" : 5678,"
        "\"num_of_partitions\"   : 16,"
        "\"series_id\"               : 100,"
        "\"saved_at_as_unixtime\"    : 1514732400,"
        "\"timeout_at_as_unixtime\"  : 1514732401,"
        "\"finished_at_as_unixtime\" : 1514732402,"
        "\"component_files\": ["
        "{"
            "\"id\" : 101,"
            "\"index\" : 1,"
            "\"datatype\" : \"meta\","
            "\"status\" : \"hand_over\","
            "\"archive_size\": 32,"
            "\"encoded_archive_digest\" : \"hzZnQoSxtyJF6Qr72cSQww==\""
        "},"
        "{"
            "\"id\" : 102,"
            "\"index\" : 2,"
            "\"datatype\" : \"savea\","
            "\"status\" : \"hand_over\","
            "\"archive_size\" : 128,"
            "\"encoded_archive_digest\" : \"ILadtj0OkJMPxwbWDzhwuQ==\""
        "}"
    "]}}",
    // index 異常
    "{ \"save_data_archive\": {"
        "\"id\"                      : 1234,"
        "\"nsa_id\"                  : \"aaaaaaaaaaaaaaaa\","
        "\"application_id\"          : \"bbbbbbbbbbbbbbbb\","
        "\"ua_id\"                   : \"cccccccccccccccc\","
        "\"device_id\"               : \"dddddddddddddddd\","
        "\"data_size\"               : 123456789,"
        "\"auto_backup\"             : false,"
        "\"encoded_digest\"          : \"bD1gyr6Kck4t7RExh_MPDPfVK-3aoCYb7Cbp0FuBrX8=\","
        "\"status\"                  : \"fixed\","
        "\"has_thumbnail\"           : false,"
        "\"launch_required_version\" : 5678,"
        "\"num_of_partitions\"   : 16,"
        "\"series_id\"               : 100,"
        "\"saved_at_as_unixtime\"    : 1514732400,"
        "\"timeout_at_as_unixtime\"  : 1514732401,"
        "\"finished_at_as_unixtime\" : 1514732402,"
        "\"component_files\": ["
        "{"
            "\"id\" : 101,"
            "\"index\" : 4294967296,"
            "\"datatype\" : \"meta\","
            "\"status\" : \"hand_over\","
            "\"archive_size\": 32,"
            "\"encoded_archive_digest\" : \"hzZnQoSxtyJF6Qr72cSQww==\""
        "},"
        "{"
            "\"id\" : 102,"
            "\"index\" : 2,"
            "\"datatype\" : \"save\","
            "\"status\" : \"hand_over\","
            "\"archive_size\" : 128,"
            "\"encoded_archive_digest\" : \"ILadtj0OkJMPxwbWDzhwuQ==\""
        "}"
    "]}}",
    "{ \"save_data_archive\": {"
        "\"id\"                      : 1234,"
        "\"nsa_id\"                  : \"aaaaaaaaaaaaaaaa\","
        "\"application_id\"          : \"bbbbbbbbbbbbbbbb\","
        "\"ua_id\"                   : \"cccccccccccccccc\","
        "\"device_id\"               : \"dddddddddddddddd\","
        "\"data_size\"               : 123456789,"
        "\"auto_backup\"             : false,"
        "\"encoded_digest\"          : \"bD1gyr6Kck4t7RExh_MPDPfVK-3aoCYb7Cbp0FuBrX8=\","
        "\"status\"                  : \"fixed\","
        "\"has_thumbnail\"           : false,"
        "\"launch_required_version\" : 5678,"
        "\"num_of_partitions\"   : 16,"
        "\"series_id\"               : 100,"
        "\"saved_at_as_unixtime\"    : 1514732400,"
        "\"timeout_at_as_unixtime\"  : 1514732401,"
        "\"finished_at_as_unixtime\" : 1514732402,"
        "\"component_files\": ["
        "{"
            "\"id\" : 101,"
            "\"index\" : 1,"
            "\"datatype\" : \"meta\","
            "\"status\" : \"hand_over\","
            "\"archive_size\": 32,"
            "\"encoded_archive_digest\" : \"hzZnQoSxtyJF6Qr72cSQww==\""
        "},"
        "{"
            "\"id\" : 102,"
            "\"index\" : 4294967296,"
            "\"datatype\" : \"save\","
            "\"status\" : \"hand_over\","
            "\"archive_size\" : 128,"
            "\"encoded_archive_digest\" : \"ILadtj0OkJMPxwbWDzhwuQ==\""
        "}"
    "]}}",
    // size 異常
    "{ \"save_data_archive\": {"
        "\"id\"                      : 1234,"
        "\"nsa_id\"                  : \"aaaaaaaaaaaaaaaa\","
        "\"application_id\"          : \"bbbbbbbbbbbbbbbb\","
        "\"ua_id\"                   : \"cccccccccccccccc\","
        "\"device_id\"               : \"dddddddddddddddd\","
        "\"data_size\"               : 123456789,"
        "\"auto_backup\"             : false,"
        "\"encoded_digest\"          : \"bD1gyr6Kck4t7RExh_MPDPfVK-3aoCYb7Cbp0FuBrX8=\","
        "\"status\"                  : \"fixed\","
        "\"has_thumbnail\"           : false,"
        "\"launch_required_version\" : 5678,"
        "\"num_of_partitions\"   : 16,"
        "\"series_id\"               : 100,"
        "\"saved_at_as_unixtime\"    : 1514732400,"
        "\"timeout_at_as_unixtime\"  : 1514732401,"
        "\"finished_at_as_unixtime\" : 1514732402,"
        "\"component_files\": ["
        "{"
            "\"id\" : 101,"
            "\"index\" : 1,"
            "\"datatype\" : \"meta\","
            "\"status\" : \"hand_over\","
            "\"archive_size\": 4294967296,"
            "\"encoded_archive_digest\" : \"hzZnQoSxtyJF6Qr72cSQww==\""
        "},"
        "{"
            "\"id\" : 102,"
            "\"index\" : 2,"
            "\"datatype\" : \"save\","
            "\"status\" : \"hand_over\","
            "\"archive_size\" : 128,"
            "\"encoded_archive_digest\" : \"ILadtj0OkJMPxwbWDzhwuQ==\""
        "}"
    "]}}",
    "{ \"save_data_archive\": {"
        "\"id\"                      : 1234,"
        "\"nsa_id\"                  : \"aaaaaaaaaaaaaaaa\","
        "\"application_id\"          : \"bbbbbbbbbbbbbbbb\","
        "\"ua_id\"                   : \"cccccccccccccccc\","
        "\"device_id\"               : \"dddddddddddddddd\","
        "\"data_size\"               : 123456789,"
        "\"auto_backup\"             : false,"
        "\"encoded_digest\"          : \"bD1gyr6Kck4t7RExh_MPDPfVK-3aoCYb7Cbp0FuBrX8=\","
        "\"status\"                  : \"fixed\","
        "\"has_thumbnail\"           : false,"
        "\"launch_required_version\" : 5678,"
        "\"num_of_partitions\"   : 16,"
        "\"series_id\"               : 100,"
        "\"saved_at_as_unixtime\"    : 1514732400,"
        "\"timeout_at_as_unixtime\"  : 1514732401,"
        "\"finished_at_as_unixtime\" : 1514732402,"
        "\"component_files\": ["
        "{"
            "\"id\" : 101,"
            "\"index\" : 1,"
            "\"datatype\" : \"meta\","
            "\"status\" : \"hand_over\","
            "\"archive_size\": 32,"
            "\"encoded_archive_digest\" : \"hzZnQoSxtyJF6Qr72cSQww==\""
        "},"
        "{"
            "\"id\" : 102,"
            "\"index\" : 2,"
            "\"datatype\" : \"save\","
            "\"status\" : \"hand_over\","
            "\"archive_size\" : 4294967296,"
            "\"encoded_archive_digest\" : \"ILadtj0OkJMPxwbWDzhwuQ==\""
        "}"
    "]}}",
    // Validate エラー
    "{ \"save_data_archive\": {"
        "\"id\"                      : 1234,"
        "\"nsa_id\"                  : \"aaaaaaaaaaaaaaaa\","
        "\"application_id\"          : \"bbbbbbbbbbbbbbbb\","
        "\"ua_id\"                   : \"cccccccccccccccc\","
        "\"device_id\"               : \"dddddddddddddddd\","
        "\"data_size\"               : 123456789,"
        "\"auto_backup\"             : false,"
        "\"encoded_digest\"          : \"bD1gyr6Kck4t7RExh_MPDPfVK-3aoCYb7Cbp0FuBrX8=\","
        "\"status\"                  : \"fixed\","
        "\"has_thumbnail\"           : false,"
        "\"launch_required_version\" : 5678,"
        "\"num_of_partitions\"   : 16,"
        "\"series_id\"               : 100,"
        "\"saved_at_as_unixtime\"    : 1514732400,"
        "\"timeout_at_as_unixtime\"  : 1514732401,"
        "\"finished_at_as_unixtime\" : 1514732402,"
        "\"component_files\": ["
        "{"
            "\"id\" : 101,"
            "\"index\" : 1,"
            "\"datatype\" : \"meta\","
            "\"archive_size\": 32,"
            "\"encoded_archive_digest\" : \"hzZnQoSxtyJF6Qr72cSQww==\""
        "},"
        "{"
            "\"id\" : 102,"
            "\"index\" : 2,"
            "\"datatype\" : \"save\","
            "\"status\" : \"hand_over\","
            "\"archive_size\" : 128,"
            "\"encoded_archive_digest\" : \"ILadtj0OkJMPxwbWDzhwuQ==\""
        "}"
    "]}}",
    "{ \"save_data_archive\": {"
        "\"id\"                      : 1234,"
        "\"nsa_id\"                  : \"aaaaaaaaaaaaaaaa\","
        "\"application_id\"          : \"bbbbbbbbbbbbbbbb\","
        "\"ua_id\"                   : \"cccccccccccccccc\","
        "\"device_id\"               : \"dddddddddddddddd\","
        "\"data_size\"               : 123456789,"
        "\"auto_backup\"             : false,"
        "\"encoded_digest\"          : \"bD1gyr6Kck4t7RExh_MPDPfVK-3aoCYb7Cbp0FuBrX8=\","
        "\"status\"                  : \"fixed\","
        "\"has_thumbnail\"           : false,"
        "\"launch_required_version\" : 5678,"
        "\"num_of_partitions\"   : 16,"
        "\"series_id\"               : 100,"
        "\"saved_at_as_unixtime\"    : 1514732400,"
        "\"timeout_at_as_unixtime\"  : 1514732401,"
        "\"finished_at_as_unixtime\" : 1514732402,"
        "\"component_files\": ["
        "{"
            "\"id\" : 101,"
            "\"index\" : 1,"
            "\"datatype\" : \"meta\","
            "\"status\" : \"hand_over\","
            "\"archive_size\": 32,"
            "\"encoded_archive_digest\" : \"hzZnQoSxtyJF6Qr72cSQww==\""
        "},"
        "{"
            "\"id\" : 102,"
            "\"datatype\" : \"save\","
            "\"status\" : \"hand_over\","
            "\"archive_size\": 32,"
            "\"encoded_archive_digest\" : \"ILadtj0OkJMPxwbWDzhwuQ==\""
        "}"
    "]}}",
};


struct Cancelable
{
    bool IsCancelled() const { return false; }
};

template<typename ExpectResultType>
void TestImpl(const char json[])
{
    auto uid = nnt::olsc::GetFirstUserId();
    nn::fs::DeleteSystemSaveData(nnt::olsc::SystemSaveDataIdForTest, uid);
    srv::util::DefaultMountManager mountManager(nnt::olsc::MountInfoForTest, nnt::olsc::MountInfoForTest, nnt::olsc::MountInfoForTest);
    srv::database::SaveDataArchiveInfoCache sdaInfoCache(uid, mountManager);

    olsc::srv::adaptor::SaveDataArchiveInfoStorageOutputStream sdaOutputStream(sdaInfoCache);

    const int CfCount = 10;
    olsc::srv::ComponentFileInfo cfList[CfCount];
    olsc::srv::adaptor::ComponentFileInfoMemoryOutputStream cfOutputStream(cfList, CfCount);

    olsc::srv::adaptor::MixedInfoAdaptor adaptor(&sdaOutputStream, &cfOutputStream);

    nnt::olsc::MemoryInputStreamForRapidJson inputStream;
    inputStream.Set(json, JsonLen);

    const Cancelable c;
    NNT_EXPECT_RESULT_SUCCESS(http::json::ImportJsonByRapidJson<http::json::DefaultJsonErrorMap>(adaptor, inputStream, &c));
    NNT_EXPECT_RESULT_FAILURE(ExpectResultType, adaptor.GetResult());
}

template<typename ExpectResultType>
void RunTest(const char json[][JsonLen], int count)
{
    for (int i = 0; i < count; ++i)
    {
        TestImpl<ExpectResultType>(json[i]);
    }
}

TEST(OlscJsonAdaptorTest, ComponentFileMetaFail)
{
    RunTest<olsc::ResultSaveDataArchiveUnacceptableContent>(SaveDataArchiveParseError, sizeof(SaveDataArchiveParseError) / JsonLen);
    RunTest<olsc::ResultComponentFileUnacceptableContent>(CompomentFileParseError, sizeof(CompomentFileParseError) / JsonLen);
}
}
