﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/result/testResult_Assert.h>

#include <nn/fs.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/fs_SystemSaveDataPrivate.h>
#include <nn/fs/fs_SaveDataTypes.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/olsc/olsc_Result.h>
#include <nn/olsc/srv/database/olsc_SortedDataArray.h>
#include <nn/olsc/srv/util/olsc_MountManager.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/time.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_FormatString.h>

#include "testOlsc_Stopwatch.h"

#include <random>

using namespace nn;

namespace {
    const int64_t   SaveDataSize                = 0x00200000;   // 2MB
    const int64_t   JournalSize                 = 0x00200000;   // 2MB
    const int       SaveDataFlag                = 0;
    const char*     MetaFilePath                = "test_db_meta";
    const char*     EntryFilePath               = "test_db_data";
    const fs::SystemSaveDataId SystemSaveDataId = 0x8000000000004000;

    olsc::srv::util::MountInfo TestSaveInfo = {
        SystemSaveDataId,
        SaveDataSize,
        JournalSize,
        SaveDataFlag
    };

    class OlscSortedDataArrayTest : public testing::Test
    {
    protected:
        virtual void SetUp()
        {
            NNT_ASSERT_RESULT_SUCCESS(InitializeSaveData());
        }

        virtual void TearDown()
        {
            NNT_ASSERT_RESULT_SUCCESS(FinalizeSaveData());
        }

        static void SetUpTestCase()
        {
            fs::SetEnabledAutoAbort(false);
            fs::DisableAutoSaveDataCreation();
            NNT_ASSERT_RESULT_SUCCESS(time::Initialize());
        }

        static void TearDownTestCase()
        {
            NNT_ASSERT_RESULT_SUCCESS(time::Finalize());
        }
    private:

        static Result InitializeSaveData()
        {
            NN_RESULT_TRY(fs::DeleteSystemSaveData(fs::SaveDataSpaceId::System, SystemSaveDataId, {}))
                NN_RESULT_CATCH(fs::ResultTargetNotFound)
                {
                }
            NN_RESULT_END_TRY;

            NN_RESULT_SUCCESS;
        }

        static Result FinalizeSaveData()
        {
            NN_RESULT_TRY(fs::DeleteSystemSaveData(fs::SaveDataSpaceId::System, SystemSaveDataId, {}))
                NN_RESULT_CATCH(fs::ResultTargetNotFound)
                {
                }
            NN_RESULT_END_TRY;

            NN_RESULT_SUCCESS;
        }

    };

#if defined(NN_BUILD_CONFIG_OS_WIN)
    const int MaxEntryCount = 32;
#else
    const int MaxEntryCount = 2048;
#endif
    const int ReadBufferCount = MaxEntryCount / 8;

    using SimpleDatastoreForTestBase = olsc::srv::database::SortedDataArray<int, int, ReadBufferCount>;
    class SimpleDatastoreForTest : public SimpleDatastoreForTestBase
    {
    public:
        NN_IMPLICIT SimpleDatastoreForTest(olsc::srv::util::DefaultMountManager& mountManager) :
            SimpleDatastoreForTestBase(Compare, &m_ReadBuffer, MaxEntryCount), m_MountManager(mountManager)
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(InitializeUnsafe());
        }

    protected:
        virtual const char* GetMetadataFileRelativePath() const NN_NOEXCEPT NN_OVERRIDE
        {
            return MetaFilePath;
        }

        virtual const char* GetEntryFileRelativePath() const NN_NOEXCEPT NN_OVERRIDE
        {
            return EntryFilePath;
        }

        virtual olsc::srv::util::ReadMount AcquireReadMount() const NN_NOEXCEPT NN_OVERRIDE
        {
            return m_MountManager.AcquireDeviceSaveForRead();
        }

        virtual olsc::srv::util::WriteMount AcquireWriteMount() NN_NOEXCEPT NN_OVERRIDE
        {
            return m_MountManager.AcquireDeviceSaveForWrite();
        }

    private:
        olsc::srv::util::DefaultMountManager& m_MountManager;
        static int Compare(const int& lhs, const int& rhs) NN_NOEXCEPT
        {
            return lhs - rhs;
        }
        ReadBuffer m_ReadBuffer;
    };

    void CreateInputPattern(int out[], int maxCount) NN_NOEXCEPT
    {
        for (int i = 0; i < maxCount; ++i)
        {
            out[i] = i;
        }
        std::random_device seed_gen;
        std::mt19937 engine(seed_gen());
        std::shuffle(&out[0], &out[maxCount], engine);
    }
}


TEST_F(OlscSortedDataArrayTest, Basic)
{
    nn::olsc::srv::util::DefaultMountManager mountManager(TestSaveInfo, TestSaveInfo, TestSaveInfo);

    SimpleDatastoreForTest db(mountManager);

    std::lock_guard<decltype(db)> lock(db);

    EXPECT_EQ(0, db.GetCount());

    std::unique_ptr<int[]> inputPattern(new int[MaxEntryCount]);
    CreateInputPattern(inputPattern.get(), MaxEntryCount);

    {
        nnt::olsc::Stopwatch s(true, "Add entries");
        for (int i = 0; i < MaxEntryCount; ++i)
        {
            auto key = inputPattern[i];
            auto value = inputPattern[i];
            nnt::olsc::Stopwatch s2(true, "Add entry[%d]: value = %d", i, value);
            EXPECT_TRUE(db.Add(key, value));
            EXPECT_EQ(value, db.Find(key));
        }
    }

    EXPECT_EQ(MaxEntryCount, db.GetCount());

    {
        nnt::olsc::Stopwatch s(true, "Replace entries");
        for (int i = 0; i < MaxEntryCount; ++i)
        {
            nnt::olsc::Stopwatch s2(true, "Replace entry[%d]", i);
            int value = i + 1;
            EXPECT_TRUE(db.Add(i, value));
        }
    }

    {
        nnt::olsc::Stopwatch s(true, "Get entries");
        for (int i = 0; i < MaxEntryCount; ++i)
        {
            nnt::olsc::Stopwatch s2(true, "Get entry[%d]", i);
            EXPECT_EQ(i + 1, db.Find(i));
        }
    }

    {
        nnt::olsc::Stopwatch s(true, "Remove entries");
        for (int i = 0; i < MaxEntryCount; ++i)
        {
            nnt::olsc::Stopwatch s2(true, "Remove entry[%d]", i);
            db.Remove(i);
            EXPECT_FALSE(db.Find(i));
        }
    }

    EXPECT_EQ(0, db.GetCount());
}
