﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/result/testResult_Assert.h>

#include <nn/fs.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/fs_SystemSaveDataPrivate.h>
#include <nn/fs/fs_SaveDataTypes.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/olsc/olsc_Result.h>
#include <nn/olsc/srv/database/olsc_DataArray.h>
#include <nn/olsc/srv/util/olsc_MountManager.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/time.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_FormatString.h>

using namespace nn;

//#define NN_OLSC_SIMPLE_DATABASE_MAX_ENTRY_TEST

namespace {
    const int64_t   SaveDataSize                = 0x00200000;   // 2MB
    const int64_t   JournalSize                 = 0x00200000;   // 2MB
    const int       SaveDataFlag                = 0;
    const char*     MetaFilePath                = "test_db_meta";
    const char*     EntryFilePath               = "test_db_data";
    const fs::SystemSaveDataId SystemSaveDataId = 0x8000000000004000;

    olsc::srv::util::MountInfo TestSaveInfo = {
        SystemSaveDataId,
        SaveDataSize,
        JournalSize,
        SaveDataFlag
    };

    class OlscDataArrayTest : public testing::Test
    {
    protected:
        virtual void SetUp()
        {
            NNT_ASSERT_RESULT_SUCCESS(InitializeSaveData());
        }

        virtual void TearDown()
        {
            NNT_ASSERT_RESULT_SUCCESS(FinalizeSaveData());
        }

        static void SetUpTestCase()
        {
            fs::SetEnabledAutoAbort(false);
            fs::DisableAutoSaveDataCreation();
            NNT_ASSERT_RESULT_SUCCESS(time::Initialize());
        }

        static void TearDownTestCase()
        {
            NNT_ASSERT_RESULT_SUCCESS(time::Finalize());
        }
    private:

        static Result InitializeSaveData()
        {
            NN_RESULT_TRY(fs::DeleteSystemSaveData(fs::SaveDataSpaceId::System, SystemSaveDataId, {}))
                NN_RESULT_CATCH(fs::ResultTargetNotFound)
                {
                }
            NN_RESULT_END_TRY;

            NN_RESULT_SUCCESS;
        }

        static Result FinalizeSaveData()
        {
            NN_RESULT_TRY(fs::DeleteSystemSaveData(fs::SaveDataSpaceId::System, SystemSaveDataId, {}))
                NN_RESULT_CATCH(fs::ResultTargetNotFound)
                {
                }
            NN_RESULT_END_TRY;

            NN_RESULT_SUCCESS;
        }

    };

    class Stopwatch
    {
    public:
        Stopwatch(bool enable, const char* fmt, ...) : m_Enable(enable)
        {
            std::va_list vlist;
            va_start(vlist, fmt);
            util::VSNPrintf(m_Title, sizeof(m_Title), fmt, vlist);
            va_end(vlist);

            m_Begin = os::GetSystemTick();
        }
        ~Stopwatch()
        {
            if (m_Enable)
            {
                auto end = os::GetSystemTick();
                NN_LOG("%s: %lld ms\n", m_Title, (end - m_Begin).ToTimeSpan().GetMilliSeconds());
            }
        }
    private:
        char m_Title[128];
        bool m_Enable;
        os::Tick m_Begin;
    };


    void GetExpectedListPattern(int out[], int count)
    {
        NN_SDK_ASSERT(count % 2 == 0);

        for (int i = 0; i < count / 2; ++i)
        {
            out[i] = count - i * 2 - 1;
        }

        for (int i = 0; i < count / 2; ++i)
        {
            out[count / 2 + i] = i * 2;
        }
    }

#if defined(NN_OLSC_SIMPLE_DATABASE_MAX_ENTRY_TEST)
    const int MaxEntryCount = 16 * 1024;
#else
    const int MaxEntryCount = 3 * 1024;
#endif

    const int CacheCount = 1024;
    const int ReadBufferCount = 256;


    class DataArrayForTest : public olsc::srv::database::DataArray<olsc::srv::util::DefaultMountManager, int, MaxEntryCount, CacheCount, ReadBufferCount>
    {
    public:
        using Base = olsc::srv::database::DataArray<olsc::srv::util::DefaultMountManager, int, MaxEntryCount, CacheCount, ReadBufferCount>;
        DataArrayForTest(CacheBuffer* cacheBuffer, ReadBuffer* readBuffer, olsc::srv::util::DefaultMountManager& mountManager) : Base(cacheBuffer, readBuffer, mountManager)
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(InitializeUnsafe());
        }

    protected:
        virtual const char* GetMetadataFileRelativePath() const NN_NOEXCEPT NN_OVERRIDE
        {
            return MetaFilePath;
        }

        virtual const char* GetEntryFileRelativePath() const NN_NOEXCEPT NN_OVERRIDE
        {
            return EntryFilePath;
        }

        virtual olsc::srv::util::ReadMount AcquireReadMount(olsc::srv::util::DefaultMountManager& mounter) const NN_NOEXCEPT NN_OVERRIDE
        {
            return mounter.AcquireDeviceSaveForRead();
        }

        virtual olsc::srv::util::WriteMount AcquireWriteMount(olsc::srv::util::DefaultMountManager& mounter) NN_NOEXCEPT NN_OVERRIDE
        {
            return mounter.AcquireDeviceSaveForWrite();
        }
    };

    class NonCachedDataArrayForTest : public olsc::srv::database::DataArray<olsc::srv::util::DefaultMountManager, int, MaxEntryCount, 0, ReadBufferCount>
    {
    public:
        using Base = olsc::srv::database::DataArray<olsc::srv::util::DefaultMountManager, int, MaxEntryCount, 0, ReadBufferCount>;
        NonCachedDataArrayForTest(CacheBuffer* cacheBuffer, ReadBuffer* readBuffer, olsc::srv::util::DefaultMountManager& mountManager) : Base(cacheBuffer, readBuffer, mountManager)
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(InitializeUnsafe());
        }

    protected:
        virtual const char* GetMetadataFileRelativePath() const NN_NOEXCEPT NN_OVERRIDE
        {
            return "non_cached_db_meta";
        }

        virtual const char* GetEntryFileRelativePath() const NN_NOEXCEPT NN_OVERRIDE
        {
            return "non_cached_db_entry";
        }

        virtual olsc::srv::util::ReadMount AcquireReadMount(olsc::srv::util::DefaultMountManager& mounter) const NN_NOEXCEPT NN_OVERRIDE
        {
            return mounter.AcquireDeviceSaveForRead();
        }

        virtual olsc::srv::util::WriteMount AcquireWriteMount(olsc::srv::util::DefaultMountManager& mounter) NN_NOEXCEPT NN_OVERRIDE
        {
            return mounter.AcquireDeviceSaveForWrite();
        }
    };

    template<typename DatabaseType>
    void BasicTest(DatabaseType& db, int logOutputInterval) NN_NOEXCEPT
    {
        std::lock_guard<decltype(db)> lock(db);

        // PushBack, PushFront を組み合わせて順番が正しくなることを確認
        // 偶奇が分かれるパターンにする。
        // 例えば count = 10 なら
        // 9,7,5,3,1,0,2,4,6,8  となる
        for (int i = 0; i < MaxEntryCount; ++i)
        {
            if (i % 2 == 0)
            {
                Stopwatch s(i % logOutputInterval == 0, "PushBack: i = %d", i);
                NNT_EXPECT_RESULT_SUCCESS(db.PushBack(i));
            }
            else
            {
                Stopwatch s(i % logOutputInterval == 1, "PushFront: i = %d", i);
                NNT_EXPECT_RESULT_SUCCESS(db.PushFront(i));
            }
            EXPECT_EQ(i + 1, db.GetCount());
        }

        // エントリ単品で取得
        int expected[MaxEntryCount];
        GetExpectedListPattern(expected, MaxEntryCount);
        for (int i = 0; i < MaxEntryCount; ++i)
        {
            NN_LOG("Checking: %d\n", i);
            int entry;
            {
                Stopwatch s(i % logOutputInterval == 0, "At: i = %d", i);
                NNT_EXPECT_RESULT_SUCCESS(db.At(&entry, i));
            }
            EXPECT_EQ(expected[i], entry);
            EXPECT_TRUE(db.HasEntry(entry));
        }

        // リストアップ
        static int listBuffer[MaxEntryCount];
        int listCount;
        {
            Stopwatch s(true, "List:");
            listCount = db.List(listBuffer, MaxEntryCount, 0);
        }
        EXPECT_EQ(MaxEntryCount, listCount);
        for (int i = 0; i < MaxEntryCount; ++i)
        {
            EXPECT_EQ(expected[i], listBuffer[i]);
        }

        // 上限
#if defined(NN_OLSC_SIMPLE_DATABASE_MAX_ENTRY_TEST)
        int entry = MaxEntryCount;
        NNT_EXPECT_RESULT_FAILURE(olsc::ResultDatabaseKeyListOutOfSpace, db.PushBack(entry));
        NNT_EXPECT_RESULT_FAILURE(olsc::ResultDatabaseKeyListOutOfSpace, db.PushFront(entry));
#endif

        // 奇数を削除
        NNT_EXPECT_RESULT_SUCCESS(db.RemoveAll([](const int& entry) {
            return entry % 2 == 1;
        }));
        EXPECT_EQ(MaxEntryCount / 2, db.GetCount());

        // 偶数を削除
        for (int i = 0; i < MaxEntryCount / 2; ++i)
        {
            auto entry = i * 2;
            NNT_EXPECT_RESULT_SUCCESS(db.Remove(entry));
            NNT_EXPECT_RESULT_FAILURE(olsc::ResultDatabaseEntryNotFound, db.Remove(entry));
            EXPECT_FALSE(db.HasEntry(entry));
        }
        EXPECT_EQ(0, db.GetCount());
    }
}


TEST_F(OlscDataArrayTest, Basic)
{
    using Database = DataArrayForTest;
    static Database::CacheBuffer cacheBuffer = {};
    static Database::ReadBuffer readBuffer = {};
    olsc::srv::util::DefaultMountManager mountManager(TestSaveInfo, TestSaveInfo, TestSaveInfo);

    Database db(&cacheBuffer, &readBuffer, mountManager);

    BasicTest(db, CacheCount);
}



TEST_F(OlscDataArrayTest, NonCache)
{
    using Database = NonCachedDataArrayForTest;
    static Database::CacheBuffer cacheBuffer = {};
    static Database::ReadBuffer readBuffer = {};
    olsc::srv::util::DefaultMountManager mountManager(TestSaveInfo, TestSaveInfo, TestSaveInfo);

    Database db(&cacheBuffer, &readBuffer, mountManager);
    BasicTest(db, ReadBufferCount);
}
