﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/result/testResult_Assert.h>

#include <nn/account/account_Api.h>
#include <nn/account/account_ApiForSystemServices.h>
#include <nn/account/account_ApiForApplications.h>
#include <nn/crypto/crypto_Sha256Generator.h>
#include <nn/fs.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/fs_SystemSaveDataPrivate.h>
#include <nn/fs/fs_SaveDataTypes.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/olsc/olsc_Result.h>
#include <nn/olsc/srv/database/olsc_AutoUploadSettingDatabase.h>
#include <nn/olsc/srv/util/olsc_MountManager.h>
#include <nn/olsc/srv/util/olsc_SaveData.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_FormatString.h>

#include <random>
#include "testOlsc_Stopwatch.h"

using namespace nn;

namespace {
    const int64_t   SaveDataSize                = 0x00200000;   // 2MB
    const int64_t   JournalSize                 = 0x00200000;   // 2MB
    const int       SaveDataFlag                = 0;
    const fs::SystemSaveDataId SystemSaveDataId = 0x8000000000004000;

    olsc::srv::util::MountInfo TestSaveInfo = {
        SystemSaveDataId,
        SaveDataSize,
        JournalSize,
        SaveDataFlag,
    };

    class OlscSimpleDatabaseTest : public testing::Test
    {
    protected:
        virtual void SetUp()
        {
            NNT_ASSERT_RESULT_SUCCESS(InitializeSaveData());
        }

        virtual void TearDown()
        {
        }

        static void SetUpTestCase()
        {
            account::InitializeForSystemService();
            fs::SetEnabledAutoAbort(false);
            fs::DisableAutoSaveDataCreation();
        }

        static void TearDownTestCase()
        {
        }
    private:

        static Result InitializeSaveData()
        {
            NN_RESULT_TRY(fs::DeleteSystemSaveData(fs::SaveDataSpaceId::System, SystemSaveDataId, {}))
                NN_RESULT_CATCH(fs::ResultTargetNotFound)
                {
                }
            NN_RESULT_END_TRY;

            NN_RESULT_SUCCESS;
        }
    };

    Result GetUid(account::Uid* out, int index)
    {
        account::Uid uids[account::UserCountMax];

        int listCount;
        NN_RESULT_DO(account::ListAllUsers(&listCount, uids, account::UserCountMax));
        NN_ABORT_UNLESS(index < listCount);

        *out = uids[index];

        NN_RESULT_SUCCESS;
    }

    void CreateAppIdList(ApplicationId out[], int count)
    {
        for (int i = 0; i < count; ++i)
        {
            out[i] = { static_cast<Bit64>(i) };
        }

        std::random_device seed_gen;
        std::mt19937 engine(seed_gen());
        std::shuffle(&out[0], &out[count], engine);
    }
}

using AutoUploadSettingDatabase = olsc::srv::database::AutoUploadSettingDatabase;
using ApplicationAutoUploadSetting = AutoUploadSettingDatabase::ApplicationAutoUploadSetting;

TEST_F(OlscSimpleDatabaseTest, ApplicationSetting)
{
    account::Uid uid;
    NNT_EXPECT_RESULT_SUCCESS(GetUid(&uid, 0));

    olsc::srv::util::DefaultMountManager mountManager(TestSaveInfo, TestSaveInfo, TestSaveInfo);
    AutoUploadSettingDatabase db(uid, mountManager);

    const int MaxAppCount = olsc::srv::MaxApplicationCount;

    std::unique_ptr<ApplicationId[]> appIdList(new ApplicationId[MaxAppCount]);
    CreateAppIdList(appIdList.get(), MaxAppCount);

    // 書き・読み
    for (int i = 0; i < MaxAppCount; ++i)
    {
        auto& appId = appIdList[i];
        {
            nnt::olsc::Stopwatch s(true, "Confirm not registered: [%d]: %016llx", i, appId.value);
            EXPECT_EQ(ApplicationAutoUploadSetting::NotSet, db.GetApplicationAutoUploadSetting(appId));
        }
        {
            nnt::olsc::Stopwatch s(true, "Set disable: [%d]: %016llx", i, appId.value);
            db.SetApplicationAutoUploadSetting(appId, ApplicationAutoUploadSetting::Disabled);
        }

        {
            nnt::olsc::Stopwatch s(true, "Check disable: [%d]: %016llx", i, appId.value);
            EXPECT_EQ(ApplicationAutoUploadSetting::Disabled, db.GetApplicationAutoUploadSetting(appId));
        }

        {
            nnt::olsc::Stopwatch s(true, "Set enable: [%d]: %016llx", i, appId.value);
            db.SetApplicationAutoUploadSetting(appId, ApplicationAutoUploadSetting::Enabled);
        }

        {
            nnt::olsc::Stopwatch s(true, "Check enable: [%d]: %016llx", i, appId.value);
            EXPECT_EQ(ApplicationAutoUploadSetting::Enabled, db.GetApplicationAutoUploadSetting(appId));
        }
    }

    // 削除・読み
    for (int i = 0; i < MaxAppCount; ++i)
    {
        auto& appId = appIdList[i];
        {
            nnt::olsc::Stopwatch s(true, "Check enable: [%d]: %016llx", i, appId.value);
            EXPECT_EQ(ApplicationAutoUploadSetting::Enabled, db.GetApplicationAutoUploadSetting(appId));
        }
        {
            nnt::olsc::Stopwatch s(true, "Remove: [%d]: %016llx", i, appId.value);
            db.RemoveApplicationAutoUploadSetting(appId);
        }
        {
            nnt::olsc::Stopwatch s(true, "Check disable: [%d]: %016llx", i, appId.value);
            EXPECT_EQ(ApplicationAutoUploadSetting::NotSet, db.GetApplicationAutoUploadSetting(appId));
        }
    }
}

TEST_F(OlscSimpleDatabaseTest, GlobalSetting)
{
    account::Uid uid;
    NNT_EXPECT_RESULT_SUCCESS(GetUid(&uid, 0));

    olsc::srv::util::DefaultMountManager mountManager(TestSaveInfo, TestSaveInfo, TestSaveInfo);
    nn::olsc::srv::database::AutoUploadSettingDatabase db(uid, mountManager);

    const int MaxAppCount = olsc::srv::MaxApplicationCount;

    std::unique_ptr<ApplicationId[]> appIdList(new ApplicationId[MaxAppCount]);
    CreateAppIdList(appIdList.get(), MaxAppCount);

    NN_LOG("Importing test data\n");
    // false の AppId 設定を追加
    {
        nnt::olsc::Stopwatch s(true, "%d entries imported. ", MaxAppCount);
        for (int i = 0; i < MaxAppCount; ++i)
        {
            auto& appId = appIdList[i];
            {
                db.SetApplicationAutoUploadSetting(appId, ApplicationAutoUploadSetting::Disabled);
            }
        }
    }

    // GlobalSetting を true にして appId 設定が全て true になることを確認
    {
        nnt::olsc::Stopwatch s(true, "GlobalSetting -> true: ");
        db.SetGlobalSetting(true);
    }

    EXPECT_TRUE(db.GetGlobalSetting());

    for (int i = 0; i < MaxAppCount; ++i)
    {
        auto& appId = appIdList[i];
        {
            EXPECT_EQ(ApplicationAutoUploadSetting::Enabled, db.GetApplicationAutoUploadSetting(appId));
        }
    }


    // GlobalSetting を false にして appId 設定が全て false になることを確認
    {
        nnt::olsc::Stopwatch s(true, "GlobalSetting -> false: ");
        db.SetGlobalSetting(false);
    }

    EXPECT_FALSE(db.GetGlobalSetting());

    for (int i = 0; i < MaxAppCount; ++i)
    {
        auto& appId = appIdList[i];
        {
            EXPECT_EQ(ApplicationAutoUploadSetting::Disabled, db.GetApplicationAutoUploadSetting(appId));
        }
    }

}
