﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/result/testResult_Assert.h>

#include <nn/account/account_Api.h>
#include <nn/account/account_ApiForApplications.h>
#include <nn/account/account_ApiForAdministrators.h>
#include <nn/account/account_ApiForSystemServices.h>
#include <nn/fs.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/fs_SystemSaveDataPrivate.h>
#include <nn/fs/fs_SaveDataTypes.h>
#include <nn/fs/fs_Utility.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/olsc/olsc_Result.h>
#include <nn/olsc/srv/olsc_AutoUploadSettingManager.h>
#include <nn/olsc/srv/olsc_DaemonControllerImpl.h>
#include <nn/olsc/srv/util/olsc_MountManager.h>
#include <nn/olsc/srv/util/olsc_MountTypes.h>
#include <nn/time.h>

#include <nn/result/result_HandlingUtility.h>

#include "testOlsc_MemoryResource.h"
#include "testOlsc_SaveDataUtil.h"
#include "testOlsc_Stopwatch.h"

using namespace nn;
using namespace nn::olsc;

namespace {
    const fs::SystemSaveDataId SystemSaveDataId = 0x8000000000004000;
    const int64_t   SystemSaveDataSize = 4 * 1024 * 1024;
    const int64_t   SystemSaveJournalSize = 4 * 1024 * 1024;
    const uint32_t  SystemSaveDataFlags = 0;

    olsc::srv::util::MountInfo TestDeviceSave = {
        SystemSaveDataId,
        SystemSaveDataSize,
        SystemSaveJournalSize,
        SystemSaveDataFlags
    };

    olsc::srv::util::MountInfo TestUserSettingSave = {
        SystemSaveDataId + 1,
        SystemSaveDataSize,
        SystemSaveJournalSize,
        SystemSaveDataFlags
    };

    olsc::srv::util::MountInfo TestUserSeriesInfoSave = {
        SystemSaveDataId + 2,
        SystemSaveDataSize,
        SystemSaveJournalSize,
        SystemSaveDataFlags
    };

    class OlscDaemonControllerImplTest : public testing::Test
    {
    protected:
    protected:
        virtual void SetUp()
        {
            NNT_ASSERT_RESULT_SUCCESS(InitializeSaveData({}));
        }

        virtual void TearDown()
        {
        }

        static void SetUpTestCase()
        {
            fs::SetEnabledAutoAbort(false);
            fs::DisableAutoSaveDataCreation();
            NNT_ASSERT_RESULT_SUCCESS(time::Initialize());

#if defined( NN_BUILD_CONFIG_OS_WIN )
            nn::account::InitializeForAdministrator();

            nnt::olsc::CleanupUsers();
            nn::account::Uid uid;
            nnt::olsc::CreateUsers(&uid, 1);
#else
            account::InitializeForSystemService();
#endif
            NNT_ASSERT_RESULT_SUCCESS(nifm::InitializeSystem());
        }

        static void TearDownTestCase()
        {
            NNT_ASSERT_RESULT_SUCCESS(time::Finalize());
        }

        static Result InitializeSaveData(fs::UserId fsUser)
        {
            fs::SystemSaveDataId toDelete[] = {
                TestDeviceSave.systemSaveDataId,
                TestUserSettingSave.systemSaveDataId,
                TestUserSeriesInfoSave.systemSaveDataId,
            };
            for (auto id : toDelete)
            {
                NN_RESULT_TRY(fs::DeleteSystemSaveData(fs::SaveDataSpaceId::System, id, fsUser))
                    NN_RESULT_CATCH(fs::ResultTargetNotFound)
                {
                    NN_LOG("NotFound: %016llx\n", id);
                }
                NN_RESULT_END_TRY;
            }

            NN_RESULT_SUCCESS;
        }
    private:

    };

    srv::TransferTaskConfig CreateUploadTransferTaskConfig(bool isAutoUpload)
    {
        srv::TransferTaskConfig config;
        config.kind = TransferTaskKind::Upload;
        config.ulInfo.force = isAutoUpload;
        config.cachedSi = {};
        return config;
    }
}

TEST_F(OlscDaemonControllerImplTest, GlobalSetting)
{
    auto uid = nnt::olsc::GetFirstUserId();

    InitializeSaveData(fs::ConvertAccountUidToFsUserId(uid));

    olsc::srv::util::DefaultMountManager mountManager(TestDeviceSave, TestUserSettingSave, TestUserSeriesInfoSave);
    olsc::srv::database::TransferTaskDatabaseManager ttdm(mountManager);
    olsc::srv::AutoUploadSettingManager settingManager(mountManager);
    olsc::srv::DaemonControllerImpl dc(nnt::olsc::DefaultMemoryResource::GetInstance(), settingManager, ttdm);

    // 全体設定をオンからオフにする場合に自動 UL タスクが削除されることを確認するためのダミータスク登録
    ApplicationId autoUlTaskAppId = { 1 };
    TransferTaskId autoUlTaskId;
    srv::TransferTaskConfig autoUlConfig = CreateUploadTransferTaskConfig(false);
    NN_ABORT_UNLESS_RESULT_SUCCESS(ttdm.RegisterTransferTask(&autoUlTaskId, uid, autoUlTaskAppId, autoUlConfig, TransferTaskRank::Postponed));

    // 全体設定をオンからオフにする場合に非自動 UL タスクが削除されないことを確認するためのダミーダスク登録
    ApplicationId manualUlTaskAppId = { 2 };
    TransferTaskId manualUlTaskId;
    srv::TransferTaskConfig manualUlConfig = CreateUploadTransferTaskConfig(true);
    NN_ABORT_UNLESS_RESULT_SUCCESS(ttdm.RegisterTransferTask(&manualUlTaskId, uid, manualUlTaskAppId, manualUlConfig, TransferTaskRank::Postponed));

    dc.SetGlobalAutoUploadSetting(uid, true);
    dc.SetApplicationAutoUploadSetting(uid, autoUlTaskAppId, true);
    dc.SetApplicationAutoUploadSetting(uid, manualUlTaskAppId, true);
    dc.SetGlobalAutoUploadSetting(uid, false);

    auto autoUlTask = ttdm.FindDetailInfo([autoUlTaskAppId](const olsc::srv::TransferTaskDetailInfo& di) -> bool {
        return di.appId == autoUlTaskAppId;
    });
    EXPECT_FALSE(autoUlTask);

    auto manualUlTask = ttdm.FindDetailInfo([manualUlTaskAppId](const olsc::srv::TransferTaskDetailInfo& di) -> bool {
        return di.appId == manualUlTaskAppId;
    });
    EXPECT_TRUE(manualUlTask);
}
