﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nnt/nntest.h>
#include <nn/os.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/applet/applet.h>

#include <nv_GemControl.h>
#include <nv_GemCoreDump.h>

#include "Application.h"
#include "AppClass.h"

void GpuCoreDumpReaderThreadFunction(void*);
void GpuCoreDumpWaiterThreadFunction(void*);
void GpuCoreDumpAppThreadFunction(void*);

#define READER_LOG(...) NN_LOG("[Reader] "); NN_LOG(__VA_ARGS__);
#define WAITER_LOG(...) NN_LOG("[Waiter] "); NN_LOG(__VA_ARGS__);
#define APP_LOG(...) NN_LOG("[App] "); NN_LOG(__VA_ARGS__);

namespace
{
    enum FuncType
    {
        FuncType_Reader,
        FuncType_Waiter,
        FuncType_App,
        FuncType_Count,
    };

    struct ThreadUserData
    {
        AppBase* pApp;
    };

    const size_t ThreadStackSize = 32 * 1024 * 1024;

    NN_OS_ALIGNAS_THREAD_STACK char s_ThreadStacks[FuncType_Count][ThreadStackSize];
    nn::os::ThreadType  s_Threads[FuncType_Count];

    // for reader and waiter
    nn::os::EventType s_StartCoreDumpingEvent;
    nn::os::EventType s_FinishCoreDumpingEvent;

    // for reader
    GcdBlock s_GcdBlock;
}

void GpuCoreDumpReaderThreadFunction(void*)
{
    READER_LOG("launched.\n");

    nv::gem::Result result;

    READER_LOG("initialize nv::gemcoredump::client.\n");
    nv::gemcoredump::Client client;
    result = client.Initialize();
    NN_ASSERT(result == nv::gem::Result_Success, "Failed to initialize gcd reader client.\n");

    nn::applet::AppletResourceUserId aruidSelf = nn::applet::GetAppletResourceUserId();
    READER_LOG("application's aruid is 0x%016llx.\n", aruidSelf.lower);

    while (true)
    {
        READER_LOG("wait event to start core dumping.\n");

        if (!nn::os::TimedWaitEvent(&s_StartCoreDumpingEvent, nn::TimeSpan::FromSeconds(1)))
        {
            READER_LOG("core dump starting event is not signaled.\n");
            continue;
        }

        READER_LOG("core dump starting event is signaled.\n");
        break;
    }

    READER_LOG("get crashed application's aruid.\n");
    nn::applet::AppletResourceUserId aruid;
    result = client.GetAruid(&aruid.lower);
    NN_ASSERT(result == nv::gem::Result_Success, "Failed to get aruid.\n");
    NN_ASSERT(aruidSelf.lower == aruid.lower,
        "crashed application's aruid (0x%016llx) should be same as self aruid (0x%016llx).\n",
        aruidSelf.lower,
        aruid.lower
        );

    READER_LOG("crashed application's aruid is 0x%016llx.\n", aruid.lower);

    while (true)
    {
        READER_LOG("read next block of core dump\n");
        std::memset(s_GcdBlock.data, 0, MAX_GCD_BLOCK_SIZE);
        result = client.ReadNextBlock(&s_GcdBlock);
        NN_ASSERT(result == nv::gem::Result_Success, "Failed to get block.\n");

        READER_LOG("an block size of gpu core dump is %d byte.\n", s_GcdBlock.size);

        if (s_GcdBlock.size == 0)
        {
            READER_LOG("finish gpu core dump.\n");
            break;
        }

        READER_LOG("Dump an block of gpu core dump.\n");
        {
            char *ptr;
            uint32_t i = 0;
            while (i < s_GcdBlock.size)
            {
                ptr = reinterpret_cast<char*>(&s_GcdBlock.data[i]);
                READER_LOG("%s", ptr);
                i += (strlen(ptr) + 1);
            }
        }
    }

    READER_LOG("signal event to notify waiter to complete core dumping.\n");
    nn::os::SignalEvent(&s_FinishCoreDumpingEvent);

    READER_LOG("Done.\n");
}

void GpuCoreDumpWaiterThreadFunction(void*)
{
    WAITER_LOG("launched.\n");

    nv::gem::Result result;

    WAITER_LOG("initialize nv:gemcontrol::client.\n");
    nv::gemcontrol::Client client;
    result = client.Initialize();
    NN_ASSERT(result == nv::gem::Result_Success, "Failed to initialize gcd waiter client.\n");

    WAITER_LOG("disable gpu hang notification.\n");
    result = client.ControlNotification(false);
    NN_ASSERT(result == nv::gem::Result_Success, "Failed to ControlNotification to false\n");

    WAITER_LOG("get nvgem event.\n");
    nn::os::SystemEventType gcdEvent;
    result = client.GetEventHandle(&gcdEvent);
    NN_ASSERT(result == nv::gem::Result_Success, "Failed to get event.\n");

    WAITER_LOG("enable gpu hang notification.\n");
    result = client.ControlNotification(true);
    NN_ASSERT(result == nv::gem::Result_Success, "Failed to ControlNotification to true.\n");

    nn::applet::AppletResourceUserId aruidSelf = nn::applet::GetAppletResourceUserId();
    WAITER_LOG("application's aruid is 0x%016llx.\n", aruidSelf.lower);

    WAITER_LOG("enable target application's gpu hang notification.\n");
    result = client.SetNotificationPerm(aruidSelf.lower, true);
    NN_ASSERT(result == nv::gem::Result_Success, "Failed to SetNotificationPerm to true.\n");

    WAITER_LOG("enable target application's gpu core dumping.\n");
    result = client.SetCoreDumpPerm(aruidSelf.lower, true);
    NN_ASSERT(result == nv::gem::Result_Success, "Failed to SetCoreDumpPerm to true.\n");

    while (true)
    {
        WAITER_LOG("wait gpu hang event.\n");

        if (!nn::os::TimedWaitSystemEvent(&gcdEvent, nn::TimeSpan::FromSeconds(1)))
        {
            WAITER_LOG("gpu hang event is not signaled.\n");
            continue;
        }

        WAITER_LOG("gpu hang event is signaled.\n");
        break;
    }

    while (true)
    {
        // Event is signaled when:
        //
        // - The first GPU error occurs and there is no other errors.
        // - nv::gemcontrol::Client::Reset() is called when more than one error is pending.
        //
        // So, if you handle all of GPU errors at one time, you needs to clear the event for each handling.
        nn::os::ClearSystemEvent(&gcdEvent);

        nn::applet::AppletResourceUserId aruid;
        bool hasCoreDump;

        WAITER_LOG("get crashed application's aruid and a flag if core dump exists or not.\n");
        result = client.GetAruid(&aruid.lower, &hasCoreDump);
        NN_ASSERT(result == nv::gem::Result_Success);

        if (aruid.lower == 0)
        {
            WAITER_LOG("there is no errors.\n");
            break;
        }

        NN_ASSERT(aruidSelf.lower == aruid.lower,
            "crashed application's aruid (0x%016llx) should be same as self aruid (0x%016llx).\n",
            aruidSelf.lower,
            aruid.lower
        );

        WAITER_LOG("target applicatin's aruid is 0x%016llx.\n", aruid.lower);

        if (hasCoreDump)
        {
            WAITER_LOG("signal to start core dumping.\n");
            nn::os::SignalEvent(&s_StartCoreDumpingEvent);

            while (true)
            {
                WAITER_LOG("wait to finish core dumping.\n");

                if (!nn::os::TimedWaitEvent(&s_FinishCoreDumpingEvent, nn::TimeSpan::FromSeconds(1)))
                {
                    WAITER_LOG("core dumping finished event is not signaled.\n");
                    continue;
                }

                WAITER_LOG("core dumping finished event is signaled.\n");
                break;
            }
        }
        else
        {
            WAITER_LOG("core dump does not exist.\n");
        }

        WAITER_LOG("reset gpu hang status.\n");
        result = client.Reset();
        NN_ASSERT(result == nv::gem::Result_Success, "Failed to Reset.\n");
    }

    WAITER_LOG("disable target application's gpu core dumping.\n");
    result = client.SetCoreDumpPerm(aruidSelf.lower, false);
    NN_ASSERT(result == nv::gem::Result_Success, "Failed to SetCoreDumpPerm to false.\n");

    WAITER_LOG("disable target application's gpu hang notification.\n");
    result = client.SetNotificationPerm(aruidSelf.lower, false);
    NN_ASSERT(result == nv::gem::Result_Success, "Failed to SetNotificationPerm to false.\n");

    WAITER_LOG("disable gpu hang notification.\n");
    result = client.ControlNotification(false);
    NN_ASSERT(result == nv::gem::Result_Success, "Failed to ControlNotification to false.\n");

    WAITER_LOG("Done.\n");
}

void GpuCoreDumpAppThreadFunction(void* data)
{
    APP_LOG("launched.\n");

    ThreadUserData* pUserData = reinterpret_cast<ThreadUserData*>(data);

    Application::Run(pUserData->pApp);

    APP_LOG("Done.\n");
}

TEST(TestGpuCoreDump, Simple)
{
    nn::Result result;

    nn::os::InitializeEvent(&s_StartCoreDumpingEvent, false, nn::os::EventClearMode_AutoClear);
    nn::os::InitializeEvent(&s_FinishCoreDumpingEvent, false, nn::os::EventClearMode_AutoClear);

    AppClass app(1920, 1080, "Application");

    ThreadUserData userData;
    userData.pApp = &app;

    nn::os::ThreadFunction functions[FuncType_Count] =
    {
        GpuCoreDumpReaderThreadFunction,
        GpuCoreDumpWaiterThreadFunction,
        GpuCoreDumpAppThreadFunction,
    };

    FuncType targets[] =
    {
        FuncType_Reader,
        FuncType_Waiter,
        FuncType_App,
    };

    int targetCount = sizeof(targets) / sizeof(targets[0]);

    for (int i = 0; i < targetCount; i++)
    {
        FuncType target = targets[i];

        result = nn::os::CreateThread(&s_Threads[target], functions[target], &userData, s_ThreadStacks[target], ThreadStackSize, nn::os::DefaultThreadPriority);
        NN_ASSERT(result.IsSuccess(), "Failed to create Thread%d\n", target);
    }

    for (int i = 0; i < targetCount; i++)
    {
        FuncType target = targets[i];

        nn::os::StartThread(&s_Threads[target]);
    }

    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

    // GPU ハングを発生させるようリクエスト
    app.RequestGpuHang();

    nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));

    // アプリを終了させるようリクエスト
    app.RequestExit();

    for (int i = 0; i < targetCount; i++)
    {
        FuncType target = targets[i];

        nn::os::WaitThread(&s_Threads[target]);
    }

    for (int i = 0; i < targetCount; i++)
    {
        FuncType target = targets[i];

        nn::os::DestroyThread(&s_Threads[target]);
    }
}
