﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>

#include <nn/nsd/nsd_Result.h>
#include <nn/nsd/detail/json/nsd_JsonParser.h>
#include <nn/nsd/detail/json/nsd_RapidJsonEventAccepterForHeader.h>

#include <nn/util/util_StringUtil.h>

using namespace nn::nsd::detail::json;

template <size_t StringBufferSize, size_t DocumentBufferSize>
class Stream : public InputStreamForRapidJson
{
private:
    char m_StringBuffer[StringBufferSize];
    char m_DocumentBuffer[DocumentBufferSize];

    const char* m_pSrcBuffer;
    const size_t m_SrcBufferSize;
    size_t m_PositionInSrcBuffer;

protected:
    virtual nn::Result RequestBufferImpl(size_t *pOutActualRead, void *buffer, size_t bufferSize) NN_NOEXCEPT NN_OVERRIDE
    {
        if( m_SrcBufferSize <= m_PositionInSrcBuffer )
        {
            *pOutActualRead = 0;
            NN_RESULT_SUCCESS;
        }

        size_t copySize = bufferSize;
        if( m_SrcBufferSize < m_PositionInSrcBuffer + copySize )
        {
            copySize  = m_SrcBufferSize - m_PositionInSrcBuffer;
        }
        std::memcpy(buffer, &m_pSrcBuffer[m_PositionInSrcBuffer], copySize);
        m_PositionInSrcBuffer += copySize;
        *pOutActualRead = copySize;
        NN_RESULT_SUCCESS;
    }

public:
    Stream(const char* pBuffer, size_t bufferSize):
        InputStreamForRapidJson(
            m_StringBuffer, StringBufferSize,
            m_DocumentBuffer, DocumentBufferSize),
        m_pSrcBuffer(pBuffer),
        m_SrcBufferSize(bufferSize),
        m_PositionInSrcBuffer(0)
    {}
};

TEST(NsdUnitTest_JwtHeaderParser, CorrectData)
{
    struct TestParseHeaderParameter
    {
        const char* pHeader;
        const char* pTypAnswer;
        const char* pAlgAnswer;
    };
    const TestParseHeaderParameter testcase[] =
    {
        {"{\"typ\":\"JWT\",\"alg\":\"RS256\"}", "JWT", "RS256"},
        {"{\"typ\":\"JWT\",\"alg\":\"HS256\"}", "JWT", "HS256"},
        {"{\"typ\":\"JWT\",\"alg\":\"none\"}", "JWT", "none"},

        {"{\"alg\":\"RS256\",\"typ\":\"JWT\"}", "JWT", "RS256"},
        {"{\"alg\":\"HS256\",\"typ\":\"JWT\"}", "JWT", "HS256"},
        {"{\"alg\":\"none\",\"typ\":\"JWT\"}", "JWT", "none"},

        // ParseHeader は中身の正当性はチェックしない
        {"{\"typ\":\"aaaaaaa\",\"alg\":\"RS256\"}", "aaaaaaa", "RS256"},
        {"{\"alg\":\"aaaaaaa\",\"typ\":\"JWT\"}", "JWT", "aaaaaaa"},
    };
    for(auto& t : testcase)
    {
        nn::nsd::detail::jwt::JwtHeaderData parsedData;

        Stream<512u, 512u> stream(t.pHeader, strlen(t.pHeader));
        RapidJsonEventAccepterForHeader handler(&parsedData);
        NNT_EXPECT_RESULT_SUCCESS( Parse(&stream, &handler) );

        EXPECT_STREQ(t.pTypAnswer, parsedData.typ.value);
        EXPECT_STREQ(t.pAlgAnswer, parsedData.alg.value);
    }

    for(auto& t : testcase)
    {
        nn::nsd::detail::jwt::JwtHeaderData parsedData;

        Stream<8u, 8u> stream(t.pHeader, strlen(t.pHeader));
        RapidJsonEventAccepterForHeader handler(&parsedData);
        NNT_EXPECT_RESULT_SUCCESS( Parse(&stream, &handler) );

        EXPECT_STREQ(t.pTypAnswer, parsedData.typ.value);
        EXPECT_STREQ(t.pAlgAnswer, parsedData.alg.value);
    }
}

TEST(NsdUnitTest_JwtHeaderParser, HeaderTypeBufferNotEnough)
{
    const char* testcase[] =
    {
        "{\"typ\":\"aaaaaaaa\",\"alg\":\"RS256\"}",
        "{\"typ\":\"aaaaaaaaa\",\"alg\":\"RS256\"}",
    };

    for(auto& t : testcase)
    {
        nn::nsd::detail::jwt::JwtHeaderData parsedData;

        Stream<512u, 512u> stream(t, strlen(t));
        RapidJsonEventAccepterForHeader handler(&parsedData);
        NNT_ASSERT_RESULT_FAILURE(nn::nsd::ResultStringTooLong, Parse(&stream, &handler) );
    }

    for(auto& t : testcase)
    {
        nn::nsd::detail::jwt::JwtHeaderData parsedData;

        Stream<8u, 8u> stream(t, strlen(t));
        RapidJsonEventAccepterForHeader handler(&parsedData);
        NNT_ASSERT_RESULT_FAILURE(nn::nsd::ResultStringTooLong, Parse(&stream, &handler) );
    }
}

TEST(NsdUnitTest_JwtHeaderParser, HeaderTypeNotFound)
{
    const char* testcase[] =
    {
        "{\"alg\":\"RS256\"}",
    };
    for(auto& t : testcase)
    {
        nn::nsd::detail::jwt::JwtHeaderData parsedData;

        Stream<512u, 512u> stream(t, strlen(t));
        RapidJsonEventAccepterForHeader handler(&parsedData);
        NNT_ASSERT_RESULT_FAILURE(nn::nsd::ResultRequiredItemNotFound, Parse(&stream, &handler) );
    }
    for(auto& t : testcase)
    {
        nn::nsd::detail::jwt::JwtHeaderData parsedData;

        Stream<8u, 8u> stream(t, strlen(t));
        RapidJsonEventAccepterForHeader handler(&parsedData);
        NNT_ASSERT_RESULT_FAILURE(nn::nsd::ResultRequiredItemNotFound, Parse(&stream, &handler) );
    }
}

TEST(NsdUnitTest_JwtHeaderParser, HeaderAlgorithmBufferNotEnough)
{
    const char* testcase[] =
    {
        "{\"typ\":\"JWT\",\"alg\":\"aaaaaaaa\"}"
        "{\"typ\":\"JWT\",\"alg\":\"aaaaaaaaa\"}"
    };
    for(auto& t : testcase)
    {
        nn::nsd::detail::jwt::JwtHeaderData parsedData;

        Stream<512u, 512u> stream(t, strlen(t));
        RapidJsonEventAccepterForHeader handler(&parsedData);
        NNT_ASSERT_RESULT_FAILURE(nn::nsd::ResultStringTooLong, Parse(&stream, &handler) );
    }
    for(auto& t : testcase)
    {
        nn::nsd::detail::jwt::JwtHeaderData parsedData;

        Stream<8u, 8u> stream(t, strlen(t));
        RapidJsonEventAccepterForHeader handler(&parsedData);
        NNT_ASSERT_RESULT_FAILURE(nn::nsd::ResultStringTooLong, Parse(&stream, &handler) );
    }
}

TEST(NsdUnitTest_JwtHeaderParser, HeaderAlgorithmNotFound)
{
    const char* testcase[] =
    {
        "{\"typ\":\"JWT\"}"
    };
    for(auto& t : testcase)
    {
        nn::nsd::detail::jwt::JwtHeaderData parsedData;

        Stream<512u, 512u> stream(t, strlen(t));
        RapidJsonEventAccepterForHeader handler(&parsedData);
        NNT_ASSERT_RESULT_FAILURE(nn::nsd::ResultRequiredItemNotFound, Parse(&stream, &handler) );
    }
    for(auto& t : testcase)
    {
        nn::nsd::detail::jwt::JwtHeaderData parsedData;

        Stream<8u, 8u> stream(t, strlen(t));
        RapidJsonEventAccepterForHeader handler(&parsedData);
        NNT_ASSERT_RESULT_FAILURE(nn::nsd::ResultRequiredItemNotFound, Parse(&stream, &handler) );
    }
}

TEST(NsdUnitTest_JwtHeaderParser, UnexpectedHeaderParam)
{
    const char* testcase[] =
    {
        "{\"Unexpected\":\"test\",\"typ\":\"JWT\",\"alg\":\"RS256\"}",
        "{\"typ\":\"JWT\",\"Unexpected\":\"test\",\"alg\":\"RS256\"}",
        "{\"typ\":\"JWT\",\"alg\":\"RS256\",\"Unexpected\":\"test\"}"
    };
    for(auto& t : testcase)
    {
        nn::nsd::detail::jwt::JwtHeaderData parsedData;

        Stream<512u, 512u> stream(t, strlen(t));
        RapidJsonEventAccepterForHeader handler(&parsedData);
        NNT_ASSERT_RESULT_FAILURE(nn::nsd::ResultUnexpectedJsonItem, Parse(&stream, &handler) );
    }
    for(auto& t : testcase)
    {
        nn::nsd::detail::jwt::JwtHeaderData parsedData;

        Stream<8u, 8u> stream(t, strlen(t));
        RapidJsonEventAccepterForHeader handler(&parsedData);
        NNT_ASSERT_RESULT_FAILURE(nn::nsd::ResultUnexpectedJsonItem, Parse(&stream, &handler) );
    }
}
