﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>

#include <nn/nsd/nsd_Result.h>
#include <nn/nsd/nsd_ResultPrivate.h>
#include <nn/nsd/detail/json/nsd_JsonParser.h>
#include <nn/nsd/detail/json/nsd_RapidJsonEventAccepterForHeader.h>

#include <nn/util/util_StringUtil.h>

using namespace nn::nsd::detail::json;

template <size_t StringBufferSize, size_t DocumentBufferSize>
class Stream : public InputStreamForRapidJson
{
private:
    char m_StringBuffer[StringBufferSize];
    char m_DocumentBuffer[DocumentBufferSize];

    const char* m_pSrcBuffer;
    const size_t m_SrcBufferSize;
    size_t m_PositionInSrcBuffer;

protected:
    virtual nn::Result RequestBufferImpl(size_t *pOutActualRead, void *buffer, size_t bufferSize) NN_NOEXCEPT NN_OVERRIDE
    {
        if( m_SrcBufferSize <= m_PositionInSrcBuffer )
        {
            *pOutActualRead = 0;
            NN_RESULT_SUCCESS;
        }

        size_t copySize = bufferSize;
        if( m_SrcBufferSize < m_PositionInSrcBuffer + copySize )
        {
            copySize  = m_SrcBufferSize - m_PositionInSrcBuffer;
        }
        std::memcpy(buffer, &m_pSrcBuffer[m_PositionInSrcBuffer], copySize);
        m_PositionInSrcBuffer += copySize;
        *pOutActualRead = copySize;
        NN_RESULT_SUCCESS;
    }

public:
    Stream(const char* pBuffer, size_t bufferSize):
        InputStreamForRapidJson(
            m_StringBuffer, StringBufferSize,
            m_DocumentBuffer, DocumentBufferSize),
        m_pSrcBuffer(pBuffer),
        m_SrcBufferSize(bufferSize),
        m_PositionInSrcBuffer(0)
    {}
};

class EmptyEventAccepter : public JsonEventAccepter
{
public:
};

TEST(NsdUnitTest_JsonParser, CorrectJson)
{
    const char* CorrectJson[] =
    {
        "null",
        "true",
        "false",
        "[]",
        "[1,2,3,4,5]",
        "\"key\":null",
        "{}",
        "{\"key\":null}",
        "{\"key\":1}",
        "{\"key\":1.5}",
        "{\"key\":true}",
        "{\"key\":false}",
        "{\"key\":\"hoge\"}",
        "{\"key\":[\"a\",\"b\",\"c\",\"d\"]}",
        "{\"key\":[1,2,3,4,5]}",
        "[]]",// nne::rapidjson::kParseStopWhenDoneFlag 指定のため成功扱い
        "\"LongKye_aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\":null", // StringBuffer 足りない場合は切れるだけでパースは成功する
        "\"key\":\"LongValue_aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\"", // StringBuffer 足りない場合は切れるだけでパースは成功する
        "\"LongKye_aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\":\"LongValue_aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\"", // StringBuffer 足りない場合は切れるだけでパースは成功する
        "{\"key1\":{\"key2\":null}}",
    };
    for(auto json : CorrectJson)
    {
        SCOPED_TRACE(json);
        Stream<512u, 512u> stream(json, strlen(json));
        EmptyEventAccepter handler;
        NNT_EXPECT_RESULT_SUCCESS( Parse(&stream, &handler) );
    }

    for(auto json : CorrectJson)
    {
        SCOPED_TRACE(json);
        Stream<4u, 4u> stream(json, strlen(json)); // バッファ小さくてもパースできることの確認
        EmptyEventAccepter handler;
        NNT_EXPECT_RESULT_SUCCESS( Parse(&stream, &handler) );
    }
}
TEST(NsdUnitTest_JsonParser, BadJson)
{
    const char* BadJson[] =
    {
        "}",
        "{",
        "[",
        "]",
        "[[]",
        "key:1",
        "'key':1",
    };
    for(auto json : BadJson)
    {
        SCOPED_TRACE(json);
        Stream<512u, 512u> stream(json, strlen(json));
        EmptyEventAccepter handler;
        auto result = Parse(&stream, &handler);
        EXPECT_TRUE(nn::nsd::ResultJsonCorrupted::Includes(result));
    }

    for(auto json : BadJson)
    {
        SCOPED_TRACE(json);
        Stream<4u, 4u> stream(json, strlen(json)); // バッファ小さくてもパースできることの確認
        EmptyEventAccepter handler;
        auto result = Parse(&stream, &handler);

        EXPECT_TRUE(nn::nsd::ResultJsonCorrupted::Includes(result));
    }
}

TEST(NsdUnitTest_Base64UrlSafeEncodedJsonInputStream, CorrectData)
{
    const char* CorrectData[] =
    {
        "bnVsbA==", // null
        "dHJ1ZQ==", // true
        "ZmFsc2U=", // false
        "W10=", // []
        "WzEsMiwzLDQsNV0=", // [1,2,3,4,5]
        "ImtleSI6bnVsbA==", // "key":null
        "e30=", // {}
        "eyJrZXkiOm51bGx9", // {\"key\":null}
        "eyJrZXkiOjF9", // {\"key\":1}
        "eyJrZXkiOjEuNX0=", // {\"key\":1.5}
        "eyJrZXkiOnRydWV9", // {\"key\":true}
        "eyJrZXkiOmZhbHNlfQ==", // {\"key\":false}
        "eyJrZXkiOiJob2dlIn0=", // "{\"key\":\"hoge\"}",
        "eyJrZXkiOlsiYSIsImIiLCJjIiwiZCJdfQ==", // {\"key\":[\"a\",\"b\",\"c\",\"d\"]}
        "eyJrZXkiOlsxLDIsMyw0LDVdfQ==", // {\"key\":[1,2,3,4,5]}
        "W11d", // []]  nne::rapidjson::kParseStopWhenDoneFlag 指定のため成功扱い
        "IkxvbmdLeWVfYWFhYWFhYWFhYWFhYWFhYWFhYWFhYWFhYWFhYWFhYWFhYWFhYWFhIjpudWxs", //"\"LongKye_aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\":null", // 切れるだけでパースは成功する
        "ImtleSI6IkxvbmdWYWx1ZV9hYWFhYWFhYWFhYWFhYWFhYWFhYWFhYWFhYWFhYWFhYWFhIg==", //"\"key\":\"LongValue_aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\"", // 切れるだけでパースは成功する
        "IkxvbmdLeWVfYWFhYWFhYWFhYWFhYWFhYWFhYWFhYWFhYWFhYWFhYWFhYWFhYWFhIjoiTG9uZ1ZhbHVlX2FhYWFhYWFhYWFhYWFhYWFhYWFhYWFhYWFhYWFhYWFhYWEi", // "\"LongKye_aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\":\"LongValue_aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\"", // 切れるだけでパースは成功する
        "eyJrZXkxIjp7ImtleTIiOm51bGx9fQ==", // {\"key1\":{\"key2\":null}}
    };
    for(auto json : CorrectData)
    {
        SCOPED_TRACE(json);
        Base64UrlSafeEncodedJsonInputStream<16u, 16u> stream(json, strlen(json));
        EmptyEventAccepter handler;
        NNT_EXPECT_RESULT_SUCCESS( Parse(&stream, &handler) );
    }
}
TEST(NsdUnitTest_Base64UrlSafeEncodedJsonInputStream, BadData)
{
    const char* BadData[] =
    {
        "bnVsbA==", // null
        "dHJ1ZQ==", // true
        "ZmFsc2U=", // false
        "W10=", // []
        "WzEsMiwzLDQsNV0=", // [1,2,3,4,5]
        "ImtleSI6bnVsbA==", // "key":null
        "e30=", // {}
        "eyJrZXkiOm51bGx9", // {\"key\":null}
        "eyJrZXkiOjF9", // {\"key\":1}
        "eyJrZXkiOjEuNX0=", // {\"key\":1.5}
        "eyJrZXkiOnRydWV9", // {\"key\":true}
        "eyJrZXkiOmZhbHNlfQ==", // {\"key\":false}
        "eyJrZXkiOiJob2dlIn0=", // "{\"key\":\"hoge\"}",
        "eyJrZXkiOlsiYSIsImIiLCJjIiwiZCJdfQ==", // {\"key\":[\"a\",\"b\",\"c\",\"d\"]}
        "eyJrZXkiOlsxLDIsMyw0LDVdfQ==", // {\"key\":[1,2,3,4,5]}
        "W11d", // []]  nne::rapidjson::kParseStopWhenDoneFlag 指定のため成功扱い
        "IkxvbmdLeWVfYWFhYWFhYWFhYWFhYWFhYWFhYWFhYWFhYWFhYWFhYWFhYWFhYWFhIjpudWxs", //"\"LongKye_aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\":null", // 切れるだけでパースは成功する
        "ImtleSI6IkxvbmdWYWx1ZV9hYWFhYWFhYWFhYWFhYWFhYWFhYWFhYWFhYWFhYWFhYWFhIg==", //"\"key\":\"LongValue_aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\"", // 切れるだけでパースは成功する
        "IkxvbmdLeWVfYWFhYWFhYWFhYWFhYWFhYWFhYWFhYWFhYWFhYWFhYWFhYWFhYWFhIjoiTG9uZ1ZhbHVlX2FhYWFhYWFhYWFhYWFhYWFhYWFhYWFhYWFhYWFhYWFhYWEi", // "\"LongKye_aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\":\"LongValue_aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\"", // 切れるだけでパースは成功する
        "eyJrZXkxIjp7ImtleTIiOm51bGx9fQ==", // {\"key1\":{\"key2\":null}}
    };
    for(auto json : BadData)
    {
        SCOPED_TRACE(json);

        char buffer[512] = {};
        nn::util::Strlcpy(buffer, json, sizeof(buffer));

        // CorrectData を適当に壊す
        buffer[0] = '/'; // base64 urlsfe 禁止文字

        Base64UrlSafeEncodedJsonInputStream<16u, 16u> stream(buffer, strlen(buffer));
        EmptyEventAccepter handler;
        NNT_EXPECT_RESULT_FAILURE(nn::nsd::ResultBase64DecodeBadData, Parse(&stream, &handler));
    }
}


