﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <string>

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nsutil/nsutil_InstallUtils.h>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <curl/curl.h>
#include <nn/os.h>
#include <nn/fs.h>
#include <nn/fs/fs_Bis.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_FormatString.h>
#include <nn/socket.h>

#include <nn/nifm.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/nim/nim_NetworkInstallManagerApi.h>
#include <nn/ovln/ovln_ForDevelop.h>
#include <nn/ovln/ovln_SenderForOverlay.h>
#include <nn/ns/ns_Result.h>
#include <nn/ns/ns_ApplicationEntitySystemApi.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/ns/ns_ApplicationManagerSystemApi.h>
#include <nn/ns/ns_ApplicationVersionSystemApi.h>
#include <nn/ns/ns_ApiForDfc.h>
#include <nn/ns/ns_InitializationApi.h>
#include <nn/ns/srv/ns_VersionManagementDatabase.h>
#include <nn/ncm/ncm_ApplicationInfo.h>

// noclean でシステムセーブデータが消えない間のワークアラウンド
#include <nn/fs/fs_SaveDataManagement.h>

#undef DeleteFile

namespace {
    const int64_t MaxJsonBufferSize = 5 * 1024 * 1024;
    char g_JsonBuffer[MaxJsonBufferSize];

    class VersionListTest : public testing::Test, public nnt::nsutil::ApplicationInstaller
    {
    public:
        static const nn::Bit64 VersionListSaveDataId = 0x8000000000000045ull;
        static const nn::Bit64 TestSystemUpdateId = 0x0005000c11000000ull;
    protected:
        virtual void SetUp()
        {
            nn::ns::Initialize();
            m_VersionListPath = nnt::GetHostArgv()[1];
            m_NspPath = nnt::GetHostArgv()[2];
            ClearSaveData();
        }

        virtual void TearDown()
        {
            nn::ns::Finalize();
            DeleteSystemUpdate();
        }

        static void DeleteSystemUpdate()
        {
            nn::ncm::ContentMetaDatabase db;
            nn::ncm::ContentStorage storage;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ncm::OpenContentMetaDatabase(&db, nn::ncm::StorageId::BuildInSystem));
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ncm::OpenContentStorage(&storage, nn::ncm::StorageId::BuildInSystem));
            nn::ncm::ContentManagerAccessor accessor(&db, &storage);

            accessor.DeleteAll(TestSystemUpdateId);
        }

        static void DeleteApplication(nn::Bit64 id)
        {
            const auto flags = nn::ns::ApplicationEntityFlag_All | nn::ns::ApplicationEntityFlag_SkipGameCardCheck::Mask | nn::ns::ApplicationEntityFlag_SkipRunningCheck::Mask;
            nn::ncm::ApplicationId appId = { id };
            nn::ns::DeleteApplicationCompletelyForDebug(appId, flags);
        }

        static void ClearSaveData() {
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountSystemSaveData("save", VersionListSaveDataId));
            NN_UTIL_SCOPE_EXIT{ nn::fs::Unmount("save"); };
            nn::fs::DeleteFile("save:/req_version_db");
            nn::fs::DeleteFile("save:/req_version_header");
            nn::fs::DeleteFile("save:/version_list_db");
            nn::fs::DeleteFile("save:/version_list_header");
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::CommitSaveData("save"));
        }

        static void SetUpTestCase()
        {
#if defined( NN_BUILD_CONFIG_OS_WIN )
            nn::ns::InitializeDependenciesForDfc();
#elif defined( NN_BUILD_CONFIG_OS_HORIZON)
            nn::ncm::Initialize();
#endif
            NN_ASSERT(nnt::GetHostArgc() > 2);
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountHostRoot());

// noclean でシステムセーブデータが消えない間のワークアラウンド
#if defined( NN_BUILD_CONFIG_OS_WIN)
            nn::fs::DeleteSaveData(0x8000000000000041ull);
#endif
        }

        static void TearDownTestCase()
        {
#if defined( NN_BUILD_CONFIG_OS_WIN )
            nn::ns::FinalizeDependenciesForDfc();
#elif defined( NN_BUILD_CONFIG_OS_HORIZON )
            nn::ncm::Finalize();
#endif

            nn::fs::UnmountHostRoot();
        }

        std::string m_VersionListPath;
        std::string m_NspPath;
        nnt::nsutil::ApplicationInstaller m_ApplicationInstaller;

        void ReadVersionList(void* out, int64_t maxReadSize, const char* fileName) {
            std::string path = m_VersionListPath + "/" + fileName;
            NN_LOG("VersionListPath: %s\n", path.c_str());

            nn::fs::FileHandle file;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&file, path.c_str(), nn::fs::OpenMode::OpenMode_Read));
            NN_UTIL_SCOPE_EXIT{ nn::fs::CloseFile(file); };
            int64_t fileSize;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::GetFileSize(&fileSize, file));
            NN_ASSERT(fileSize <= maxReadSize);
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadFile(file, 0, out, static_cast<size_t>(fileSize)));

        }


        void TestImportedVersionList(nn::Bit64 startId, uint32_t startVersion, uint32_t startRequiredVersion, int32_t entryCount)
        {
            static nn::ns::VersionListEntry s_VersionListEntryBuffer[nn::ns::srv::VersionListDatabaseConfig::MaxEntryCount];
            nn::ns::srv::VersionListDatabase db("vmdb", s_VersionListEntryBuffer, sizeof(s_VersionListEntryBuffer));

            NNT_ASSERT_RESULT_SUCCESS(db.Load());
            auto actualEntryCount = db.GetEntryCount();
            EXPECT_EQ(entryCount, actualEntryCount);

            for (int i = 0; i < entryCount; ++i)
            {
                nn::Bit64 appId = startId + i;
                auto pEntry = db.Find(appId);
                NN_ASSERT_NOT_NULL(pEntry);
                EXPECT_EQ(appId, pEntry->id);
                EXPECT_EQ((startVersion + i) << 16, pEntry->recommendedVersion);
                EXPECT_EQ((startRequiredVersion + i) << 16, pEntry->notifiedVersion);
            }
        }


        void TestRequiredVersionDatabase(nn::Bit64 startId, uint32_t startRequiredVersion, int32_t entryCount)
        {
            static nn::ns::RequiredVersionEntry s_RequiredVersionDatabaseBuffer[nn::ns::srv::RequiredVersionDatabaseConfig::MaxEntryCount];
            nn::ns::srv::RequiredVersionDatabase db("vmdb", s_RequiredVersionDatabaseBuffer, sizeof(s_RequiredVersionDatabaseBuffer));

            NNT_ASSERT_RESULT_SUCCESS(db.Load());
            auto actualEntryCount = db.GetEntryCount();
            EXPECT_EQ(entryCount, actualEntryCount);

            for (int i = 0; i < entryCount; ++i)
            {
                nn::Bit64 appId = startId + i ;
                auto pEntry = db.Find(appId);
                NN_ASSERT_NOT_NULL(pEntry);
                EXPECT_EQ(appId, pEntry->id);
                EXPECT_EQ((startRequiredVersion + i) << 16, pEntry->requiredVersion);
            }
        }

        void InstallNsp(nn::Bit64 id, nn::ncm::StorageId storage = nn::ncm::StorageId::BuildInUser)
        {
            DeleteApplication(id);

            char filePath[32];
            nn::util::SNPrintf(filePath, sizeof(filePath), "/0x%016llx.nsp", id);
            std::vector<nn::ncm::ApplicationContentMetaKey> installedKeys;
            std::string nspPath = m_NspPath + filePath;
            NN_LOG("NspPath: %s\n", nspPath.c_str());
            NNT_ASSERT_RESULT_SUCCESS(m_ApplicationInstaller.InstallOne(&installedKeys, (m_NspPath + filePath).c_str(), storage));
        }

        nn::Result TestCheckLaunchApplicationVersion(const char* versionList, nn::Bit64 applicationId, uint32_t requiredVersion)
        {
            nn::ncm::ApplicationId id = { applicationId };

            ReadVersionList(g_JsonBuffer, MaxJsonBufferSize, versionList);
            NNT_EXPECT_RESULT_SUCCESS(nn::ns::UpdateVersionList(g_JsonBuffer, MaxJsonBufferSize));
            nn::ns::PushLaunchVersion(id, requiredVersion << 16);
            NN_LOG("\n");
            return nn::ns::CheckApplicationLaunchVersion(id);
        }
    };

}

TEST_F(VersionListTest, Full)
{
    nn::ns::InvalidateAllApplicationControlCache();

    ReadVersionList(g_JsonBuffer, MaxJsonBufferSize, "DummyVersionList_Full.json");
    NNT_EXPECT_RESULT_SUCCESS(nn::ns::UpdateVersionList(g_JsonBuffer, MaxJsonBufferSize));
    TestImportedVersionList(0x0005000c10000000ull, 0, 1, 16 * 1024);
}

TEST_F(VersionListTest, Over)
{
    nn::ns::InvalidateAllApplicationControlCache();

    ReadVersionList(g_JsonBuffer, MaxJsonBufferSize, "DummyVersionList_Over.json");
    NNT_EXPECT_RESULT_FAILURE(nn::ns::ResultOutOfMaxVersionListCount, nn::ns::UpdateVersionList(g_JsonBuffer, MaxJsonBufferSize));
}

TEST_F(VersionListTest, NoRequiredVersion)
{
    nn::ns::InvalidateAllApplicationControlCache();

    ReadVersionList(g_JsonBuffer, MaxJsonBufferSize, "DummyVersionList_NoRequiredVersion.json");
    NNT_EXPECT_RESULT_SUCCESS(nn::ns::UpdateVersionList(g_JsonBuffer, MaxJsonBufferSize));
    TestImportedVersionList(0x0005000c10000000ull, 1, 1, 1);
}

TEST_F(VersionListTest, LaunchVersion)
{
    nn::ns::InvalidateAllApplicationControlCache();

    const int EntryCount = 16;
    for (int i = 0; i < EntryCount; ++i)
    {
        nn::ncm::ApplicationId id = { static_cast<nn::Bit64>(i) };
        nn::ns::PushLaunchVersion(id, (i + 1) << 16);
    }

    TestRequiredVersionDatabase(0, 1, EntryCount);
}

TEST_F(VersionListTest, CheckApplicationLaunchVersion)
{
    nn::Bit64 appId = 0x0005000c10000007ull;
    InstallNsp(appId);

    NNT_EXPECT_RESULT_FAILURE(
        nn::ns::ResultApplicationUpdateRequiredByRequiredVersion, TestCheckLaunchApplicationVersion("DummyVersionList_UpdateRecommended.json", appId, 2));

    NNT_EXPECT_RESULT_FAILURE(
        nn::ns::ResultApplicationUpdateRequiredByRequiredVersion, TestCheckLaunchApplicationVersion("DummyVersionList_NoNeedUpdate.json", appId, 3));

    NNT_EXPECT_RESULT_FAILURE(
        nn::ns::ResultApplicationUpdateRecommended, TestCheckLaunchApplicationVersion("DummyVersionList_UpdateRecommended.json", appId, 0));

    NNT_EXPECT_RESULT_SUCCESS(TestCheckLaunchApplicationVersion("DummyVersionList_NoNeedUpdate.json", appId, 0));
    NNT_EXPECT_RESULT_SUCCESS(TestCheckLaunchApplicationVersion("DummyVersionList_NoNeedUpdate.json", appId, 1));
}

TEST_F(VersionListTest, CheckApplicationLaunchVersionRequiredSystemUpdate)
{
    nn::Bit64 appId = 0x0005000c10000008ull;
    InstallNsp(appId);

    NNT_EXPECT_RESULT_FAILURE(
        nn::ns::ResultSystemUpdateRequired, TestCheckLaunchApplicationVersion("DummyVersionList_UpdateRecommended.json", appId, 1));

    NNT_EXPECT_RESULT_FAILURE(
        nn::ns::ResultSystemUpdateRequired, TestCheckLaunchApplicationVersion("DummyVersionList_NoNeedUpdate.json", appId, 3));

    NNT_EXPECT_RESULT_FAILURE(
        nn::ns::ResultSystemUpdateRequired, TestCheckLaunchApplicationVersion("DummyVersionList_UpdateRecommended.json", appId, 0));

    NNT_EXPECT_RESULT_FAILURE(
        nn::ns::ResultSystemUpdateRequired, TestCheckLaunchApplicationVersion("DummyVersionList_NoNeedUpdate.json", appId, 0));
}

/*
TEST_F(VersionListTest, CheckApplicationLaunchVersionRequiredSystemUpdate_SystemUpdated)
{
    nn::Bit64 appId = 0x0005000c10000008ull;
    InstallNsp(appId);
    InstallNsp(VersionListTest::TestSystemUpdateId, nn::ncm::StorageId::BuildInSystem);

    NNT_EXPECT_RESULT_FAILURE(
        nn::ns::ResultApplicationUpdateRequiredByRequiredVersion, TestCheckLaunchApplicationVersion("DummyVersionList_UpdateRecommended.json", appId, 2));

    NNT_EXPECT_RESULT_FAILURE(
        nn::ns::ResultApplicationUpdateRequiredByRequiredVersion, TestCheckLaunchApplicationVersion("DummyVersionList_NoNeedUpdate.json", appId, 3));

    NNT_EXPECT_RESULT_FAILURE(
        nn::ns::ResultApplicationUpdateRecommended, TestCheckLaunchApplicationVersion("DummyVersionList_UpdateRecommended.json", appId, 0));

    NNT_EXPECT_RESULT_SUCCESS(TestCheckLaunchApplicationVersion("DummyVersionList_NoNeedUpdate.json", appId, 0));
    NNT_EXPECT_RESULT_SUCCESS(TestCheckLaunchApplicationVersion("DummyVersionList_NoNeedUpdate.json", appId, 1));
}
*/
