﻿<#
    .SYNOPSIS
        Private option utilities module

    .DESCRIPTION
        This file defines functions for private operations

    e.g. )
    {
        $global:defExistAdvisoryFile = $true

        # ファイル検知監視準備
        # ScriptBlock はメインスレッドにInvokeされまするので、メインで RunOnTarget してたら取り逃す.
        # かといって RunOnTarget を別プロセス( Start-Job )にしたらログどうなる？
        # 別プロセスではログとれない
        $defWatchHandle = Start-FileWatcher -ScriptBlock {
            Param ( $sender, $ev )
            if ( $ev.ChangeType -eq 'Deleted' -or $ev.ChangeType -eq 'Renamed' )
            {
                $global:defExistAdvisoryFile = $false
            }
        } -WatchedPath ${defAdvisoryFileDirectoryName} -WatchedFilter ${defAdvisoryFileName}
    }
#>

# 基本パス関連 ( 全部文字列型らしい [string] )
$MyScriptPath          = $MyInvocation.MyCommand.Path
$MyScriptDirectoryPath = [System.IO.Path]::GetDirectoryName($MyScriptPath)

# -----------------------------------------------------------------------
# Base resource container for the file watch event signature.
# -----------------------------------------------------------------------
$StaticFileWatchEventSignature = "SigloTestNs_FileWatcher"

# -----------------------------------------------------------------------
# Create the string of event signature.
# -----------------------------------------------------------------------
function New-FileWatchEventSignature( [string]$defEventType, [string]$defWatchedPath, [string]$defWatchedFilter )
{
    return "${StaticFileWatchEventSignature}_${defEventType}_${defWatchedPath}\${defWatchedFilter}"
}

# -----------------------------------------------------------------------
# Remove suffix character
# -----------------------------------------------------------------------
function Remove-SuffixCharacter( [string]$defValue )
{
    if ( -not [string]::IsNullOrEmpty( ${defValue} ) )
    {
        while ( $defValue.EndsWith( "\" ) )
        {
            $defPosition = $defValue.Length - 1
            $defValue = $defValue.Remove( ${defPosition}, 1 )
        }
    }
    return ${defValue}
}

# -----------------------------------------------------------------------
# Remove the event jobs.
# -----------------------------------------------------------------------
function Remove-JobEvents()
{
    Param (
        [string] $JobName = ''
    )
    if ( -not [string]::IsNullOrEmpty( ${JobName} ) )
    {
        Unregister-Event ${JobName}
        Remove-Job ${JobName}
    }
}

# -----------------------------------------------------------------------
# Remove the file watch events.
# -----------------------------------------------------------------------
function Remove-FileWatcher()
{
    Param (
        [string] $WatchedPath,
        [string] $WatchedFilter = '*.*'
    )
    if ( [string]::IsNullOrEmpty( ${WatchedPath} ) -or [string]::IsNullOrEmpty( ${WatchedFilter} ) )
    {
        Write-Host "Could not remove watcher as `"${WatchedPath}\${WatchedFilter}`"."
    }
    else
    {
        $WatchedFilter.Trim() | Out-Null
        $WatchedPath.Trim() | Out-Null
        $WatchedPath = Remove-SuffixCharacter( ${WatchedPath} )
        Remove-JobEvents( $(New-FileWatchEventSignature "Created" ${WatchedPath} ${WatchedFilter}) )
        Remove-JobEvents( $(New-FileWatchEventSignature "Changed" ${WatchedPath} ${WatchedFilter}) )
        Remove-JobEvents( $(New-FileWatchEventSignature "Deleted" ${WatchedPath} ${WatchedFilter}) )
        Remove-JobEvents( $(New-FileWatchEventSignature "Renamed" ${WatchedPath} ${WatchedFilter}) )
    }
}
Export-ModuleMember -Function Remove-FileWatcher

# -----------------------------------------------------------------------
# Start the file watch events.
# ScriptBlock はメインスレッドにInvokeされます.
# メインスレッドが高負荷処理中( RunOnTarget など )の場合、イベント受信が失敗します.
# e.g.)
<#
    # 開始
    $global:defExistFile = $true
    $defWatchHandle = Start-FileWatcher -ScriptBlock {
        Param ( $sender, $ev )
        if ( $ev.ChangeType -eq 'Deleted' -or $ev.ChangeType -eq 'Renamed' )
        {
            $global:defExistFile = $false
        }
    } -WatchedPath ${DirectoryPath} -WatchedFilter ${FileName} -NotifyFilter [System.IO.NotifyFilters]::FileName

    # 解放
    if ( $null -ne $defWatchHandle )
    {
        &$defWatchHandle
    }
#>
# -----------------------------------------------------------------------
function Start-FileWatcher()
{
    Param (
        $ScriptBlock,
        [string] $WatchedPath,
        [string] $WatchedFilter = '*.*',
        [switch] $IncludeSubdirectories,
        [int] $InternalBufferSize = 8192,
        [System.IO.NotifyFilters] $NotifyFilter = [System.IO.NotifyFilters]::LastWrite -bor [System.IO.NotifyFilters]::FileName -bor [System.IO.NotifyFilters]::DirectoryName,
        $MessageData = $null
    )
    if ( [string]::IsNullOrEmpty( ${WatchedPath} ) -or [string]::IsNullOrEmpty( ${WatchedFilter} ) )
    {
        Write-Host "Could not start watcher as `"${WatchedPath}\${WatchedFilter}`"."
    }
    else
    {
        $WatchedFilter.Trim() | Out-Null
        $WatchedPath.Trim() | Out-Null
        $WatchedPath = Remove-SuffixCharacter( ${WatchedPath} )

        $defWatcher = New-Object System.IO.FileSystemWatcher
        $defWatcher.Path = ${WatchedPath}
        $defWatcher.Filter = ${WatchedFilter}
        $defWatcher.IncludeSubdirectories = ${IncludeSubdirectories}
        $defWatcher.InternalBufferSize = ${InternalBufferSize}
        $defWatcher.NotifyFilter = ${NotifyFilter}
        $defWatcher.EnableRaisingEvents = $true

        Register-ObjectEvent -InputObject ${defWatcher} -EventName Created -SourceIdentifier ( $(New-FileWatchEventSignature "Created" ${WatchedPath} ${WatchedFilter}) ) -Action ${ScriptBlock} -MessageData ${MessageData} | Out-Null
        Register-ObjectEvent -InputObject ${defWatcher} -EventName Changed -SourceIdentifier ( $(New-FileWatchEventSignature "Changed" ${WatchedPath} ${WatchedFilter}) ) -Action ${ScriptBlock} -MessageData ${MessageData} | Out-Null
        Register-ObjectEvent -InputObject ${defWatcher} -EventName Deleted -SourceIdentifier ( $(New-FileWatchEventSignature "Deleted" ${WatchedPath} ${WatchedFilter}) ) -Action ${ScriptBlock} -MessageData ${MessageData} | Out-Null
        Register-ObjectEvent -InputObject ${defWatcher} -EventName Renamed -SourceIdentifier ( $(New-FileWatchEventSignature "Renamed" ${WatchedPath} ${WatchedFilter}) ) -Action ${ScriptBlock} -MessageData ${MessageData} | Out-Null

        return { Remove-FileWatcher -WatchedPath ${WatchedPath} -WatchedFilter ${WatchedFilter} }.GetNewClosure()
    }
}
Export-ModuleMember -Function Start-FileWatcher
