﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <string>
#include <vector>

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nsutil/nsutil_InstallUtils.h>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/fs.h>
#include <nn/fs/fs_Bis.h>

#include <nn/ns/ns_Result.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/ns/ns_InitializationApi.h>
#include <nn/ns/ns_SystemUpdateApi.h>
#include <nn/ovln/ovln_ForDevelop.h>
#include <nn/ovln/ovln_SenderForOverlay.h>
#include "libraries/testNs_MountHost.h"
#include "testNs_NcmStorageVerify.h"

//#define WILL_CLEAN_SPECIFIED_CONTENTS_AT_BEFORE_TEST

#define LOG_OUT( ... )  NN_LOG( "[NsSystemUpdate] " __VA_ARGS__ )

namespace {

#if defined( NN_BUILD_CONFIG_OS_WIN )

    //!--------------------------------------------------------------------------------------
    //! @brief The implementation in C11 rules for aligned memory allocations.@n
    //!        Memory allocator with alignment for start address of rules on the windows operating system.
    //!--------------------------------------------------------------------------------------
    NN_FORCEINLINE static void* aligned_alloc( size_t beginAlignment, size_t size ) NN_NOEXCEPT
    {
        return ::_aligned_malloc( size, beginAlignment );
    }

    //!--------------------------------------------------------------------------------------
    //! @brief The implementation for aligned memory release rules on the windows operating system.
    //!--------------------------------------------------------------------------------------
    NN_FORCEINLINE static void aligned_free( void* address ) NN_NOEXCEPT
    {
        ::_aligned_free( address );
    }

#else   // NX, Cafe, and C++11 supported platforms...

    //!--------------------------------------------------------------------------------------
    //! @brief The implementation for aligned memory release that has been imitated on the rules of the windows operating system.
    //!--------------------------------------------------------------------------------------
    NN_FORCEINLINE static void aligned_free( void* address ) NN_NOEXCEPT
    {
        ::free( address );
    }

#endif

    //!--------------------------------------------------------------------------------------
    //! 配列要素数算出.
    //!--------------------------------------------------------------------------------------
    template< typename TYPE, size_t SIZE >
    inline static unsigned CountOf( const TYPE( &)[ SIZE ] ) {
        return SIZE;
    }

    //!--------------------------------------------------------------------------------------
    //! Case Setup
    //!--------------------------------------------------------------------------------------
    class CupCase : public testing::Test, public ::testing::WithParamInterface<bool>, public nnt::nsutil::ApplicationInstaller
    {
    public:
        static const size_t WorkingBufferSize = 128 * 1024 * 1024;

        static void SetUpTestCase()
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS( testns::MountHostFileSystem( "Contents" ) );
        }

        static void TearDownTestCase()
        {
            testns::UnmountHostFileSystem();
        }

    protected:
        virtual void SetUp()
        {
            nn::ncm::Initialize();

#if defined( NN_BUILD_CONFIG_OS_WIN )
            nn::ovln::PrepareSenderAndReceiverForDevelop();
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ovln::InitializeSenderLibraryForOverlay());
#endif

            nn::ns::Initialize();

            auto& verifierInstance = nn::util::Get( m_NcmStorageVerifier );
            new ( &verifierInstance ) NcmStorageVerifier( nn::ncm::StorageId::BuildInSystem );
            verifierInstance.Initialize();

            DebugForDeleteContents( verifierInstance );
        }

        virtual void TearDown()
        {
            nn::util::Get( m_NcmStorageVerifier ).Finalize();
            nn::ns::Finalize();

#if defined( NN_BUILD_CONFIG_OS_WIN )
            nn::ovln::FinalizeSenderLibraryForOverlay();
            nn::ovln::ReleasePreparedSenderAndReceiverForDevelop();
#endif

            nn::ncm::Finalize();
        }

#if defined( WILL_CLEAN_SPECIFIED_CONTENTS_AT_BEFORE_TEST )
        // ローカルテスト用.
        // 強制的にコンテンツ消すよー
        void DebugForDeleteContents( const NcmStorageVerifier& verifier )
        {
            auto& storageId = verifier.GetStorageId();
            auto& collection = verifier.GetContentMetaKeyCollection();
            for ( auto it = collection.begin(); it != collection.end(); ++it )
            {
                auto& metaKey = *it;
                DeleteAllContents( metaKey.id, storageId );
            }
        }
#else
        void DebugForDeleteContents( const NcmStorageVerifier& verifier )
        {
            // none operated.
            NN_UNUSED( verifier );
        }
#endif // defined( WILL_CLEAN_SPECIFIED_CONTENTS_AT_BEFORE_TEST )

    protected:
        nn::util::TypedStorage< NcmStorageVerifier, sizeof( NcmStorageVerifier ), NN_ALIGNOF( NcmStorageVerifier ) > m_NcmStorageVerifier;
    };



    // AsyncUtils
    template< typename THandle, typename TControl = nn::ns::SystemUpdateControl >
    class RequestSynchronizer
    {
    public:
        typedef TControl    ControlType;
        typedef THandle     HandleType;

        HandleType& GetHandle() NN_NOEXCEPT
        {
            return m_Handle;
        }

        const nn::TimeSpan GetElapsedTime() const NN_NOEXCEPT
        {
            return m_ElapsedTime;
        }

        nn::Result WaitResponse() NN_NOEXCEPT
        {
            const nn::TimeSpan before = nn::os::GetSystemTick().ToTimeSpan();
            auto result = OnWaitResponse( m_Control, m_Handle );
            m_ElapsedTime = nn::os::GetSystemTick().ToTimeSpan() - before;
            LOG_OUT( "Response wait : %lld ms.\n", m_ElapsedTime.GetMilliSeconds() );
            return result;
        }

    protected:
        explicit RequestSynchronizer( ControlType& control ) NN_NOEXCEPT
            : m_Control( control )
        {
        }

        virtual nn::Result OnWaitResponse( ControlType& control, HandleType& handle ) NN_NOEXCEPT = 0;

    private:
        ControlType&    m_Control;
        HandleType      m_Handle;
        nn::TimeSpan    m_ElapsedTime;
    };

    // RequestPrepareCardUpdate
    class NewlyUpdatePreparer : public RequestSynchronizer< nn::ns::AsyncResult >
    {
    public:
        typedef RequestSynchronizer< nn::ns::AsyncResult > BaseType;

        explicit NewlyUpdatePreparer( ControlType& control ) NN_NOEXCEPT
            : BaseType( control )
        {
        }

    protected:
        virtual nn::Result OnWaitResponse( ControlType& control, HandleType& handle ) NN_NOEXCEPT NN_OVERRIDE
        {
            while ( !handle.TryWait() )
            {
                auto progress = control.GetPrepareCardUpdateProgress();
                LOG_OUT( "loaded %lld, total %lld\n", progress.loaded, progress.total );
                nn::os::SleepThread( nn::TimeSpan::FromSeconds( 1 ) );
            }
            auto progress = control.GetPrepareCardUpdateProgress();
            LOG_OUT( "loaded %lld, total %lld\n", progress.loaded, progress.total );
            return handle.Get();
        }
    };

    // ApplyCardUpdate
    class NewlyUpdateInstaller : public RequestSynchronizer< int >
    {
    public:
        typedef RequestSynchronizer< int > BaseType;

        explicit NewlyUpdateInstaller( ControlType& control ) NN_NOEXCEPT
            : BaseType( control )
        {
        }

    protected:
        virtual nn::Result OnWaitResponse( ControlType& control, HandleType& handle ) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED( handle );

            control.ApplyCardUpdate();
            NN_RESULT_SUCCESS;
        }
    };
}

TEST_P( CupCase, Simple )
{
    // ストレージ上のインストールコンテンツ及び本体更新対象コンテンツを含むコレクション生成
    auto& verifier = nn::util::Get( m_NcmStorageVerifier );
    verifier.PrepareUnexpectedContentOnStorage();

    // 本体更新制御の占有開始.
    nn::ns::SystemUpdateControl control;
    NNT_ASSERT_RESULT_SUCCESS( control.Occupy() );

    // カード本体更新の事前準備用バッファ確保.
    const size_t bufferSize = nn::util::align_up( WorkingBufferSize, nn::os::MemoryPageSize );
    void* const pWorkingBuffer = aligned_alloc( nn::os::MemoryPageSize, bufferSize );
    ASSERT_NE( nullptr, pWorkingBuffer );
    LOG_OUT( "Did allocate the working buffer for CUP which have size of [ %lld ] byte.\n", bufferSize );

    // カード本体更新の事前準備.
    if (GetParam())
    {
        NNT_ASSERT_RESULT_SUCCESS( control.SetupCardUpdateViaSystemUpdater( pWorkingBuffer, bufferSize ) );
    }
    else
    {
        NNT_ASSERT_RESULT_SUCCESS( control.SetupCardUpdate( pWorkingBuffer, bufferSize ) );
    }

    {
        bool prepared = false;
        {
            // ワーク領域書き込み. RequestPrepareCardUpdate
            LOG_OUT( "Start latest update preparing... On API [nn::ns::SystemUpdateControl::RequestPrepareCardUpdate]\n" );
            NewlyUpdatePreparer preparer( control );
            NNT_EXPECT_RESULT_SUCCESS( control.RequestPrepareCardUpdate( &preparer.GetHandle() ) );
            NNT_EXPECT_RESULT_SUCCESS( preparer.WaitResponse() );
            LOG_OUT( "Complete latest update preparing.\n" );
            prepared = control.HasPreparedCardUpdate();
            EXPECT_EQ( true, prepared );
        }
        if ( prepared )
        {
            // インストール. ApplyCardUpdate
            LOG_OUT( "Start install prepared update... On API [nn::ns::SystemUpdateControl::ApplyCardUpdate]\n" );
            NewlyUpdateInstaller installer( control );
            NNT_EXPECT_RESULT_SUCCESS( installer.WaitResponse() );
            LOG_OUT( "Complete install prepared update.\n" );

            // NCMデータベースベリファイ
            verifier.VerifyHasContentOnStorage();
            verifier.VerifyUnnecessaryPlaceHolderOnStorage();
            verifier.VerifyUnexpectedContentOnStorage();
        }
    }

    // ワークバッファ解放.
    aligned_free( pWorkingBuffer );

} // NOLINT(impl/function_size)


INSTANTIATE_TEST_CASE_P(Simple,
                        CupCase,
                        ::testing::Values(false));

INSTANTIATE_TEST_CASE_P(ViaUpdaterSimple,
                        CupCase,
                        ::testing::Values(true));
