﻿<#
    $ powershell -Version 2.0 -ExecutionPolicy Unrestricted GetDeviceId.ps1
#>

# main 引数
param (
    # Qualify the platforms
    [string] $Platform = "NX-NXFP2-a64",

    # Qualify the build types
    [string] $BuildType = "Develop",

    # Qualify the input file path of cup configurations.
    [string] $InputCupConfigFile,

    [string]
    $TargetName,

    [string]
    $TargetAddress
)

# スクリプトパス関連 ( 全部文字列型らしい [string] )
$MyScriptPath           = $MyInvocation.MyCommand.Path
$MyScriptDirectoryPath  = [System.IO.Path]::GetDirectoryName($MyScriptPath)

Import-Module "${MyScriptDirectoryPath}\Modules\PlatformUtilities"

# SDK ROOTパス
$NintendoSdkRootPath = $(Find-NintendoSdkRootPath)

# NintendoSDK Integrate script module インポート
Import-Module "${NintendoSdkRootPath}\Integrate\Scripts\Modules\Error"
Import-Module "${NintendoSdkRootPath}\Integrate\Scripts\Modules\HostBridge"
Import-Module "${NintendoSdkRootPath}\Integrate\Scripts\Modules\Path"
Import-Module "${MyScriptDirectoryPath}\Modules\FileUtilities"
Import-Module "${MyScriptDirectoryPath}\Modules\SdkToolUtilities"

if ([string]::IsNullOrEmpty($TargetAddress)) {
    if ([string]::IsNullOrEmpty($TargetName)) {
        throw "TargetAddress and TargetName not specified"
    } else {
        $TargetAddress = Get-TargetAddressFromSerial $TargetName
    }
}

# ================================
# main
# ================================
try {
    # CIジョブでのTestRunnerは、nact経由でステップ実行し、nactはUTF8で出力を受信している。
    # そのため、強制的にコンソールをUTF-8としている。
    # 但し通常のコード932のDOS窓上では本スクリプトから、Write-Hostなどに日本語指定が使えないので注意。
    $defUseEncoding = New-Object System.Text.UTF8Encoding($False)
    $defBackupConsoleEncoding = [System.Console]::OutputEncoding
    $defBackupEncode = $OutputEncoding
    [System.Console]::OutputEncoding = ${defUseEncoding}
    $OutputEncoding = ${defUseEncoding}

    [string] $defConsoleEncode = [System.Console]::OutputEncoding.EncodingName
    [string] $defStartupEncode = $defBackupEncode.EncodingName
    [string] $defCurrentEncode = $OutputEncoding.EncodingName
    Write-Host "[Console]::OutputEncoding : ${defConsoleEncode}"
    Write-Host "Startup `$OutputEncoding   : ${defStartupEncode}"
    Write-Host "Current `$OutputEncoding   : ${defCurrentEncode}"

    # プラットフォームチェック
    $Platform = Resolve-PlatformConstants( ${Platform} )

    # ConfigFile 確認
    $(Test-ExistFile ${InputCupConfigFile})

    Write-Host "============== Script configuration. =============="
    Write-Host "Platform        : ${Platform}"
    Write-Host "BuildType       : ${BuildType}"
    Write-Host "SDK Root        : ${NintendoSdkRootPath}"
    Write-Host "ScriptPath      : ${MyScriptPath}"
    Write-Host "ScriptDirectory : ${MyScriptDirectoryPath}"
    Write-Host "CupConfig File  : ${InputCupConfigFile}"

    # DevMenuCommandSystem.nsp の検索
    [string] $expectNsp = $(Search-AvailableDevMenuCommandSystem ${NintendoSdkRootPath} ${Platform} ${BuildType})
    Write-Host "GameCardWriter  : ${expectNsp}"

    # コンフィグファイルの読み込み
    $configs = @{}
    Import-Csv -Path ${InputCupConfigFile} -Header "Key","Value" -Delimiter "=" | % {
        $configs.Add( $_.Key.Trim(), $_.Value.Trim() )
    }
    [int] $defCupVersion = $configs.CupVersion
    [string] $defCupNsp = $configs.CupNsp
    [string] $defAppNsp = $configs.AppNsp

    Write-Host "============== GameCardWrite configuration. =============="
    Write-Host "CupVersion : ${defCupVersion}"
    Write-Host "CupNsp : ${defCupNsp}"
    Write-Host "AppNsp : ${defAppNsp}"

    # GameCard Writing.
    Invoke-Expression "& `"${NintendoSdkRootPath}\Tools\CommandLineTools\RunOnTarget.exe`" `"--target`" `"${TargetAddress}`" `"${expectNsp}`" `"--`" `"gamecard`" `"write`" `"${defAppNsp}`" `"--update-partition`" `"${defCupNsp}`""

} catch [Exception] {

    Write-ErrorRecord $_
    $LastExitCode = 1

} finally {

    $OutputEncoding = $defBackupEncode
    [System.Console]::OutputEncoding = $defBackupConsoleEncoding

}
exit $LastExitCode
