﻿<#
    $ powershell -Version 2.0 -ExecutionPolicy Unrestricted GetDeviceId.ps1
#>

# main 引数
param (
    # Qualify the platforms
    [string] $Platform = "NX-NXFP2-a64",

    # Qualify the build types
    [string] $BuildType = "Develop",

    # Qualify the input file path of cup configurations.
    [string] $InputCupConfigFile,

    # Qualify the intermediate working folder path
    [string] $IntermediateDirectory
)

# スクリプトパス関連 ( 全部文字列型らしい [string] )
$MyScriptPath           = $MyInvocation.MyCommand.Path
$MyScriptDirectoryPath  = [System.IO.Path]::GetDirectoryName($MyScriptPath)

Import-Module "${MyScriptDirectoryPath}\Modules\PlatformUtilities"

# SDK ROOTパス
$NintendoSdkRootPath = $(Find-NintendoSdkRootPath)

# AuthoringTool
[string] $defAuthoringTool = "${NintendoSdkRootPath}\Tools\CommandLineTools\AuthoringTool\AuthoringTool.exe"
[string] $defDescFile = "${NintendoSdkRootPath}\Tests\Common\Resources\TestApplication.autogen.desc"

# TestProgram
[string] $defProgramName = "testNs_CupViaUpdaterSimple"

# NintendoSDK Integrate script module インポート
Import-Module "${NintendoSdkRootPath}\Integrate\Scripts\Modules\Error"
Import-Module "${NintendoSdkRootPath}\Integrate\Scripts\Modules\Path"
Import-Module "${MyScriptDirectoryPath}\Modules\FileUtilities"
Import-Module "${MyScriptDirectoryPath}\Modules\SdkToolUtilities"

function Search-TestProgram( [string]$defSdkRoot, [string]$defPlatform, [string]$defBuildType )
{
    # 指定のプラットフォーム + ビルドタイプで存在するか
    [string] $expectNsp = "${defSdkRoot}\Tests\Outputs\${Platform}\Tests\${defProgramName}\${BuildType}\${defProgramName}.nsp"
    if ( Test-Path "${expectNsp}" )
    {
        return "${expectNsp}"
    }
    $message = "${expectNsp} not found."
    throw New-Object "System.IO.FileNotFoundException" $message
}

function Extract-CupNsp( [string]$cupNsp, [string]$outputDir )
{
    Remove-Item "${outputDir}\*"
    Invoke-Expression "& `"${defAuthoringTool}`" extractnsp -o `"${outputDir}`" `"${cupNsp}`"" | Out-Null
    Remove-Item "${outputDir}\*.xml"
}

function Rebuild-TestProgram( [string]$defSdkRoot, [string]$targetNsp, [string]$uppRomDir )
{
    [string] $metaFile = "${defSdkRoot}\Tests\Outputs\${Platform}\Tests\${defProgramName}\${BuildType}\${defProgramName}.nmeta"
    [string] $codeDir = ${targetNsp}.Replace("${defProgramName}.nsp", "code")
    Invoke-Expression "& `"${defAuthoringTool}`" creatensp -o `"${targetNsp}`" --type Application --meta `"${metaFile}`" --desc `"${defDescFile}`" --program `"${codeDir}`" `"${uppRomDir}`""
}

# ================================
# main
# ================================
try {
    # CIジョブでのTestRunnerは、nact経由でステップ実行し、nactはUTF8で出力を受信している。
    # そのため、強制的にコンソールをUTF-8としている。
    # 但し通常のコード932のDOS窓上では本スクリプトから、Write-Hostなどに日本語指定が使えないので注意。
    $defUseEncoding = New-Object System.Text.UTF8Encoding($False)
    $defBackupConsoleEncoding = [System.Console]::OutputEncoding
    $defBackupEncode = $OutputEncoding
    [System.Console]::OutputEncoding = ${defUseEncoding}
    $OutputEncoding = ${defUseEncoding}

    [string] $defConsoleEncode = [System.Console]::OutputEncoding.EncodingName
    [string] $defStartupEncode = $defBackupEncode.EncodingName
    [string] $defCurrentEncode = $OutputEncoding.EncodingName
    Write-Host "[Console]::OutputEncoding : ${defConsoleEncode}"
    Write-Host "Startup `$OutputEncoding   : ${defStartupEncode}"
    Write-Host "Current `$OutputEncoding   : ${defCurrentEncode}"

    # 中間ファイル作成作業フォルダ存在確認
    [string] $IntermediateDirectory = $(Get-AbsoluteFullPath ${IntermediateDirectory})
    $(Edit-MakeDirectory ${IntermediateDirectory})

    # プラットフォームチェック
    $Platform = Resolve-PlatformConstants( ${Platform} )

    # ConfigFile 確認
    $(Test-ExistFile ${InputCupConfigFile})

    Write-Host "============== Script configuration. =============="
    Write-Host "Platform        : ${Platform}"
    Write-Host "BuildType       : ${BuildType}"
    Write-Host "SDK Root        : ${NintendoSdkRootPath}"
    Write-Host "ScriptPath      : ${MyScriptPath}"
    Write-Host "ScriptDirectory : ${MyScriptDirectoryPath}"
    Write-Host "CupConfig File  : ${InputCupConfigFile}"

    # 対象 nsp の決定
    [string] $expectNsp = $(Search-TestProgram ${NintendoSdkRootPath} ${Platform} ${BuildType})
    Write-Host "SystemUpdater  : ${expectNsp}"

    # コンフィグファイルの読み込み
    $configs = @{}
    Import-Csv -Path ${InputCupConfigFile} -Header "Key","Value" -Delimiter "=" | % {
        $configs.Add( $_.Key.Trim(), $_.Value.Trim() )
    }
    [int] $defCupVersion = $configs.CupVersion
    [string] $defCupNsp = $configs.CupNsp
    [string] $defAppNsp = $configs.AppNsp

    Write-Host "============== SystemUpdater configuration. =============="
    Write-Host "CupVersion : ${defCupVersion}"
    Write-Host "CupNsp : ${defCupNsp}"

    # SystemUpdater をリビルド
    [string] $uppRomDir = "${IntermediateDirectory}\upprom"
    $(Edit-MakeDirectory ${uppRomDir})
    Extract-CupNsp ${defCupNsp} ${uppRomDir}
    Rebuild-TestProgram ${NintendoSdkRootPath} ${expectNsp} ${uppRomDir}

} catch [Exception] {

    Write-ErrorRecord $_
    $LastExitCode = 1

} finally {

    $OutputEncoding = $defBackupEncode
    [System.Console]::OutputEncoding = $defBackupConsoleEncoding

}
exit $LastExitCode