﻿<#
    .SYNOPSIS
        Private option utilities module

    .DESCRIPTION
        This file defines functions for private operations
#>

# 基本パス関連 ( 全部文字列型らしい [string] )
$MyScriptPath          = $MyInvocation.MyCommand.Path
$MyScriptDirectoryPath = [System.IO.Path]::GetDirectoryName($MyScriptPath)

Import-Module "${MyScriptDirectoryPath}\PlatformUtilities"
Import-Module "${MyScriptDirectoryPath}\OptionUtilities"

# SDK ROOTパス
$NintendoSdkRootPath = $(Find-NintendoSdkRootPath)

# NintendoSDK Integrate script module インポート
Import-Module "${MyScriptDirectoryPath}\FileUtilities"

# -----------------------------------------------------------------------
# メタID ( [a-fA-F0-9]{16} ) 整形.
# 入力文字列の "0x" を取り除く。
# 変換不可能な場合は例外を発生します。
# -----------------------------------------------------------------------
function Select-ExpectStringMetaId( [string]$defInputString )
{
    [System.UInt64] $defIntValue = [System.Convert]::ToInt64( ${defInputString}, 16 );
    return [string] $defIntValue.ToString( "X16" )
}
Export-ModuleMember -Function Select-ExpectStringMetaId

# -----------------------------------------------------------------------
# make  nsp file base name.
# -----------------------------------------------------------------------
function Get-ContentNspFileBaseName( [string]$defProgramId, [int]$defVersion, [string]$defArchSignature )
{
    [string] $defNspBaseName = "${defProgramId}_v${defVersion}"
    if ( $defArchSignature.Length -gt 0 )
    {
        [string] $defNspBaseName = "${defNspBaseName}.${defArchSignature}"
    }
    return ${defNspBaseName}
}
Export-ModuleMember -Function Get-ContentNspFileBaseName

# -----------------------------------------------------------------------
# Extract nsp.
# -----------------------------------------------------------------------
function Expand-ContentNspFromFile( [string]$defExpandFolderPath, [string]$defNspFilePath )
{
    # ファイルの存在確認
    $(Test-ExistFile ${defNspFilePath})
    $(Remove-ExistDirectory ${defExpandFolderPath})
    $(Edit-MakeDirectory ${defExpandFolderPath})

    # nsp extract
    [string] $defAuthoringTool = "${NintendoSdkRootPath}\Tools\CommandLineTools\AuthoringTool\AuthoringTool.exe"
    [string] $defKeyConfig = "${NintendoSdkRootPath}\Tools\CommandLineTools\AuthoringTool\AuthoringTool.ocean.keyconfig.xml"
    Invoke-Expression "& `"${defAuthoringTool}`" extractnsp -o `"${defExpandFolderPath}`" `"${defNspFilePath}`" --keyconfig `"${defKeyConfig}`"" | Out-Null
    return ${defExpandFolderPath}
}
Export-ModuleMember -Function Expand-ContentNspFromFile

# -----------------------------------------------------------------------
# Extract nsp.
# -----------------------------------------------------------------------
function Expand-ContentNspFromName( [string]$defIntermediateDirectory, [string]$defExpandFolderName, [string]$defNspBaseName )
{
    [string] $defOutPath = "${defIntermediateDirectory}\${defExpandFolderName}"
    [string] $defNspFile = "${defIntermediateDirectory}\${defNspBaseName}.nsp"
    return $(Expand-ContentNspFromFile ${defOutPath} ${defNspFile})
}
Export-ModuleMember -Function Expand-ContentNspFromName

# -----------------------------------------------------------------------
# Extract nsp.
# -----------------------------------------------------------------------
function Expand-ContentNsp( [string]$defIntermediateDirectory, [string]$defProgramId, [int]$defVersion, [string]$defArchSignature )
{
    [string] $defExpandName = $(Get-ContentNspFileBaseName ${defProgramId} ${defVersion} "")
    [string] $defNspBaseName = $(Get-ContentNspFileBaseName ${defProgramId} ${defVersion} ${defArchSignature})
    return $(Expand-ContentNspFromName ${defIntermediateDirectory} ${defExpandName} ${defNspBaseName})
}
Export-ModuleMember -Function Expand-ContentNsp

# -----------------------------------------------------------------------
# 利用可能な DevMenuCommandSystem.nsp のパスを返す。
# -----------------------------------------------------------------------
function Search-AvailableDevMenuCommandSystem( [string]$defSdkRoot, [string]$defPlatform, [string]$defBuildType )
{
    # 指定のプラットフォーム + ビルドタイプで存在するか
    [string] $expectNsp = "${defSdkRoot}\Programs\Eris\Outputs\${defPlatform}\TargetTools\DevMenuCommandSystem\${defBuildType}\DevMenuCommandSystem.nsp"
    if ( Test-Path "${expectNsp}" )
    {
        return "${expectNsp}"
    }

    # 指定がなければ可能な範囲で探す
    $defBuildTypes = @( "Debug", "Develop", "Release" )
    foreach( $type in $defBuildTypes )
    {
        [string] $expectNsp = "${defSdkRoot}\Programs\Eris\Outputs\${defPlatform}\TargetTools\DevMenuCommandSystem\${type}\DevMenuCommandSystem.nsp"
        if ( Test-Path "${expectNsp}" )
        {
            Write-Host "Not found the DevMenuCommandSystem.nsp on specified platform and build type : ${defPlatform}, ${defBuildType}."
            Write-Host "Therefor will used the alternate nca which has existed in ${defPlatform}/${type}."
            return "${expectNsp}"
        }
    }
    $message = "DevMenuCommandSystem.nsp not found."
    throw New-Object "System.IO.FileNotFoundException" $message
}
Export-ModuleMember -Function Search-AvailableDevMenuCommandSystem

# -----------------------------------------------------------------------
# 利用可能な ControlTarget.exe のパスを返す。
# -----------------------------------------------------------------------
function Search-AvailableControlTarget( [string]$defSdkRoot )
{
    [string] $defExpectExe = "${defSdkRoot}\Tools\CommandLineTools\ControlTarget.exe"
    if ( Test-Path "${defExpectExe}" )
    {
        return "${defExpectExe}"
    }
    $message = "ControlTarget.exe not found."
    throw New-Object "System.IO.FileNotFoundException" $message
}
Export-ModuleMember -Function Search-AvailableControlTarget

# -----------------------------------------------------------------------
# 利用可能な RunOnTarget.exe のパスを返す。
# -----------------------------------------------------------------------
function Search-AvailableRunOnTarget( [string]$defSdkRoot )
{
    [string] $defExpectExe = "${defSdkRoot}\Tools\CommandLineTools\RunOnTarget.exe"
    if ( Test-Path "${defExpectExe}" )
    {
        return "${defExpectExe}"
    }
    $message = "RunOnTarget.exe not found."
    throw New-Object "System.IO.FileNotFoundException" $message
}
Export-ModuleMember -Function Search-AvailableRunOnTarget

# -----------------------------------------------------------------------
# 利用可能な RunOnTargetPrivate.exe のパスを返す。
# -----------------------------------------------------------------------
function Search-AvailableRunOnTargetPrivate( [string]$defSdkRoot )
{
    [string] $defExpectExe = "${defSdkRoot}\Tools\CommandLineTools\RunOnTargetPrivate.exe"
    if ( Test-Path "${defExpectExe}" )
    {
        return "${defExpectExe}"
    }
    $message = "RunOnTargetPrivate.exe not found."
    throw New-Object "System.IO.FileNotFoundException" $message
}
Export-ModuleMember -Function Search-AvailableRunOnTargetPrivate

# -----------------------------------------------------------------------
# 利用可能なテストコード nsp のパスを返す。
# -----------------------------------------------------------------------
function Search-AvailableTestModuleNsp( [string]$defSdkRoot, [string]$defPlatform, [string]$defBuildType, [string]$defTestModuleName )
{
    # 指定のプラットフォーム + ビルドタイプで存在するか
    [string] $expectNsp = "${defSdkRoot}\Tests\Outputs\${defPlatform}\Tests\${defTestModuleName}\${defBuildType}\${defTestModuleName}.nsp"
    if ( Test-Path "${expectNsp}" )
    {
        return "${expectNsp}"
    }

    # 指定がなければ可能な範囲で探す
    $defBuildTypes = @( "Debug", "Develop", "Release" )
    foreach( $type in $defBuildTypes )
    {
        [string] $expectNsp = "${defSdkRoot}\Tests\Outputs\${defPlatform}\Tests\${defTestModuleName}\${type}\${defTestModuleName}.nsp"
        if ( Test-Path "${expectNsp}" )
        {
            Write-Host "Not found the ${defTestModuleName}.nsp on specified platform and build type : ${defPlatform}, ${defBuildType}."
            Write-Host "Therefor will used the alternate nca which has existed in ${defPlatform}/${type}."
            return "${expectNsp}"
        }
    }
    $message = "${defTestModuleName}.nsp not found."
    throw New-Object "System.IO.FileNotFoundException" $message
}
Export-ModuleMember -Function Search-AvailableTestModuleNsp

# -----------------------------------------------------------------------
# Check integrity of uploaded nsp on server.
# AuthoringTool extract で抽出した *.cnmt.xml と、rops list-contents コマンドで取得した nsp ステータスを比較します.
# 比較対象は cnmt.xml の <Content> です.
# -----------------------------------------------------------------------
function Test-IntegrityOfUploadedNspOnServer( [string]$defRopsExec, [string]$defAuthToken, [string]$defRopsEnv, [string]$defCnmtSearchPath, [string]$defMetaId, [int]$defMetaVersion )
{
    Write-Host "`n===== Check integrity of uploaded nsp[${defMetaId}:${defMetaVersion}] on server. ====="

    # rops list-contents.
    [string] $defRopsCommand = "& ${defRopsExec} list-contents --token `"${defAuthToken}`" -e ${defRopsEnv} -t ${defMetaId} -v ${defMetaVersion}"
    [string] $defRopsResponse = Invoke-Expression "${defRopsCommand} 2>&1"
    Write-Host "===== Get response of `"list-contents`". ======"
    Write-Host "${defRopsResponse}"
    Write-Host "===== Try verification. ====="

    [bool] $defResult = $True

    # 展開先の*.cnmt.xml から、type, id が合致するものを検出。
    Get-ChildItem ${defCnmtSearchPath} -include "*.cnmt.xml" -recurse -force | % {
        [System.IO.FileInfo] $defCnmtFilePath = $_
        $defXmlCnmt = [xml] $(Get-Content ${defCnmtFilePath})
        if (( $defXmlCnmt.ContentMeta.Id -eq "0x${defMetaId}" ) -and
            ( $defXmlCnmt.ContentMeta.Version -eq "${defMetaVersion}" ))
        {
            Write-Host "Has content in [${defMetaId}:${defMetaVersion}] = {"

            # 検出した cnmt.xml から<Content>を抽出。
            [array] $defXmlCnmtContentMetaNodes = $defXmlCnmt.ContentMeta.Content
            ${defXmlCnmtContentMetaNodes} | % {
                [System.Xml.XmlElement] $defXmlCnmtContentMetaNode = $_
                [string] $defContentId = $defXmlCnmtContentMetaNode.id
                [string] $defContentType = $defXmlCnmtContentMetaNode.type
                Write-Host "    Client expects <Id=`"${defContentId}`", Type=`"${defContentType}`">"

                # 期待 content id の検出
                $defExpectContent = [regex]::Match( ${defRopsResponse}, "`"content_id`"\:[\s]*`"${defContentId}`"" )
                if ( $defExpectContent.Success -eq $False )
                {
                    Write-Host "      -> [NG] Could not found on the server."
                    $defResult = $False
                }
                else
                {
                    Write-Host "      -> [OK] Found on the server."
                }
            } | Out-Null

            Write-Host "}"
        }
    } | Out-Null

    if ( $False -eq ${defResult} )
    {
        Write-Host "Failed to test of integrity with the client contents and the server contents."
    }
    Write-Host "===== End of check integrity of uploaded nsp[${defMetaId}:${defMetaVersion}] on server. ====="
    return ${defResult}
}
Export-ModuleMember -Function Test-IntegrityOfUploadedNspOnServer

# -----------------------------------------------------------------------
# Gets the ROM-ID from rops upload response.
# -----------------------------------------------------------------------
function Get-RomIdFromRopsUploadResponse( [string]$defResponseStream )
{
    if ( $False -eq [string]::IsNullOrEmpty( ${defResponseStream} ) )
    {
        $defMatchResult = [regex]::Match( ${defResponseStream}, "Upload has been completed successfully\. ROM ID = [A-Fa-f0-9]+" )
        if ( $defMatchResult.Success )
        {
            $defMatchedValue = $defMatchResult.Value
            $defRomIdLineValues = ${defMatchedValue} -csplit "ROM ID = "
            if ( $defRomIdLineValues.Length -gt 1 ) 
            {
                $defRomId = $defRomIdLineValues[1]
                return [string] ${defRomId}
            }
        }
    }
    Write-Host "Were not found the available ROM-ID in the specified input stream."
    return [string] ""
}
Export-ModuleMember -Function Get-RomIdFromRopsUploadResponse

# -----------------------------------------------------------------------
# Make submit command option string from rops upload response.
# -----------------------------------------------------------------------
function Make-SubmitOptions( [string]$defUploadedRomId, [string]$defIntermediateDirectory, [string]$defMetaId, [int]$defReleaseVersion, [int]$defSubmitVersion )
{
    [string] $defXmlFilePath = $(Out-MakeRopsSubmitXml ${defIntermediateDirectory} ${defMetaId} ${defReleaseVersion} ${defSubmitVersion})
    return [string] "-r `"${defUploadedRomId}`" -i `"${defXmlFilePath}`""
}

# -----------------------------------------------------------------------
# rom status check
# Will retry until success or timeout breaking.
# Specified the timeout value must be than 30 seconds.
# -----------------------------------------------------------------------
function Test-RomStatus( [string]$defExpectStatus, [string]$defRopsExec, [string]$defAuthToken, [string]$defEnv, [string]$defRomId, [double]$defTimeOutSeconds )
{
    [string] $defRopsCommand = "& ${defRopsExec} list-roms --token `"${defAuthToken}`" -e ${defEnv} -r `"${defRomId}`""
    Write-Host "Check rom status... [ ${defRopsCommand} ]"

    [string] $defMatchedString = "`"status`": `"${defExpectStatus}`""
    [DateTime] $defStartTime = $(Get-Date)
    while ( $(New-TimeSpan ${defStartTime} $(Get-Date)).TotalSeconds -lt ${defTimeOutSeconds} )
    {
        [string]$defResponseStream = Invoke-Expression "${defRopsCommand} 2>&1"
        $defMatchResult = [regex]::Match( ${defResponseStream}, "${defMatchedString}" )
        if ( $defMatchResult.Success )
        {
            Write-Host "Cound detect the expected status as `"${defExpectStatus}`"."
            return [bool] $True
        }
        Write-Host "Retry the rom status check, That response = "
        Write-Host "${defResponseStream}"
        Start-Sleep -s 30
    }
    Write-Host "The rom status check was timeout in ${defTimeOutSeconds} seconds."
    Write-Host "Failed the rom status check, Could not detect the expected status as `"${defExpectStatus}`"."
    return [bool] $False
}
Export-ModuleMember -Function Test-RomStatus

# -----------------------------------------------------------------------
# Submit with retry.
# -----------------------------------------------------------------------
function Submit-RomByRops( [string]$defRopsSubmitCommand, [double]$defTimeOutSeconds )
{
    Write-Host "Invoke rops submit: ${defRopsSubmitCommand}"
    [DateTime] $defStartTime = $(Get-Date)
    while ( $(New-TimeSpan ${defStartTime} $(Get-Date)).TotalSeconds -lt ${defTimeOutSeconds} )
    {
        # Server workflow finish waits.
        Start-Sleep -s 30

        # submit execution
        # 2>&1 で標準エラーを標準出力へリダイレクト.
        [string]$defSubmitResponse = Invoke-Expression "${defRopsSubmitCommand} 2>&1"
        $defSubmitResult = [regex]::Match( ${defSubmitResponse}, "Submit has been completed successfully\." )
        if ( $defSubmitResult.Success -eq $True )
        {
            # Successful.
            Write-Host "${defSubmitResponse}"
            return [bool] $True
        }
        else
        {
            Write-Host "Failed submit contents, response = ${defSubmitResponse}"
            $defFailCode = [regex]::Match( ${defSubmitResponse}, "`"code`"\:`"000\-0509`"" )
            if ( $defFailCode.Success -eq $True )
            {
                # retry from the submit stage.
                Write-Host "Detect the error-code with `"000-0509`" on submit contents."
            }
            else
            {
                # needs retry from the upload stage.
                break
            }
        }
        Write-Host "Retry submit contents..."
    }
    return [bool] $False
}
Export-ModuleMember -Function Submit-RomByRops

# -----------------------------------------------------------------------
# Get value of `access_token` from CLI token as json.
# -----------------------------------------------------------------------
function Export-CliTokenFromJson( [string]$defJsonStream, [string]$defTokenTypeKey )
{
    $matchResult = [regex]::Match( ${defJsonStream}, "`"${defTokenTypeKey}`":`"[A-Fa-f0-9]+`"," )
    if ( $matchResult.Success )
    {
        $matchToken = [regex]::Match( $matchResult.Value, "`"[A-Fa-f0-9]+`"" )
        if ( $matchToken.Success )
        {
            [string] $defTokenValue = [regex]::Replace( $matchToken.Value, "^`"", "" )
            [string] $defTokenValue = [regex]::Replace( ${defTokenValue}, "`"$", "" )
            return ${defTokenValue}
        }
    }
    throw "Could not found the ``${defTokenTypeKey}`` field in [ ${defJsonStream} ]"
}
Export-ModuleMember -Function Export-CliTokenFromJson

# -----------------------------------------------------------------------
# Get CLI basic token as json.
# -----------------------------------------------------------------------
function Get-CliBasicTokenAsJson( [string]$defProxyOption, [string]$defEnvironment )
{
    [string] $defNDID = "pux_tsutsumi.daisuke"
    [string] $defPASS = "NGEzYmM0YTUtODBk"
    [string] $defURI = "https://star.debug.${defEnvironment}.d4c.nintendo.net/v1/ndid/users/@me/token"

    # cURL.exe のパス取得
    [string] $defCurlExec = $(Get-CurlExec)
    [string] $defCommand = "& ${defCurlExec} ${defProxyOption} -k --user `"${defNDID}:${defPASS}`" `"${defURI}`""
    Write-Host "Request query basic token command : ${defCommand}"
    [string] $defResponse = Invoke-Expression ${defCommand}
    Write-Host "Response : ${defResponse}"
    return ${defResponse}
}
Export-ModuleMember -Function Get-CliBasicTokenAsJson

# -----------------------------------------------------------------------
# Get CLI basic token with write to file.
# -----------------------------------------------------------------------
function Get-CliBasicToken( [string]$defIntermediateDirectory, [string]$defProxyOption, [string]$defEnvironment )
{
    # Json 受信
    [string] $defResponse = $(Get-CliBasicTokenAsJson ${defProxyOption} ${defEnvironment})

    # BOMなしUTF8出力
    [string] $defOut = "${defIntermediateDirectory}\cli.auth.token.json"
    $defEncodeUtf8NoBom = New-Object System.Text.UTF8Encoding( $false )
    [System.IO.File]::WriteAllLines( ${defOut}, ${defResponse}, ${defEncodeUtf8NoBom} )
    return ${defOut}
}
Export-ModuleMember -Function Get-CliBasicToken

# -----------------------------------------------------------------------
# Upload and submit.
# -----------------------------------------------------------------------
function Publish-RomByRops( [string]$defIntermediateDirectory, [string]$defProxy, [string]$defEnvironment, [string]$defNspId, [string]$defNspFile, [int]$defVersion )
{
    # プロキシオプション( --proxy http://XXX )
    [string] $defProxyOption = $(New-ProxyOption ${defProxy})
    # CLIトークン
    [string] $defAuthToken = $(Get-CliBasicToken ${defIntermediateDirectory} ${defProxyOption} ${defEnvironment})
    # rops upload command
    [string] $defRopsExec = "`"${NintendoSdkRootPath}\Tools\CommandLineTools\rops\rops.exe`" --insecure --suppress-warning http-fallback ${defProxyOption}"
    [string] $defUploadCommand = "& ${defRopsExec} upload --token `"${defAuthToken}`" -e ${defEnvironment} -a 0x${defNspId} -s `"${defNspFile}`""

    # uploading.
    Write-Host "`n>>>>>>> Start uploading contents by rops... [ ${defNspId}:${defVersion} ]"
    Write-Host "`nInvoke rops submit: ${defUploadCommand}"

    [int] $defUploadRetryCount = 3
    [double] $defSubmitRetryTimeout = 20 * 60
    [double] $defRomStatusCheckTimeout = 20 * 60
    for ( [int] $retryCount = ${defUploadRetryCount}; $retryCount -gt 0; )
    {
        # () で括る事で[ Invoke-Expression "" | tee -Variable defUploadResponse ]と同じようにコンソール＆変数への出力が可能.
        # 但し、標準出力されるので function の場合、返値に含まれる点に注意.
        [string] $defUploadResponse = Invoke-Expression "${defUploadCommand} 2>&1"
        Write-Host "${defUploadResponse}"
        [string] $defRomId = $(Get-RomIdFromRopsUploadResponse ${defUploadResponse})
        if ( $defRomId.Length -ne 0 )
        {
            # Server screening waits.
            Start-Sleep -s 30

            # rom status check to "submittable".
            if ( $True -eq $(Test-RomStatus "SUBMITTABLE" ${defRopsExec} ${defAuthToken} ${defEnvironment} ${defRomId} ${defRomStatusCheckTimeout}) )
            {
                # make the invokable submit commands.
                [string] $defSubmitOption = $(Make-SubmitOptions ${defRomId} ${defIntermediateDirectory} ${defNspId} ${defVersion} ${defVersion})
                [string] $defSubmitCommand = "& ${defRopsExec} submit --token `"${defAuthToken}`" -e ${defEnvironment} ${defSubmitOption}"
                if ( $True -eq $(Submit-RomByRops ${defSubmitCommand} ${defSubmitRetryTimeout}) )
                {
                    # Successful.
                    # Next check to rom status of "checking", that status be the publish succeeded.
                    Start-Sleep -s 30

                    # rom status check to "checking".
                    if ( $True -eq $(Test-RomStatus "CHECKING" ${defRopsExec} ${defAuthToken} ${defEnvironment} ${defRomId} ${defRomStatusCheckTimeout}) )
                    {
                        # Successful.
                        # Check integrity for uploaded nsp.
                        Start-Sleep -s 10

                        Write-Host "`n===== Extract nsp ====="
                        Write-Host "  -> nsp : `"${defNspFile}`""
                        [string] $defNspExpandName = $(Get-ContentNspFileBaseName ${defNspId} ${defVersion} "")
                        [string] $defCnmtExtractPath = "${defIntermediateDirectory}\${defNspExpandName}"
                        [string] $defCnmtExtractPath = $(Expand-ContentNspFromFile ${defCnmtExtractPath} ${defNspFile})
                        if ( $True -eq $(Test-IntegrityOfUploadedNspOnServer ${defRopsExec} ${defAuthToken} ${defEnvironment} ${defCnmtExtractPath} ${defNspId} ${defVersion}) )
                        {
                            return
                        }
                    }
                }
            }
        }
        $retryCount = ${retryCount} - 1
        Write-Host "Retry uploading contents... [ ${defNspId}:${defVersion} ], left as [ ${retryCount} / ${defUploadRetryCount} ] "
    }
    throw "Failed uploading contents by rops... [ ${defNspId}:${defVersion} ]"
}
Export-ModuleMember -Function Publish-RomByRops

# -----------------------------------------------------------------------
# 指定されたNsp配列を含むNfaを作成します。
# 返値として作成されたNfaファイルのフルパス文字列を返します。
# -----------------------------------------------------------------------
function Out-MakeContentsNfa( [string]$defIntermediateDirectory, [string]$defCupBaseName, [string[]]$defNspContents )
{
    [string] $defNfaYmlFile = "${defIntermediateDirectory}\${defCupBaseName}.yml"
    [string] $defNfaFile = "${defIntermediateDirectory}\${defCupBaseName}.contents.nfa"

    Write-Host "YML: ${defNfaYmlFile}"
    Write-Host "NFA: ${defNfaFile}"
    $(Remove-ExistFile ${defNfaYmlFile})

    # 上書きモードファイルオープン
    $fstream = New-Object System.IO.StreamWriter( $defNfaYmlFile, $False, [Text.Encoding]::GetEncoding( "UTF-8" ) )
    $fstream.WriteLine( "Attributes:" ) | Out-Null
    $fstream.WriteLine( "- Name: SYSTEM" ) | Out-Null
    $fstream.WriteLine( "  Path:" ) | Out-Null

    foreach( $defNsp in $defNspContents )
    {
        # - ""で囲みたいが、囲むと MakeFirmwareArchive.exe が死ぬ.
        $fstream.WriteLine( "  - ${defNsp}" ) | Out-Null
    }
    $fstream.Close() | Out-Null
    $(Test-ExistFile ${defNfaYmlFile})

    # nfa
    $(Remove-ExistFile ${defNfaFile})
    [string] $defMakeFirmwareArchive = "${NintendoSdkRootPath}\Tools\CommandLineTools\MakeFirmwareArchive\MakeFirmwareArchive.exe"
    Invoke-Expression "& `"${defMakeFirmwareArchive}`" createnfa -o `"${defNfaFile}`" -i `"${defNfaYmlFile}`"" | Write-Host
    $(Test-ExistFile ${defNfaFile})

    return ${defNfaFile}
}
Export-ModuleMember -Function Out-MakeContentsNfa

# -----------------------------------------------------------------------
# CUPコンフィグファイルを作成します。
# -----------------------------------------------------------------------
function Out-MakeCupConfigFile( [string]$defIntermediateDirectory, [string]$defApplicationNsp, [string]$defCupNsp, [int]$defCupVersion )
{
    # CUPコンフィグファイル
    [string] $defCupConfigFileName = "CupConfig_v${defCupVersion}.txt"
    [string] $defPathCupConfigFile = "${defIntermediateDirectory}\${defCupConfigFileName}"

    # 出力ファイル削除
    $(Remove-ExistFile ${defPathCupConfigFile})

    Write-Host "Out-MakeCupConfigFile: ${defPathCupConfigFile}"

    # 上書きモードファイルオープン
    $fstream = New-Object System.IO.StreamWriter( ${defPathCupConfigFile}, $False, [Text.Encoding]::GetEncoding( "UTF-8" ) )
    $fstream.WriteLine( "CupVersion=${defCupVersion}" ) | Out-Null
    $fstream.WriteLine( "CupNsp=`"${defCupNsp}`"" ) | Out-Null
    $fstream.WriteLine( "AppNsp=`"${defApplicationNsp}`"" ) | Out-Null
    $fstream.Close() | Out-Null

    return ${defPathCupConfigFile}
}
Export-ModuleMember -Function Out-MakeCupConfigFile

# -----------------------------------------------------------------------
# Make dummy application
# -----------------------------------------------------------------------
function New-DummyUserApplication( [string]$defIntermediateDirectory, [string]$defApplicationId, [int]$defVersion, [string]$defArchSignature )
{
    # MakeTestApplication control path
    [string] $defMakeTestApplication = "${NintendoSdkRootPath}\Tools\CommandLineTools\MakeTestApplication\MakeTestApplication.exe"
    [string] $defWorkDirectory = "${defIntermediateDirectory}\_NspWorkTemporary_"

    # application nsp base name 生成
    [string] $defApplicationBaseName = "${defApplicationId}_v${defVersion}"
    if ( $False -eq [string]::IsNullOrEmpty( ${defArchSignature} ) )
    {
        [string] $defApplicationBaseName = "${defApplicationBaseName}.${defArchSignature}"
    }
    [string] $defApplicationNsp = "${defIntermediateDirectory}\${defApplicationBaseName}.nsp"
    $(Edit-MakeDirectory ${defWorkDirectory})

    # application nsp 生成
    Invoke-Expression "& `"${defMakeTestApplication}`" --type Application --id 0x${defApplicationId} --ver 0 --output-file-name ${defApplicationBaseName} -o `"${defIntermediateDirectory}`" --work-directory `"${defWorkDirectory}`"" | Write-Host
    $(Test-ExistFile ${defApplicationNsp})

    return ${defApplicationNsp}
}
Export-ModuleMember -Function New-DummyUserApplication

# -----------------------------------------------------------------------
# Make content nsp by the MakeTestApplication tool.
# -----------------------------------------------------------------------
function Out-CreateContentNspByConfigXml( [string]$defOutputDirectoryPath, [string]$defFilePathConfig )
{
    # 入力コンフィグファイルの存在確認
    $(Test-ExistFile ${defFilePathConfig})

    # content nsp 生成
    [string] $defMakeTestApplication = "${NintendoSdkRootPath}\Tools\CommandLineTools\MakeTestApplication\MakeTestApplication.exe"
    [string] $defWorkDirectory = "${defOutputDirectoryPath}\_NspWorkTemporary_"
    $(Edit-MakeDirectory ${defWorkDirectory})
    Invoke-Expression "& `"${defMakeTestApplication}`" -j 4 -o `"${defOutputDirectoryPath}`" --config-xml `"${defFilePathConfig}`" --work-directory `"${defWorkDirectory}`""  | Write-Host
}
Export-ModuleMember -Function Out-CreateContentNspByConfigXml

# -----------------------------------------------------------------------
# Make system update nsp by the AuthoringTool.
# -----------------------------------------------------------------------
function Out-CreateSystemUpdateNsp( [string]$defOutputDirectoryPath, [string]$defTitleBaseName, [string]$defMetaFilePath )
{
    # 入力メタファイルの存在確認
    $(Test-ExistFile ${defMetaFilePath})

    # system updatensp 生成
    [string] $defNspFile = "${defOutputDirectoryPath}\${defTitleBaseName}.nsp"
    [string] $defAuthoringTool = "${NintendoSdkRootPath}\Tools\CommandLineTools\AuthoringTool\AuthoringTool.exe"
    [string] $defKeyConfig = "${NintendoSdkRootPath}\Tools\CommandLineTools\AuthoringTool\AuthoringTool.ocean.keyconfig.xml"
    Invoke-Expression "& `"${defAuthoringTool}`" creatensp --type SystemUpdate --meta `"${defMetaFilePath}`" --keyconfig `"${defKeyConfig}`" -o `"${defNspFile}`"" | Write-Host
    $(Test-ExistFile ${defNspFile})
    return ${defNspFile}
}
Export-ModuleMember -Function Out-CreateSystemUpdateNsp
