﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.Collections.Generic;
using System.Linq;
using System.Net;

namespace CsTestAssistants
{
    //!----------------------------------------------------------------------------
    /// <summary>
    /// ろーぐ
    /// </summary>
    /// <see cref="System.Console.OutputEncoding"/>
    //!----------------------------------------------------------------------------
    public static class Log
    {
        public static bool Verbose { set; get; } = true;
        public static bool WithTrace { set; get; } = true;
        public static string Prefix { set; get; } = "[CsTestAssistants]";

        public static void WriteLine( string message, params object[] args )
        {
            if ( Verbose )
            {
                string text = string.Format( string.IsNullOrEmpty( Prefix ) ? message : ( Prefix + " " + message ), args );
                System.Console.WriteLine( text );
                if ( WithTrace )
                {
                    System.Diagnostics.Trace.WriteLine( text );
                }
            }
        }

        public static void WriteLine( string message )
        {
            if ( Verbose )
            {
                string text = string.IsNullOrEmpty( Prefix ) ? message : ( Prefix + " " + message );
                System.Console.WriteLine( text );
                if ( WithTrace )
                {
                    System.Diagnostics.Trace.WriteLine( text );
                }
            }
        }

        public static void WriteLineAsIs( string message, params object[] args )
        {
            if ( Verbose )
            {
                var text = string.Format( message, args );
                System.Console.WriteLine( text );
                if ( WithTrace )
                {
                    System.Diagnostics.Trace.WriteLine( text );
                }
            }
        }

        public static void WriteLineAsIs( string message )
        {
            if ( Verbose )
            {
                System.Console.WriteLine( message );
                if ( WithTrace )
                {
                    System.Diagnostics.Trace.WriteLine( message );
                }
            }
        }
    }

    public class ScopeLog : System.IDisposable
    {
        public string BeginFormat { get; }
        public string EndFormat { get; }
        public string DisposeForamt { get; }
        public string ScopeName { get; }
        private List<System.Action> Disposer;

        public ScopeLog( string begin, string end, string dispose, string name )
        {
            BeginFormat = begin;
            EndFormat = end;
            DisposeForamt = dispose;
            ScopeName = name;

            Disposer = new List<System.Action>();
            WriteBeginLabel();
        }

        public void Dispose()
        {
            if ( Disposer.Count > 0 )
            {
                Log.WriteLineAsIs( System.Environment.NewLine );
                WriteDisposeLabel();
                foreach ( var action in Disposer )
                {
                    ThrowFrameworks.SkipThrow( action );
                }
                Disposer.Clear();
            }
            WriteEndLabel();
        }

        /// <summary>
        /// 後始末処理を追加します。Dispose メソッドが呼ばれた際に実行されます。
        /// </summary>
        /// <param name="action">後始末処理</param>
        public void AddDisposer( System.Action action )
        {
            Disposer.Add( action );
        }

        /// <summary>
        /// 開始ラベルを出力します。
        /// </summary>
        public virtual void WriteBeginLabel()
        {
            Log.WriteLine( BeginFormat, ScopeName );
        }

        /// <summary>
        /// 終了ラベルを出力します。
        /// </summary>
        public virtual void WriteEndLabel()
        {
            Log.WriteLine( EndFormat, ScopeName );
        }

        /// <summary>
        /// 後始末ラベルを出力します。
        /// </summary>
        public virtual void WriteDisposeLabel()
        {
            Log.WriteLine( DisposeForamt, ScopeName );
        }

        /// <summary>
        /// 接頭辞にスコープ名とタイムスタンプを付けてログを出力します。引数の最後尾にスコープ名が付与されます。
        /// </summary>
        /// <param name="format">書式</param>
        /// <param name="args">引数</param>
        public void WriteLineWithTimeStamp( string format, params object[] args )
        {
            Log.WriteLine( $"{ScopeName} {System.DateTime.Now} " + format, args.Concat( new object[] { ScopeName } ).ToArray() );
        }

        /// <summary>
        /// 接頭辞を付けてログを出力します。引数の最後尾にスコープ名が付与されます。
        /// </summary>
        /// <param name="format">書式</param>
        /// <param name="args">引数</param>
        public void WriteLine( string format, params object[] args )
        {
            Log.WriteLine( format, args.Concat( new object[] { ScopeName } ).ToArray() );
        }

        /// <summary>
        /// 接頭辞なしでログを出力します。引数の最後尾にスコープ名が付与されます。
        /// </summary>
        /// <param name="format">書式</param>
        /// <param name="args">引数</param>
        public void WriteLineAsIs( string format, params object[] args )
        {
            Log.WriteLineAsIs( format, args.Concat( new object[] { ScopeName } ).ToArray() );
        }
    }

    //!----------------------------------------------------------------------------
    /// <summary>
    /// 期待外例外
    /// </summary>
    //!----------------------------------------------------------------------------
    public class UnexpectFailureException : System.Exception
    {
        public UnexpectFailureException() : base( "Unexpect failure happend." )
        {
        }

        public UnexpectFailureException( string message ) : base( message )
        {
        }
    }

    //!----------------------------------------------------------------------------
    /// <summary>
    /// 評価支援
    /// 条件を満たさなければ throw new TException() が発生します。
    /// </summary>
    /// <typeparam name="TException">次の条件を満たした例外クラス型。
    /// * System.Exception を継承している。
    /// * デフォルトコンストラクタを持つ。
    /// </typeparam>
    //!----------------------------------------------------------------------------
    public class ThrowEvaluator<TException> where TException : System.Exception, new()
    {
        /// <summary>
        /// actual が true でなければ throw が発生します
        /// </summary>
        /// <param name="actual"></param>
        public static void IsTrue( bool actual )
        {
            IsEqual( true, actual );
        }

        /// <summary>
        /// actual が false でなければ throw が発生します
        /// </summary>
        /// <param name="actual"></param>
        public static void IsFalse( bool actual )
        {
            IsEqual( false, actual );
        }

        /// <summary>
        /// ( expect == actual ) が成立しなければ throw が発生します
        /// </summary>
        /// <typeparam name="T"></typeparam>
        /// <param name="expect"></param>
        /// <param name="actual"></param>
        public static void IsEqual<T>( T expect, T actual ) where T : System.IEquatable<T>
        {
            if ( false == expect.Equals( actual ) )
            {
                throw new TException();
            }
        }

        /// <summary>
        /// ( lhs &gt; rhs ) が成立しなければ throw が発生します
        /// </summary>
        /// <typeparam name="T"></typeparam>
        /// <param name="lhs"></param>
        /// <param name="rhs"></param>

        public static void IsGreater<T>( T lhs, T rhs ) where T : System.IComparable<T>
        {
            if ( lhs.CompareTo( rhs ) <= 0 )
            {
                throw new TException();
            }
        }

        /// <summary>
        /// ( lhs &gt;= rhs ) が成立しなければ throw が発生します
        /// </summary>
        /// <typeparam name="T"></typeparam>
        /// <param name="lhs"></param>
        /// <param name="rhs"></param>
        public static void IsGreaterThan<T>( T lhs, T rhs ) where T : System.IComparable<T>
        {
            if ( lhs.CompareTo( rhs ) < 0 )
            {
                throw new TException();
            }
        }

        /// <summary>
        /// ( lhs &lt; rhs ) が成立しなければ throw が発生します
        /// </summary>
        /// <typeparam name="T"></typeparam>
        /// <param name="lhs"></param>
        /// <param name="rhs"></param>
        public static void IsLesser<T>( T lhs, T rhs ) where T : System.IComparable<T>
        {
            if ( lhs.CompareTo( rhs ) >= 0 )
            {
                throw new TException();
            }
        }

        /// <summary>
        /// ( lhs &lt;= rhs ) が成立しなければ throw が発生します
        /// </summary>
        /// <typeparam name="T"></typeparam>
        /// <param name="lhs"></param>
        /// <param name="rhs"></param>
        public static void IsLesserThan<T>( T lhs, T rhs ) where T : System.IComparable<T>
        {
            if ( lhs.CompareTo( rhs ) > 0 )
            {
                throw new TException();
            }
        }
    }

    //!----------------------------------------------------------------------------
    /// <summary>
    /// 例外を用いたフレームワーク
    /// </summary>
    //!----------------------------------------------------------------------------
    public static class ThrowFrameworks
    {
        //!----------------------------------------------------------------------------
        /// <summary>
        /// doAction で指定した処理中の例外を System.Exception でキャッチして例外の放出をスキップします。
        /// </summary>
        /// <param name="doAction">例外発生を無視したい処理ブロック</param>
        //!----------------------------------------------------------------------------
        public static void SkipThrow( System.Action doAction )
        {
            try
            {
                doAction();
            }
            catch ( System.Exception e )
            {
                Log.WriteLine( "Catched exception in the scope of skip throws : {0}", e.Message );
            }
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// 遅延実行
        /// </summary>
        //!----------------------------------------------------------------------------
        public static System.Threading.Tasks.Task DelayRun( int millisecondsTimeout, System.Action doAction )
        {
            return System.Threading.Tasks.Task.Run( () =>
            {
                // 実施前遅延
                if ( 0 < millisecondsTimeout )
                {
                    System.Threading.Thread.Sleep( millisecondsTimeout );
                }
                try
                {
                    doAction();
                }
                catch ( System.Exception e )
                {
                    Log.WriteLine( $"Detected exception on sub thread => {e.ToString()}" );
                }
            } );
        }
    }

    //!----------------------------------------------------------------------------
    /// <summary>
    /// 範囲定義支援.
    /// Min &lt;= Max が保証されます。
    /// </summary>
    /// <typeparam name="TValue">要素値型</typeparam>
    //!----------------------------------------------------------------------------
    public class Range<TValue> where TValue : System.IComparable<TValue>
    {
        public TValue Min { get; }
        public TValue Max { get; }

        public Range( TValue min, TValue max )
        {
            ThrowEvaluator<UnexpectFailureException>.IsGreaterThan( max, min );
            Min = min;
            Max = max;
        }

        public Range( Range<TValue> other )
            : this( other.Min, other.Max )
        {
        }
    }

    //!----------------------------------------------------------------------------
    /// <summary>
    /// ベーシック認証用アカウント構成
    /// </summary>
    //!----------------------------------------------------------------------------
    public class BasicAccount
    {
        public string ID { get; }
        public string PASSWORD { get; }

        public BasicAccount( string id, string password )
        {
            ID = id;
            PASSWORD = password;
        }

        public bool Equals( BasicAccount other )
        {
            return ( this == other || ( null != other && ID.Equals( other.ID ) && PASSWORD.Equals( other.PASSWORD ) ) );
        }

        public string ToAuthorizationBase64( string encoding = "ISO-8859-1" )
        {
            var b = new System.Text.StringBuilder().Append( ID ).Append( ':' ).Append( PASSWORD );
            return System.Convert.ToBase64String( System.Text.Encoding.GetEncoding( encoding ).GetBytes( b.ToString() ) );
        }
    }

    //!----------------------------------------------------------------------------
    /// <summary>
    /// コンソールエンコーディングコンテキスト。
    /// 局所的なエンコーディング変更などに用います。
    /// Console のエンコーディング状態との同期保証はありません。
    /// </summary>
    //!----------------------------------------------------------------------------
    public class EncodeContext
    {
        public static readonly System.Text.Encoding ConsoleInitialEncoding = System.Console.OutputEncoding;
        public static readonly System.Text.Encoding UTF8_NoBOM = new System.Text.UTF8Encoding( false );
        public static readonly System.Text.Encoding UTF8 = System.Text.Encoding.UTF8;
        public static readonly System.Text.Encoding ASCII = System.Text.Encoding.ASCII;
        public static readonly System.Text.Encoding UTF16 = System.Text.Encoding.Unicode;
        public static readonly System.Text.Encoding UTF32 = System.Text.Encoding.UTF32;


        /// <summary>
        /// このコンテキストが持つ復元対象エンコーディング。
        /// </summary>
        private System.Text.Encoding RestoreEncoding;

        /// <summary>
        /// このコンテキストが持つ現在設定しているエンコーディング。
        /// </summary>
        public System.Text.Encoding CurrentEncoding { get; set; }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// コンストラクタ
        /// </summary>
        //!----------------------------------------------------------------------------
        public EncodeContext()
        {
            RestoreEncoding = null;
            CurrentEncoding = System.Console.OutputEncoding;
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// デストラクタ
        /// </summary>
        //!----------------------------------------------------------------------------
        ~EncodeContext()
        {
            Restore();
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// Console.OutputEncoding の状態を指定のエンコーディングに設定します。
        /// Restore メソッドで以前のエンコーディングに復帰できます。
        /// ネストはできません。
        /// </summary>
        /// <param name="newEncoding">新しく設定したいエンコーディングを指定します。</param>
        /// <returns>変更前の復元対象エンコーディングが返されます。</returns>
        /// <see cref="System.Console.OutputEncoding"/>
        /// <exception cref="System.IO.IOException">OutputEncodingアクセス時のIO例外</exception>
        /// <exception cref="System.Security.SecurityException">OutputEncodingアクセス時のセキュリティ例外</exception>
        //!----------------------------------------------------------------------------
        public System.Text.Encoding Apply( System.Text.Encoding newEncoding )
        {
            var pre = System.Console.OutputEncoding;
            System.Console.OutputEncoding = newEncoding;
            CurrentEncoding = newEncoding;
            RestoreEncoding = pre;
            return pre;
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// Console.OutputEncoding の状態を復元対象エンコーディングに設定します。
        /// ネストはできません。
        /// </summary>
        /// <returns>復元したエンコーディング, null の場合、復元対象が設定されていません。</returns>
        /// <see cref="System.Console.OutputEncoding"/>
        /// <exception cref="System.IO.IOException">OutputEncodingアクセス時のIO例外</exception>
        /// <exception cref="System.Security.SecurityException">OutputEncodingアクセス時のセキュリティ例外</exception>
        //!----------------------------------------------------------------------------
        public System.Text.Encoding Restore()
        {
            var restore = RestoreEncoding;
            if ( null != restore )
            {
                RestoreEncoding = null;
                System.Console.OutputEncoding = restore;
            }
            return restore;
        }
    }

    //!----------------------------------------------------------------------------
    /// <summary>
    /// 文字列出力ストリームコンテキスト
    /// </summary>
    //!----------------------------------------------------------------------------
    public class OutputStreams
    {
        public class Stream
        {
            /// <summary>
            /// デフォルトの文字列ストリームデータ格納バッファ初期サイズ( 文字数単位 )
            /// </summary>
            public const int DefaultInitialCapacity = 64 * 1024;

            /// <summary>
            /// 文字列ストリームデータ格納バッファ
            /// </summary>
            /// <remarks>
            /// 本クラスのメソッド群は、本バッファオブジェクトインスタンスにより排他制御を実施しています。
            /// 本クラスメソッド運用時での明示的な排他同期を行いたい場合は、本オブジェクトを lock してください。
            /// </remarks>
            public System.Text.StringBuilder Buffer { get; }

            /// <summary>
            /// デフォルトコンストラクタ
            /// </summary>
            /// <param name="initialCapacity">データ格納バッファ初期サイズ( 文字数単位 )</param>
            public Stream( int initialCapacity = DefaultInitialCapacity )
                : this( new System.Text.StringBuilder( initialCapacity ) )
            {
            }

            /// <summary>
            /// 明示的な受信バッファ指定型コンストラクタ
            /// </summary>
            /// <param name="buffer">データ格納バッファインスタンス</param>
            public Stream( System.Text.StringBuilder buffer )
            {
                Buffer = ( null != buffer ) ? buffer : new System.Text.StringBuilder( DefaultInitialCapacity );
            }

            /// <summary>
            /// コピーコンストラクタ
            /// </summary>
            /// <param name="other">コピー元インスタンス</param>
            public Stream( Stream other )
            {
                if ( null == other )
                {
                    throw new System.ArgumentNullException( "Could not construct the new instance by the specified argument because the null object." );
                }
                Buffer = new System.Text.StringBuilder( other.Buffer.Capacity );
            }

            /// <summary>
            /// バッファ文字列取得.
            /// </summary>
            /// <returns></returns>
            public override string ToString()
            {
                lock ( Buffer )
                {
                    return Buffer.ToString();
                }
            }

            /// <summary>
            /// バッファクリーン.
            /// </summary>
            /// <returns>
            /// 自身インスタンス参照を返します.
            /// </returns>
            public Stream Clear()
            {
                lock ( Buffer )
                {
                    Buffer.Clear();
                }
                return this;
            }

            public Stream AppendLine( string value )
            {
                lock ( Buffer )
                {
                    Buffer.AppendLine( value );
                }
                return this;
            }

            public Stream AppendFormat( string format, params object[] args )
            {
                lock ( Buffer )
                {
                    Buffer.AppendFormat( format, args );
                }
                return this;
            }

            public Stream AppendFormat( string format, object arg0 )
            {
                lock ( Buffer )
                {
                    Buffer.AppendFormat( format, arg0 );
                }
                return this;
            }

            public Stream AppendFormat( string format, object arg0, object arg1 )
            {
                lock ( Buffer )
                {
                    Buffer.AppendFormat( format, arg0, arg1 );
                }
                return this;
            }

            public Stream AppendFormat( string format, object arg0, object arg1, object arg2 )
            {
                lock ( Buffer )
                {
                    Buffer.AppendFormat( format, arg0, arg1, arg2 );
                }
                return this;
            }

            /// <summary>
            /// データ受信イベントレシーバ.
            /// </summary>
            /// <param name="sender">The source of the event.</param>
            /// <param name="e">A System.Diagnostics.DataReceivedEventArgs that contains the event data.</param>
            /// <see cref="System.Diagnostics.DataReceivedEventHandler"/>
            public void OutputDataReceiver( object sender, System.Diagnostics.DataReceivedEventArgs e )
            {
                // The redirect receiver that for work around the buffer overflow.
                // default buffer is 4096 bytes.
                string data;
                if ( false == string.IsNullOrEmpty( data = e.Data ) )
                {
                    AppendLine( data );
                }
            }
        }

        /// <summary>
        /// 標準出力カテゴリ文字列ストリームバッファ
        /// </summary>
        public Stream Standard { get; }

        /// <summary>
        /// 標準エラー出力カテゴリ文字列ストリームバッファ
        /// </summary>
        public Stream Error { get; }

        /// <summary>
        /// デフォルトコンストラクタ
        /// </summary>
        /// <param name="initialCapacityOnStandard">標準出力カテゴリの受信データバッファサイズ( 文字数単位 )</param>
        /// <param name="initialCapacityOnError">標準エラー出力カテゴリの受信データバッファサイズ( 文字数単位 )</param>
        public OutputStreams( int initialCapacityOnStandard = Stream.DefaultInitialCapacity, int initialCapacityOnError = Stream.DefaultInitialCapacity )
            : this( new Stream( initialCapacityOnStandard ), new Stream( initialCapacityOnError ) )
        {
        }

        /// <summary>
        /// 明示的な受信バッファ指定型コンストラクタ
        /// </summary>
        /// <param name="standard">標準出力カテゴリの受信データバッファインスタンス</param>
        /// <param name="error">標準エラー出力カテゴリの受信データバッファインスタンス</param>
        /// <see cref="Stream"/>
        public OutputStreams( Stream standard, Stream error )
        {
            Standard = ( null != standard ) ? standard : new Stream();
            Error = ( null != error ) ? error : new Stream();
        }

        /// <summary>
        /// コピーコンストラクタ
        /// </summary>
        /// <param name="other"></param>
        public OutputStreams( OutputStreams other )
        {
            if ( null == other )
            {
                throw new System.ArgumentNullException( "Could not construct the new instance by the specified argument because the null object." );
            }
            Standard = new Stream( other.Standard );
            Error = new Stream( other.Error );
        }

        /// <summary>
        /// ストリームクリーン.
        /// </summary>
        /// <returns></returns>
        public OutputStreams Clear()
        {
            Standard.Clear();
            Error.Clear();
            return this;
        }
    }

    //!----------------------------------------------------------------------------
    /// <summary>
    /// 外部コマンド実行支援
    /// </summary>
    //!----------------------------------------------------------------------------
    public static class CommandLineExecutor
    {
        //!----------------------------------------------------------------------------
        /// <summary>
        /// オプション
        /// </summary>
        //!----------------------------------------------------------------------------
        public class Options
        {
            /// <summary>
            /// コマンド実行結果のログをコンソールに出力するかを設定します.
            /// </summary>
            public bool EnableLogsOutputToConsole { get; set; }

            /// <summary>
            /// Process.StartInfo.StandardOutputEncoding, Process.StartInfo.StandardErrorEncoding に渡すエンコードを指定します.
            /// </summary>
            public readonly System.Text.Encoding LogsEncoding;

            //!----------------------------------------------------------------------------
            /// <summary>
            /// デフォルトコンストラクタ
            /// </summary>
            /// <param name="enableLogOut">プロセスの出力内容を Logコンソールに自動出力するかを設定します.</param>
            /// <param name="logsEncoding">Process.StartInfo.StandardOutputEncoding, Process.StartInfo.StandardErrorEncoding に渡すエンコードを指定します.
            /// null の場合、Console の初期値が設定されます.
            /// 現在の Console.OutputEncoding ではない事に注意してください.
            /// これは、実行対象プロセスで動作するプログラムがホストコンソールエンコードに依存している事が多いためです.
            /// 本エンコードで受信した対象プロセスから出力は、本クラスを運用する自プロセスの Console出力を用いる事で、
            /// 自プロセスの Consoleエンコードに変換して出力されます.
            /// ExecuteOnProcess ( ConsoleInitialEncoding, 仮にCP932 )
            ///   ↓ CP932 で Unicode StringBuilder へ受信.
            /// Console.WriteLine( 仮に出力を UTF8 )
            ///   ↓ UTF8 で Consoleへ出力.
            /// Host Console.
            /// </param>
            //!----------------------------------------------------------------------------
            public Options( bool enableLogOut = true, System.Text.Encoding logsEncoding = null )
            {
                EnableLogsOutputToConsole = enableLogOut;
                LogsEncoding = ( null != logsEncoding ) ? logsEncoding : EncodeContext.ConsoleInitialEncoding;
            }
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// デフォルトオプション
        /// </summary>
        //!----------------------------------------------------------------------------
        public static readonly Options DefaultOptions = new Options();

        //!----------------------------------------------------------------------------
        /// <summary>
        /// コマンドプロセス実行
        /// </summary>
        /// <param name="executionFileName">実行したいコマンドパス.</param>
        /// <param name="executionArguments">実行したいコマンドに対して渡す引数.</param>
        /// <param name="output">null の場合、内部で一時バッファが生成されます.</param>
        /// <param name="options">null の場合、DefaultOptions が採用されます.</param>
        /// <returns></returns>
        //!----------------------------------------------------------------------------
        public static int ExecuteOnProcess( string executionFileName, string executionArguments, OutputStreams output = null, Options options = null )
        {
            options = ( null != options ) ? options : DefaultOptions;
            output = ( null != output ) ? output : new OutputStreams();
            var standardOutput = output.Standard.Clear();
            var standardError = output.Error.Clear();

            // 実施対象の処理が長いと結果表示までかかるので、まず開始したことをTraceだけに通知。
            // Console は自動テスト用の整形です。
            if ( true == Log.WithTrace )
            {
                System.Diagnostics.Trace.WriteLine( string.Format( "\nExe={0}\nArg={1}\n", executionFileName, executionArguments ) );
            }

            using ( var process = new System.Diagnostics.Process() )
            {
                // CommandLine execution application.
                process.StartInfo.FileName = executionFileName;

                // CommandLine arguments.
                process.StartInfo.Arguments = executionArguments;

                // Process configurations.
                process.StartInfo.CreateNoWindow = true;
                process.StartInfo.UseShellExecute = false;
                process.StartInfo.RedirectStandardError = true;
                process.StartInfo.RedirectStandardOutput = true;
                process.StartInfo.StandardErrorEncoding = options.LogsEncoding;
                process.StartInfo.StandardOutputEncoding = options.LogsEncoding;
                process.OutputDataReceived += standardOutput.OutputDataReceiver;
                process.ErrorDataReceived += standardError.OutputDataReceiver;

                // Start process.
                process.Start();
                process.BeginOutputReadLine();
                process.BeginErrorReadLine();
                int processId = process.Id;
                standardOutput.AppendFormat(
                    ">>> process [ {0} ], start.\nFile Name = {1}\nArguments = {2}\n",
                    processId, executionFileName, executionArguments
                );

                // Wait for finish that the process.
                process.WaitForExit();
                int exitCode = process.ExitCode;
                standardOutput.AppendFormat(
                    ">>> process [ {0} ], finish, ExitCode = {1}.",
                    processId, exitCode
                );

                if ( options.EnableLogsOutputToConsole )
                {
                    var stdout = standardOutput.ToString();
                    if ( false == string.IsNullOrEmpty( stdout ) )
                    {
                        Log.WriteLine( "========== Standard Output Begin ==========" );
                        Log.WriteLineAsIs( "{0}", stdout ); // stdout内に {} があるとエラーになる
                        Log.WriteLine( "========== Standard Output End ==========" );
                    }
                    var stderr = standardError.ToString();
                    if ( false == string.IsNullOrEmpty( stderr ) )
                    {
                        Log.WriteLine( "========== Standard Error Begin ==========" );
                        Log.WriteLineAsIs( "{0}", stderr ); // stderr内に {} があるとエラーになる
                        Log.WriteLine( "========== Standard Error End ==========" );
                    }
                }
                return exitCode;
            }
        }
    }

    //!----------------------------------------------------------------------------
    /// <summary>
    /// HttpWebRequest 制御支援
    /// </summary>
    //!----------------------------------------------------------------------------
    public static class HttpExtension
    {
        //!----------------------------------------------------------------------------
        /// <summary>
        /// 接続結果
        /// </summary>
        //!----------------------------------------------------------------------------
        public class ConnectionResult
        {
            /// <summary>
            /// HTTPステータス ( enum )
            /// </summary>
            public HttpStatusCode StatusCode { get; }

            /// <summary>
            /// レスポンステキスト
            /// </summary>
            public string Text { get; }

            /// <summary>
            /// コンストラクタ
            /// </summary>
            internal ConnectionResult( HttpStatusCode status, string text )
            {
                StatusCode = status;
                Text = text;
            }

            /// <summary>
            /// ToString
            /// </summary>
            /// <returns></returns>
            public override string ToString()
            {
                return string.Format( "{0} => [ Status={1}, Text={2} ]", base.ToString(), StatusCode, Text );
            }
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// レスポンスオブジェクトのストリーム読み取り
        /// </summary>
        /// <param name="response"></param>
        /// <param name="requestConditionMessage"></param>
        /// <returns></returns>
        //!----------------------------------------------------------------------------
        private static ConnectionResult ReadResponse( WebResponse response, string requestConditionMessage )
        {
            using ( var httpResponse = response as HttpWebResponse )
            {
                var status = httpResponse.StatusCode;
                Log.WriteLine( $"{requestConditionMessage} => Received response: [ StatusCode={status} ]" );
                using ( var reader = new System.IO.StreamReader( httpResponse.GetResponseStream() ) )
                {
                    return new ConnectionResult( status, reader.ReadToEnd() );
                }
            }
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// 同期型サーバ接続要求実施.
        /// </summary>
        /// <param name="request">接続要求ハンドル</param>
        /// <param name="result">レスポンスストリームが読み取れた場合、結果が格納されます。読み取れない場合は null が格納されます。</param>
        /// <returns>
        /// 例外により要求が失敗した場合の例外ステータスコードが返されます。
        /// レスポンスストリームが存在する場合は result に格納されます。
        /// </returns>
        //!----------------------------------------------------------------------------
        public static WebExceptionStatus ConnectToServer( this HttpWebRequest request, out ConnectionResult result )
        {
            string requestConditionMessage = $"HttpWebRequest: [ Method={request.Method}, URI={request.Address} ]";
            Log.WriteLine( $"{requestConditionMessage} => Connecting..." );
            try
            {
                result = ReadResponse( request.GetResponse(), requestConditionMessage );
                return WebExceptionStatus.Success;
            }
            catch ( WebException e )
            {
                WebResponse r;
                result = ( null != ( r = e.Response ) && r is HttpWebResponse ) ? ReadResponse( e.Response, requestConditionMessage ) : null;
                return e.Status;
            }
        }

        /// <summary>
        /// 接続要求ストリームへの書き出し支援メソッド.
        /// Web例外以外は外部へスローされます.
        /// </summary>
        /// <param name="request">接続要求ハンドル</param>
        /// <param name="formData">書き出し対象データ文字列</param>
        /// <returns>Web例外により要求が失敗した場合の例外ステータスコードが返されます。</returns>
        public static WebExceptionStatus WriteToStream( this HttpWebRequest request, string formData )
        {
            if ( false == string.IsNullOrEmpty( formData ) )
            {
                try
                {
                    byte[] buffer = System.Text.Encoding.UTF8.GetBytes( formData );
                    using ( var stream = request.GetRequestStream() )
                    {
                        stream.Write( buffer, 0, buffer.Length );
                    }
                }
                catch ( WebException e )
                {
                    string requestConditionMessage = $"HttpWebRequest: [ Method={request.Method}, URI={request.Address} ]";
                    Log.WriteLine( $"{requestConditionMessage} => Write to stream failed..." );

                    WebResponse r;
                    if ( null != ( r = e.Response ) && r is HttpWebResponse )
                    {
                        var result = ReadResponse( r, requestConditionMessage );
                        Log.WriteLine( $"WebException: [Code={result.StatusCode}, Text={result.Text}" );
                    }
                    return e.Status;
                }
            }
            return WebExceptionStatus.Success;
        }
    }

    //!----------------------------------------------------------------------------
    /// <summary>
    /// タイムアウト制御支援
    /// </summary>
    //!----------------------------------------------------------------------------
    public static class Timeout
    {
        public const double Infinity = 0.0;

        /// <summary>
        /// タイムアウト監視
        /// </summary>
        public class Observer
        {
            private static readonly System.TimeSpan SpanInfinity = System.TimeSpan.Zero;

            public System.TimeSpan TimeoutSpan { get; }
            private System.DateTime ElapsedStart { get; set; }

            public static Observer FromSeconds( double seconds )
            {
                return new Observer( System.TimeSpan.FromSeconds( seconds ) );
            }

            public static Observer FromMilliseconds( double milliseconds )
            {
                return new Observer( System.TimeSpan.FromMilliseconds( milliseconds ) );
            }

            internal Observer( System.TimeSpan timeoutSpan )
            {
                TimeoutSpan = ( null == timeoutSpan ) ? SpanInfinity : timeoutSpan;
                ElapsedStart = System.DateTime.Now;
            }

            public void Reset()
            {
                ElapsedStart = System.DateTime.Now;
            }

            public bool IsTimeout()
            {
                return ( TimeoutSpan > SpanInfinity && ( System.DateTime.Now - ElapsedStart ) >= TimeoutSpan );
            }
        }
    }

    //!----------------------------------------------------------------------------
    /// <summary>
    /// 再試行管理支援
    /// </summary>
    //!----------------------------------------------------------------------------
    public static class Retry
    {
        public class Watcher
        {
            /// <summary>
            /// 開始時刻
            /// </summary>
            public System.DateTime StartTime { get; } = System.DateTime.Now;
            public System.TimeSpan ElapsedTime { get { return System.DateTime.Now - StartTime; } }

            /// <summary>
            /// 終了時刻
            /// </summary>
            private System.DateTime _endtime = System.DateTime.MinValue;
            public System.DateTime EndTime { get { return _endtime; } }

            /// <summary>
            /// 実行間隔
            /// </summary>
            private int _interval = 0;  // sec
            public int IntervalSeconds { get { return _interval; } }
            public int IntervalMilliseconds { get { return _interval * 1000; } }

            /// <summary>
            /// タイムアウト
            /// </summary>
            private int _timeout = 0;   // sec
            public int TimeoutSeconds { get { return _timeout; } }
            public int TimeoutMilliseconds { get { return _timeout * 1000; } }
            public int RemainingSeconds
            {
                get { return _timeout > 0 ? ( int )( TimeoutSeconds - ElapsedTime.TotalSeconds ) : int.MaxValue; }
            }
            public int RemainingMilliseconds
            {
                get { return _timeout > 0 ? ( int )( TimeoutMilliseconds - ElapsedTime.TotalMilliseconds ) : int.MaxValue; }
            }

            /// <summary>
            /// 試行回数
            /// </summary>
            private int _count = 0;
            private int _limit = 0;
            public int TryCount { get { return _count; } }
            public int LimitCount { get { return _limit; } }
            public int RemainingCount
            {
                get { return _limit > 0 ? ( _limit - _count ) : int.MaxValue; }
            }

            /// <summary>
            /// 継続判定
            /// </summary>
            public bool IsContinuable
            {
                get
                {
                    return ( RemainingMilliseconds > 0 ) && ( RemainingCount > 0 );
                }
            }

            /// <summary>
            /// コンストラクタ。タイムアウトに 0 を指定すると永続的に継続可能となります。
            /// </summary>
            /// <param name="interval">実行間隔</param>
            /// <param name="timeout">タイムアウト秒</param>
            public Watcher( int interval, int timeout )
            {
                if ( interval < 0 )
                {
                    throw new System.ArgumentOutOfRangeException( "interval" );
                }
                if ( timeout < 0 )
                {
                    throw new System.ArgumentOutOfRangeException( "timeout" );
                }
                _interval = interval;
                _timeout = timeout;
            }

            /// <summary>
            /// コンストラクタ。上限回数、タイムアウトに 0 を指定すると永続的に継続可能となります。
            /// </summary>
            /// <param name="interval">実行間隔</param>
            /// <param name="limit">上限回数</param>
            /// <param name="timeout">タイムアウト秒</param>
            public Watcher( int interval, int limit, int timeout )
            {
                if ( interval < 0 )
                {
                    throw new System.ArgumentOutOfRangeException( "interval" );
                }
                if ( limit < 0 )
                {
                    throw new System.ArgumentOutOfRangeException( "limit" );
                }
                if ( timeout < 0 )
                {
                    throw new System.ArgumentOutOfRangeException( "timeout" );
                }
                _interval = interval;
                _limit = limit;
                _timeout = timeout;
            }

            /// <summary>
            /// 実行間隔の待機します。
            /// </summary>
            public void WaitInterval()
            {
                System.Threading.Thread.Sleep( IntervalMilliseconds );
            }

            /// <summary>
            /// 試行回数を増やして、継続確認を行います。
            /// </summary>
            /// <returns>継続可能なら true を返します。</returns>
            public bool TryContinue()
            {
                var next = IsContinuable;
                _count++;
                return next;
            }

            /// <summary>
            /// 終了時刻を記録します。
            /// </summary>
            public void RecordEndTime()
            {
                _endtime = System.DateTime.Now;
            }

            /// <summary>
            /// 文字列化
            /// </summary>
            /// <returns>インスタンスの文字列表現を返します。</returns>
            public override string ToString()
            {
                var sb = new System.Text.StringBuilder();
                sb.Append( $"StartTime:          {StartTime}\n" );
                sb.Append( $"EndTime:            {EndTime}\n" );
                sb.Append( $"ElapsedTime:        {ElapsedTime}\n" );
                sb.Append( $"IntervalSeconds:    {IntervalSeconds}\n" );
                sb.Append( $"TimeoutSeconds:     {TimeoutSeconds}\n" );
                sb.Append( $"RemainingSeconds:   {RemainingSeconds}\n" );
                sb.Append( $"TryCount:           {TryCount}\n" );
                sb.Append( $"LimitCount:         {LimitCount}\n" );
                sb.Append( $"RemainingCount:     {RemainingCount}\n" );
                return sb.ToString();
            }
        }

        /// <summary>
        /// 要素単位リクエスト管理支援.
        /// </summary>
        public class Observer<TRetryValue> : Dictionary<TRetryValue, int>
        {
            public int LeftoverTrialCount { get; }
            public List<TRetryValue> Queue { get; }

            public Observer( int leftoverTrialCount, int initialQueueCapacity = 32 )
            {
                LeftoverTrialCount = leftoverTrialCount;
                Queue = new List<TRetryValue>( initialQueueCapacity );
            }

            private bool RequestQueue( TRetryValue value, bool requestQueueTop = false )
            {
                if ( false == Queue.Contains( value ) )
                {
                    if ( requestQueueTop )
                    {
                        Queue.Insert( 0, value );
                    }
                    else
                    {
                        Queue.Add( value );
                    }
                    return true;
                }
                return false;
            }

            /// <summary>
            /// 再試行要求。
            /// </summary>
            /// <param name="path">再試行対象値</param>
            /// <param name="requestQueueTop">キュートップに登録します</param>
            /// <returns>残り試行可能回数を返します。負値の場合、試行できなかった事を意味します。</returns>
            public int TryRetry( TRetryValue value, bool requestQueueTop = false )
            {
                value = OnCorrectValue( value );
                int count;
                if ( false == TryGetValue( value, out count ) )
                {
                    count = LeftoverTrialCount;
                    if ( RequestQueue( value, requestQueueTop ) )
                    {
                        Add( value, --count );
                    }
                }
                else if ( count > 0 )
                {
                    if ( RequestQueue( value, requestQueueTop ) )
                    {
                        this[ value ] = --count;
                    }
                }
                else
                {
                    --count;
                }
                return count;
            }

            /// <summary>
            /// 次の試行対象パスを取得します。
            /// </summary>
            /// <returns></returns>
            public TRetryValue QueryNext()
            {
                TRetryValue result = default( TRetryValue );
                if ( Queue.Count > 0 )
                {
                    result = Queue[ 0 ];
                    Queue.RemoveAt( 0 );
                }
                return result;
            }

            /// <summary>
            /// 入力試行値補正を行います。
            /// </summary>
            /// <param name="inValue"></param>
            /// <returns></returns>
            protected virtual TRetryValue OnCorrectValue( TRetryValue inValue )
            {
                return inValue;
            }
        }
    }
}
