﻿<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        起床中に BGNUP を開始してスリープ ---> 即半起床後にダウンロードタスクが完了するか
        のテストを行います。

    .DESCRIPTION
        Setup-TargetEnvironment, Setup-SystemUpdate の事前実行が必要です。

    .PARAMETER Target
        ControlTarget, RunOnTarget の -t オプションに渡す文字列
#>

param
(
    [Parameter(Mandatory = $true)]
    [string]
    $Target,

    [switch]
    $AgingTest
)

$ErrorActionPreference = 'Stop'

Set-StrictMode -Version Latest

$ScriptPath      = $MyInvocation.MyCommand.Path
$ScriptDirectory = Split-Path $ScriptPath

# --------------------------------------------------------------------------
# 共通モジュール読み込み & 定数定義など
# --------------------------------------------------------------------------
Import-Module "$ScriptDirectory\Common"
. "$ScriptDirectory\Constants.ps1"
. "$ScriptDirectory\Functions.ps1"


# --------------------------------------------------------------------------
# テスト
# --------------------------------------------------------------------------
$TestSuccess = $false

try
{
    # 不具合調査用にデバイス ID を表示させる
    Write-InfoLog "Device ID = $(Get-TargetDeviceId $Target)"

    # NUP が必要であることを確認
    Test-SystemUpdateIsRequired $Target

    # BGNUP 開始
    # ダウンロードの開始を待つ
    Start-SystemUpdateDownload $Target

    for ($i = 0; ; ++$i)
    {
        # 寝る
        Invoke-TargetSleep $Target

        # 何度かスリープさせてから起こす
        if ($AgingTest)
        {
            Wait-SleepCycle $Target -SleepCount 1
            Invoke-TargetWakeForAging $Target
        }
        else
        {
            Wait-SleepCycle $Target
            Invoke-TargetWake $Target
        }

        # 本体更新はダウンロードされているか
        if (Test-SystemUpdateDownloadComplete $Target)
        {
            break
        }
        else
        {
            Write-InfoLog "Download hasn't completed yet, but maybe because of some temporary network failures. Will enter sleep again."
        }
    }

    # ダウンロードされた NUP を適用
    Invoke-DevMenuCommandSystem $Target systemupdate,apply-downloaded

    Write-SuccessLog "TEST SUCCESS"
    $TestSuccess = $true
}
catch
{
    Write-FailureLog "TEST FAILURE"
    throw
}
finally
{
    # スリープから復帰しないことがあるので、テストに成功していないならリセットをかける
    if (-not $TestSuccess)
    {
        Invoke-TargetReset $Target
    }

    Teardown-SystemUpdate $Target
}
