﻿<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        スリープ中に DTL 更新 ---> 途中起床してダウンロードが行われるか
        のテストを行います。

    .DESCRIPTION
        Setup-TargetEnvironment, Setup-TestApplication の事前実行が必要です。

    .PARAMETER Target
        ControlTarget, RunOnTarget の -t オプションに渡す文字列
#>

param
(
    [Parameter(Mandatory = $true)]
    [string]
    $Target,

    [switch]
    $AgingTest
)

$ErrorActionPreference = 'Stop'

Set-StrictMode -Version Latest

$ScriptPath      = $MyInvocation.MyCommand.Path
$ScriptDirectory = Split-Path $ScriptPath

# --------------------------------------------------------------------------
# 共通モジュール読み込み & 定数定義など
# --------------------------------------------------------------------------
Import-Module "$ScriptDirectory\Common"
. "$ScriptDirectory\Constants.ps1"
. "$ScriptDirectory\Functions.ps1"

$DownloadTaskListUpdateWait = 20


# --------------------------------------------------------------------------
# テスト
# --------------------------------------------------------------------------
$TestSuccess = $false

try
{
    # 不具合調査用にデバイス ID を表示させる
    Write-InfoLog "Device ID = $(Get-TargetDeviceId $Target)"

    # NA ID Token 取得
    $NaIdToken = Get-NetworkAccountIdToken $Target
    Write-InfoLog "NA ID Token = $NaIdToken"

    # 寝る
    Invoke-TargetSleep $Target

    # 適当な時間待つ
    Write-InfoLog "idle sleeping for $DownloadTaskListUpdateWait sec."
    Start-Sleep $DownloadTaskListUpdateWait

    # オフデバイス購入時、サーバ証明書の警告が出るのを回避
    [System.Net.ServicePointManager]::ServerCertificateValidationCallback = { $true }

    # オフデバイス購入
    Invoke-OffDeviceDemoDownload $TestApplicationNsUid $NaIdToken

    for ($i = 0; ; ++$i)
    {
        # 何度かスリープさせてから起こす
        if ($AgingTest)
        {
            Wait-SleepCycle $Target -SleepCount 2
            Invoke-TargetWakeForAging $Target
        }
        else
        {
            Wait-SleepCycle $Target -SleepCount 3
            Invoke-TargetWake $Target
        }

        # アプリはダウンロードされているか、バージョンは想定通りか
        if (Test-ApplicationDownloadComplete $Target $TestApplicationId $TestApplicationV0Version -AllowNoAppExistence)
        {
            break
        }
        else
        {
            Write-InfoLog "Download hasn't completed yet, but maybe because of some temporary network failures. Will enter sleep again."
        }

        # 寝る
        Invoke-TargetSleep $Target
    }

    # アプリがちゃんと起動するか確認
    Invoke-RunOnTarget $Target "--failure-timeout",10,$TestApplicationId

    Write-SuccessLog "TEST SUCCESS"
    $TestSuccess = $true
}
catch
{
    Write-FailureLog "TEST FAILURE"
    throw
}
finally
{
    # スリープから復帰しないことがあるので、テストに成功していないならリセットをかける
    if (-not $TestSuccess)
    {
        Invoke-TargetReset $Target
    }

    Teardown-DemoDownload $Target $TestApplicationId
}
