﻿<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        半起床のテストを行うために共通して必要な開発機の準備を行います。

    .DESCRIPTION
        開発機は初期化直後の状態になっていることを想定しています。

    .PARAMETER Target
        ControlTarget, RunOnTarget の -t オプションに渡す文字列

    .PARAMETER NetworkAccountId
        テストに使用する NNID のユーザ ID。無指定の場合は、CI テスト用の NNID が使用されます。

    .PARAMETER NetworkAccountPassword
        テストに使用する NNID のパスワード。無指定の場合は、CI テスト用の NNID が使用されます。

    .PARAMETER NetworkSettingsPath
        SettingsManager にインポートさせるネットワーク接続設定を記述した json ファイルのパス。
        無指定の場合、スクリプトと同じディレクトリに NetworkSettings.local.json というファイルが
        あればそれを使用します。無ければ CI テスト用の json ファイルを使用します。

    .PARAMETER TrialCount
        ネットワーク接続を伴う DevMenuCommand 処理の試行回数。
        2 以上を指定すると、その回数を上限として実行に成功するまで再試行を行います。1 ならば
        再試行を行いません。一時的なネットワーク接続の問題を回避するために使用します。
#>

param
(
    [Parameter(Mandatory = $true)]
    [string]
    $Target,

    [string]
    $NetworkAccountId,

    [string]
    $NetworkAccountPassword,

    [string]
    $NetworkSettingsPath,

    [int]
    $TrialCount
)

$ErrorActionPreference = 'Stop'

Set-StrictMode -Version Latest

$ScriptPath      = $MyInvocation.MyCommand.Path
$ScriptDirectory = Split-Path $ScriptPath

if (-not $NetworkSettingsPath)
{
    if (Test-Path "$ScriptDirectory\NetworkSettings.local.json")
    {
        $NetworkSettingsPath = "$ScriptDirectory\NetworkSettings.local.json"
    }
    elseif (Test-Path "$ScriptDirectory\NetworkSettings.ci.json")
    {
        $NetworkSettingsPath = "$ScriptDirectory\NetworkSettings.ci.json"
    }
}
$NetworkSettingsPath = Resolve-Path $NetworkSettingsPath


# --------------------------------------------------------------------------
# 共通モジュール読み込み & 定数定義
# --------------------------------------------------------------------------
Import-Module "$ScriptDirectory\Common"
. "$ScriptDirectory\Constants.ps1"


# --------------------------------------------------------------------------
# テスト準備 (SDEV 編)
# --------------------------------------------------------------------------
# AP 設定のインポート
Invoke-SettingsManager $Target Import,$NetworkSettingsPath
Invoke-TargetReset $Target

# サービスディスカバリ設定のインポート
Invoke-DevMenuCommandSystem $Target servicediscovery,import-all,td1pass -TrialCount $TrialCount
Invoke-TargetReset $Target

# DA 登録を解除
Invoke-DevMenuCommandSystem $Target shop,unlink-device-all -TrialCount $TrialCount
Invoke-DevMenuCommandSystem $Target shop,unregister-device-account -TrialCount $TrialCount

# アカウント設定
Invoke-DevMenuCommandSystem $Target account,add,--register_nsa -TrialCount $TrialCount
Invoke-DevMenuCommandSystem $Target account,link,--index,0,--id,$NetworkAccountId,--password,$NetworkAccountPassword -TrialCount $TrialCount

# DA 登録、NA の既存権利放棄
Invoke-DevMenuCommandSystem $Target shop,register-device-account -TrialCount $TrialCount
Invoke-DevMenuCommandSystem $Target shop,shop-account-status,0 -TrialCount $TrialCount
Invoke-DevMenuCommandSystem $Target shop,delete-all-rights,0 -TrialCount $TrialCount
Invoke-DevMenuCommandSystem $Target shop,link-device,0 -TrialCount $TrialCount

# DTL が空になっていることを確認
# FIXME: TrialCount に連動して Timeout を設定するべきではない
Wait-ServerDownloadTaskListCleared $Target -Timeout ($TrialCount * 60)

# 半起床の間隔を短くする
Invoke-DevMenuCommandSystem $Target npns,set-sleep-periodic-interval,20
# 半起床の間隔が極端に短い場合は、充電状態の監視機能を切らないと上手く設定されない
Invoke-DevMenuCommandSystem $Target bgtc,config-battery-saver,disable
# 通知受信後のランダム待ちを固定1秒に変更
Invoke-DevMenuCommandSystem $Target debug,set-integer-fwdbg,--name,ns.notification,--key,debug_waiting_limit,1
# 起動時のリクエスト処理を無効化
Invoke-DevMenuCommandSystem $Target debug,set-boolean-fwdbg,--name,ns.notification,--key,enable_request_on_cold_boot,false

Invoke-TargetReset $Target
