﻿<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        本体更新のテストを行うための準備をします。

    .PARAMETER Target
        ControlTarget, RunOnTarget の -t オプションに渡す文字列
#>

param
(
    [Parameter(Mandatory = $true)]
    [string]
    $Target
)

$ErrorActionPreference = 'Stop'

Set-StrictMode -Version Latest

$ScriptPath      = $MyInvocation.MyCommand.Path
$ScriptDirectory = Split-Path $ScriptPath


# --------------------------------------------------------------------------
# 共通モジュール読み込み & 定数定義
# --------------------------------------------------------------------------
Import-Module "$ScriptDirectory\Common"
. "$ScriptDirectory\Constants.ps1"

$AuthoringToolKeyConfigPath = Resolve-Path "$NintendoSdkRootPath\Tools\CommandLineTools\AuthoringTool\AuthoringTool.ocean.keyconfig.xml"
$SystemUpdateNmetaPath = Resolve-Path "$ScriptDirectory\SystemUpdateMeta.nmeta"
$SystemUpdateContentsNmetaPath = Resolve-Path "$ScriptDirectory\SystemUpdateContents.nmeta"
$RopsSubmitRequestEntryPath = Resolve-Path "$ScriptDirectory\RopsSubmitRequestEntry.xml"

$SystemUpdateMetaNspPath = "$IntermediatesDirectory\SystemUpdateMeta.nsp"
$SystemUpdateContentsDirectoryPath = "$IntermediatesDirectory\SystemUpdateDummyData"
$SystemUpdateContentsNspPath = "$IntermediatesDirectory\SystemUpdateContents.nsp"

$SystemUpdateDummyDataSize = 24 * 1024 * 1024

$RopsAuthTokenPath = "$IntermediatesDirectory\RopsAuthToken.json"

function Get-RopsRomIdFromLogs
{
    param
    (
        [Parameter(Mandatory = $true, ValueFromPipeline = $true)]
        [AllowEmptyString()]
        [string[]]
        $Logs
    )

    begin
    {
        $RomId = $null
    }
    process
    {
        $Logs | % {
            if ($_ -match "ROM ID = ([0-9a-f]{32})")
            {
                $RomId = $matches[1]
            }
        }
    }
    end
    {
        if ($RomId)
        {
            $RomId
        }
    }
}

function Upload-Nsp
{
    param
    (
        [Parameter(Mandatory = $true)]
        [string]
        $NspPath,

        [Parameter(Mandatory = $true)]
        [string]
        $NspApplicationId
    )

    # id の最初が 0x で始まっていたら、それを削る。rops は出力で 0x を扱わないので、入力も揃える
    if ($NspApplicationId.StartsWith("0x"))
    {
        $NspApplicationId = $NspApplicationId.Substring("0x".Length)
    }

    $RomId = Invoke-Rops upload,-e,dev6,--token,$RopsAuthTokenPath,-a,$NspApplicationId,-s,$NspPath | Get-RopsRomIdFromLogs

    for ($i = 0; ; ++$i)
    {
        if ($i -ge 10)
        {
            throw "rops upload timeout"
        }

        $ListRomsLogs = Invoke-Rops list-roms,-e,dev6,--token,$RopsAuthTokenPath,-r,$RomId
        $Roms = ConvertFrom-Json20 ($ListRomsLogs -join "`n")
        $Rom = $Roms.roms | ? { $_.application_id -eq $NspApplicationId }

        if ($Rom.status -eq "SUBMITTABLE")
        {
            break
        }

        # 適当に休んでおく
        Start-Sleep 5
    }

    Invoke-Rops submit,-e,dev6,--token,$RopsAuthTokenPath,-r,$RomId,-i,$RopsSubmitRequestEntryPath
}


# --------------------------------------------------------------------------
# テスト準備 (本体更新データアップロード編)
# --------------------------------------------------------------------------
# 古い一時ディレクトリを削除
if (Test-Path $IntermediatesDirectory)
{
    Remove-Item $IntermediatesDirectory -Recurse
}
New-Item $IntermediatesDirectory -ItemType Directory | Out-Null

# システムアップデートメタをつくる
Invoke-AuthoringTool creatensp,--type,SystemUpdate,--meta,$SystemUpdateNmetaPath,-o,$SystemUpdateMetaNspPath,--keyconfig,$AuthoringToolKeyConfigPath

# システムデータに含めるダミーのリソースデータを作る
if (-not (Test-Path $SystemUpdateContentsDirectoryPath))
{
    New-Item $SystemUpdateContentsDirectoryPath -ItemType Directory | Out-Null
}
$FileStream = [System.IO.File]::Open("$SystemUpdateContentsDirectoryPath\SystemUpdateDummyData", "Create")
try
{
    for ($i = 0; $i -lt $SystemUpdateDummyDataSize; ++$i)
    {
        $FileStream.WriteByte(0x24)
    }
}
finally
{
    if ($FileStream)
    {
        $FileStream.Dispose()
    }
}

# システムデータを作る
Invoke-AuthoringTool creatensp,--type,SystemData,--meta,$SystemUpdateContentsNmetaPath,--data,$SystemUpdateContentsDirectoryPath,-o,$SystemUpdateContentsNspPath,--keyconfig,$AuthoringToolKeyConfigPath

# NSD サーバへのアクセス時、サーバ証明書の警告が出るのを回避
# Get-RopsOauthToken の他、Set-SystemUpdateDeliveryConfiguration でも必要
[System.Net.ServicePointManager]::ServerCertificateValidationCallback = { $true }

# rops で使用するトークンを取得
Get-RopsOauthToken -UserName $RopsAuthUserName -SecurePassword (ConvertTo-SecureString $RopsAuthPassword -AsPlainText -Force) | Set-Content $RopsAuthTokenPath

# rops で nsp をアップロード
# ContentsUploader でできるようにしたい...
Upload-Nsp $SystemUpdateMetaNspPath $SystemUpdateId
Upload-Nsp $SystemUpdateContentsNspPath $SystemUpdateContentsId

# デバイス ID を取得して、本体更新の配信設定を書きかえる
$DeviceId = Get-TargetDeviceId $Target
Set-SystemUpdateDeliveryConfiguration $DeliveryConfigurationId $DeviceId $SystemUpdateId $SystemUpdateVersion
