﻿<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        半起床のテストに使用する共通実装を集めたモジュールです。
#>

Set-StrictMode -Version Latest

$ScriptPath      = $MyInvocation.MyCommand.Path
$ScriptDirectory = Split-Path $ScriptPath


# --------------------------------------------------------------------------
# ログ出力系関数
# --------------------------------------------------------------------------
function Write-Log
{
    param
    (
        [Parameter(Mandatory = $true)]
        [string]
        $Log
    )
    Write-Host "$(Get-Date -Format "G") [HostPowerShell] $Log"
}
Export-ModuleMember -Function Write-Log

function Write-InfoLog
{
    param
    (
        [Parameter(Mandatory = $true)]
        [string]
        $Log
    )
    Write-Host -ForegroundColor DarkGray "$(Get-Date -Format "G") [HostPowerShell] $Log"
}
Export-ModuleMember -Function Write-InfoLog

function Write-SuccessLog
{
    param
    (
        [Parameter(Mandatory = $true)]
        [string]
        $Log
    )
    Write-Host -ForegroundColor Green "$(Get-Date -Format "G") [HostPowerShell] $Log"
}
Export-ModuleMember -Function Write-SuccessLog

function Write-FailureLog
{
    param
    (
        [Parameter(Mandatory = $true)]
        [string]
        $Log
    )
    Write-Host -ForegroundColor Red "$(Get-Date -Format "G") [HostPowerShell] $Log"
}
Export-ModuleMember -Function Write-FailureLog

# --------------------------------------------------------------------------
# Invoke-Command
# --------------------------------------------------------------------------
function Invoke-Command
{
    param
    (
        [Parameter(Mandatory = $true)]
        [string[]]
        $Command,

        [Parameter(Mandatory = $true)]
        [string[]]
        $Arguments,

        [switch]
        $NoExitCodeCheck,

        [int[]]
        $SuccessExitCodes
    )

    if (-not $SuccessExitCodes)
    {
        $SuccessExitCodes = @(0)
    }

    Write-InfoLog ("$Command {0}" -f ($Arguments -join " "))

    & $Command @Arguments 2>&1 | % { "$_" }

    if (-not $NoExitCodeCheck -and $SuccessExitCodes -notcontains $LastExitCode)
    {
        throw "Unexpected exit code ($LastExitCode): $Command $($Arguments -join " ")"
    }
}

# --------------------------------------------------------------------------
# NintendoSdkRootPath
# --------------------------------------------------------------------------
function Find-NintendoSdkRootPath
{
    param
    (
    )

    $d = $ScriptDirectory
    while ($d)
    {
        if (Test-Path "$d\NintendoSdkRootMark")
        {
            return $d
        }
        $d = Split-Path $d
    }
    throw "NintendoSdkRootMark cannot be found"
}

$NintendoSdkRootPath = Find-NintendoSdkRootPath
Export-ModuleMember -Variable NintendoSdkRootPath

# --------------------------------------------------------------------------
# プロキシ定義
# --------------------------------------------------------------------------
$HttpProxy = "http://proxy.nintendo.co.jp:8080"

# --------------------------------------------------------------------------
# 各種ツール類のパス
# --------------------------------------------------------------------------
$RunOnTargetPath = Resolve-Path "$NintendoSdkRootPath\Tools\CommandLineTools\RunOnTarget.exe"
$ControlTargetPath = Resolve-Path "$NintendoSdkRootPath\Tools\CommandLineTools\ControlTarget.exe"

$DevMenuCommandSystemPath = Resolve-Path "$NintendoSdkRootPath\Programs\Eris\Outputs\NX-NXFP2-a64\TargetTools\DevMenuCommandSystem\Develop\DevMenuCommandSystem.nsp"

$AuthoringToolPath = Resolve-Path "$NintendoSdkRootPath\Tools\CommandLineTools\AuthoringTool\AuthoringTool.exe"
$ContentsUploaderPath = Resolve-Path "$NintendoSdkRootPath\Tools\CommandLineTools\ContentsUploader\ContentsUploader.exe"
$MakeTestApplicationPath = Resolve-Path "$NintendoSdkRootPath\Tools\CommandLineTools\MakeTestApplication\MakeTestApplication.exe"
$RopsPath = Resolve-Path "$NintendoSdkRootPath\Tools\CommandLineTools\rops\rops.exe"

$SettingsManager = Resolve-Path "$NintendoSdkRootPath\Programs\Eris\Outputs\NX-NXFP2-a64\TargetTools\SettingsManager\Develop\SettingsManager.nsp"

$HalfAwakeLogMonitorPath = Resolve-Path "$NintendoSdkRootPath\Programs\Chris\Outputs\AnyCPU\Tools\NsHalfAwakeTestTools\HalfAwakeLogMonitor\Release\HalfAwakeLogMonitor.exe"

$DefaultAllResultsInfoXmlPath = "$NintendoSdkRootPath\Documents\Outputs\Api\HtmlFull\AllResultsInfo.xml"

# --------------------------------------------------------------------------
# Invoke-RunOnTarget
# --------------------------------------------------------------------------
function Invoke-RunOnTarget
{
    param
    (
        [Parameter(Mandatory = $true)]
        [string]
        $Target,

        [Parameter(Mandatory = $true)]
        [string[]]
        $Arguments,

        [switch]
        $NoExitCodeCheck
    )

    Invoke-Command $RunOnTargetPath ("--target",$Target + $Arguments) -NoExitCodeCheck:$NoExitCodeCheck
}
Export-ModuleMember -Function Invoke-RunOnTarget

# --------------------------------------------------------------------------
# Invoke-DevMenuCommandSystem
# --------------------------------------------------------------------------
function Invoke-DevMenuCommandSystem
{
    param
    (
        [Parameter(Mandatory = $true)]
        [string]
        $Target,

        [Parameter(Mandatory = $true)]
        [string[]]
        $Arguments,

        [int]
        $Timeout,

        [int]
        $TrialCount,

        [int]
        $TrialInterval,

        [switch]
        $NoPatternCheck,

        [switch]
        $NoExitCodeCheck,

        [switch]
        $RetryWithTargetReset
    )

    $Timeout = if ($Timeout) { $Timeout } else { 60 }
    $TrialCount = if ($TrialCount) { $TrialCount } else { 2 }
    $TrialInterval = if ($TrialInterval) { $TrialInterval } else { 60 }

    $FailureTimeoutArgs = if ($Timeout -gt 0) { @("--failure-timeout",$Timeout) } else { @() }
    $PatternCheckArgs = if ($NoPatternCheck) { @() } else { @("--pattern-not-found-failure","\[SUCCESS\]") }
    $RunOnTargetArguments = $FailureTimeoutArgs + $PatternCheckArgs + @("--target",$Target,$DevMenuCommandSystemPath,"--") + $Arguments

    for ($Trial = 1; ; ++$Trial)
    {
        $Logs = Invoke-Command $RunOnTargetPath $RunOnTargetArguments -NoExitCodeCheck
        $Logs

        if ($LastExitCode -eq 0 -or $NoExitCodeCheck)
        {
            break
        }

        $ResultCode = $null
        $ResultIdentifier = "unknown"
        $Logs | % {
            if ($_ -match 'Unexpected error\. result = (0x[0-9a-f]{8})\.')
            {
                $ResultCode = $matches[1]
                $ErrorResult = Find-ResultDefinition $ResultCode
                if ($ErrorResult)
                {
                    $ResultIdentifier = "$($ErrorResult.Namespace)::$($ErrorResult.Name)"
                }
            }
        }

        if ($ResultCode)
        {
            $ErrorMessage = "DevMenuCommandSystem failed. Arguments = '$($Arguments -join " ")', Result = $ResultIdentifier ($ResultCode)"
        }
        else
        {
            $ErrorMessage = "Unexpected exit code ($LastExitCode): $RunOnTargetPath $($RunOnTargetArguments -join " ")"
        }

        if ($Trial -ge $TrialCount)
        {
            throw $ErrorMessage
        }
        else
        {
            Write-InfoLog $ErrorMessage
            Write-InfoLog "DevMenuCommandSystem failed. Will try again in $TrialInterval sec."
        }
        Start-Sleep $TrialInterval

        if ($RetryWithTargetReset)
        {
            Invoke-TargetReset $Target
        }
    }
}
Export-ModuleMember -Function Invoke-DevMenuCommandSystem

# --------------------------------------------------------------------------
# Invoke-TargetSleep
# --------------------------------------------------------------------------
function Invoke-TargetSleep
{
    param
    (
        [Parameter(Mandatory = $true)]
        [string]
        $Target
    )

    Invoke-Command $ControlTargetPath ("press-power-button","--target",$Target)
}
Export-ModuleMember Invoke-TargetSleep

# --------------------------------------------------------------------------
# Invoke-TargetWake
# --------------------------------------------------------------------------
function Invoke-TargetWake
{
    param
    (
        [Parameter(Mandatory = $true)]
        [string]
        $Target,

        [switch]
        $NoWait
    )

    Invoke-Command $ControlTargetPath ("press-power-button","--target",$Target)

    if (-not $NoWait)
    {
        for ($i = 0; ; ++$i)
        {
            if ($i -ge 5)
            {
                throw "Could not connect to target after wake."
            }

            Invoke-Command $ControlTargetPath "connect","--target",$Target -NoExitCodeCheck
            if ($LastExitCode -eq 0)
            {
                break
            }
        }
    }
}
Export-ModuleMember Invoke-TargetWake

# --------------------------------------------------------------------------
# Invoke-TargetReset
# --------------------------------------------------------------------------
function Invoke-TargetReset
{
    param
    (
        [Parameter(Mandatory = $true)]
        [string]
        $Target,

        [switch]
        $NoWait
    )

    Invoke-Command $ControlTargetPath ("reset","--target",$Target)

    if (-not $NoWait)
    {
        # reset 直後に connect しようとすると、うまく接続できずにハングしたような状態となる
        # ことがあるため、少し待つ
        Start-Sleep 2

        for ($i = 0; ; ++$i)
        {
            if ($i -ge 10)
            {
                throw "Could not connect to target after reset."
            }

            Invoke-Command $ControlTargetPath "connect","--target",$Target -NoExitCodeCheck
            if ($LastExitCode -eq 0)
            {
                Start-Sleep 10
                break
            }
        }
    }
}
Export-ModuleMember Invoke-TargetReset

# --------------------------------------------------------------------------
# Wait-NextSleep
# --------------------------------------------------------------------------
function Wait-NextSleep
{
    param
    (
        [Parameter(Mandatory = $true)]
        [string]
        $Target,

        [int]
        $Timeout = 0
    )

    $TargetIpAddress = Get-TargetIpAddress $Target

    Write-InfoLog "waiting for uart log pattern 'OYASUMI'"

    # TODO: ログもスリープに入ったかどうかも両方取得したい場合は？
    $FailureTimeoutArgs = if ($Timeout -gt 0) { @("--failure-timeout",$Timeout) } else { @() }
    Invoke-HalfAwakeLogMonitor $TargetIpAddress ($FailureTimeoutArgs + @('--pattern-success-exit','OYASUMI','--pattern-failure-exit','"=== Skipped SC7 ==="')) -AllowFailurePatternExitCode | Out-Host

    if ($LastExitCode -eq 1)
    {
        # --pattern-failure-exit に一致して終了した、つまりスリープに入らず起床した
        Write-InfoLog "Skipped SC7 detected."
        return $false
    }

    return $true
}
Export-ModuleMember -Function Wait-NextSleep

# --------------------------------------------------------------------------
# Wait-NextAwake
# --------------------------------------------------------------------------
function Wait-NextAwake
{
    param
    (
        [Parameter(Mandatory = $true)]
        [string]
        $Target,

        [int]
        $Timeout = 180
    )

    $TargetIpAddress = Get-TargetIpAddress $Target

    Write-InfoLog "waiting for uart log pattern 'OHAYO'"
    Invoke-HalfAwakeLogMonitor $TargetIpAddress --pattern-success-exit,OHAYO,--failure-timeout,$Timeout
}
Export-ModuleMember -Function Wait-NextAwake

# --------------------------------------------------------------------------
# Invoke-TargetEnableCradle
# --------------------------------------------------------------------------
function Invoke-TargetEnableCradle
{
    param
    (
        [Parameter(Mandatory = $true)]
        [string]
        $Target,

        [switch]
        $NoExitCodeCheck
    )

    Invoke-Command $ControlTargetPath enable-cradle,--target,$Target -NoExitCodeCheck:$NoExitCodeCheck
}
Export-ModuleMember -Function Invoke-TargetEnableCradle

# --------------------------------------------------------------------------
# Invoke-TargetDisableCradle
# --------------------------------------------------------------------------
function Invoke-TargetDisableCradle
{
    param
    (
        [Parameter(Mandatory = $true)]
        [string]
        $Target,

        [switch]
        $NoExitCodeCheck
    )

    Invoke-Command $ControlTargetPath disable-cradle,--target,$Target -NoExitCodeCheck:$NoExitCodeCheck
}
Export-ModuleMember -Function Invoke-TargetDisableCradle

# --------------------------------------------------------------------------
# Invoke-MakeTestApplication
# --------------------------------------------------------------------------
function Invoke-MakeTestApplication
{
    param
    (
        [Parameter(Mandatory = $true)]
        [string[]]
        $Arguments
    )

    Invoke-Command $MakeTestApplicationPath $Arguments
}
Export-ModuleMember -Function Invoke-MakeTestApplication

# --------------------------------------------------------------------------
# Invoke-AuthoringTool
# --------------------------------------------------------------------------
function Invoke-AuthoringTool
{
    param
    (
        [Parameter(Mandatory = $true)]
        [string[]]
        $Arguments
    )

    Invoke-Command $AuthoringToolPath $Arguments
}
Export-ModuleMember -Function Invoke-AuthoringTool

# --------------------------------------------------------------------------
# Invoke-ContentsUploader
# --------------------------------------------------------------------------
function Invoke-ContentsUploader
{
    param
    (
        [Parameter(Mandatory = $true)]
        [string[]]
        $Arguments
    )

    Invoke-Command $ContentsUploaderPath ($Arguments + @("--proxy",$HttpProxy))
}
Export-ModuleMember -Function Invoke-ContentsUploader

# --------------------------------------------------------------------------
# Invoke-SettingsManager
# --------------------------------------------------------------------------
function Invoke-SettingsManager
{
    param
    (
        [Parameter(Mandatory = $true)]
        [string]
        $Target,

        [Parameter(Mandatory = $true)]
        [string[]]
        $Arguments,

        [int]
        $Timeout = 10
    )

    Invoke-Command $RunOnTargetPath ("--target",$Target,"--failure-timeout",$Timeout,$SettingsManager,"--" + $Arguments)
}
Export-ModuleMember -Function Invoke-SettingsManager

# --------------------------------------------------------------------------
# ConvertFrom-Json20
# --------------------------------------------------------------------------
function ConvertFrom-Json20
{
    param
    (
        [Parameter(Mandatory = $true)]
        [string]
        $Item
    )

    Add-Type -Assembly System.Web.Extensions
    $Serializer = New-Object System.Web.Script.Serialization.JavaScriptSerializer
    return $Serializer.DeserializeObject($Item)
}
Export-ModuleMember ConvertFrom-Json20

# --------------------------------------------------------------------------
# ConvertTo-Json20
# --------------------------------------------------------------------------
function ConvertTo-Json20()
{
    param
    (
        [Parameter(Mandatory = $true)]
        [object]
        $Item
    )

    Add-Type -Assembly System.Web.Extensions
    $Serializer = New-Object System.Web.Script.Serialization.JavaScriptSerializer
    return $Serializer.Serialize($Item)
}
Export-ModuleMember -Function ConvertTo-Json20

# --------------------------------------------------------------------------
# Invoke-Rops
# --------------------------------------------------------------------------
function Invoke-Rops
{
    param
    (
        [Parameter(Mandatory = $true)]
        [string[]]
        $Arguments
    )

    Invoke-Command $RopsPath (@("--insecure","--proxy",$HttpProxy) + $Arguments)
}
Export-ModuleMember -Function Invoke-Rops

# --------------------------------------------------------------------------
# Get-RopsOauthToken
# --------------------------------------------------------------------------
function Get-RopsOauthToken
{
    param
    (
        [Parameter(Mandatory = $true)]
        [string]
        $UserName,

        [Parameter(Mandatory = $true)]
        [System.Security.SecureString]
        $SecurePassword
    )

    $Credential = New-Object System.Management.Automation.PSCredential -ArgumentList $UserName,$SecurePassword

    $WebClient = New-Object System.Net.WebClient
    $WebClient.Credentials = $Credential.GetNetworkCredential()

    $RequestUrl = "https://star.debug.dev6.d4c.nintendo.net/v1/ndid/users/@me/token"
    $WebClient.DownloadString($RequestUrl)
}
Export-ModuleMember -Function Get-RopsOauthToken

# --------------------------------------------------------------------------
# Set-SystemUpdateDeliveryConfiguration
# --------------------------------------------------------------------------
function Set-SystemUpdateDeliveryConfiguration
{
    param
    (
        [Parameter(Mandatory = $true)]
        [string]
        $ConfigurationId,

        [Parameter(Mandatory = $true)]
        [string]
        $DeviceId,

        [Parameter(Mandatory = $true)]
        [string]
        $TitleId,

        [Parameter(Mandatory = $true)]
        [string]
        $TitleVersion
    )

    if ($TitleId.StartsWith("0x"))
    {
        $TitleId = $TitleId.Substring("0x".Length)
    }

    $WebClient = New-Object System.Net.WebClient

    $SaveUrl = "https://weather.wc.dev6.d4c.nintendo.net/v1/delivery_configurations/$ConfigurationId"
    $UploadData = "{`"device_ids`":[`"$DeviceId`"], `"title_id`":`"$TitleId`", `"title_version`":`"$TitleVersion`"}"

    $WebClient.Headers.Add("Content-type", "application/json")
    $WebClient.UploadString($SaveUrl, "PUT", $UploadData) | Out-Null

    $ApproveUrl = "$SaveUrl/approve"

    $WebClient.Headers.Remove("Content-type")
    $WebClient.UploadData($ApproveUrl, "PUT", (New-Object byte[] 0)) | Out-Null
}
Export-ModuleMember -Function Set-SystemUpdateDeliveryConfiguration

# --------------------------------------------------------------------------
# Invoke-HalfAwakeLogMonitor
# --------------------------------------------------------------------------
function Invoke-HalfAwakeLogMonitor
{
    param
    (
        [Parameter(Mandatory = $true)]
        [string]
        $HostName,

        [Parameter(Mandatory = $true)]
        [string[]]
        $Arguments,

        [switch]
        $AllowFailurePatternExitCode
    )

    $SuccessExitCodes = ,0
    if ($AllowFailurePatternExitCode)
    {
        $SuccessExitCodes = 0,1
    }

    Invoke-Command $HalfAwakeLogMonitorPath (@("--hostname",$HostName) + $Arguments) -SuccessExitCodes $SuccessExitCodes
}
Export-ModuleMember -Function Invoke-HalfAwakeLogMonitor

# --------------------------------------------------------------------------
# Start-HalfAwakeLogMonitor
# --------------------------------------------------------------------------
function Start-HalfAwakeLogMonitor
{
    param
    (
        [Parameter(Mandatory = $true)]
        [string]
        $HostName,

        [Parameter(Mandatory = $true)]
        [string[]]
        $Arguments
    )

    # 以下のコードでは、終了コードを拾えない問題がある
    #     $Process = Start-Process -FilePath hoge -PassThru
    #     $Process.WaitForExit()
    #     $Process.ExitCode
    # 回避方法が分からないので、Start-Process に頼るのは止めて .NET Framework を使う

    $ProcessStartInfo = New-Object "System.Diagnostics.ProcessStartInfo"
    $ProcessStartInfo.FileName = $HalfAwakeLogMonitorPath
    $ProcessStartInfo.Arguments = (@("--hostname",$HostName) + $Arguments) -join " "

    # CreateNoWindow = true, RedirectStandardOutput/Error = false にして Start-Job でテストを実行すると、
    # ジョブから以下の例外が返る
    #     There is an error processing data from the background process.
    #     Error reported: Cannot process an element with node type "Text".
    #     Only Element and EndElement node types are supported..
    # PowerShell のジョブから IPC で受信するデータは特定の形式に沿っていなければならないが、起動した
    # プロセスの標準出力がそれに沿わずに送信されてくる？
    # 回避のため、標準出力をリダイレクトする
    $ProcessStartInfo.CreateNoWindow = $true
    $ProcessStartInfo.UseShellExecute = $false
    $ProcessStartInfo.RedirectStandardInput = $false
    $ProcessStartInfo.RedirectStandardOutput = $true
    $ProcessStartInfo.RedirectStandardError = $true

    $Process = New-Object "System.Diagnostics.Process"
    $Process.StartInfo = $ProcessStartInfo

    Write-InfoLog "Starting Process: $($Process.StartInfo.FileName) $($Process.StartInfo.Arguments)"
    $Process.Start() >$null

    # 標準出力を随時読み取っていかないとデッドロックする (出力自体はとりあえず捨てる)
    $Process.BeginOutputReadLine()
    $Process.BeginErrorReadLine()

    return $Process
}
Export-ModuleMember -Function Start-HalfAwakeLogMonitor

# --------------------------------------------------------------------------
# Wait-HalfAwakeLogMonitor
# --------------------------------------------------------------------------
function Wait-HalfAwakeLogMonitor
{
    param
    (
        [System.Diagnostics.Process]
        $Process,

        [switch]
        $AllowFailurePatternExitCode
    )

    $SuccessExitCodes = ,0
    if ($AllowFailurePatternExitCode)
    {
        $SuccessExitCodes = 0,1
    }

    $Process.WaitForExit()
    Write-InfoLog "Exited Process ($($Process.ExitCode)): $($Process.StartInfo.FileName) $($Process.StartInfo.Arguments)"

    if ($SuccessExitCodes -contains $Process.ExitCode)
    {
        $Process.ExitCode
    }
    else
    {
        throw "Unexpected exit code ($($Process.ExitCode)): $($Process.StartInfo.FileName) $($Process.StartInfo.Arguments)"
    }
}
Export-ModuleMember -Function Wait-HalfAwakeLogMonitor

# --------------------------------------------------------------------------
# Get-TargetIpAddress
# --------------------------------------------------------------------------
function Get-TargetIpAddress
{
    param
    (
        [Parameter(Mandatory = $true)]
        [string]
        $Target
    )

    Invoke-Command $ControlTargetPath list-target | % {
        if ($_.IndexOf($Target) -ge 0)
        {
            if ($_ -match '\d+\.\d+\.\d+\.\d+')
            {
                $matches[0]
            }
        }
    }
}
Export-ModuleMember -Function Get-TargetIpAddress

# --------------------------------------------------------------------------
# Format-Result
# --------------------------------------------------------------------------
function Format-Result
{
    param
    (
        [Parameter(Mandatory = $true)]
        [string]
        $Result
    )

    if ($Result -match '^2?([0-9]{3})-([0-9]{4})$')
    {
        $Module = [convert]::ToInt32($matches[1], 10)
        $Description = [convert]::ToInt32($matches[2], 10)
    }
    elseif ($Result -match '^(0x)?([0-9a-f]{8})$')
    {
        $Hex = [convert]::ToInt32($matches[2], 16)
        $Module = ($Hex -band 0x1ff)
        $Description = ($Hex -band 0x3ffe00) * [Math]::Pow(2, -9)  # -shr 9 の代わり
    }
    else
    {
        throw 'invalid result'
    }

    New-Object PSObject -Property @{Module = $Module; Description = $Description}
}
Export-ModuleMember -Function Format-Result

# --------------------------------------------------------------------------
# Get-ResultDefinition
# --------------------------------------------------------------------------
function Get-ResultDefinition
{
    param
    (
        [Parameter(Mandatory = $true)]
        [object]
        $FormattedResult,

        [xml]
        $AllResultInfo,

        [string]
        $AllResultsInfoXmlPath
    )

    if (-not $AllResultInfo)
    {
        $AllResultInfo = [xml] (Get-Content $AllResultsInfoXmlPath)
    }

    $ResultModule = $AllResultInfo.ResultDefinitions.ResultModules |
        % { $_.ResultModule } |
        ? { $_.Number -eq $FormattedResult.Module }

    if ($ResultModule)
    {
        $AllResultInfo.ResultDefinitions.ErrorResults |
            % { $_.ErrorResult } |
            ? { $_.ModuleName -eq $ResultModule.Name -and $_.DescriptionValue -eq $FormattedResult.Description }
    }
}
Export-ModuleMember -Function Get-ResultDefinition

# --------------------------------------------------------------------------
# Find-ResultDefinition
# --------------------------------------------------------------------------
function Find-ResultDefinition
{
    param
    (
        [Parameter(Mandatory = $true)]
        [string]
        $Result
    )

    $FormattedResult = Format-Result $Result

    if (Test-Path $DefaultAllResultsInfoXmlPath)
    {
        Get-ResultDefinition $FormattedResult -AllResultsInfoXmlPath $DefaultAllResultsInfoXmlPath
    }
}
Export-ModuleMember -Function Find-ResultDefinition

# --------------------------------------------------------------------------
# Get-DownloadTaskListData
# --------------------------------------------------------------------------
function Process-RequestDownloadTaskListDataJson
{
    param
    (
        [Parameter(Mandatory = $true, ValueFromPipeline = $true)]
        [AllowEmptyString()]
        [string[]]
        $Logs
    )

    begin
    {
        $JsonProcessStarted = $false
        $JsonProcessFinished = $false
    }
    process
    {
        # FIXME: json の出力途中で他のログを入れられるとうまく解析できない可能性がある
        $Logs | % {
            if ($_ -eq '{' -or $_ -eq '{}')
            {
                $JsonProcessStarted = $true
            }
            if ($_ -eq '[SUCCESS]')
            {
                $JsonProcessFinished = $true
            }

            if ($JsonProcessStarted -and -not $JsonProcessFinished)
            {
                if ($_[0] -eq ' ' -or $_ -eq '{' -or $_ -eq '}' -or $_ -eq '{}')
                {
                    $_
                }
            }
        }
    }
}
function Get-DownloadTaskListData
{
    param
    (
        [Parameter(Mandatory = $true)]
        [string]
        $Target
    )

    $Logs = Invoke-DevMenuCommandSystem $Target application,request-download-task-list-data
    $Logs | Write-Host

    $JsonString = ($Logs | Process-RequestDownloadTaskListDataJson) -join "`n"
    Write-InfoLog $JsonString

    try
    {
        return ConvertFrom-Json20 $JsonString
    }
    catch
    {
        # ログから json 出力をうまく解析できなかった
        return $null
    }
}
Export-ModuleMember -Function Get-DownloadTaskListData

# --------------------------------------------------------------------------
# Wait-ServerDownloadTaskListCleared
# --------------------------------------------------------------------------
function Wait-ServerDownloadTaskListCleared
{
    param
    (
        [Parameter(Mandatory = $true)]
        [string]
        $Target,

        [int]
        $Timeout
    )

    $Timeout = if ($Timeout) { $Timeout } else { 300 }

    $StartDate = Get-Date

    while ($true)
    {
        $DownloadTaskListData = Get-DownloadTaskListData $Target
        if ($DownloadTaskListData -and $DownloadTaskListData.tasks.Length -eq 0)
        {
            break
        }

        $Diff = (Get-Date) - $StartDate
        if ($Diff.TotalSeconds -gt $Timeout)
        {
            throw "After $Timeout seconds, tasks of DTL on server still aren't cleared."
        }
    }
}
Export-ModuleMember -Function Wait-ServerDownloadTaskListCleared

# --------------------------------------------------------------------------
# Get-TargetDeviceId
# --------------------------------------------------------------------------
function Get-TargetDeviceId
{
    param
    (
        [Parameter(Mandatory = $true)]
        [string]
        $Target
    )

    $DeviceId = $null

    $GetDeviceIdLog = Invoke-DevMenuCommandSystem -Target $Target -Arguments systemupdate,get-device-id -RetryWithTargetReset
    $GetDeviceIdLog | Write-Host

    $GetDeviceIdLog | % {
        if ($_ -match "^[0-9a-f]{16}$")
        {
            $DeviceId = $matches[0]
        }
    }

    if ($DeviceId)
    {
        return $DeviceId
    }
    else
    {
        throw "Could not get target device id via DevMenuCommandSystem systemupdate get-device-id command."
    }
}
Export-ModuleMember -Function Get-TargetDeviceId

# --------------------------------------------------------------------------
# Get-NetworkAccountIdToken
# --------------------------------------------------------------------------
function Get-NetworkAccountIdToken
{
    param
    (
        [Parameter(Mandatory = $true)]
        [string]
        $Target
    )

    $NaIdToken = $null

    $NaIdTokenLog = Invoke-DevMenuCommandSystem -Target $Target -Arguments shop,na-id-token -RetryWithTargetReset
    $NaIdTokenLog | Write-Host

    $NaIdTokenLog | % {
        # ID Token は JSON Web Token フォーマットらしい
        # http://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=162183953
        #
        # JWT は Base64Url エンコードされた文字列をピリオドで繋げたものなので、以下の正規表現で
        # 認識できるはず。本当か？
        if ($_ -match '^[\w\.\-_]+$')
        {
            $NaIdToken = $matches[0]
        }
    }

    if ($NaIdToken)
    {
        return $NaIdToken
    }
    else
    {
        throw "Could not get NA ID Token via DevMenuCommandSystem shop na-id-token command."
    }
}
Export-ModuleMember -Function Get-NetworkAccountIdToken

# --------------------------------------------------------------------------
# Invoke-OffDeviceDemoDownload
# --------------------------------------------------------------------------
function Invoke-OffDeviceDemoDownload
{
    param
    (
        [Parameter(Mandatory = $true)]
        [string]
        $ApplicationNsUid,

        [Parameter(Mandatory = $true)]
        [string]
        $NaIdToken,

        [string]
        $Language = "ja"
    )

    # curl -X POST --header 'Content-Type: application/x-www-form-urlencoded' --header 'Accept: application/json' --header 'Authorization: Bearer $NaIdToken' 'https://ashigaru.hac.td1.eshop.nintendo.net/shogun/v1/demos/$TestApplicationNsUid/download?lang=ja&shop_id=4'

    $WebClient = New-Object System.Net.WebClient

    $DownloadUrl = "https://ashigaru.hac.td1.eshop.nintendo.net/shogun/v1/demos/$ApplicationNsUid/download?lang=$Language&shop_id=4"

    $WebClient.Headers.Add("Content-type", "application/x-www-form-urlencoded")
    $WebClient.Headers.Add("Accept", "application/json")
    $WebClient.Headers.Add("Authorization", "Bearer $NaIdToken")

    $WebClient.UploadData($DownloadUrl, "POST", (New-Object byte[] 0)) >$null
}
Export-ModuleMember -Function Invoke-OffDeviceDemoDownload
