﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/fs.h>

namespace fsutil {

    /**
     * @brief   指定したパスが存在するかどうかを確認する関数
     *          ファイル、ディレクトリのどちらにも対応しているはず
     */
    bool IsExistPath(const char* inPath) NN_NOEXCEPT;

    bool IsAbsolutePath(const char* path) NN_NOEXCEPT;
    bool IsNspFile(const char* path) NN_NOEXCEPT;

// -----------------------------------------------------------------------

    // FSオブジェクト用抽象クラス
    // 若干無理やり用意した基底クラスだが、C++コーディング規約のオーバーライド関数実装のサンプルとなれば幸い。
    class FsObject
    {
        // コピー禁止宣言
        // C++コーディング規約的にコピーが必要でない場合は宣言"必須"な模様
        NN_DISALLOW_COPY(FsObject);

    public:
        // コンストラクタ
        FsObject() NN_NOEXCEPT
            : m_Result( nn::ResultSuccess() )
        {}

        // デストラクタ
        virtual ~FsObject() NN_NOEXCEPT {}

        // 最後に実行したFS操作の Result を返す
        nn::Result GetLastResult() const NN_NOEXCEPT
        {
            return m_Result;
        }

        // FS操作に共通する関数の純粋仮想関数
        // 継承クラスはこの関数を実装する必要あり
        virtual bool IsOpened() NN_NOEXCEPT = 0;
        virtual nn::Result Open(const char* inPath, int mode) NN_NOEXCEPT = 0;
        virtual void Close() NN_NOEXCEPT = 0;

    protected:
        // 内部関数
        // Result値をメンバ変数に設定する
        void SetResult(const nn::Result& inResult) NN_NOEXCEPT
        {
            m_Result = inResult;
        }

    private:
        nn::Result m_Result;
    };

// -----------------------------------------------------------------------

    // ディレクトリ関連のFS関数をラップするクラス
    // 基本的にSDKのFS操作はハンドル(nn::fs::DirectoryHandle)を使用者が管理しなければならないが
    // その管理を極力意識させないためのラッパークラス。自動Close機能を提供する意図もあり。
    class Directory : public FsObject
    {
        // コピー禁止宣言
        // C++コーディング規約的にコピーが必要でない場合は宣言必須な模様
        NN_DISALLOW_COPY(Directory);

    public:
        // コンストラクタ
        // 特に何もしない
        Directory() NN_NOEXCEPT;

        // パス指定のコンストラクタ
        // インスタンス生成と共に、指定されたディレクトリパスを開こうとする。
        // エラーが発生した場合は、GetLastResult() でエラー内容を取得できるかもしれない
        explicit Directory(const char* inDirPath) NN_NOEXCEPT;

        // デストラクタ
        // インスタンスが Open() している場合は、Close() を実行する
        virtual ~Directory() NN_NOEXCEPT;

        // 本インスタンスがOpen状態かどうかの確認用関数
        // 仮想関数をオーバライドする場合は、必ず明示的に virtual と NN_OVERRIDE の両方を付加する必要あり
        virtual bool IsOpened() NN_NOEXCEPT NN_OVERRIDE;

        // nn::fs::OpenDirectory() のラップ関数
        // 仮想関数をオーバライドする場合は、必ず明示的に virtual と NN_OVERRIDE の両方を付加する必要あり
        virtual nn::Result Open(const char* inPath, int mode = nn::fs::OpenDirectoryMode_All) NN_NOEXCEPT NN_OVERRIDE;

        // nn::fs::CloseDirectory() のラップ関数
        // 仮想関数をオーバライドする場合は、必ず明示的に virtual と NN_OVERRIDE の両方を付加する必要あり
        virtual void Close() NN_NOEXCEPT NN_OVERRIDE;

        // nn::fs::ReadDirectory() のラップ関数
        nn::Result Read(int64_t* outValue, nn::fs::DirectoryEntry* entryBuffer, int64_t entryBufferCount) NN_NOEXCEPT;

        // nn::fs::GetDirectoryEntryCount() のラップ関数
        int64_t GetEntryCount() NN_NOEXCEPT;

    private:
        nn::fs::DirectoryHandle m_Handle;
    };

// -----------------------------------------
    // ファイル関連のFS関数をラップするクラス
    // 基本的にSDKのFS操作はハンドル(nn::fs::FileHandle)を使用者が管理しなければならないが
    // その管理を極力意識させないためのラッパークラス。自動Close機能を提供する意図もあり。
    class File : public FsObject
    {
        // コピー禁止宣言
        // C++コーディング規約的にコピーが必要でない場合は宣言必須な模様
        NN_DISALLOW_COPY(File);

    public:
        // コンストラクタ
        // 特に何もしない
        File() NN_NOEXCEPT;

        // パス指定のコンストラクタ
        // インスタンス生成と共に指定されたファイルパスを開こうとする。
        // エラーが発生した場合は、GetLastResult() でエラー内容を取得できるかもしれない
        explicit File(const char* inFilePath) NN_NOEXCEPT;

        // デストラクタ
        // インスタンスが Open() している場合は、Close() を実行する
        virtual ~File() NN_NOEXCEPT;

        // 本インスタンスがOpen状態かどうかの確認用関数
        // 仮想関数をオーバライドする場合は、必ず明示的に virtual と NN_OVERRIDE の両方を付加する必要あり
        virtual bool IsOpened() NN_NOEXCEPT NN_OVERRIDE;

        // nn::fs::OpenFile() のラップ関数
        // 仮想関数をオーバライドする場合は、必ず明示的に virtual と NN_OVERRIDE の両方を付加する必要あり
        virtual nn::Result Open(const char* inPath, int mode = (nn::fs::OpenMode_Read | nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend) ) NN_NOEXCEPT NN_OVERRIDE;

        // nn::fs::CloseFile() のラップ関数
        // 仮想関数をオーバライドする場合は、必ず明示的に virtual と NN_OVERRIDE の両方を付加する必要あり
        virtual void Close() NN_NOEXCEPT NN_OVERRIDE;

        // nn::fs::ReadFile() のラップ関数
        nn::Result Read(int64_t offset, void* buffer, size_t size, const nn::fs::ReadOption& option) NN_NOEXCEPT;

        // nn::fs::ReadFile() のラップ関数
        nn::Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT;

        // nn::fs::ReadFile() のラップ関数
        nn::Result Read(size_t* outValue, int64_t offset, void* buffer, size_t size, const nn::fs::ReadOption& option) NN_NOEXCEPT;

        // nn::fs::ReadFile() のラップ関数
        nn::Result Read(size_t* outValue, int64_t offset, void* buffer, size_t size) NN_NOEXCEPT;

        // nn::fs::WriteFile() のラップ関数
        nn::Result Write(int64_t offset, void* buffer, size_t size, const nn::fs::WriteOption& option) NN_NOEXCEPT;

        // nn::fs::FlushFile() のラップ関数
        nn::Result Flush() NN_NOEXCEPT;

        // nn::fs::SetFileSize() のラップ関数
        nn::Result SetSize(int64_t size) NN_NOEXCEPT;

        // nn::fs::GetFileSize() のラップ関数
        int64_t GetSize() NN_NOEXCEPT;

        // nn::fs::GetFileOpenMode() のラップ関数
        int GetFileOpenMode() NN_NOEXCEPT;

        nn::fs::FileHandle GetHandle() NN_NOEXCEPT
        {
            return this->m_Handle;
        }

    private:
        nn::fs::FileHandle m_Handle;
    };

} // namespace fsutil
