﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <cstdlib>
#include <cstring>
#include <string>
#include <mutex>

#include <nn/account.h>
#include <nn/account/account_Api.h>
#include <nn/account/account_ApiPrivate.h>
#include <nn/account/account_ApiForApplications.h>
#include <nn/account/account_ApiForAdministrators.h>
#include <nn/account/account_Result.h>
#include <nn/account/account_ResultForAdministrators.h>
#include <nn/friends/friends_ApiAdmin.h>

#include <nn/ec.h>
#include <nn/fs.h>
#include <nn/os.h>
#include <nn/oe.h>
#include <nn/nifm.h>
#include <nn/nifm/nifm_ApiClientManagement.h>
#include <nn/ssl.h>
#include <nn/ssl/ssl_Context.h>
#include <nn/socket.h>

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/util/util_Uuid.h>
#include <nn/util/util_Base64.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_CharacterEncoding.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/fwdbg/settings_SettingsSetterApi.h>

#include <curl/curl.h>

#include <nn/ec/ec_ShopServiceAccessor.h>
#include <nn/ec/ec_ShopServiceAccessorForDebug.h>
#include <nn/nim/nim_ShopServiceAccessConfig.h>
#include "accounts/testNim_AccountUtil.h"
#include "testNim_AccountServiceUtil.h"
#include "testNim_ResponseAdaptor.h"

#include "nimsrv/nim_ShopServiceAccessPrivate.h"
#include "nimsrv/nim_HttpJsonUtil.h"

// time ライブラリの GetCurrentTime が Windows のマクロと競合するため
#ifdef GetCurrentTime
#undef GetCurrentTime
#endif

#include <nn/time.h>

//!----------------------------------------------------------------------------
// テスト環境説明
//  - 強制的にユーザインデクス 0～2 に対して、テスト用ニンテンドーアカウントで NSA 連携を行います。
//  - ユーザの自動生成について
//    - ターゲットにユーザが存在する場合は、自動生成は行なわず既存ユーザを上書きします。
//    - ターゲットにユーザが存在しない場合は自動生成を行い、テスト終了時に NSA 連携解除、及びユーザの削除を行います。
//!----------------------------------------------------------------------------

//!----------------------------------------------------------------------------
//  Sugar サーバー用テストを有効化します。
//!----------------------------------------------------------------------------
#define ENABLE_SUGAR_TEST_FIXTURE

//!----------------------------------------------------------------------------
//  Civil サーバー用テストを有効化します。
//!----------------------------------------------------------------------------
#define ENABLE_CIVIL_TEST_FIXTURE

//!----------------------------------------------------------------------------
//  Civil サーバーが全機能搭載されレスポンスが正しい場合のテストを有効化します。
//!----------------------------------------------------------------------------
#define ENABLE_CIVIL_TEST_AS_FINISHED_FEATURES

//!----------------------------------------------------------------------------
// ダミー権利情報の登録を Sugar に対してリクエストします。
// 既存ユーザが既に NSA 連携済のユーザである場合
//  - NsaId が同じものが使われるため、同じアイテムへの権利情報が多重登録されます。
//  - 新敷く追加された権利はレスポンスJSONの先頭要素に挿入されていきます。( 2018/01/19 現在 )
// ユーザの自動生成を行った場合
//  - テスト終了時にNSA連携の自動解除も行われるため、ENABLE_REGISTER_DUMMY_RIGHTS を有効にして都度アイテムへの権利登録が必要な点ご注意ください。
#define ENABLE_REGISTER_DUMMY_RIGHTS

//!----------------------------------------------------------------------------
#if defined( NN_SDK_BUILD_DEBUG ) || defined( NN_SDK_BUILD_DEVELOP )
#define PRIVATE_LOG(...) NN_LOG( "[ShopServiceAccessor::TEST] " __VA_ARGS__ )
#define PRIVATE_LOG_OUT_FOR(...)    LogOutFor(__VA_ARGS__)
#else
#define PRIVATE_LOG(...)            static_cast<void>(0)
#define PRIVATE_LOG_OUT_FOR(...)    static_cast<void>(0)
#endif

#if defined( NN_BUILD_CONFIG_TOOLCHAIN_GCC )
#define PRIVATE_STR_TO_LL  strtoll
#define PRIVATE_STR_TO_ULL strtoull
#elif defined( NN_BUILD_CONFIG_OS_WIN )
#define PRIVATE_STR_TO_LL  _strtoi64
#define PRIVATE_STR_TO_ULL _strtoui64
#else
#define PRIVATE_STR_TO_LL  std::strtoll
#define PRIVATE_STR_TO_ULL std::strtoull
#endif

//!----------------------------------------------------------------------------
#if defined( NN_BUILD_CONFIG_OS_HORIZON )

using namespace nn;

namespace {

//!----------------------------------------------------------------------------
class StatefulInitializer
{
public:
    StatefulInitializer() NN_NOEXCEPT : m_IsInitialized(false) {}

    template<typename Predicate>
    void Initialize(const char* pLogPrefix, Predicate executor) NN_NOEXCEPT
    {
        PRIVATE_LOG("==== Initializing %s\n", pLogPrefix);
        executor();
        m_IsInitialized = true;
    }

    template<typename Predicate>
    void Finalize(const char* pLogPrefix, Predicate executor) NN_NOEXCEPT
    {
        PRIVATE_LOG("==== Cleaning up %s\n", pLogPrefix);
        if (m_IsInitialized)
        {
            m_IsInitialized = false;
            executor();
        }
    }

private:
    bool    m_IsInitialized;
};

//!----------------------------------------------------------------------------
class Environment
{
private:
    socket::ConfigDefaultWithMemory     m_SocketConfig;
    friends::DaemonSuspension           m_FriendSuspension;
    nnt::account::UserContext           m_Users[account::UserCountMax];

    StatefulInitializer m_InitialLibcurl;
    StatefulInitializer m_InitialSocket;
    StatefulInitializer m_InitialNifm;
    StatefulInitializer m_InitialSsl;

public:
    void Initialize() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(friends::SuspendDaemon(&m_FriendSuspension));

#if defined( NN_BUILD_CONFIG_OS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
        oe::Initialize();
#endif

        m_InitialNifm.Initialize("NIFM", []() -> void
        {
            NNT_ASSERT_RESULT_SUCCESS(nifm::Initialize());
            nifm::SubmitNetworkRequestAndWait();
        });
        m_InitialSsl.Initialize("SSL", []() -> void
        {
            NNT_ASSERT_RESULT_SUCCESS(ssl::Initialize());
        });
        m_InitialSocket.Initialize("socket", [this]() -> void
        {
            NNT_ASSERT_RESULT_SUCCESS(socket::Initialize(m_SocketConfig));
        });
        m_InitialLibcurl.Initialize("libcurl", []() -> void
        {
            ASSERT_EQ(CURLE_OK, curl_global_init(CURL_GLOBAL_ALL));
        });

        PrepareUsers(3);
    }

    void Finalize() NN_NOEXCEPT
    {
        FinishUsers();

        m_InitialLibcurl.Finalize("libcurl", []() -> void
        {
            curl_global_cleanup();
        });

        m_InitialSocket.Finalize("socket", []() -> void
        {
            NNT_ASSERT_RESULT_SUCCESS(socket::Finalize());
        });
        m_InitialSsl.Finalize("SSL", []() -> void
        {
            NNT_ASSERT_RESULT_SUCCESS(ssl::Finalize());
        });
        m_InitialNifm.Finalize("NIFM", []() -> void
        {
            nifm::CancelNetworkRequest();
        });
    }

    Result AcquireNsaId(int index, account::NetworkServiceAccountId* pOutValue) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pOutValue);

        const auto& user = m_Users[index];
        NN_ASSERT(user.id);
        NN_ASSERT(user.isOpened);

        // ネットワークサービスアカウントの確定 ( NA連携 )
        NN_RESULT_DO(account::EnsureNetworkServiceAccountAvailable(user.open));
        PRIVATE_LOG("Success: EnsureNetworkServiceAccount.\n");

        // ネットワークサービスアカウントの情報
        NN_RESULT_DO(account::GetNetworkServiceAccountId(pOutValue, user.open));
        PRIVATE_LOG("Success: Acquire Network Service Account ID [ %016llx ].\n", pOutValue->id);

        NN_RESULT_SUCCESS;
    }

    const ::nn::account::Uid& GetUser(int index) const NN_NOEXCEPT
    {
        return m_Users[index].id;
    }

private:
    void RegisterNewUsers(int createdCount, int startCount) NN_NOEXCEPT
    {
        account::Finalize();
        NN_UTIL_SCOPE_EXIT
        {
            account::Initialize();
        };

        nnt::account::CreateUsers(m_Users, createdCount, startCount);
    }

    void UnregisterNewUsers() NN_NOEXCEPT
    {
        account::Finalize();
        nnt::account::Cleanup(m_Users);
        account::Initialize();
    }

    void LinkNa(const int linkUserCount) NN_NOEXCEPT
    {
        const char* const accountInfoID[] =
        {
            "eshop-test+puxdd1_001@exmx.nintendo.co.jp" ,
            "koshiyama_ssdtest+pux001@exmx.nintendo.co.jp",
            "eca-child-login",
        };
        const char* const accountInfoPassword[] =
        {
            "puxdd1_001" ,
            "qc13-test",
            "eca-child-password",
        };

        account::InitializeForAdministrator();
        NN_UTIL_SCOPE_EXIT
        {
            account::Finalize();
        };

        for (int i = 0; i < linkUserCount; ++i)
        {
            const auto uid = m_Users[i].id;

            account::NetworkServiceAccountAdministrator admin;
            NNT_ASSERT_RESULT_SUCCESS(::nn::account::GetNetworkServiceAccountAdministrator(&admin, uid));
            if (!nnt::account::IsNetworkServiceAccountRegistered(admin))
            {
                NNT_ASSERT_RESULT_SUCCESS(nnt::account::RegisterNetworkServiceAccount(admin));
            }
            ASSERT_TRUE(accounts::LinkNa(uid, accountInfoID[i], accountInfoPassword[i]));
        }
    }

    void PrepareUsers(int createdCount) NN_NOEXCEPT
    {
        ASSERT_GT(createdCount, 0);

        int nUsers;
        account::Uid users[account::UserCountMax];
        NNT_ASSERT_RESULT_SUCCESS(account::ListAllUsers(&nUsers, users, NN_ARRAY_SIZE(users)));
        if (createdCount > nUsers)
        {
            int differenceCreatedCount = createdCount - nUsers;
            RegisterNewUsers(differenceCreatedCount, nUsers);
        }
        CreateUserContext(m_Users, users, nUsers);

        account::Finalize();
        LinkNa(createdCount);
        account::Initialize();

        for (int i = 0; i < createdCount; ++i)
        {
            m_Users[i].Open();
        }
    }

    void FinishUsers() NN_NOEXCEPT
    {
        const auto n = NN_ARRAY_SIZE(m_Users);
        for (int i = 0; i < n; ++i)
        {
            m_Users[i].Close();
        }
        UnregisterNewUsers();
    }
};

//!------------------------------------------------------------------------------------------------------
//! デバッグ機能支援
//!------------------------------------------------------------------------------------------------------
namespace DebugResponse
{
    char BasicResponseLong[(12 * 1024) - 11];
    const size_t SizeofBasicResponseLong = sizeof(BasicResponseLong);

    const char BasicResponse1[] = "This is debug response as basic(1)!! Are you happy? I'm hungry...";
    const char BasicResponse2[] = "This is debug response as basic(2)!! Are you genius? I'm fine thank you...";
    const size_t SizeofBasicResponse1 = sizeof(BasicResponse1) - 1;
    const size_t SizeofBasicResponse2 = sizeof(BasicResponse2) - 1;

    //! @brief  fwdbg 設定、および即時反映。
    //! @note   ec::DebugForShopServiceAccessor::Initialize() の呼び出しが必須です。
    static void EnableFirmwareSettings(bool request) NN_NOEXCEPT
    {
        bool enabled = request;
        settings::fwdbg::SetSettingsItemValue("nim.errorsimulate", "enable_error_simulate", &enabled, sizeof(enabled));
        ec::DebugForShopServiceAccessor::RefreshAvailability();
    }

    //! @brief  全てのデバッグ設定をクリーンして支援機能を無効化します。
    static void Cleanup() NN_NOEXCEPT
    {
        ec::DebugForShopServiceAccessor::ScopedSession session;
        ec::DebugForShopServiceAccessor::Initialize(&session);
        ec::DebugForShopServiceAccessor::ClearResponse();
        EnableFirmwareSettings(false);
    }

    //! @brief  テスト用の長大レスポンスを作成します。
    static void CreateLongResponse() NN_NOEXCEPT
    {
        // 全て '-' で埋める。
        auto pBuf = BasicResponseLong;
        std::fill_n(pBuf, sizeof(BasicResponseLong), '-');
        // 区切りポジションにキーワード入れる。
        pBuf[0] = '0';
        std::memcpy(&pBuf[4096], "4096", 4);
        std::memcpy(&pBuf[8192], "8192", 4);
        pBuf[sizeof(BasicResponseLong) - 1] = 'E';
    }

    //! @brief  長大レスポンスの一致比較を実施します。
    static bool VerifyLongResponse(const char* pVerifySource) NN_NOEXCEPT
    {
        return (0 == std::memcmp(BasicResponseLong, pVerifySource, sizeof(BasicResponseLong)));
    }
};

//!------------------------------------------------------------------------------------------------------
class TestForShopServiceAccessor : public testing::Test
{
private:
    static Environment& GetEnvironment() NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC(Environment, s_Instance);
        return s_Instance;
    }

protected:
    CURL*   pCurlHandle;

    virtual void SetUp()
    {
        DebugResponse::Cleanup();
        pCurlHandle = curl_easy_init();
        NN_ABORT_UNLESS(nullptr != pCurlHandle);
    }

    virtual void TearDown()
    {
        if (nullptr != pCurlHandle)
        {
            curl_easy_cleanup(pCurlHandle);
        }
    }

    static void SetUpTestCase()
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(time::Initialize());

#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        oe::Initialize();
#endif

        account::Initialize();
        GetEnvironment().Initialize();
        DebugResponse::CreateLongResponse();
    }

    static void TearDownTestCase()
    {
        GetEnvironment().Finalize();
        account::Finalize();
        time::Finalize();
    }

public:
    TestForShopServiceAccessor() : pCurlHandle(nullptr) {}

    Result AcquireNsaId(int index, account::NetworkServiceAccountId* pOutValue) NN_NOEXCEPT
    {
        return GetEnvironment().AcquireNsaId(index, pOutValue);
    }

    const ::nn::account::Uid& GetUser(int index) const NN_NOEXCEPT
    {
        return GetEnvironment().GetUser(index);
    }
};

//!------------------------------------------------------------------------------------------------------
std::string ToBase64(const std::string& source) NN_NOEXCEPT
{
    const auto capacity = nn::util::align_up(source.length() * 9 / 6, 4) + 1; // 1.5倍(論理値は133%)
    std::string::allocator_type allocator = source.get_allocator();
    std::string::pointer buffer = allocator.allocate(capacity);
    nn::util::Base64::Status result;
    result = nn::util::Base64::ToBase64String(buffer, capacity, source.c_str(), source.length(), nn::util::Base64::Mode_UrlSafe);
    NN_ABORT_UNLESS(nn::util::Base64::Status_Success == result);

    std::string out(buffer);
    allocator.deallocate(buffer, capacity);
    return out;
}

//!------------------------------------------------------------------------------------------------------
#define DO_RESULT_CURLCODE(curlCode_)   \
{                                       \
    if (curlCode_ != CURLE_OK)          \
    {                                   \
        ToCurlErrorHandling(curlCode_); \
        return curlCode_;               \
    }                                   \
}

//!------------------------------------------------------------------------------------------------------
class CurlExecutor
{
private:
    struct SslCtxFunctionContext
    {
        bool isCalled;
        Result result;

        void Reset() NN_NOEXCEPT
        {
            isCalled = false;
            result = nn::ResultSuccess();
        }
    };

    void ToCurlErrorHandling(const CURLcode code) NN_NOEXCEPT
    {
        PRIVATE_LOG("CurlError: %u\n", static_cast<Bit32>(code));
    }

public:
    struct Response
    {
        char*   pBuffer;
        size_t  capacity;
        size_t  receivedSize;
        int     code;

        explicit Response(size_t initialCapacity) NN_NOEXCEPT
            : pBuffer(nullptr), capacity(initialCapacity), receivedSize(0), code(0)
        {
            pBuffer = reinterpret_cast<char*>(std::malloc(initialCapacity));
            NN_ABORT_UNLESS(nullptr != pBuffer);
            pBuffer[0] = '\0';
        }

        ~Response() NN_NOEXCEPT
        {
            if (nullptr != pBuffer)
            {
                capacity = 0;
                std::free(pBuffer);
            }
            Reset();
        }

        void Reset() NN_NOEXCEPT
        {
            code = 0;
            receivedSize = 0;
        }

        void Append(const char* pSource, size_t sourceSize) NN_NOEXCEPT
        {
            NN_ABORT_UNLESS((receivedSize + sourceSize) < capacity);
            std::memcpy(&pBuffer[receivedSize], pSource, sourceSize);
            receivedSize += sourceSize;
            pBuffer[receivedSize] = '\0';
        }
    };

    const CURLcode Post(Response* pOutResponse, CURL* pCurlHandle, const char* pUrl, const char* pPostField, const char* pHeaders[], const int headerCount, long timeoutSeconds) NN_NOEXCEPT
    {
        NN_ASSERT(pOutResponse);
        NN_ASSERT(pCurlHandle);
        NN_ASSERT(pPostField);
        NN_ASSERT(pUrl);

        m_SslFunctionContext.Reset();
        curl_easy_reset(pCurlHandle);

        DO_RESULT_CURLCODE(curl_easy_setopt(pCurlHandle, CURLOPT_XFERINFOFUNCTION, nullptr));
        DO_RESULT_CURLCODE(curl_easy_setopt(pCurlHandle, CURLOPT_XFERINFODATA, nullptr));
        DO_RESULT_CURLCODE(curl_easy_setopt(pCurlHandle, CURLOPT_NOPROGRESS, 1));
        DO_RESULT_CURLCODE(curl_easy_setopt(pCurlHandle, CURLOPT_URL, pUrl));

        DO_RESULT_CURLCODE(curl_easy_setopt(pCurlHandle, CURLOPT_POST, 1));
        DO_RESULT_CURLCODE(curl_easy_setopt(pCurlHandle, CURLOPT_COPYPOSTFIELDS, pPostField));

        curl_slist* pSlist = nullptr;
        NN_UTIL_SCOPE_EXIT{curl_slist_free_all(pSlist);};

        for (int i = 0; i < headerCount; ++i)
        {
            pSlist = curl_slist_append(pSlist, pHeaders[i]);
            NN_ASSERT(nullptr != pSlist);
        }

        DO_RESULT_CURLCODE(curl_easy_setopt(pCurlHandle, CURLOPT_HTTPHEADER, pSlist));

        DO_RESULT_CURLCODE(curl_easy_setopt(pCurlHandle, CURLOPT_HEADERDATA, nullptr));
        DO_RESULT_CURLCODE(curl_easy_setopt(pCurlHandle, CURLOPT_HEADERFUNCTION, nullptr));

        DO_RESULT_CURLCODE(curl_easy_setopt(pCurlHandle, CURLOPT_WRITEDATA, pOutResponse));
        DO_RESULT_CURLCODE(curl_easy_setopt(pCurlHandle, CURLOPT_WRITEFUNCTION, WriteCallback));
        DO_RESULT_CURLCODE(curl_easy_setopt(pCurlHandle, CURLOPT_NOBODY, 0));
        DO_RESULT_CURLCODE(curl_easy_setopt(pCurlHandle, CURLOPT_TIMEOUT, timeoutSeconds));

        DO_RESULT_CURLCODE(curl_easy_setopt(pCurlHandle, CURLOPT_SSL_CTX_FUNCTION, DefaultSslCtxFunction));
        DO_RESULT_CURLCODE(curl_easy_setopt(pCurlHandle, CURLOPT_SSL_CTX_DATA, &m_SslFunctionContext));

        DO_RESULT_CURLCODE(curl_easy_perform(pCurlHandle));
        DO_RESULT_CURLCODE(curl_easy_getinfo(pCurlHandle, CURLINFO_RESPONSE_CODE, &pOutResponse->code));
        return CURLE_OK;
    }

protected:
    static size_t WriteCallback(char* pSource, size_t blockUnitSize, size_t blockCount, void* pUserData) NN_NOEXCEPT
    {
        const auto totalSize = blockUnitSize * blockCount;
        if (nullptr != pSource && nullptr != pUserData)
        {
            auto pHandle = static_cast<Response*>(pUserData);
            pHandle->Append(pSource, totalSize);
        }
        return totalSize;
    }

    static size_t DefaultSslCtxFunction(CURL* curlHandle, void* pSslContext, SslCtxFunctionContext* pFunctionContext) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pFunctionContext);
        pFunctionContext->isCalled = true;
        if (!pFunctionContext->result.IsSuccess())
        {
            return 1;
        }
        auto& context = *reinterpret_cast<nn::ssl::Context*>(pSslContext);
        auto r = context.Create(nn::ssl::Context::SslVersion_Auto);
        if (r.IsSuccess())
        {
            ssl::CertStoreId clientCertStoreId;
            r = context.RegisterInternalPki(&clientCertStoreId, ssl::Context::InternalPki_DeviceClientCertDefault);
        }
        if (!r.IsSuccess())
        {
            pFunctionContext->result = r;
            return 1;
        }
        return 0;
    }

private:
    SslCtxFunctionContext m_SslFunctionContext;
};

//!------------------------------------------------------------------------------------------------------
class SugarRegister : private CurlExecutor
{
public:

    SugarRegister(CURL* const pCurlHandle, const char* const pClientSecret) NN_NOEXCEPT
        : m_pCurlHandle(pCurlHandle)
        , m_BaseUrl("https://sugar.ws.%.ss.nintendo.net/v1/applications/01002ad001e32000/")
        , m_AuthClient(std::string("Authorization: Basic ") + ToBase64(pClientSecret))
    {
        // URL: 01002ad001e32000 はコンテンツ配信自動テストの OMAS払い出し済 ID。
    }

    // テスト用ダミーアイテム登録 ( 消費型 )
    void Register(const char* const pItemName) NN_NOEXCEPT
    {
        const char* pHeaders[] = {
            "Content-Type:application/json",
            m_AuthClient.c_str()
        };
        const char* const pFieldBase = "[{"
            "\"item_id\": \"%s\""
            "}]";

        char postData[1024];
        NN_ABORT_UNLESS(nn::util::SNPrintf(postData, sizeof(postData), pFieldBase, pItemName) < sizeof(postData));

        Response response(8 * 1024);
        const auto url = m_BaseUrl + "items";
        NN_ABORT_UNLESS(CURLE_OK == Post(&response, m_pCurlHandle, url.c_str(), postData, pHeaders, NN_ARRAY_SIZE(pHeaders), 60));

        PRIVATE_LOG("<SugerRegister>::Register:\n    HEAD: %s\n    URL: %s\n    POST: %s\n    RESPONSE: %s\n\n"
            , m_AuthClient.c_str(), url.c_str(), postData, response.pBuffer
        );
    }

    // テスト用ダミーアイテム権利情報登録 ( 消費型 )
    void Purchase(const char* const pItemName, const nn::account::NetworkServiceAccountId& nsaId) NN_NOEXCEPT
    {
        const char* const pFieldBase = "{"
            "\"item_id\": \"%s\", "
            "\"country\": \"JP\""
            "}";

        char postData[1024];
        NN_ABORT_UNLESS(nn::util::SNPrintf(postData, sizeof(postData), pFieldBase, pItemName) < sizeof(postData));

        char path[128];
        NN_ABORT_UNLESS(nn::util::SNPrintf(path, sizeof(path), "accounts/%016llx/rights", nsaId.id) < sizeof(path));

        char contentLength[128];
        NN_ABORT_UNLESS(nn::util::SNPrintf(contentLength, sizeof(contentLength), "Content-Length:%u", std::strlen(postData)) < sizeof(path));

        const char* pHeaders[] = {
            "Content-Type:application/json",
            m_AuthClient.c_str(),
            contentLength
        };

        Response response(8 * 1024);
        const auto url = m_BaseUrl + path;
        NN_ABORT_UNLESS(CURLE_OK == Post(&response, m_pCurlHandle, url.c_str(), postData, pHeaders, NN_ARRAY_SIZE(pHeaders), 60));

        PRIVATE_LOG("<SugerRegister>::Update:\n    HEAD: %s\n    URL: %s\n    POST: %s\n    RESPONSE: %s\n\n"
            , m_AuthClient.c_str(), url.c_str(), postData, response.pBuffer
        );
    }

private:
    CURL* const m_pCurlHandle;
    const std::string m_BaseUrl;
    const std::string m_AuthClient;
};

//!------------------------------------------------------------------------------------------------------
//! 権利取得レスポンスパーサーアダプタ
typedef ::nnt::eca::RightReponseAdaptor RightReponseAdaptor;

//!------------------------------------------------------------------------------------------------------
//! Civil カタログレスポンスパーサーアダプタ
typedef ::nnt::eca::ConsumablesReponseAdaptor ConsumablesReponseAdaptor;

//!------------------------------------------------------------------------------------------------------
//! Civil 消費アイテムレスポンスパーサーアダプタ
typedef ::nnt::eca::ConsumableItemsReponseAdaptor ConsumableItemsReponseAdaptor;

//!----------------------------------------------------------------------------
#if defined( NN_SDK_BUILD_DEBUG ) || defined( NN_SDK_BUILD_DEVELOP )
void LogOutFor(const char* pBuffer, const size_t bufferSize) NN_NOEXCEPT
{
    auto pRear = const_cast<char*>(&pBuffer[bufferSize - 1]);
    const auto rearCode = *pRear;
    *pRear = '\0';
    PRIVATE_LOG("Response(%zu) =>\n%s%c\n\n", bufferSize, pBuffer, rearCode);
    *pRear = rearCode;
}
#endif

#if defined(ENABLE_SUGAR_TEST_FIXTURE)
//!------------------------------------------------------------------------------------------------------
Result RequestAsGetRightsInformation(ec::ShopServiceAccessor& accessor
    , ec::ShopServiceAccessor::AsyncResponse* pResponse
    , const account::Uid& user
    , int pageIndex
    , bool useInvalidParameter = false) NN_NOEXCEPT
{
    char pPath[256];
    const char* const pBasePath = (useInvalidParameter)
        ? "/v1/rights/my?app"
        : "/v1/rights/my?application_id={applicationId}&page=%d&per_page=10";
    NN_ABORT_UNLESS(nn::util::SNPrintf(pPath, sizeof(pPath), pBasePath, pageIndex) < sizeof(pPath));

    NN_RESULT_DO(accessor.Request(pResponse, user, ec::ShopService::Method_Get, pPath));
    NN_RESULT_SUCCESS;
}

//!------------------------------------------------------------------------------------------------------
Result RequestAsPostConsume(ec::ShopServiceAccessor& accessor
    , ec::ShopServiceAccessor::AsyncResponse* pResponse
    , const account::Uid& user
    , const char* pRightId
    , const char* pConsumptionRequestId) NN_NOEXCEPT
{
    const char RightPostBase[] =
    "{"
        "\"rights\": ["
            "{"
                "\"right_id\":\"%s\","
                "\"consumption_request_id\":\"%s\""
            "}"
        "]"
    "}";

    char pRightPostData[512];
    NN_ABORT_UNLESS(nn::util::SNPrintf(pRightPostData, sizeof(pRightPostData), RightPostBase, pRightId, pConsumptionRequestId) < sizeof(pRightPostData));
    PRIVATE_LOG("PostData[%s]\n", pRightPostData);

    ec::ShopServiceAccessor::PostData postData = {pRightPostData, std::strlen(pRightPostData)};
    NN_RESULT_DO(accessor.Request(pResponse, user, ec::ShopService::Method_Post, "/v1/rights/consume", postData));
    NN_RESULT_SUCCESS;
}
#endif  // defined(ENABLE_SUGAR_TEST_FIXTURE)

#if defined(ENABLE_CIVIL_TEST_FIXTURE)
//!------------------------------------------------------------------------------------------------------
Result RequestAsGetCatalog(ec::ShopServiceAccessor& accessor
    , ec::ShopServiceAccessor::AsyncResponse* pResponse
    , const account::Uid& user
    , const char* pLanguage = nullptr)
{
    char pPath[256];
    pLanguage = (nullptr == pLanguage) ? "ja" : pLanguage;
    const char* const pBasePath = "/v1/applications/{applicationId}/consumables?country={country}&lang=%s";
    NN_ABORT_UNLESS(nn::util::SNPrintf(pPath, sizeof(pPath), pBasePath, pLanguage) < sizeof(pPath));

    NN_RESULT_DO(accessor.Request(pResponse, user, ec::ShopService::Method_Get, pPath));
    NN_RESULT_SUCCESS;
}

//!------------------------------------------------------------------------------------------------------
Result RequestAsGetConsumableItem(ec::ShopServiceAccessor& accessor
    , ec::ShopServiceAccessor::AsyncResponse* pResponse
    , const account::Uid& user
    , const char* pConsumableId
    , const char* pLanguage = nullptr)
{
    char pPath[256];
    pLanguage = (nullptr == pLanguage) ? "ja" : pLanguage;
    const char* const pBasePath = "/v1/consumables/%s/items?country={country}&lang=%s";
    NN_ABORT_UNLESS(nn::util::SNPrintf(pPath, sizeof(pPath), pBasePath, pConsumableId, pLanguage) < sizeof(pPath));

    NN_RESULT_DO(accessor.Request(pResponse, user, ec::ShopService::Method_Get, pPath));
    NN_RESULT_SUCCESS;
}
#endif  // defined(ENABLE_CIVIL_TEST_FIXTURE)

//!------------------------------------------------------------------------------------------------------
template<typename JsonPathType>
Result ParseErrorCodeResponse(Bit32* pOutCode, const char* pSource) NN_NOEXCEPT
{
    nim::srv::JsonParser::ErrorReponseAdaptor<JsonPathType> adaptor("$.error.code");
    nim::srv::HttpJson::Stream::MemoryInputWithStaticOutput<1024> stream(pSource, std::strlen(pSource));
    NN_RESULT_DO(::nn::http::json::ImportJsonByRapidJson<::nn::http::json::DefaultJsonErrorMap>(adaptor, stream, static_cast<nim::srv::HttpJson::Canceler*>(nullptr)));
    PRIVATE_LOG("Detected error code( %u )\n", adaptor.GetErrorCode());
    *pOutCode = adaptor.GetErrorCode();
    NN_RESULT_SUCCESS;
}

//!------------------------------------------------------------------------------------------------------
// 全てのメモリブロックが指定値と一致するかチェック
template<typename BlockType = Bit8>
bool VerifyMemoryAsValue(const BlockType* pBlockTop, size_t countOfBlock, BlockType value) NN_NOEXCEPT
{
    const auto pEnd = &pBlockTop[countOfBlock];
    return (pEnd == std::find_if_not(pBlockTop, pEnd, [&value](BlockType blockValue) -> bool
    {
        return value == blockValue;
    }));
}

//!------------------------------------------------------------------------------------------------------
// リクエストワーク
NN_ALIGNAS(os::MemoryPageSize) Bit8 g_WorkMemory[os::MemoryPageSize * 16];

}   // ~::unnamed

//!------------------------------------------------------------------------------------------------------
#if defined(ENABLE_REGISTER_DUMMY_RIGHTS)
TEST_F(TestForShopServiceAccessor, DummyRegistration)
{
    PRIVATE_LOG("Try acquire network service account id.\n");
    nn::account::NetworkServiceAccountId nsaId;
    NNT_ASSERT_RESULT_SUCCESS(AcquireNsaId(0, &nsaId));

    PRIVATE_LOG("Try register the dummy items.\n");
    SugarRegister executor(pCurlHandle, "c632ada668f04556:XiOYjfyghJGfFRVs");

    // テスト用ダミーアイテム登録 ( 消費型 )
    // アイテム登録は Admin権限の ClientID/Secret が必要なため、このテストからの登録は除外。
    //executor.Register("eca0000");

    // テスト用ダミーアイテム権利情報登録 ( 消費型 )
    //  所謂 eca0000 アイテムの購入履歴が一つ増えます。
    //  権利一覧取得で以下のような履歴構成になる。
    //  {"total_results":2,
    //      "rights" : [
    //          {"right_id":"d5a2fdb5-3099-485e-a97e-36b5138fc605", "item_id" : "eca0000", "nsa_id" : "a4b9678e7041f9ab", "purchased_date_time" : 1513760114000, "status" : "PURCHASED", "country" : "JP"},
    //          {"right_id":"f98b113e-5f24-4e3f-9d02-82ebb98c72d9", "item_id" : "eca0000", "nsa_id" : "a4b9678e7041f9ab", "purchased_date_time" : 1513758859000, "status" : "PURCHASED", "country" : "JP"}
    //      ]
    //  }
    executor.Purchase("eca0000", nsaId);
}
#endif

//!------------------------------------------------------------------------------------------------------
//! DecimalConverter テスト
TEST_F(TestForShopServiceAccessor, DecimalConverter)
{
    char pBuffer[64];
    char* pEnd = &pBuffer[sizeof(pBuffer) - 1];
    for (size_t v = 0; v < 10000U; ++v)
    {
        char* pPtr;
        nim::srv::ShopServiceAccess::ToDecimalStringFrom(pBuffer, pEnd, v);
        EXPECT_EQ(v, PRIVATE_STR_TO_ULL(pBuffer, &pPtr, 10));
    }
}

//!------------------------------------------------------------------------------------------------------
//! JSONパーサーテスト
TEST_F(TestForShopServiceAccessor, ResponseParse)
{
    typedef ::nn::http::json::JsonPath<8, 64> JsonPathType;
    Bit32 errorCode = 0;
    char jsonSourceBuffer[4 * 1024];

    // ダミーエラーコードレスポンス
    // http://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=277358966 に基づく。
    const char ErrorCodeResponseBase[] =
    "{"
        "\"error\": {"
            "\"code\": \"%s\","
            "\"message\": \"%s\""
        "}"
    "}";

    errorCode = 0;
    ASSERT_TRUE(sizeof(jsonSourceBuffer) > util::SNPrintf(jsonSourceBuffer, sizeof(jsonSourceBuffer), ErrorCodeResponseBase, "1500", "Invalid parameters."));
    NNT_ASSERT_RESULT_SUCCESS(ParseErrorCodeResponse<JsonPathType>(&errorCode, jsonSourceBuffer));
    ASSERT_EQ(errorCode, 1500);

    // 壊れたレスポンス( かなりてきとう )
    const char InvalidResponse[] =
    "{"
            "\"message\": \"%s\""
        "}"
    "}";
    errorCode = 0;
    ASSERT_TRUE(sizeof(jsonSourceBuffer) > util::SNPrintf(jsonSourceBuffer, sizeof(jsonSourceBuffer), InvalidResponse, "Invalid parameters."));
    NNT_ASSERT_RESULT_SUCCESS(ParseErrorCodeResponse<JsonPathType>(&errorCode, jsonSourceBuffer));
    ASSERT_EQ(errorCode, 0);
}

#if defined(ENABLE_SUGAR_TEST_FIXTURE)
//!------------------------------------------------------------------------------------------------------
//! 初期化/終了単体呼び出しテスト
//! トランスファーメモリのクリーニングテストも一緒に行います。
TEST_F(TestForShopServiceAccessor, InitializeAndFinalize)
{
    std::memset(g_WorkMemory, 0xAA, sizeof(g_WorkMemory));
    ASSERT_TRUE(VerifyMemoryAsValue(g_WorkMemory, sizeof(g_WorkMemory), static_cast<Bit8>(0xAA)));

    NNT_ASSERT_RESULT_SUCCESS(ec::InitializeForShopServiceAccessors(g_WorkMemory, sizeof(g_WorkMemory)));
    {
        ec::ShopServiceAccessor withFinal;
        NNT_ASSERT_RESULT_SUCCESS(withFinal.Initialize(ec::ShopService(ec::ShopService::Type_OwnedConsumableServiceItem)));
    }
    ec::ShopServiceAccessor accessor;
    NNT_ASSERT_RESULT_SUCCESS(accessor.Initialize(ec::ShopService(ec::ShopService::Type_OwnedConsumableServiceItem)));
    NNT_ASSERT_RESULT_SUCCESS(ec::FinalizeForShopServiceAccessors());

    ASSERT_TRUE(VerifyMemoryAsValue(g_WorkMemory, sizeof(g_WorkMemory), static_cast<Bit8>(0x00)));
}

//!------------------------------------------------------------------------------------------------------
//! 接続要求未提出状態要求実施テスト
TEST_F(TestForShopServiceAccessor, NotNetworkSubmitRequest)
{
    nifm::CancelNetworkRequest();

    // 単一要求→即時 Service finalize
    NNT_ASSERT_RESULT_SUCCESS(ec::InitializeForShopServiceAccessors(g_WorkMemory, sizeof(g_WorkMemory)));
    ec::ShopServiceAccessor accessor;
    NNT_ASSERT_RESULT_SUCCESS(accessor.Initialize(ec::ShopService(ec::ShopService::Type_OwnedConsumableServiceItem)));
    ec::ShopServiceAccessor::AsyncResponse response;
    NNT_ASSERT_RESULT_FAILURE(ec::ResultInternetRequestNotAccepted, RequestAsGetRightsInformation(accessor, &response, GetUser(0), 0));
    NNT_ASSERT_RESULT_SUCCESS(ec::FinalizeForShopServiceAccessors());

    nifm::SubmitNetworkRequestAndWait();
}

//!------------------------------------------------------------------------------------------------------
//! 初期化/終了単体呼び出しテスト
TEST_F(TestForShopServiceAccessor, InitializeAndFinalize2)
{
    auto& user = GetUser(0);
    {
        // 単一要求→即時 Service finalize
        NNT_ASSERT_RESULT_SUCCESS(ec::InitializeForShopServiceAccessors(g_WorkMemory, sizeof(g_WorkMemory)));
        ec::ShopServiceAccessor accessor;
        NNT_ASSERT_RESULT_SUCCESS(accessor.Initialize(ec::ShopService(ec::ShopService::Type_OwnedConsumableServiceItem)));
        ec::ShopServiceAccessor::AsyncResponse response;
        NNT_ASSERT_RESULT_SUCCESS(RequestAsGetRightsInformation(accessor, &response, GetUser(0), 0));
        NNT_ASSERT_RESULT_SUCCESS(ec::FinalizeForShopServiceAccessors());
    }
    {
        // 多量要求→即時 Service finalize
        NNT_ASSERT_RESULT_SUCCESS(ec::InitializeForShopServiceAccessors(g_WorkMemory, sizeof(g_WorkMemory)));
        ec::ShopServiceAccessor accessor;
        NNT_ASSERT_RESULT_SUCCESS(accessor.Initialize(ec::ShopService(ec::ShopService::Type_OwnedConsumableServiceItem)));
        const int requestCount = 10;
        ec::ShopServiceAccessor::AsyncResponse responseHandle[requestCount];
        for (int pageIndex = 0; pageIndex < requestCount; ++pageIndex)
        {
            NNT_ASSERT_RESULT_SUCCESS(RequestAsGetRightsInformation(accessor, &responseHandle[pageIndex], user, pageIndex));
        }
        NNT_ASSERT_RESULT_SUCCESS(ec::FinalizeForShopServiceAccessors());
    }
}

//!------------------------------------------------------------------------------------------------------
//! 同期型単体要求テスト( GET )
TEST_F(TestForShopServiceAccessor, SynchronizeRequestOnce)
{
    auto& user = GetUser(0);

    NNT_ASSERT_RESULT_SUCCESS(ec::InitializeForShopServiceAccessors(g_WorkMemory, sizeof(g_WorkMemory)));

    {
        ec::ShopServiceAccessor accessor;
        NNT_ASSERT_RESULT_SUCCESS(accessor.Initialize(ec::ShopService(ec::ShopService::Type_OwnedConsumableServiceItem)));

        ec::ShopServiceAccessor::AsyncResponse response;
        NNT_ASSERT_RESULT_SUCCESS(RequestAsGetRightsInformation(accessor, &response, user, 0));

        PRIVATE_LOG("Accessor Requested.\n");
        response.Wait();
        PRIVATE_LOG("Accessor Wait completed.\n");

        size_t responseSize;
        NNT_ASSERT_RESULT_SUCCESS(response.GetSize(&responseSize));
        if (responseSize > 0)
        {
            // サイズ未満要求のテスト
            char pFailureBuffer[8]; // 該当APIでは確実に不足するレスポンスサイズ。
            NNT_ASSERT_RESULT_FAILURE(ec::ResultShopServiceAccessInsufficientBuffer, response.Get(pFailureBuffer, sizeof(pFailureBuffer)));

            // ジャストサイズ要求で再取得検査
            CurlExecutor::Response responseBuffer(responseSize);
            NNT_ASSERT_RESULT_SUCCESS(response.Get(responseBuffer.pBuffer, responseSize));

            // レスポンスログ出力
            PRIVATE_LOG_OUT_FOR(responseBuffer.pBuffer, responseSize);
        }
        const auto invalidErrorCode = ::nn::err::ErrorCode::GetInvalidErrorCode();
        const auto errorCode = response.GetErrorCode();
        ASSERT_EQ(invalidErrorCode.category, errorCode.category);
        ASSERT_EQ(invalidErrorCode.number, errorCode.number);
    }

    PRIVATE_LOG("Service finalize calling.\n");
    NNT_ASSERT_RESULT_SUCCESS(ec::FinalizeForShopServiceAccessors());

    ASSERT_TRUE(VerifyMemoryAsValue(g_WorkMemory, sizeof(g_WorkMemory), static_cast<Bit8>(0x00)));
}

//!------------------------------------------------------------------------------------------------------
//! 非同期多重要求テスト( GET )
//! 準備済 Proxyオブジェクト上限利用で稼働するか確認。
//! ※解放する前に要求を nim::ShopServiceAccessConfiguration::AvailableAsyncAccessMax + 1 以上にしたら、Request に失敗する想定。
TEST_F(TestForShopServiceAccessor, ManyRequest)
{
    auto& user = GetUser(0);

    NNT_ASSERT_RESULT_SUCCESS(ec::InitializeForShopServiceAccessors(g_WorkMemory, sizeof(g_WorkMemory)));

    ec::ShopServiceAccessor accessor;
    NNT_ASSERT_RESULT_SUCCESS(accessor.Initialize(ec::ShopService(ec::ShopService::Type_OwnedConsumableServiceItem)));

    // 想定要求上限.
    const int requestCount = static_cast<int>(nim::ShopServiceAccessConfiguration::AvailableAsyncAccessMax);
    char responseBuffer[4096];
    {
        ec::ShopServiceAccessor::AsyncResponse responseHandle[requestCount];

        // 一気に要求
        for (int pageIndex = 0; pageIndex < requestCount; ++pageIndex)
        {
            NNT_ASSERT_RESULT_SUCCESS(RequestAsGetRightsInformation(accessor, &responseHandle[pageIndex], user, pageIndex));
            PRIVATE_LOG("Accessor requested( %d ).\n", pageIndex);
        }

        // 各同期。
        for (int pageIndex = 0; pageIndex < requestCount; ++pageIndex)
        {
            size_t responseSize;
            auto& response = responseHandle[pageIndex];
            NNT_EXPECT_RESULT_SUCCESS(response.GetSize(&responseSize));
            NNT_EXPECT_RESULT_SUCCESS(response.Get(responseBuffer, sizeof(responseBuffer)));
            responseBuffer[responseSize] = '\0';

            EXPECT_GT(static_cast<size_t>(sizeof(responseBuffer)), responseSize);
            PRIVATE_LOG("Page(%d), Response(%zu byte) =>\n%s\n\n", pageIndex, responseSize, responseBuffer);
        }

        // 解放前なので要求失敗。
        PRIVATE_LOG("Verify over request failure start.\n");
        ec::ShopServiceAccessor::AsyncResponse failureResponse;
        NNT_ASSERT_RESULT_FAILURE(ec::ResultShopServiceAccessOverRequest, RequestAsGetRightsInformation(accessor, &failureResponse, user, 0));
    }

    {
        PRIVATE_LOG("Immediate discard handle start.\n");

        // 解放後なら要求成功。
        for (int pageIndex = 0; pageIndex < requestCount; ++pageIndex)
        {
            ec::ShopServiceAccessor::AsyncResponse responseHandle;
            NNT_ASSERT_RESULT_SUCCESS(RequestAsGetRightsInformation(accessor, &responseHandle, user, pageIndex));
            PRIVATE_LOG("Accessor requested( %d ).\n", pageIndex);
            // 即閉じ。
        }

        PRIVATE_LOG("Immediate discard handle done.\n");
    }

    PRIVATE_LOG("Accessor finalize calling.\n");
    NNT_ASSERT_RESULT_SUCCESS(accessor.Finalize());

    PRIVATE_LOG("Service finalize calling.\n");
    NNT_ASSERT_RESULT_SUCCESS(ec::FinalizeForShopServiceAccessors());

    ASSERT_TRUE(VerifyMemoryAsValue(g_WorkMemory, sizeof(g_WorkMemory), static_cast<Bit8>(0x00)));
}

//!------------------------------------------------------------------------------------------------------
//! 権利確認～消費テスト( GET / POST )
TEST_F(TestForShopServiceAccessor, SynchronizeRequestAsConsume)
{
    auto& user = GetUser(0);

    NNT_ASSERT_RESULT_SUCCESS(ec::InitializeForShopServiceAccessors(g_WorkMemory, sizeof(g_WorkMemory)));

    char responseBuffer[4096];
    RightReponseAdaptor adaptor(0);
    ec::ShopServiceAccessor accessor;
    NNT_ASSERT_RESULT_SUCCESS(accessor.Initialize(ec::ShopService(ec::ShopService::Type_OwnedConsumableServiceItem)));

    {
        ec::ShopServiceAccessor::AsyncResponse response;
        NNT_ASSERT_RESULT_SUCCESS(RequestAsGetRightsInformation(accessor, &response, user, 0));
        size_t responseSize;
        NNT_EXPECT_RESULT_SUCCESS(response.GetSize(&responseSize));
        NNT_EXPECT_RESULT_SUCCESS(response.Get(responseBuffer, sizeof(responseBuffer)));
        responseBuffer[responseSize] = '\0';

        EXPECT_GT(static_cast<size_t>(sizeof(responseBuffer)), responseSize);
        PRIVATE_LOG("Response(%zu) =>\n%s\n\n", responseSize, responseBuffer);

        nim::srv::HttpJson::Stream::MemoryInputWithStaticOutput<1024> stream(responseBuffer, responseSize);
        NNT_EXPECT_RESULT_SUCCESS(http::json::ImportJsonByRapidJson<http::json::DefaultJsonErrorMap>(adaptor, stream, static_cast<nim::srv::HttpJson::Canceler*>(nullptr)));
        ASSERT_TRUE(std::strlen(adaptor.GetRightId()) > 0);
    }
#if defined(ENABLE_REGISTER_DUMMY_RIGHTS)
    {
        // AsyncResponse の再利用のチェックします。
        ec::ShopServiceAccessor::AsyncResponse response;
        {
            const char* pRightId = adaptor.GetRightId();
            const char* pConsumptionId = adaptor.GetConsumptionId();
            NNT_ASSERT_RESULT_SUCCESS(RequestAsPostConsume(accessor, &response, user, pRightId, pConsumptionId));
            size_t responseSize;
            auto result = response.GetSize(&responseSize);
            ASSERT_TRUE(result.IsSuccess());
            ASSERT_EQ(responseSize, 0U);

            responseBuffer[0] = '\0';
            result = response.Get(responseBuffer, 0);
            ASSERT_TRUE(result.IsSuccess());

            EXPECT_GT(static_cast<size_t>(sizeof(responseBuffer)), responseSize);
            PRIVATE_LOG("Response as first consumed(%zu) =>\n%s\n\n", responseSize, responseBuffer);
        }
        {
            const char* pRightId = adaptor.GetRightId();
            const char* pConsumptionId = adaptor.GetConsumptionId();
            NNT_ASSERT_RESULT_SUCCESS(RequestAsPostConsume(accessor, &response, user, pRightId, pConsumptionId));
            size_t responseSize;
            auto result = response.GetSize(&responseSize);
            ASSERT_TRUE(ec::ResultOwnedConsumableServiceItemConsumedRights::Includes(result));
            result = response.Get(responseBuffer, sizeof(responseBuffer));
            ASSERT_TRUE(ec::ResultOwnedConsumableServiceItemConsumedRights::Includes(result));
            responseBuffer[responseSize] = '\0';

            EXPECT_GT(static_cast<size_t>(sizeof(responseBuffer)), responseSize);
            PRIVATE_LOG("Response as re-consumed(%zu) =>\n%s\n\n", responseSize, responseBuffer);
        }
    }
#else   // defined(ENABLE_REGISTER_DUMMY_RIGHTS)
    {
        ec::ShopServiceAccessor::AsyncResponse response;
        const char* pRightId = adaptor.GetRightId();
        const char* pConsumptionId = adaptor.GetConsumptionId();
        NNT_ASSERT_RESULT_SUCCESS(RequestAsPostConsume(accessor, &response, user, pRightId, pConsumptionId));
        size_t responseSize;
        auto result = response.GetSize(&responseSize);
        ASSERT_TRUE(result.IsSuccess() || ec::ResultOwnedConsumableServiceItemConsumedRights::Includes(result));
        result = response.Get(responseBuffer, sizeof(responseBuffer));
        ASSERT_TRUE(result.IsSuccess() || ec::ResultOwnedConsumableServiceItemConsumedRights::Includes(result));
        responseBuffer[responseSize] = '\0';

        EXPECT_GT(static_cast<size_t>(sizeof(responseBuffer)), responseSize);
        PRIVATE_LOG("Response(%zu) =>\n%s\n\n", responseSize, responseBuffer);
    }
#endif  // defined(ENABLE_REGISTER_DUMMY_RIGHTS)

    NNT_ASSERT_RESULT_SUCCESS(ec::FinalizeForShopServiceAccessors());

    ASSERT_TRUE(VerifyMemoryAsValue(g_WorkMemory, sizeof(g_WorkMemory), static_cast<Bit8>(0x00)));
}

//!------------------------------------------------------------------------------------------------------
//! 不正要求テスト
TEST_F(TestForShopServiceAccessor, InvalidParameterRequest)
{
    char responseBuffer[4096];
    NNT_ASSERT_RESULT_SUCCESS(ec::InitializeForShopServiceAccessors(g_WorkMemory, sizeof(g_WorkMemory)));
    ec::ShopServiceAccessor accessor;
    NNT_ASSERT_RESULT_SUCCESS(accessor.Initialize(ec::ShopService(ec::ShopService::Type_OwnedConsumableServiceItem)));

    // 不正URLで要求を実施。
    ec::ShopServiceAccessor::AsyncResponse response;
    NNT_ASSERT_RESULT_SUCCESS(RequestAsGetRightsInformation(accessor, &response, GetUser(0), 0, true));

    // サーバーエラーコード 1500 (400 BadRequest)で失敗する想定。
    size_t responseSize;
    NNT_ASSERT_RESULT_FAILURE(ec::ResultShowErrorCodeRequired, response.GetSize(&responseSize));
    NNT_ASSERT_RESULT_FAILURE(ec::ResultShowErrorCodeRequired, response.Get(responseBuffer, sizeof(responseBuffer)));
    ASSERT_EQ(1500, response.GetErrorCode().number);

    responseBuffer[responseSize] = '\0';
    PRIVATE_LOG("Response =>\n%s\n\n", responseBuffer);

    NNT_ASSERT_RESULT_SUCCESS(ec::FinalizeForShopServiceAccessors());
}

//!------------------------------------------------------------------------------------------------------
//! 空ボディPOST / PUTテスト
//! @note   現状 Sugar の API には空ボディを受け取る API はありません。
//!         そのため、"/v1/rights/consume" を試験運用しており、レスポンスはサーバー次第としています。
//!         サーバーに対する要求到達の確認を目的としたテストとしています。
TEST_F(TestForShopServiceAccessor, PostAndPutWithoutBody)
{
    char responseBuffer[4096];
    NNT_ASSERT_RESULT_SUCCESS(ec::InitializeForShopServiceAccessors(g_WorkMemory, sizeof(g_WorkMemory)));
    ec::ShopServiceAccessor accessor;
    NNT_ASSERT_RESULT_SUCCESS(accessor.Initialize(ec::ShopService(ec::ShopService::Type_OwnedConsumableServiceItem)));

    ec::ShopServiceAccessor::AsyncResponse response;
    ec::ShopServiceAccessor::PostData postData = {nullptr, 0};
    NNT_ASSERT_RESULT_SUCCESS(accessor.Request(&response, GetUser(0), ec::ShopService::Method_Post, "/v1/rights/consume", postData));

    // 何らかのサーバーエラーコードで失敗する想定。
    size_t responseSize;
    NNT_ASSERT_RESULT_FAILURE(ec::ResultShowErrorCodeRequired, response.GetSize(&responseSize));
    NNT_ASSERT_RESULT_FAILURE(ec::ResultShowErrorCodeRequired, response.Get(responseBuffer, sizeof(responseBuffer)));

    NNT_ASSERT_RESULT_SUCCESS(ec::FinalizeForShopServiceAccessors());
}
#endif  // defined(ENABLE_SUGAR_TEST_FIXTURE)

#if defined(ENABLE_CIVIL_TEST_FIXTURE)

//!------------------------------------------------------------------------------------------------------
namespace {

#if defined(ENABLE_CIVIL_TEST_AS_FINISHED_FEATURES)
#define CIVIL_EXPECT_RESULT_FAILURE(actualResult_)    NNT_ASSERT_RESULT_FAILURE(ec::ResultShowErrorCodeRequired, actualResult_)
#else
#define CIVIL_EXPECT_RESULT_FAILURE(actualResult_)    NNT_ASSERT_RESULT_SUCCESS(actualResult_)
#endif  // defined(ENABLE_CIVIL_TEST_AS_FINISHED_FEATURES)

Result SimpleResponseCheck(ec::ShopServiceAccessor::AsyncResponse& response) NN_NOEXCEPT
{
    size_t responseSize;
    NN_RESULT_DO(response.GetSize(&responseSize));
    EXPECT_GT(responseSize, 0U);

    if (responseSize > 0)
    {
        // ジャストサイズ要求で再取得検査
        CurlExecutor::Response responseBuffer(responseSize);
        NN_RESULT_DO(response.Get(responseBuffer.pBuffer, responseSize));

        // レスポンスログ出力
        PRIVATE_LOG_OUT_FOR(responseBuffer.pBuffer, responseSize);
    }
    NN_RESULT_SUCCESS;
}

template<typename TAdaptor, size_t ConstantStringBufferCapacity = 1024U>
Result ParseByAdaptor(ec::ShopServiceAccessor::AsyncResponse& async, TAdaptor& adaptor) NN_NOEXCEPT
{
    size_t responseSize;
    NN_RESULT_DO(async.GetSize(&responseSize));
    if (responseSize > 0)
    {
        CurlExecutor::Response responseBuffer(responseSize);
        NN_RESULT_DO(async.Get(responseBuffer.pBuffer, responseSize));
        nim::srv::HttpJson::Stream::MemoryInputWithStaticOutput<ConstantStringBufferCapacity> stream(responseBuffer.pBuffer, responseSize);
        NN_RESULT_DO(http::json::ImportJsonByRapidJson<http::json::DefaultJsonErrorMap>(adaptor, stream, static_cast<nim::srv::HttpJson::Canceler*>(nullptr)));
        PRIVATE_LOG_OUT_FOR(responseBuffer.pBuffer, responseSize);
    }
    NN_RESULT_SUCCESS;
}

}

//!------------------------------------------------------------------------------------------------------
//! Civil: "/v1/applications/{applicationId}/consumables?country={country}&lang=%s";
TEST_F(TestForShopServiceAccessor, CivilGetCatalog)
{
    NNT_ASSERT_RESULT_SUCCESS(ec::InitializeForShopServiceAccessors(g_WorkMemory, sizeof(g_WorkMemory)));
    ec::ShopServiceAccessor accessor;
    NNT_ASSERT_RESULT_SUCCESS(accessor.Initialize(ec::ShopService(ec::ShopService::Type_Catalog)));

    ec::ShopServiceAccessor::AsyncResponse response;
    NNT_ASSERT_RESULT_SUCCESS(RequestAsGetCatalog(accessor, &response, GetUser(0)));

    NNT_ASSERT_RESULT_SUCCESS(SimpleResponseCheck(response));

    NNT_ASSERT_RESULT_SUCCESS(ec::FinalizeForShopServiceAccessors());

    ASSERT_TRUE(VerifyMemoryAsValue(g_WorkMemory, sizeof(g_WorkMemory), static_cast<Bit8>(0x00)));
}

//!------------------------------------------------------------------------------------------------------
//! Civil: "/v1/applications/{applicationId}/consumables?country={country}&lang=%s";
TEST_F(TestForShopServiceAccessor, CivilQuebecAgeRestriction)
{
    NNT_ASSERT_RESULT_SUCCESS(ec::InitializeForShopServiceAccessors(g_WorkMemory, sizeof(g_WorkMemory)));
    ec::ShopServiceAccessor accessor;
    NNT_ASSERT_RESULT_SUCCESS(accessor.Initialize(ec::ShopService(ec::ShopService::Type_Catalog)));

    // ケベック州且つ13歳以上のユーザー⇒成功
    ec::ShopServiceAccessor::AsyncResponse response;
    NNT_ASSERT_RESULT_SUCCESS(RequestAsGetCatalog(accessor, &response, GetUser(1)));
    NNT_ASSERT_RESULT_SUCCESS(SimpleResponseCheck(response));

    // ケベック州且つ13歳未満のユーザー⇒失敗
    NNT_ASSERT_RESULT_SUCCESS(RequestAsGetCatalog(accessor, &response, GetUser(2)));
    NNT_ASSERT_RESULT_FAILURE(ec::ResultAgeRestriction, SimpleResponseCheck(response));

    NNT_ASSERT_RESULT_SUCCESS(ec::FinalizeForShopServiceAccessors());

    ASSERT_TRUE(VerifyMemoryAsValue(g_WorkMemory, sizeof(g_WorkMemory), static_cast<Bit8>(0x00)));
}

//!------------------------------------------------------------------------------------------------------
//! Civil: "/v1/consumables/%s/items?country={country}&lang=%s"
TEST_F(TestForShopServiceAccessor, CivilGetConsumableItem)
{
    NNT_ASSERT_RESULT_SUCCESS(ec::InitializeForShopServiceAccessors(g_WorkMemory, sizeof(g_WorkMemory)));
    ec::ShopServiceAccessor accessor;
    NNT_ASSERT_RESULT_SUCCESS(accessor.Initialize(ec::ShopService(ec::ShopService::Type_Catalog)));

    ec::ShopServiceAccessor::AsyncResponse response;
    NNT_ASSERT_RESULT_SUCCESS(RequestAsGetConsumableItem(accessor, &response, GetUser(0), "123"));

    CIVIL_EXPECT_RESULT_FAILURE(SimpleResponseCheck(response));

    NNT_ASSERT_RESULT_SUCCESS(ec::FinalizeForShopServiceAccessors());

    ASSERT_TRUE(VerifyMemoryAsValue(g_WorkMemory, sizeof(g_WorkMemory), static_cast<Bit8>(0x00)));
}

//!------------------------------------------------------------------------------------------------------
//! 一つの Accessor で複数のAPIをコール。
TEST_F(TestForShopServiceAccessor, MultipleApiUseOnCivil)
{
    NNT_ASSERT_RESULT_SUCCESS(ec::InitializeForShopServiceAccessors(g_WorkMemory, sizeof(g_WorkMemory)));
    ec::ShopServiceAccessor accessor;
    NNT_ASSERT_RESULT_SUCCESS(accessor.Initialize(ec::ShopService(ec::ShopService::Type_Catalog)));

    auto& user = GetUser(0);
    ec::ShopServiceAccessor::AsyncResponse listAsync;
    ec::ShopServiceAccessor::AsyncResponse itemAsync;
    NNT_ASSERT_RESULT_SUCCESS(RequestAsGetCatalog(accessor, &listAsync, user));
    NNT_ASSERT_RESULT_SUCCESS(RequestAsGetConsumableItem(accessor, &itemAsync, user, "123"));

    // カタログは成功、アイテムは失敗する。
    NNT_ASSERT_RESULT_SUCCESS(SimpleResponseCheck(listAsync));
    CIVIL_EXPECT_RESULT_FAILURE(SimpleResponseCheck(itemAsync));

    {
        // カタログから正しい consumablesId 取得して再度 Item 一覧要求
        ConsumablesReponseAdaptor catalogAdaptor(0);
        NNT_ASSERT_RESULT_SUCCESS(ParseByAdaptor(listAsync, catalogAdaptor));
        ASSERT_TRUE(std::strlen(catalogAdaptor.GetConsumableId()) > 0);
        PRIVATE_LOG("Detected consumableId => [%s]\n", catalogAdaptor.GetConsumableId());

        // Item ステータスのチェック。
        ConsumableItemsReponseAdaptor itemAdaptor;
        ec::ShopServiceAccessor::AsyncResponse item;
        NNT_ASSERT_RESULT_SUCCESS(RequestAsGetConsumableItem(accessor, &item, user, catalogAdaptor.GetConsumableId()));
        NNT_ASSERT_RESULT_SUCCESS(ParseByAdaptor(item, itemAdaptor));

        struct VerifyData
        {
            const char* item_id;
            const char* sales_status;
            const char* raw_value;
        } verifyDatas[] =
        {
            {"eca0000", "onsale", "100"},
            {"eca0001", "onsale", "100"},
        };
        ASSERT_EQ(std::extent<decltype(verifyDatas)>::value, itemAdaptor.GetTotalCount());

        int index = 0;
        const auto& properties = itemAdaptor.GetProperties();
        for (auto p : properties)
        {
            PRIVATE_LOG("Item property[%d] {item_id=\"%s\", ns_uid=%llu, sales_status=\"%s\", raw_value=\"%s\"}\n",
                index, p.item_id.store, p.ns_uid.store, p.sales_status.store, p.raw_value.store
            );
            const auto& verify = verifyDatas[index];
            ASSERT_NE(0, p.ns_uid.store);
            ASSERT_EQ(0, std::strcmp(p.item_id.store, verify.item_id));
            ASSERT_EQ(0, std::strcmp(p.sales_status.store, verify.sales_status));
            ASSERT_EQ(0, std::strcmp(p.raw_value.store, verify.raw_value));
            ++index;
        }
    }
    NNT_ASSERT_RESULT_SUCCESS(ec::FinalizeForShopServiceAccessors());

    ASSERT_TRUE(VerifyMemoryAsValue(g_WorkMemory, sizeof(g_WorkMemory), static_cast<Bit8>(0x00)));
}
#endif  // defined(ENABLE_CIVIL_TEST_FIXTURE)

#if defined(ENABLE_CIVIL_TEST_FIXTURE) && defined(ENABLE_SUGAR_TEST_FIXTURE)
//!------------------------------------------------------------------------------------------------------
//! Civil と Sugar の同時利用。
TEST_F(TestForShopServiceAccessor, CivilAndSugar)
{
    NNT_ASSERT_RESULT_SUCCESS(ec::InitializeForShopServiceAccessors(g_WorkMemory, sizeof(g_WorkMemory)));

    ec::ShopServiceAccessor sugar;
    ec::ShopServiceAccessor civil;

    NNT_ASSERT_RESULT_SUCCESS(sugar.Initialize(ec::ShopService(ec::ShopService::Type_OwnedConsumableServiceItem)));
    NNT_ASSERT_RESULT_SUCCESS(civil.Initialize(ec::ShopService(ec::ShopService::Type_Catalog)));

    auto& user = GetUser(0);

    // Civil access.
    ec::ShopServiceAccessor::AsyncResponse listAsync;
    NNT_ASSERT_RESULT_SUCCESS(RequestAsGetCatalog(civil, &listAsync, user));

    // Sugar consume.
    {
        RightReponseAdaptor adaptor(0);

        ec::ShopServiceAccessor::AsyncResponse rights;
        ec::ShopServiceAccessor::AsyncResponse consume;

        size_t rightsResponseSize;
        size_t consumeResponseSize;

        // get rights.
        NNT_ASSERT_RESULT_SUCCESS(RequestAsGetRightsInformation(sugar, &rights, user, 0));
        NNT_ASSERT_RESULT_SUCCESS(rights.GetSize(&rightsResponseSize));
        CurlExecutor::Response rightsResponse(rightsResponseSize);
        NNT_ASSERT_RESULT_SUCCESS(rights.Get(rightsResponse.pBuffer, rightsResponseSize));
        PRIVATE_LOG_OUT_FOR(rightsResponse.pBuffer, rightsResponseSize);

        nim::srv::HttpJson::Stream::MemoryInputWithStaticOutput<1024> stream(rightsResponse.pBuffer, rightsResponseSize);
        NNT_EXPECT_RESULT_SUCCESS(http::json::ImportJsonByRapidJson<http::json::DefaultJsonErrorMap>(adaptor, stream, static_cast<nim::srv::HttpJson::Canceler*>(nullptr)));

        // consume.
        const char* pRightId = adaptor.GetRightId();
        ASSERT_TRUE(std::strlen(pRightId) > 0);
        const char* pConsumptionId = adaptor.GetConsumptionId();
        ASSERT_TRUE(std::strlen(pConsumptionId) > 0);

        NNT_ASSERT_RESULT_SUCCESS(RequestAsPostConsume(sugar, &consume, user, pRightId, pConsumptionId));
        auto result = consume.GetSize(&consumeResponseSize);
        ASSERT_TRUE(result.IsSuccess() || ec::ResultOwnedConsumableServiceItemConsumedRights::Includes(result));

        char consumeResponse[4096];
        result = consume.Get(consumeResponse, sizeof(consumeResponse));
        ASSERT_TRUE(result.IsSuccess() || ec::ResultOwnedConsumableServiceItemConsumedRights::Includes(result));
        consumeResponse[consumeResponseSize] = '\0';
        EXPECT_GT(static_cast<size_t>(sizeof(consumeResponse)), consumeResponseSize);
        PRIVATE_LOG("Response(%zu) =>\n%s\n\n", consumeResponseSize, consumeResponse);
    }

    // Civil
    {
        size_t responseSize;
        NNT_ASSERT_RESULT_SUCCESS(SimpleResponseCheck(listAsync));
        NNT_ASSERT_RESULT_SUCCESS(listAsync.GetSize(&responseSize));
        if (responseSize > 0)
        {
            ConsumablesReponseAdaptor adaptor(0);
            CurlExecutor::Response responseBuffer(responseSize);
            NNT_ASSERT_RESULT_SUCCESS(listAsync.Get(responseBuffer.pBuffer, responseSize));
            nim::srv::HttpJson::Stream::MemoryInputWithStaticOutput<1024> stream(responseBuffer.pBuffer, responseSize);
            NNT_EXPECT_RESULT_SUCCESS(http::json::ImportJsonByRapidJson<http::json::DefaultJsonErrorMap>(adaptor, stream, static_cast<nim::srv::HttpJson::Canceler*>(nullptr)));
            ASSERT_TRUE(std::strlen(adaptor.GetConsumableId()) > 0);
            PRIVATE_LOG("Detected consumableId => [%s]\n", adaptor.GetConsumableId());

            ec::ShopServiceAccessor::AsyncResponse itemAsync;
            NNT_ASSERT_RESULT_SUCCESS(RequestAsGetConsumableItem(civil, &itemAsync, user, adaptor.GetConsumableId()));

            // 取得アイテム状態の検証は MultipleApiUseOnCivil テストで行っているので、こちらはしない。
            NNT_ASSERT_RESULT_SUCCESS(SimpleResponseCheck(itemAsync));
        }
    }

    NNT_ASSERT_RESULT_SUCCESS(ec::FinalizeForShopServiceAccessors());

    ASSERT_TRUE(VerifyMemoryAsValue(g_WorkMemory, sizeof(g_WorkMemory), static_cast<Bit8>(0x00)));
}
#endif  // defined(ENABLE_CIVIL_TEST_FIXTURE) && defined(ENABLE_SUGAR_TEST_FIXTURE)

//!------------------------------------------------------------------------------------------------------
//! デバッグ機能テスト ( レスポンス受信 )
TEST_F(TestForShopServiceAccessor, DebugResponse)
{
    {
        ec::DebugForShopServiceAccessor::ScopedSession session;
        ec::DebugForShopServiceAccessor::Initialize(&session);
        DebugResponse::EnableFirmwareSettings(true);

        ec::DebugForShopServiceAccessor::ClearResponse();
        ec::DebugForShopServiceAccessor::RegisterResponse(ec::ShopService(ec::ShopService::Type_OwnedConsumableServiceItem)
            , "/v1/rights/my?app01002ad00"
            , DebugResponse::BasicResponse1
            , DebugResponse::SizeofBasicResponse1
            , nn::ResultSuccess()
            , 10000
        );
        ec::DebugForShopServiceAccessor::RegisterResponse(ec::ShopService(ec::ShopService::Type_Catalog)
            , "/v1/applications/{applicationId}/consumables?"
            , DebugResponse::BasicResponse2
            , DebugResponse::SizeofBasicResponse2
            , ec::ResultShopServiceAccessOutOfResource()
            , 10000
        );
    }
    {
        auto& user = GetUser(0);

        char responseBuffer[4096];
        NNT_ASSERT_RESULT_SUCCESS(ec::InitializeForShopServiceAccessors(g_WorkMemory, sizeof(g_WorkMemory)));
        ec::ShopServiceAccessor sugar;
        ec::ShopServiceAccessor civil;
        NNT_ASSERT_RESULT_SUCCESS(sugar.Initialize(ec::ShopService(ec::ShopService::Type_OwnedConsumableServiceItem)));
        NNT_ASSERT_RESULT_SUCCESS(civil.Initialize(ec::ShopService(ec::ShopService::Type_Catalog)));

        ec::ShopServiceAccessor::AsyncResponse resSugar;
        ec::ShopServiceAccessor::AsyncResponse resCivil;
        NNT_ASSERT_RESULT_SUCCESS(sugar.Request(&resSugar, user, ec::ShopService::Method_Get, "/v1/rights/my?app{applicationId}"));
        NNT_ASSERT_RESULT_SUCCESS(civil.Request(&resCivil, user, ec::ShopService::Method_Get, "/v1/applications/{applicationId}/consumables?country={country}&lang=ja"));

        {
            // 本来はサーバーエラーコード 1500 (400 BadRequest)で失敗する想定だが、HTTP 200 で成功する。
            size_t responseSize;
            NNT_ASSERT_RESULT_SUCCESS(resSugar.GetSize(&responseSize));
            ASSERT_EQ(DebugResponse::SizeofBasicResponse1, responseSize);
            NNT_ASSERT_RESULT_SUCCESS(resSugar.Get(responseBuffer, sizeof(responseBuffer)));
            responseBuffer[responseSize] = '\0';
            PRIVATE_LOG("Response =>\n%s\n\n", responseBuffer);
            ASSERT_EQ(0, ::nn::util::Strncmp(responseBuffer, DebugResponse::BasicResponse1, responseSize));
        }
        {
            // 本来は成功する想定だが、ResultShopServiceAccessOutOfResource で失敗する。
            size_t responseSize;
            NNT_EXPECT_RESULT_FAILURE(ec::ResultShopServiceAccessOutOfResource, resCivil.GetSize(&responseSize));
            EXPECT_EQ(0, responseSize);
            NNT_EXPECT_RESULT_FAILURE(ec::ResultShopServiceAccessOutOfResource, resCivil.Get(responseBuffer, sizeof(responseBuffer)));
            responseBuffer[responseSize] = '\0';
            PRIVATE_LOG("Response =>\n%s\n\n", responseBuffer);
        }
        NNT_ASSERT_RESULT_SUCCESS(ec::FinalizeForShopServiceAccessors());
    }
}

//!------------------------------------------------------------------------------------------------------
//! デバッグ機能テスト ( 長大レスポンス受信 )
TEST_F(TestForShopServiceAccessor, DebugResponseLong)
{
    {
        ec::DebugForShopServiceAccessor::ScopedSession session;
        ec::DebugForShopServiceAccessor::Initialize(&session);
        DebugResponse::EnableFirmwareSettings(true);

        ec::DebugForShopServiceAccessor::ClearResponse();
        ec::DebugForShopServiceAccessor::RegisterResponse(ec::ShopService(ec::ShopService::Type_OwnedConsumableServiceItem)
            , "/v1/rights/my?app01002ad00"
            , DebugResponse::BasicResponseLong
            , DebugResponse::SizeofBasicResponseLong
            , nn::ResultSuccess()
            , 10000
        );
    }
    {
        auto& user = GetUser(0);
        ec::ShopServiceAccessor sugar;
        char responseBuffer[16 * 1024];
        ec::ShopServiceAccessor::AsyncResponse resSugar;
        NN_STATIC_ASSERT(sizeof(responseBuffer) > DebugResponse::SizeofBasicResponseLong);

        NNT_ASSERT_RESULT_SUCCESS(ec::InitializeForShopServiceAccessors(g_WorkMemory, sizeof(g_WorkMemory)));
        NNT_ASSERT_RESULT_SUCCESS(sugar.Initialize(ec::ShopService(ec::ShopService::Type_OwnedConsumableServiceItem)));
        NNT_ASSERT_RESULT_SUCCESS(sugar.Request(&resSugar, user, ec::ShopService::Method_Get, "/v1/rights/my?app{applicationId}"));
        {
            // 本来はサーバーエラーコード 1500 (400 BadRequest)で失敗する想定だが、HTTP 200 で成功する。
            size_t responseSize;
            NNT_ASSERT_RESULT_SUCCESS(resSugar.GetSize(&responseSize));
            ASSERT_EQ(DebugResponse::SizeofBasicResponseLong, responseSize);
            NNT_ASSERT_RESULT_SUCCESS(resSugar.Get(responseBuffer, sizeof(responseBuffer)));
            ASSERT_TRUE(DebugResponse::VerifyLongResponse(responseBuffer));
        }
        NNT_ASSERT_RESULT_SUCCESS(ec::FinalizeForShopServiceAccessors());
    }
}

#if defined(ENABLE_SUGAR_TEST_FIXTURE)

#if 0
//!------------------------------------------------------------------------------------------------------
//! クライアントアボートテスト ( 結果の手動確認が必要 )
//!     通信要求中のアボート( 異常切断 )でもサーバープロセスに影響を与えない事の保証。
//! TODO: testlist で本ABORT異常終了テスト後に、次の通常動作が行えるテスト構成を作る。
TEST_F(TestForShopServiceAccessor, AbortClient)
{
    auto& user = GetUser(0);

    NNT_ASSERT_RESULT_SUCCESS(ec::InitializeForShopServiceAccessors(g_WorkMemory, sizeof(g_WorkMemory)));

    {
        ec::ShopServiceAccessor accessor;
        NNT_ASSERT_RESULT_SUCCESS(accessor.Initialize(ec::ShopService(ec::ShopService::Type_OwnedConsumableServiceItem)));

        ec::ShopServiceAccessor::AsyncResponse response;
        NNT_ASSERT_RESULT_SUCCESS(RequestAsGetRightsInformation(accessor, &response, user, 0));

        NN_ABORT("[ShopServiceAccessor::TEST] Force abort !!");
    }

    PRIVATE_LOG("Service finalize calling.\n");
    NNT_ASSERT_RESULT_SUCCESS(ec::FinalizeForShopServiceAccessors());
}
#endif

//!------------------------------------------------------------------------------------------------------
//! Finalize 忘れテスト ( 結果の手動確認が必要 )
//!     Finalize API 呼び出し漏れでも、サーバープロセスに影響を与えない事の保証。
//!     ec ライブラリは FinalizeForShopServiceAccessors() を呼ばない場合、次の InitializeForShopServiceAccessors() で事前条件違反で ABORT します。
//!     なので、テスト結果の厳密な判断は以下で行います。
//!         - 本テストを全テストの最終で実施。
//!         - ログ中の [PASSED] 検出後、"[ShopServiceAccess::Server] Destruction: done( SchedulerStack[XXXXXXXXXXXXXXXX] )." が存在する。
//! TODO: 外部からログ確認するようなテストを作成する。
TEST_F(TestForShopServiceAccessor, LeakFinalizeByClient)
{
    auto& user = GetUser(0);

    NNT_ASSERT_RESULT_SUCCESS(ec::InitializeForShopServiceAccessors(g_WorkMemory, sizeof(g_WorkMemory)));

    ec::ShopServiceAccessor accessor;
    NNT_ASSERT_RESULT_SUCCESS(accessor.Initialize(ec::ShopService(ec::ShopService::Type_OwnedConsumableServiceItem)));

    ec::ShopServiceAccessor::AsyncResponse response;
    NNT_ASSERT_RESULT_SUCCESS(RequestAsGetRightsInformation(accessor, &response, user, 0));
}

#endif  // defined(ENABLE_SUGAR_TEST_FIXTURE)

#endif // defined( NN_BUILD_CONFIG_OS_HORIZON )
