﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <string>
#include <vector>

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nsutil/nsutil_InstallUtils.h>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <curl/curl.h>
#include <nn/os.h>
#include <nn/fs.h>
#include <nn/fs/fs_Bis.h>
#include <nn/socket.h>

#include <nn/ncm/ncm_Service.h>
#include <nn/ncm/ncm_Result.h>
#include <nn/ncm/ncm_ContentManagementUtil.h>
#include <nn/ns/ns_Result.h>
#include <nn/nifm.h>
#include <nn/nim/nim_Result.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/nim/nim_NetworkInstallManagerApi.h>
#include <nn/ec/system/ec_DeviceAuthenticationApi.h>

namespace {

    class NetworkInstallTest : public testing::Test, public nnt::nsutil::ApplicationInstaller
    {
    protected:
        virtual void SetUp()
        {
#if defined( NN_BUILD_CONFIG_OS_WIN )
            SetupBisWorkingDirectory(nnt::GetHostArgv()[1]);
            m_OutputDirectoryPath = std::string(nnt::GetHostArgv()[2]);
            NN_LOG("outputDirectoryPath %s\n", m_OutputDirectoryPath.c_str());
#endif
            nn::ncm::Initialize();
        }

        virtual void TearDown()
        {
            nn::ncm::Finalize();
#if defined( NN_BUILD_CONFIG_OS_WIN )
            NN_ABORT_UNLESS_RESULT_SUCCESS(TearDownBisWorkingDirectory());
#endif
        }

        static void SetUpTestCase()
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::Initialize());
            nn::nifm::SubmitNetworkRequestAndWait();

#if defined( NN_BUILD_CONFIG_OS_WIN )
            NN_ASSERT(nnt::GetHostArgc() > 2);
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountHostRoot());
            static NN_ALIGNAS(4096) uint8_t s_SocketMemoryPoolBuffer[nn::socket::DefaultSocketMemoryPoolSize];
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::socket::Initialize(reinterpret_cast<void*>(s_SocketMemoryPoolBuffer),
                                                                  nn::socket::DefaultSocketMemoryPoolSize,
                                                                  nn::socket::DefaultSocketAllocatorSize,
                                                                  nn::socket::DefaultConcurrencyLimit));
            auto result = curl_global_init(CURL_GLOBAL_ALL);
            NN_ASSERT_EQUAL(CURLE_OK, result);
#endif
        }

        static void TearDownTestCase()
        {
#if defined( NN_BUILD_CONFIG_OS_WIN )
            curl_global_cleanup();
            nn::socket::Finalize();
            nn::fs::UnmountHostRoot();
#endif
            nn::nifm::CancelNetworkRequest();
        }

        const char* GetOutputDirectoryPath()
        {
            return m_OutputDirectoryPath.c_str();
        }

        std::vector<nn::ncm::ContentMetaKey> GetLatestVersionList()
        {
            nn::ncm::ContentMetaKey array[] = {
                nn::ncm::ContentMetaKey::Make(0x0000000000000010, 0, nn::ncm::ContentMetaType::Application),
                nn::ncm::ContentMetaKey::Make(0x0000000000000011, 0, nn::ncm::ContentMetaType::Application),
            };

            return std::vector<nn::ncm::ContentMetaKey>(array, std::end(array));
        }

        std::vector<nn::ncm::ApplicationContentMetaKey> GetNetworkInstallList()
        {
            nn::ncm::ApplicationContentMetaKey array[] = {
                { nn::ncm::ContentMetaKey::Make(0x0000000000000810, 0, nn::ncm::ContentMetaType::Patch),{ 0x0000000000000010 } },
                { nn::ncm::ContentMetaKey::Make(0x0000000000000811, 0, nn::ncm::ContentMetaType::Patch),{ 0x0000000000000011 } }
            };

            return std::vector<nn::ncm::ApplicationContentMetaKey>(array, std::end(array));
        }

        std::vector<nn::ncm::ContentMetaKey> GetSystemContentMetaList()
        {
            nn::ncm::ContentMetaKey array[] = {
                nn::ncm::ContentMetaKey::Make(0x0000000000000004, 0, nn::ncm::ContentMetaType::SystemUpdate),
                nn::ncm::ContentMetaKey::Make(0x0000000000000005, 0, nn::ncm::ContentMetaType::SystemData),
                nn::ncm::ContentMetaKey::Make(0x0000000000000006, 0, nn::ncm::ContentMetaType::SystemData)
            };

            return std::vector<nn::ncm::ContentMetaKey>(array, std::end(array));
        }

    private:
        std::string m_OutputDirectoryPath;
    };
}

TEST_F(NetworkInstallTest, NetworkInstallTaskManager)
{
    nn::nim::InitializeForNetworkInstallManager();

    {
        nn::nim::NetworkInstallTaskId idListArray[64];
        int count = nn::nim::ListNetworkInstallTask(idListArray, sizeof(idListArray) / sizeof(idListArray[0]));
        for (int i = 0; i < count; i++)
        {
            NNT_EXPECT_RESULT_SUCCESS(nn::nim::DestroyNetworkInstallTask(idListArray[i]));
        }
    }

    {
        nn::nim::NetworkInstallTaskId listed;
        auto count = nn::nim::ListNetworkInstallTask(&listed, 1);
        EXPECT_EQ(0, count);
    }

    auto list = GetNetworkInstallList();
    for (auto key : list)
    {
        DeleteAllContents(key.key.id, nn::ncm::StorageId::BuildInUser);
        nn::nim::NetworkInstallTaskId id;
        NNT_EXPECT_RESULT_SUCCESS(nn::nim::CreateNetworkInstallTask(&id, key.applicationId, &key.key, 1, nn::ncm::StorageId::BuildInUser));
    }

    nn::nim::NetworkInstallTaskId idListArray[64];
    int count = nn::nim::ListNetworkInstallTask(idListArray, sizeof(idListArray) / sizeof(idListArray[0]));
    EXPECT_EQ(list.size(), count);

    std::vector<nn::nim::NetworkInstallTaskId> idList(idListArray, &idListArray[count]);
    std::vector<std::unique_ptr<nn::nim::AsyncResult>> asyncResultList;

    for (auto id : idList)
    {
        nn::nim::NetworkInstallTaskInfo info;
        NNT_EXPECT_RESULT_SUCCESS(nn::nim::GetNetworkInstallTaskInfo(&info, id));
        EXPECT_EQ(nn::ncm::InstallProgressState::NotPrepared, info.progress.state);

        std::unique_ptr<nn::nim::AsyncResult> asyncResult(new nn::nim::AsyncResult());
        NNT_EXPECT_RESULT_SUCCESS(nn::nim::RequestNetworkInstallTaskRun(asyncResult.get(), id));
        asyncResultList.push_back(std::move(asyncResult));
    }

    while (NN_STATIC_CONDITION(true))
    {
        for (auto id : idList)
        {
            nn::nim::NetworkInstallTaskInfo info;
            NNT_EXPECT_RESULT_SUCCESS(nn::nim::GetNetworkInstallTaskInfo(&info, id));
            NN_LOG("id 0x%016llx, installed %lld, total %lld\n", info.applicationId, info.progress.installedSize, info.progress.totalSize);
        }

        bool taskRemains = false;
        for (auto& asyncResult : asyncResultList)
        {
            if (!asyncResult->TryWait())
            {
                taskRemains = true;
                break;
            }
        }

        if (!taskRemains)
        {
            break;
        }

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }

    for (auto& asyncResult : asyncResultList)
    {
        NNT_EXPECT_RESULT_SUCCESS(asyncResult->Get());
    }

    for (auto id : idList)
    {
        nn::nim::NetworkInstallTaskInfo info;
        NNT_EXPECT_RESULT_SUCCESS(nn::nim::GetNetworkInstallTaskInfo(&info, id));

        EXPECT_EQ(nn::ncm::InstallProgressState::Downloaded, info.progress.state);
    }

    for (auto id : idList)
    {
        nn::nim::AsyncResult asyncResult;
        NNT_EXPECT_RESULT_FAILURE(nn::nim::ResultTaskStillRunning, nn::nim::RequestNetworkInstallTaskRun(&asyncResult, id));
        NNT_EXPECT_RESULT_FAILURE(nn::nim::ResultTaskStillRunning, nn::nim::CommitNetworkInstallTask(id));
        NNT_EXPECT_RESULT_FAILURE(nn::nim::ResultTaskStillRunning, nn::nim::DestroyNetworkInstallTask(id));
    }

    for (auto& asyncResult : asyncResultList)
    {
        asyncResult->Finalize();
    }

    for (auto id : idList)
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::nim::CommitNetworkInstallTask(id));
        NNT_EXPECT_RESULT_SUCCESS(nn::nim::DestroyNetworkInstallTask(id));
    }

    nn::nim::FinalizeForNetworkInstallManager();
} // NOLINT(impl/function_size)

TEST_F(NetworkInstallTest, ApplicationControl)
{
    nn::nim::InitializeForNetworkInstallManager();

    {
        auto list = GetNetworkInstallList();
        std::vector<std::unique_ptr<nn::nim::AsyncApplicationControlInfo>> asyncControlInfoList;
        for (size_t i = 0; i < list.size(); i++)
        {
            auto key = list[i];

            std::unique_ptr<nn::nim::AsyncApplicationControlInfo> async(new nn::nim::AsyncApplicationControlInfo());
            NNT_EXPECT_RESULT_SUCCESS(nn::nim::RequestApplicationControl(async.get(), key.applicationId, key.key.version));
            asyncControlInfoList.push_back(std::move(async));
        }

        while (NN_STATIC_CONDITION(true))
        {
            bool remains = false;
            for (auto& async : asyncControlInfoList)
            {
                if (!async->TryWait())
                {
                    remains = true;
                    break;
                }
            }

            if (!remains)
            {
                break;
            }

            NN_LOG("Waiting application control...\n");
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        }

        for (auto& async : asyncControlInfoList)
        {
            nn::nim::ApplicationControlInfo info;
            NNT_EXPECT_RESULT_SUCCESS(async->Get(&info));
            NN_LOG("path %s\n", info.path.string);
        }
    }

    nn::nim::FinalizeForNetworkInstallManager();
} // NOLINT(impl/function_size)

TEST_F(NetworkInstallTest, LatestVersion)
{
    nn::nim::InitializeForNetworkInstallManager();

    {
        auto list = GetLatestVersionList();

        nn::nim::AsyncLatestKeyList asyncList;
        NNT_EXPECT_RESULT_SUCCESS(nn::nim::RequestLatestVersion(&asyncList, { list.data()[0].id }));

        while (!asyncList.TryWait())
        {
            NN_LOG("Waiting latest version...\n");
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        }

        std::vector<nn::ncm::ContentMetaKey> keyList;
        keyList.resize(list.size());

        int listCount = static_cast<int>(keyList.size());
        int count;
        NNT_EXPECT_RESULT_SUCCESS(asyncList.Get(&count, keyList.data(), listCount));
        EXPECT_EQ(0, count);
    }

    nn::nim::FinalizeForNetworkInstallManager();
} // NOLINT(impl/function_size)

TEST_F(NetworkInstallTest, SystemUpdateTaskManager)
{
    nn::nim::InitializeForNetworkInstallManager();

    auto list = GetSystemContentMetaList();
    for (auto& meta : list)
    {
        DeleteAllContents(meta.id, nn::ncm::StorageId::BuildInSystem);
    }

    nn::ncm::ContentMetaKey key;
    {
        {
            nn::nim::AsyncContentMetaKey meta;
            NNT_EXPECT_RESULT_SUCCESS(nn::nim::RequestSystemUpdateMeta(&meta));
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
            meta.Cancel();

            while (!meta.TryWait())
            {
                NN_LOG("Waiting system update meta canceled...\n");

                nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
            }
            NNT_EXPECT_RESULT_FAILURE(nn::nim::ResultHttpConnectionCanceled, meta.Get(&key));
        }

        nn::nim::AsyncContentMetaKey meta;
        NNT_EXPECT_RESULT_SUCCESS(nn::nim::RequestSystemUpdateMeta(&meta));
        while (!meta.TryWait())
        {
            NN_LOG("Waiting system update meta...\n");

            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        }
        NNT_EXPECT_RESULT_SUCCESS(meta.Get(&key));
        NN_LOG("system update meta id 0x%016llx, version %u\n", key.id, key.version);
        EXPECT_EQ(nn::ncm::ContentMetaType::SystemUpdate, key.type);
    }

    nn::nim::SystemUpdateTaskId listed;
    auto count = nn::nim::ListSystemUpdateTask(&listed, 1);
    EXPECT_EQ(0, count);

    nn::nim::SystemUpdateTaskId id;
    NNT_EXPECT_RESULT_SUCCESS(nn::nim::CreateSystemUpdateTask(&id, key, false));

    count = nn::nim::ListSystemUpdateTask(&listed, 1);
    EXPECT_EQ(1, count);
    EXPECT_EQ(id, listed);

    nn::nim::SystemUpdateTaskInfo info;
    NNT_EXPECT_RESULT_SUCCESS(nn::nim::GetSystemUpdateTaskInfo(&info, id));
    EXPECT_EQ(nn::ncm::InstallProgressState::NotPrepared, info.progress.state);

    {
        nn::nim::AsyncResult asyncResult;
        NNT_EXPECT_RESULT_SUCCESS(nn::nim::RequestSystemUpdateTaskRun(&asyncResult, id));
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        asyncResult.Cancel();

        while (!asyncResult.TryWait())
        {
            NN_LOG("Waiting download canceled...\n");

            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        }
        NNT_EXPECT_RESULT_FAILURE(nn::nim::ResultHttpConnectionCanceled, asyncResult.Get());
    }

    nn::nim::AsyncResult asyncResult;
    NNT_EXPECT_RESULT_SUCCESS(nn::nim::RequestSystemUpdateTaskRun(&asyncResult, id));
    while (!asyncResult.TryWait())
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::nim::GetSystemUpdateTaskInfo(&info, id));

        NN_LOG("installed %lld, total %lld\n", info.progress.installedSize, info.progress.totalSize);

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }

    NNT_EXPECT_RESULT_SUCCESS(asyncResult.Get());
    NNT_EXPECT_RESULT_SUCCESS(nn::nim::GetSystemUpdateTaskInfo(&info, id));

    EXPECT_EQ(nn::ncm::InstallProgressState::Downloaded, info.progress.state);

    NNT_EXPECT_RESULT_FAILURE(nn::nim::ResultTaskStillRunning, nn::nim::CommitSystemUpdateTask(id));
    NNT_EXPECT_RESULT_FAILURE(nn::nim::ResultTaskStillRunning, nn::nim::DestroySystemUpdateTask(id));

    asyncResult.Finalize();

    NNT_EXPECT_RESULT_SUCCESS(nn::nim::CommitSystemUpdateTask(id));
    NNT_EXPECT_RESULT_SUCCESS(nn::nim::DestroySystemUpdateTask(id));

    {
        nn::nim::AsyncContentMetaKey meta;
        NNT_EXPECT_RESULT_SUCCESS(nn::nim::RequestSystemUpdateMeta(&meta));
        while (!meta.TryWait())
        {
            NN_LOG("Waiting system update meta...\n");

            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        }
        NNT_EXPECT_RESULT_FAILURE(nn::ncm::ResultContentAlreadyExists, meta.Get(&key));
    }

    nn::nim::FinalizeForNetworkInstallManager();
} // NOLINT(impl/function_size)

TEST_F(NetworkInstallTest, DeviceAuthenticationTokenTaskManager)
{
    nn::nim::InitializeForNetworkInstallManager();

    nn::ec::system::DeviceAuthenticationToken token;
    {
        nn::ec::system::AsyncDeviceAuthenticationToken asyncToken;
        NNT_EXPECT_RESULT_SUCCESS(nn::ec::system::RequestDeviceAuthenticationToken(&asyncToken));
        asyncToken.Cancel();

        while (!asyncToken.TryWait())
        {
            NN_LOG("Waiting device authentication token canceled...\n");

            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        }
        // TODO: ec Result 対応
        EXPECT_TRUE(asyncToken.Get(&token).IsFailure());

        asyncToken.Finalize();
    }

    nn::ec::system::AsyncDeviceAuthenticationToken asyncToken;
    NNT_EXPECT_RESULT_SUCCESS(nn::ec::system::RequestDeviceAuthenticationToken(&asyncToken));
    while (!asyncToken.TryWait())
    {
        NN_LOG("Waiting device authentication token...\n");

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }
    NNT_EXPECT_RESULT_SUCCESS(asyncToken.Get(&token));
    NN_LOG("Device authentication token is %s\n", token.data);

    asyncToken.Finalize();

    nn::nim::FinalizeForNetworkInstallManager();
}
