﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>
#include <nnt/result/testResult_Assert.h>

#include <nn/os.h>
#include <nn/fs.h>
#include <nn/init.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/socket.h>

#include <nn/nifm/nifm_Api.h>
#include <nn/nifm/nifm_ApiForMenu.h>
#include <nn/nifm/nifm_ApiForNetworkMiddleware.h>
#include <nn/nifm/nifm_ApiForTest.h>
#include <nn/nifm/nifm_ApiClientManagement.h>
#include <nn/nifm/nifm_TemporaryNetworkProfile.h>
#include <nn/nifm/nifm_Request.h>
#include <nn/nifm/nifm_ApiRequest.h>
#include <nn/nifm/nifm_NetworkConnection.h>
#include <nn/nifm/nifm_Result.h>
#include <nn/nifm/nifm_ResultPrivate.h> // TODO: こちらは不要になるべき
#include <nn/nifm/nifm_TypesRequest.h>
#include <nn/nifm/nifm_TypesNetworkProfile.h>
#include <nn/nifm/nifm_TypesNetworkInterface.h>

#include <nn/util/util_Uuid.h>

#include <curl/curl.h>
#include <algorithm>

#include "../Common/nifm_TestUtility.h"

namespace
{
    char NN_ALIGNAS(4096) g_SocketMemoryPoolBuffer[nn::socket::DefaultSocketMemoryPoolSize];

    const int64_t TIME_OUT_IN_SECONDS_FOR_SUCCESS = 60;

    void GetConntestServerResponse()
    {
        CURL *curl;
        CURLcode res;

        NNT_ASSERT_RESULT_SUCCESS( nn::socket::Initialize(g_SocketMemoryPoolBuffer, nn::socket::DefaultSocketMemoryPoolSize, nn::socket::DefaultSocketAllocatorSize, nn::socket::DefaultConcurrencyLimit) );

        //curl_global_init(CURL_GLOBAL_DEFAULT);
        curl = curl_easy_init();

        if (curl)
        {
            curl_easy_setopt(curl, CURLOPT_URL, "http://ctest.cdn.nintendo.net/");
            //curl_easy_setopt(curl, CURLOPT_URL, "http://69.25.139.140");
            curl_easy_setopt(curl, CURLOPT_VERBOSE, 1L);

            res = curl_easy_perform(curl);

            if (res != CURLE_OK)
            {
                NN_LOG("curl_easy_perform() failed\n");
            }
            curl_easy_cleanup(curl);
        }
        else
        {
            NN_LOG("curl_easy_init() failed\n");
        }

        //curl_global_cleanup();
        nn::socket::Finalize();
    }
}

class SimpleConnectCaseTest : public ::testing::Test
{
protected:
    static void SetUpTestCase()
    {
        // 本テストプロセスで独占
        nn::nifm::test::SetExclusive<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest>(true);

        curl_global_init(CURL_GLOBAL_DEFAULT);
    }

    static void TearDownTestCase()
    {
        curl_global_cleanup();

        // 本テストプロセスで独占解除
        nn::nifm::test::SetExclusive<nn::nifm::InitializeAdmin, nn::nifm::FinalizeAdminForTest>(false);
    }
};

TEST_F(SimpleConnectCaseTest, NetworkInterfaceList)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::InitializeAdmin());

    nn::nifm::NetworkInterfaceInfo networkInterfaceInfo[8];
    int bufferSize;
    int outCount;

    // 最初の scan 終了後の NIC 登録を待つ workaround
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(15));

    // 利用対象の NIC を列挙
    bufferSize = nn::nifm::NetworkInterfaceCountMax;
    NNT_ASSERT_RESULT_SUCCESS(
        nn::nifm::EnumerateNetworkInterfaces(
            networkInterfaceInfo, &outCount, bufferSize,
            nn::nifm::EnumerateNetworkInterfacesFilter_Ieee80211 | nn::nifm::EnumerateNetworkInterfacesFilter_Ethernet
        )
    );
    nn::nifm::test::PrintNetworkInterfaces(networkInterfaceInfo, bufferSize, outCount);

    // 利用対象の無線 NIC を列挙
    bufferSize = nn::nifm::NetworkInterfaceCountMax;
    NNT_ASSERT_RESULT_SUCCESS(
        nn::nifm::EnumerateNetworkInterfaces(
            networkInterfaceInfo, &outCount, bufferSize,
            nn::nifm::EnumerateNetworkInterfacesFilter_Ieee80211
        )
    );
    nn::nifm::test::PrintNetworkInterfaces(networkInterfaceInfo, bufferSize, outCount);

    // 利用対象の有線 NIC を列挙
    bufferSize = nn::nifm::NetworkInterfaceCountMax;
    NNT_ASSERT_RESULT_SUCCESS(
        nn::nifm::EnumerateNetworkInterfaces(
            networkInterfaceInfo, &outCount, bufferSize,
            nn::nifm::EnumerateNetworkInterfacesFilter_Ethernet
        )
    );
    nn::nifm::test::PrintNetworkInterfaces(networkInterfaceInfo, bufferSize, outCount);

    // 無線 NIC を全部列挙
    bufferSize = 8;
    NNT_ASSERT_RESULT_SUCCESS(
        nn::nifm::EnumerateNetworkInterfaces(
            networkInterfaceInfo, &outCount, bufferSize,
            nn::nifm::EnumerateNetworkInterfacesFilter_Ieee80211 | nn::nifm::EnumerateNetworkInterfacesFilter_Extended
        )
    );
    nn::nifm::test::PrintNetworkInterfaces(networkInterfaceInfo, bufferSize, outCount);

    // 有線 NIC を全部列挙
    bufferSize = 8;
    NNT_ASSERT_RESULT_SUCCESS(
        nn::nifm::EnumerateNetworkInterfaces(
            networkInterfaceInfo, &outCount, bufferSize,
            nn::nifm::EnumerateNetworkInterfacesFilter_Ethernet | nn::nifm::EnumerateNetworkInterfacesFilter_Extended
        )
    );
    nn::nifm::test::PrintNetworkInterfaces(networkInterfaceInfo, bufferSize, outCount);

    // NIC を全部列挙
    bufferSize = 8;
    NNT_ASSERT_RESULT_SUCCESS(
        nn::nifm::EnumerateNetworkInterfaces(
            networkInterfaceInfo, &outCount, bufferSize,
            nn::nifm::EnumerateNetworkInterfacesFilter_Ieee80211 | nn::nifm::EnumerateNetworkInterfacesFilter_Ethernet | nn::nifm::EnumerateNetworkInterfacesFilter_Extended
        )
    );
    nn::nifm::test::PrintNetworkInterfaces(networkInterfaceInfo, bufferSize, outCount);

    // フィルタを指定しなければ何も取れない
    bufferSize = nn::nifm::NetworkInterfaceCountMax;
    NNT_ASSERT_RESULT_SUCCESS(
        nn::nifm::EnumerateNetworkInterfaces(
            networkInterfaceInfo, &outCount, bufferSize,
            0
        )
    );
    ASSERT_EQ(outCount, 0);

    // 定義外のフィルタビットを立てると失敗
    bufferSize = nn::nifm::NetworkInterfaceCountMax;
    NNT_ASSERT_RESULT_FAILURE(
        nn::nifm::ResultInvalidArgument,
        nn::nifm::EnumerateNetworkInterfaces(
            networkInterfaceInfo, &outCount, bufferSize,
            ~static_cast<nn::Bit32>(0)
        )
    );
}

TEST_F(SimpleConnectCaseTest, TemporaryConnectWifi)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::InitializeAdmin());

    {
        nn::nifm::AccessPointData accessPoint;
        NNT_EXPECT_RESULT_FAILURE(nn::nifm::ResultNotConnected, nn::nifm::GetCurrentAccessPoint(&accessPoint));
    }

    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Scan());

    const int bufferSize = nn::nifm::AccessPointCountMax;
    nn::nifm::AccessPointData accessPointData[bufferSize];
    int32_t outCount;
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetScanData(accessPointData, &outCount, bufferSize));
    int count = std::min(outCount, bufferSize);
    for (int i = 0; i < count; ++i)
    {
        nn::nifm::test::PrintAccessPoint( accessPointData[i] );
    }

    {
        // imatake-wpa2aes
        const nn::nifm::WirelessSettingData wirelessSetting0 = {
            {  // ssidConfig
                {  // ssid
                    15,  // length
                    {0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x77,0x70,0x61,0x32,0x61,0x65,0x73}  // hex
                },
                false  // nonBroadcast
            },
            {  // security
                {  // authEncryption
                    nn::nifm::Authentication_Wpa2Psk,  // authentication
                    nn::nifm::Encryption_Aes  // encryption
                },
                {  // sharedKey
                    11,  // length
                    "Shi2iTaiZen"  // keyMaterial
                }
            }
        };
        // ex.DHCP
        const nn::nifm::IpSettingData ipSetting0 = {
            {  // ip
                true,  // isAuto
                {},  // ipAddress
                {},  // subnetMask
                {}  // defaultGateway
            },
            {  // dns
                true,  // isAuto
                {},  // preferredDns
                {}  // alternateDns
            },
            {  // proxy
                false,  // isEnabled
                0,  // port
                "",  // proxy
                {  // authentication
                    false,  // isEnabled
                    "",  // username
                    ""  // password
                }
            },
            1400  //mtu
        };
        const nn::nifm::NetworkProfileData networkProfile0 = {
            nn::util::InvalidUuid,  // id
            {},  // name
            nn::nifm::NetworkProfileType_Temporary, // networkProfileType
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,  // networkInterfaceType
            true, // isAutoConnect
            true, // isLargeCapacity
            {
                wirelessSetting0
            },
            ipSetting0
        };

        nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile( networkProfile0 );
        ASSERT_NE( nn::util::InvalidUuid, temporaryNetworkProfile.GetId() );

        // テンポラリ接続要求0
        nn::nifm::NetworkConnection networkConnection0;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnection0.GetRequestHandle(), nn::nifm::RequirementPreset_InternetGeneric));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection0.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection0.GetRequestHandle(), temporaryNetworkProfile.GetId()));

        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection0, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        NNT_ASSERT_RESULT_SUCCESS(networkConnection0.GetResult());

        nn::socket::InAddr inAddr;
        NNT_ASSERT_RESULT_SUCCESS( nn::nifm::GetCurrentPrimaryIpAddress( &inAddr ) );
        EXPECT_TRUE(inAddr.S_addr != 0);
        NN_LOG("IpAddress: %u.%u.%u.%u\n", (inAddr.S_addr >>  0) & 0xff, (inAddr.S_addr >>  8) & 0xff, (inAddr.S_addr >> 16) & 0xff, (inAddr.S_addr >> 24) & 0xff);

        // TODO: AP が固定される環境ができたら，具体的に内容をチェックするテストを testNifm_Connect 辺りに追加する
        nn::nifm::AccessPointData accessPoint;
        NNT_ASSERT_RESULT_SUCCESS( nn::nifm::GetCurrentAccessPoint( &accessPoint ) );
        nn::nifm::test::PrintAccessPoint( accessPoint );

        GetConntestServerResponse();
    }
}

TEST_F(SimpleConnectCaseTest, TemporaryConnectEthernet)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::InitializeAdmin());

    {
        const nn::nifm::WirelessSettingData wirelessSetting0 = nn::nifm::InvalidWirelessSettingData;
        const nn::nifm::IpSettingData ipSetting0 = { {true, {}, {}, {}}, {true, {}, {}}, {false, 0, "", {false, "", ""}}, 1400 };
        nn::nifm::NetworkProfileData networkProfile0 = { nn::util::InvalidUuid, {}, nn::nifm::NetworkProfileType_Temporary, nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ethernet, true, true, { wirelessSetting0 }, ipSetting0 };

        nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile( networkProfile0 );
        ASSERT_NE( nn::util::InvalidUuid, temporaryNetworkProfile.GetId() );

        // テンポラリ接続要求0
        nn::nifm::NetworkConnection networkConnection;
        nn::nifm::SetRequestNetworkProfileId(networkConnection.GetRequestHandle(), temporaryNetworkProfile.GetId());
        nn::nifm::SetRequestConnectionConfirmationOption(networkConnection.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired);

        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
        NNT_ASSERT_RESULT_SUCCESS(networkConnection.GetResult());

        nn::socket::InAddr inAddr;
        NNT_ASSERT_RESULT_SUCCESS( nn::nifm::GetCurrentPrimaryIpAddress( &inAddr ) );
        EXPECT_TRUE(inAddr.S_addr != 0);
        NN_LOG("IpAddress: %u.%u.%u.%u\n", (inAddr.S_addr >>  0) & 0xff, (inAddr.S_addr >>  8) & 0xff, (inAddr.S_addr >> 16) & 0xff, (inAddr.S_addr >> 24) & 0xff);

        GetConntestServerResponse();
    }
}


extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    auto ret = RUN_ALL_TESTS();

    nnt::Exit(ret);
}
