﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#if defined(NN_BUILD_CONFIG_COMPILER_VC)
#pragma execution_character_set("utf-8")
#endif

#include <nnt.h>
#include <nnt/result/testResult_Assert.h>

#include <nn/os.h>
#include <nn/fs.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/init.h>

#include <nn/nifm/nifm_Common.h>
#include <nn/nifm/nifm_Api.h>
#include <nn/nifm/nifm_ApiForMenu.h>
#include <nn/nifm/nifm_ApiNetworkConnection.h>
#include <nn/nifm/nifm_Request.h>
#include <nn/nifm/nifm_ApiRequest.h>
#include <nn/nifm/nifm_NetworkConnection.h>
#include <nn/nifm/detail/nifm_CommonDetail.h>
#include <nn/nifm/nifm_TypesRequest.h>
#include <nn/nifm/nifm_TypesNetworkProfile.h>
#include <nn/nifm/nifm_TypesNetworkInterface.h>

#include <nn/util/util_Uuid.h>
#include <nn/util/util_Base64.h>
#include <nn/util/util_StringUtil.h>

#include <nn/settings/system/settings_Network.h>

#include <cstring>

#include "../Common/nifm_TestUtility.h"

const int nn::nifm::Ssid::HexSize;

namespace
{
    const int64_t TIME_OUT_IN_SECONDS_FOR_SUCCESS = 60;
}

namespace nn
{
namespace nifm
{
namespace detail
{
    // SSID リストを修正した場合：
    // testNifm_Profile.cpp の修正必須（本ファイル）
    // testNifm_ErrorReportVerify.cpp の修正必須
    // http://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=173241023 の編集必須

    static const char g_SsidListVersionStr[] = "180111";

    struct SsidListEntry
    {
        char idStr[nn::util::Uuid::StringSize];
        char ssidStr[44]; // (32+1) * 4 / 3
        char passphrase[86];  // 64 * 4 / 3
        Authentication authentication;
        Encryption encryption;
        char name[86]; // 64 * 4 / 3
        bool isAutoConnect;
    };

    static const SsidListEntry g_SsidListEntries[] = {
        // (uuid), "mmt47182hJj", "THmgsS01aCI", "mmt47182hJjName"
        { "1a765924-cfb0-4833-a9da-aa81d83aea4a", "bW10NDcxODJoSmo=",             "VEhtZ3NTMDFhQ0k=",                             Authentication_Wpa2Psk, Encryption_Aes,  "bW10NDcxODJoSmpOYW1l",         false },
        { "cd1b2fab-7144-4ff2-8460-7465f7eda537", "WlNWU0lGUzAx",                 "OUZEMkRBRDlGMkJDNDUyREZEMTIxRjA0OTE4RTA3MTc=", Authentication_Wpa2Psk, Encryption_Aes,  "TmludGVuZG8gWm9uZQ==",         false },
        { "7c205582-060b-4ac4-8fa6-d2fb23f0bf21", "WlNWU0lTRTAx",                 "M0YxNUVGNTU0OTdGRkEwMDdDQzNFQkEzRTZEMDM5MkU=", Authentication_Wpa2Psk, Encryption_Aes,  "TmludGVuZG8gWm9uZQ==",         false },
        { "f38c76bc-2e05-4370-996c-1e3604f92369", "WlNWU0lJWTAx",                 "NjlDNjIzMTYyMjNDNjBEOTgxNEQwM0NGMUYzQTkxQ0I=", Authentication_Wpa2Psk, Encryption_Aes,  "TmludGVuZG8gWm9uZQ==",         false },
        { "a49d00de-79d7-478a-b960-5e2451e84f48", "WlNWU0lTUzAx",                 "MDk0QTE2RTJFQTJENUM0MDkzQTI5ODVDRkExRDVDNTM=", Authentication_Wpa2Psk, Encryption_Aes,  "TmludGVuZG8gWm9uZQ==",         false },
        { "141f5ecc-da97-44b7-ab55-211e07b75fe4", "bnphZTIwMDgxOA==",             "SjJNeXVuU3REYWk4TjFDbw==",                     Authentication_Wpa2Psk, Encryption_Aes,  "TmludGVuZG8gWm9uZQ==",         false },
        { "8d39adb4-de7c-4451-a4d7-8c6891e147f8", "TklDT19HVUVTVA==",             "MjVuYXRzdWJpcmFraQ==",                         Authentication_Wpa2Psk, Encryption_Aes,  "TmludGVuZG8gWm9uZQ==",         false },
        { "e052b1b6-9971-4c53-8ecc-874507e8aa0a", "cGNXbVJuU2QyM0c2TGRBOA==",     "dkQ3eVlGWHNuSzl1NTNibg==",                     Authentication_Wpa2Psk, Encryption_Aes,  "TmludGVuZG8gWm9uZQ==",         false },
        { "ac20abaa-c881-4228-9489-ecd1232fdb8d", "VERTLUEyNDYyQzdQ",             "bzZpZkd1VkE1OFpWN3c4TQ==",                     Authentication_Wpa2Psk, Encryption_Aes,  "TmludGVuZG8gWm9uZQ==",         false },
        { "f66f1041-94ff-4057-b097-eef154284e4c", "TkNMLU5aU0VSVklDRTE=",         "UTRJRlo4NTdQRlJLRzE4WlVYWVBGSTY4VVBYNUlRTFM=", Authentication_Wpa2Psk, Encryption_Aes,  "TmludGVuZG8gV2ktRmkgMQ==",     false },
        { "18cdaba6-d2a3-4a14-bf66-f25459ec58f4", "TkNMLU5aU0VSVklDRTI=",         "RlAwWllSV0xHTllWM0xIQ1JGNFMyRFJUSlY3WEM2OEg=", Authentication_Wpa2Psk, Encryption_Aes,  "TmludGVuZG8gV2ktRmkgMg==",     false },
        { "21ab7387-14d9-4156-9989-90f7f58958f8", "TkNMLU5aU0VSVklDRTM=",         "NDhPRzdJUTFIMVNRU0VCVEZEM0dCUVhVVk5OTFBES04=", Authentication_Wpa2Psk, Encryption_Aes,  "TmludGVuZG8gV2ktRmkgMw==",     false },
        { "fba1ce54-5095-4751-b37f-efaf8be14cf4", "TkNMLU5aU0VSVklDRTQ=",         "TDA1U1VUMkdTWkZUVFdKRk1PV1JGMk02TUJGVTYwOE4=", Authentication_Wpa2Psk, Encryption_Aes,  "TmludGVuZG8gV2ktRmkgNA==",     false },
        { "5ace2fba-0104-4f93-9033-5206b8cd4e2b", "TkNMLU5aU0VSVklDRTU=",         "SVg5Tlo2RllXM01FREFVVlFTSkRZUzFNVVdYNVhBQjI=", Authentication_Wpa2Psk, Encryption_Aes,  "TmludGVuZG8gV2ktRmkgNQ==",     false },
        { "c87f930f-9527-4b7b-9cfd-ea4c69f18491", "TkNMLU5aU0VSVklDRTY=",         "QUhZMkVPVlVIQ0pJWUNPQ0EzMjlPM1BISE8wM1IxNkE=", Authentication_Wpa2Psk, Encryption_Aes,  "TmludGVuZG8gV2ktRmkgNg==",     false },
        { "dadd2044-4af2-4818-9da5-64b660f027f5", "TkNMLU5aU0VSVklDRTc=",         "QzA2UzVZV0dLUkJMMUtBWktRSFZRQjg2NDFBSzlEQTc=", Authentication_Wpa2Psk, Encryption_Aes,  "TmludGVuZG8gV2ktRmkgNw==",     false },
        { "172fef24-1f6f-4e34-b140-079a81b884ed", "TkNMLU5aU0VSVklDRTg=",         "TjhOS0ZKR1ZNMjhVUjNRTE5aQUtSUEtZOUs0U081ODk=", Authentication_Wpa2Psk, Encryption_Aes,  "TmludGVuZG8gV2ktRmkgOA==",     false },
        { "2b72d58c-d051-4477-a917-11fc46bc901c", "TkNMLU5aU0VSVklDRTk=",         "NkdTSDBDM1UzR0lMOUVJRldSRVdEN0paRlRYWVRFWjE=", Authentication_Wpa2Psk, Encryption_Aes,  "TmludGVuZG8gV2ktRmkgOQ==",     false },
        { "8d9a3f55-8b3d-45ea-90ff-3094ff3ff5fc", "TkNMLU5aU0VSVklDRTEw",         "MU9UTVFBMUFWUlM5VFZLSFk5Q1JaNlFLNTVUT1kzRTk=", Authentication_Wpa2Psk, Encryption_Aes,  "TmludGVuZG8gV2ktRmkgMTA=",     false },
        { "a08cd1c5-ad1a-431c-8f60-698c53573d5b", "RlJFRVNQT1Q=",                 "",                                             Authentication_Open,    Encryption_None, "RlJFRVNQT1Q=",                 false },
        { "1eb03f86-b80a-40de-9900-c21a62b3b21d", "QmVzdGJ1eQ==",                 "QjM1dEJ1eTIwMTI=",                             Authentication_Wpa2Psk, Encryption_Aes,  "TmludGVuZG8=",                 false },
        { "69af850c-2c49-4c16-ba6c-90d48aba7bad", "QkJZVmVuZG9y",                 "QmU1dFYzbmQwcg==",                             Authentication_Wpa2Psk, Encryption_Aes,  "TmludGVuZG8=",                 false },
        { "a477b8a2-c692-432a-871b-3fd200e2af85", "TmludGVuZG9ab25lQE5XUw==",     "c3BsYXRvb24yMDE1",                             Authentication_Wpa2Psk, Encryption_Aes,  "TmludGVuZG8=",                 false },
        { "060c0ba1-38b4-4bef-a11f-70cea32bd2d8", "bm9hc3AwMQ==",                 "",                                             Authentication_Open,    Encryption_None, "TmludGVuZG8=",                 false },
        { "2c8970c1-6d87-4c54-9fc7-947586471e8c", "bm9hc3AwMg==",                 "",                                             Authentication_Open,    Encryption_None, "TmludGVuZG8=",                 false },
        { "ebaf4500-887c-48da-b9e9-a1c8e7b95a8a", "bm9hc3AwMw==",                 "Z0MoP3pBJjMrUDNbQVBO",                         Authentication_Wpa2Psk, Encryption_Aes,  "TmludGVuZG8=",                 false },
        { "e8d2e530-efe7-4373-8e0c-c0a6c66dfdf3", "bm9hc3AwNA==",                 "Z0QoP3pBJjQrUDRbQVBO",                         Authentication_Wpa2Psk, Encryption_Aes,  "TmludGVuZG8=",                 false },
        { "a3dadf1e-3866-453e-8926-472c2edeafa2", "Tk9FIDE=",                     "UzJ5UFZrUk5xajVOSktKdU5lUDNkRlNtdmREUTlYQ0w=", Authentication_Wpa2Psk, Encryption_Aes,  "TmludGVuZG8=",                 false },
        { "762ffa11-9491-4a4b-9231-8e2f1b06aa73", "Tk9FIDI=",                     "M3pFdXRHckdGbkFjZm5iQzY4cjNiS0pBRWhNSmp1RkU=", Authentication_Wpa2Psk, Encryption_Aes,  "TmludGVuZG8=",                 false },
        { "58bfcaef-148f-4851-a6a2-ff55b832561e", "Tk9FIDM=",                     "WXJUNERnUHNEc1FtTWVxNHZ4UVl0NENneWZuemZXSFI=", Authentication_Wpa2Psk, Encryption_Aes,  "TmludGVuZG8=",                 false },
        { "ca5e4713-3cb1-49f7-bc58-044fdbf357f6", "Tk9FIDQ=",                     "clpYUWd3TVpGMjdjQ2FVeTZIV3Q0bVJzSHBmWWVoeVA=", Authentication_Wpa2Psk, Encryption_Aes,  "TmludGVuZG8=",                 false },
        { "01af785e-c74c-4537-bf7c-9ad9c4741eae", "Tk9FIDU=",                     "NFA1ZU11WVV3RWFhdXRNVWZ0ZUJabVVWNUdHNW56VlM=", Authentication_Wpa2Psk, Encryption_Aes,  "TmludGVuZG8=",                 false },
        { "f896f79b-2aab-4281-bf68-6a8facf81484", "Tk9FIDY=",                     "eVR4ZkZlcTlBOVVzWjQzRGVwcERtQUJlQXZZemZBckg=", Authentication_Wpa2Psk, Encryption_Aes,  "TmludGVuZG8=",                 false },
        { "94ae0271-02f7-49ce-9ac7-16a9aec1d087", "Tk9FIDc=",                     "eWZUV3JDNlBhc25Ka0RQQVF3ZHhnNldENkJHRm5jWVQ=", Authentication_Wpa2Psk, Encryption_Aes,  "TmludGVuZG8=",                 false },
        { "25f31d4a-b492-4887-b707-d8326a30b320", "Tk9FIDg=",                     "VlBLZDV4a1UzdjdoWUtNYnNjN0FCTHY0WUFaUkNWS1A=", Authentication_Wpa2Psk, Encryption_Aes,  "TmludGVuZG8=",                 false },
        { "99b0be25-b0d5-477a-8512-2a335477ddb0", "Tk9FIDk=",                     "WGRxQmpaM1VMVVhLa1Q2ZXduY0IzOUFqU1pYQUhtN3o=", Authentication_Wpa2Psk, Encryption_Aes,  "TmludGVuZG8=",                 false },
        { "d219f268-1c05-4780-80ec-fdfd2fc6d6ef", "Tk9FIDEw",                     "dUVmclFKVTY0WG0yazh3WUJ1RXM5QnBaN0o5dWVlbUU=", Authentication_Wpa2Psk, Encryption_Aes,  "TmludGVuZG8=",                 false },
        { "c73733d8-b7a8-4cb8-818b-05562c6d3566", "bmludGVuZG9hcHBkZWJ1ZzE=",     "MTY2YWU0ZTM5ZWI2MzM5OGI4ZWY1OGRlMTY3NmRjNGU=", Authentication_Wpa2Psk, Encryption_Aes,  "TmludGVuZG8gQXBwIERlYnVnMQ==", false },
        { "6a5b8b0d-1abb-46c4-b2db-58896fd0497c", "bmludGVuZG9hcHBkZWJ1ZzI=",     "MTY2YWU0ZTM5ZWI2MzM5OGI4ZWY1OGRlMTY3NmRjNGU=", Authentication_Wpa2Psk, Encryption_Aes,  "TmludGVuZG8gQXBwIERlYnVnMg==", false },
        { "cb50975c-fefc-437d-b674-be986b030ad3", "bmludGVuZG9hcHBkZWJ1ZzM=",     "MTY2YWU0ZTM5ZWI2MzM5OGI4ZWY1OGRlMTY3NmRjNGU=", Authentication_Wpa2Psk, Encryption_Aes,  "TmludGVuZG8gQXBwIERlYnVnMw==", false },
        { "ea4140ec-860f-4486-b756-47ce5dd11566", "TkNMLU5aU0VSVklDRTEtNUdIWg==", "RENMOEJVTU5TWVVCS0RQVEZSSDYxTkFaOFJISVZZSTg=", Authentication_Wpa2Psk, Encryption_Aes,  "TmludGVuZG8gV2ktRmkgMSA1R0h6", false },
    };
}
}
}

namespace
{
    static const int g_SsidListSize = NN_ARRAY_SIZE(nn::nifm::detail::g_SsidListEntries); // プリイン時点:38

    static nn::nifm::NetworkProfileData g_SsidListNetworkProfile[g_SsidListSize];

    void BuildSsidListnetworkProfileDataForTest()
    {
        int i = 0;
        for (const auto& e : nn::nifm::detail::g_SsidListEntries)
        {
            auto&& pNp = g_SsidListNetworkProfile[i];

            // name 複号
            char decodedName[nn::nifm::NetworkProfileBasicInfo::NameSize] = {};
            size_t num;
            nn::util::Base64::Status status = nn::util::Base64::FromBase64String(&num, decodedName, nn::nifm::NetworkProfileBasicInfo::NameSize, e.name, nn::util::Base64::Mode_NormalNoLinefeed);
            ASSERT_EQ(nn::util::Base64::Status_Success, status);
            decodedName[num] = '\0';

            nn::util::Strlcpy(pNp.name, decodedName, static_cast<int>(num + 1));

            pNp.id.FromString(e.idStr);
            pNp.isAutoConnect = e.isAutoConnect;
            pNp.isLargeCapacity = true;
            pNp.networkProfileType = nn::nifm::NetworkProfileType_SsidList;
            pNp.networkInterfaceType = nn::nifm::NetworkInterfaceType_Ieee80211;

            auto&& pWs = pNp.wirelessSetting;

            pWs.ssidConfig.nonBroadcast = false;

            // ssid 復号
            char decodedSsidStr[nn::nifm::Ssid::HexSize + 1];
            status = nn::util::Base64::FromBase64String(&num, decodedSsidStr, nn::nifm::Ssid::HexSize + 1, e.ssidStr, nn::util::Base64::Mode_NormalNoLinefeed);
            ASSERT_EQ(nn::util::Base64::Status_Success, status);
            ASSERT_GT(nn::nifm::Ssid::HexSize, static_cast<int>(num));

            std::memcpy(pWs.ssidConfig.ssid.hex, decodedSsidStr, num);
            pWs.ssidConfig.ssid.length = static_cast<uint8_t>(num);

            pWs.security.authEncryption.authentication = e.authentication;
            pWs.security.authEncryption.encryption = e.encryption;

            // passphase 復号
            char decodedPassphrase[nn::nifm::SharedKey::KeyMaterialSize] = {};
            status = nn::util::Base64::FromBase64String(&num, decodedPassphrase, nn::nifm::SharedKey::KeyMaterialSize, e.passphrase, nn::util::Base64::Mode_NormalNoLinefeed);
            ASSERT_EQ(nn::util::Base64::Status_Success, status);

            pWs.security.sharedKey.length = static_cast<uint8_t>(num);
            std::memcpy(pWs.security.sharedKey.keyMaterial, decodedPassphrase, num);

            auto&& pIs = pNp.ipSetting;

            pIs.ip.isAuto = true;
            pIs.dns.isAuto = true;
            pIs.mtu = 1400;

            ++i;
        }
    }

    static const nn::nifm::NetworkProfileData g_UserNetworkProfile[] = {
        {
            { { 0x93, 0x18, 0x84, 0x40, 0xb6, 0xd5, 0x14, 0xb7, 0x05, 0x6b, 0x08, 0xcd, 0x57, 0x91, 0x09, 0x60 } },
            "nsd-imatake-01",
            nn::nifm::NetworkProfileType_User,
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,
            true,
            false,
            { { { { 14,{ 0x6e,0x73,0x64,0x2d,0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x30,0x31 } }, false },{ { nn::nifm::Authentication_Open, nn::nifm::Encryption_None },{} } } },
            { { true,{},{},{} },{ true,{},{} },{ false, 0, "",{ false, "", "" } }, 1400 },
        },
        {
            { { 0x93, 0x18, 0x84, 0x40, 0xb6, 0xd5, 0x14, 0xb7, 0x05, 0x6b, 0x08, 0xcd, 0x57, 0x91, 0x09, 0x61 } },
            "ssd-imatake-02",
            nn::nifm::NetworkProfileType_User,
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,
            true,
            true,
            { { { { 14,{ 0x73,0x73,0x64,0x2d,0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x30,0x32 } }, false },{ { nn::nifm::Authentication_Open, nn::nifm::Encryption_None },{} } } },
            { { true,{},{},{} },{ true,{},{} },{ false, 0, "",{ false, "", "" } }, 1400 },
        },
        {
            { { 0x93, 0x18, 0x84, 0x40, 0xb6, 0xd5, 0x14, 0xb7, 0x05, 0x6b, 0x08, 0xcd, 0x57, 0x91, 0x09, 0x62 } },
            "dummy-profile",
            nn::nifm::NetworkProfileType_User,
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,
            true,
            true,
            { { { { 13,{ 0x64,0x75,0x6D,0x6D,0x79,0x2D,0x70,0x72,0x6F,0x66,0x69,0x6C,0x65 } }, true },{ { nn::nifm::Authentication_Open, nn::nifm::Encryption_Wep },{ 3, "xyz" } } } },
            { { false,{ { 0xc0,0xa8,0x64,0x65 } },{ { 0xff,0xff,0xff,0x00 } },{ { 0xc0,0xa8,0x64,0x01 } } },{ false,{ { 0x08,0x08,0x08,0x08 } },{ { 0xc0,0xa8,0x64,0x01 } } },{ true, 555, "proxy.dummy.net",{ true, "dummy_user", "dummy_password" } }, 1500 },
        },
        {
            { { 0x93, 0x18, 0x84, 0x40, 0xb6, 0xd5, 0x14, 0xb7, 0x05, 0x6b, 0x08, 0xcd, 0x57, 0x91, 0x09, 0x63 } },
            "nbd-plala1wpa2",
            nn::nifm::NetworkProfileType_User,
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,
            true,
            true,
            { { { { 14,{ 0x6E,0x62,0x64,0x2D,0x70,0x6C,0x61,0x6C,0x61,0x31,0x77,0x70,0x61,0x32 } }, false },{ { nn::nifm::Authentication_Wpa2Psk, nn::nifm::Encryption_Aes },{ 8, "xxxxxxxx" } } } },
            { { true,{},{},{} },{ true,{},{} },{ false, 0, "",{ false, "", "" } }, 1400 },
        },
        {
            { { 0x93, 0x18, 0x84, 0x40, 0xb6, 0xd5, 0x14, 0xb7, 0x05, 0x6b, 0x08, 0xcd, 0x57, 0x91, 0x09, 0x64 } },
            "ethernet",
            nn::nifm::NetworkProfileType_User,
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ethernet,
            true,
            true,
            { nn::nifm::InvalidWirelessSettingData },
            { { true,{},{},{} },{ true,{},{} },{ false, 0, "",{ true, "ethernet_username", "ethernet_password" } }, 1400 },
        },
        {
            { { 0x93, 0x18, 0x84, 0x40, 0xb6, 0xd5, 0x14, 0xb7, 0x05, 0x6b, 0x08, 0xcd, 0x57, 0x91, 0x09, 0x65 } },
            "max_profile_name_string_example",
            nn::nifm::NetworkProfileType_User,
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,
            true,
            true,
            { { { { 11,{ 0x6d,0x61,0x78,0x2d,0x70,0x72,0x6f,0x66,0x69,0x6c,0x65 } }, false },{ { nn::nifm::Authentication_Wpa2Psk, nn::nifm::Encryption_Aes },{ 31, "abcdefghijklmnopqrstuvwxyz01234" } } } },
            { { true,{},{},{} },{ true,{},{} },{ true, 810, "max_proxy_name_string_01234567890123456789012345678901234567890123456789012345678901234567890123456",{ true, "max_proxy_username_string_01234", "max_proxy_password_string_01234" } }, 1400 },
        },
        {
            { { 0x93, 0x18, 0x84, 0x40, 0xb6, 0xd5, 0x14, 0xb7, 0x05, 0x6b, 0x08, 0xcd, 0x57, 0x91, 0x09, 0x66 } },
            "ダミープロファイル_06",
            nn::nifm::NetworkProfileType_User,
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ethernet,
            true,
            true,
            { nn::nifm::InvalidWirelessSettingData },
            { { true,{},{},{} },{ true,{},{} },{ false, 0, "",{ false, "", "" } }, 1400 },
        },
        {
            { { 0x93, 0x18, 0x84, 0x40, 0xb6, 0xd5, 0x14, 0xb7, 0x05, 0x6b, 0x08, 0xcd, 0x57, 0x91, 0x09, 0x67 } },
            "dummy_profile_07",
            nn::nifm::NetworkProfileType_User,
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ethernet,
            true,
            true,
            { nn::nifm::InvalidWirelessSettingData },
            { { true,{},{},{} },{ true,{},{} },{ false, 0, "",{ false, "", "" } }, 1400 },
        },
        {
            { { 0x93, 0x18, 0x84, 0x40, 0xb6, 0xd5, 0x14, 0xb7, 0x05, 0x6b, 0x08, 0xcd, 0x57, 0x91, 0x09, 0x68 } },
            "dummy_profile_08",
            nn::nifm::NetworkProfileType_User,
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ethernet,
            true,
            true,
            { nn::nifm::InvalidWirelessSettingData },
            { { true,{},{},{} },{ true,{},{} },{ false, 0, "",{ false, "", "" } }, 1400 },
        },
        {
            { { 0x93, 0x18, 0x84, 0x40, 0xb6, 0xd5, 0x14, 0xb7, 0x05, 0x6b, 0x08, 0xcd, 0x57, 0x91, 0x09, 0x69 } },
            "dummy_profile_09",
            nn::nifm::NetworkProfileType_User,
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ethernet,
            true,
            true,
            { nn::nifm::InvalidWirelessSettingData },
            { { true,{},{},{} },{ true,{},{} },{ false, 0, "",{ false, "", "" } }, 1400 },
        },
        {
            { { 0x93, 0x18, 0x84, 0x40, 0xb6, 0xd5, 0x14, 0xb7, 0x05, 0x6b, 0x08, 0xcd, 0x57, 0x91, 0x09, 0x6a } },
            "dummy_profile_10",
            nn::nifm::NetworkProfileType_User,
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ethernet,
            true,
            true,
            { nn::nifm::InvalidWirelessSettingData },
            { { true,{},{},{} },{ true,{},{} },{ false, 0, "",{ false, "", "" } }, 1400 },
        },
        {
            { { 0x93, 0x18, 0x84, 0x40, 0xb6, 0xd5, 0x14, 0xb7, 0x05, 0x6b, 0x08, 0xcd, 0x57, 0x91, 0x09, 0x6b } },
            "dummy_profile_11",
            nn::nifm::NetworkProfileType_User,
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ethernet,
            true,
            true,
            { nn::nifm::InvalidWirelessSettingData },
            { { true,{},{},{} },{ true,{},{} },{ false, 0, "",{ false, "", "" } }, 1400 },
        },
        {
            { { 0x93, 0x18, 0x84, 0x40, 0xb6, 0xd5, 0x14, 0xb7, 0x05, 0x6b, 0x08, 0xcd, 0x57, 0x91, 0x09, 0x6c } },
            "dummy_profile_12",
            nn::nifm::NetworkProfileType_User,
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ethernet,
            true,
            true,
            { nn::nifm::InvalidWirelessSettingData },
            { { true,{},{},{} },{ true,{},{} },{ false, 0, "",{ false, "", "" } }, 1400 },
        },
        {
            { { 0x93, 0x18, 0x84, 0x40, 0xb6, 0xd5, 0x14, 0xb7, 0x05, 0x6b, 0x08, 0xcd, 0x57, 0x91, 0x09, 0x6d } },
            "dummy_profile_13",
            nn::nifm::NetworkProfileType_User,
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ethernet,
            true,
            true,
            { nn::nifm::InvalidWirelessSettingData },
            { { true,{},{},{} },{ true,{},{} },{ false, 0, "",{ false, "", "" } }, 1400 },
        },
        {
            { { 0x93, 0x18, 0x84, 0x40, 0xb6, 0xd5, 0x14, 0xb7, 0x05, 0x6b, 0x08, 0xcd, 0x57, 0x91, 0x09, 0x6e } },
            "dummy_profile_14",
            nn::nifm::NetworkProfileType_User,
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ethernet,
            true,
            true,
            { nn::nifm::InvalidWirelessSettingData },
            { { true,{},{},{} },{ true,{},{} },{ false, 0, "",{ false, "", "" } }, 1400 },
        },
        {
            { { 0x93, 0x18, 0x84, 0x40, 0xb6, 0xd5, 0x14, 0xb7, 0x05, 0x6b, 0x08, 0xcd, 0x57, 0x91, 0x09, 0x6f } },
            "dummy_profile_15",
            nn::nifm::NetworkProfileType_User,
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ethernet,
            true,
            true,
            { nn::nifm::InvalidWirelessSettingData },
            { { true,{},{},{} },{ true,{},{} },{ false, 0, "",{ false, "", "" } }, 1400 },
        },
        {
            { { 0x93, 0x18, 0x84, 0x40, 0xb6, 0xd5, 0x14, 0xb7, 0x05, 0x6b, 0x08, 0xcd, 0x57, 0x91, 0x09, 0x70 } },
            "dummy_profile_16",
            nn::nifm::NetworkProfileType_User,
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ethernet,
            true,
            true,
            { nn::nifm::InvalidWirelessSettingData },
            { { true,{},{},{} },{ true,{},{} },{ false, 0, "",{ false, "", "" } }, 1400 },
        },
        {
            { { 0x93, 0x18, 0x84, 0x40, 0xb6, 0xd5, 0x14, 0xb7, 0x05, 0x6b, 0x08, 0xcd, 0x57, 0x91, 0x09, 0x71 } },
            "dummy_profile_17",
            nn::nifm::NetworkProfileType_User,
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ethernet,
            true,
            true,
            { nn::nifm::InvalidWirelessSettingData },
            { { true,{},{},{} },{ true,{},{} },{ false, 0, "",{ false, "", "" } }, 1400 },
        },
        {
            { { 0x93, 0x18, 0x84, 0x40, 0xb6, 0xd5, 0x14, 0xb7, 0x05, 0x6b, 0x08, 0xcd, 0x57, 0x91, 0x09, 0x72 } },
            "dummy_profile_18",
            nn::nifm::NetworkProfileType_User,
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ethernet,
            true,
            true,
            { nn::nifm::InvalidWirelessSettingData },
            { { true,{},{},{} },{ true,{},{} },{ false, 0, "",{ false, "", "" } }, 1400 },
        },
        {
            { { 0x93, 0x18, 0x84, 0x40, 0xb6, 0xd5, 0x14, 0xb7, 0x05, 0x6b, 0x08, 0xcd, 0x57, 0x91, 0x09, 0x73 } },
            "dummy_profile_19",
            nn::nifm::NetworkProfileType_User,
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ethernet,
            true,
            true,
            { nn::nifm::InvalidWirelessSettingData },
            { { true,{},{},{} },{ true,{},{} },{ false, 0, "",{ false, "", "" } }, 1400 },
        },
        {
            { { 0x93, 0x18, 0x84, 0x40, 0xb6, 0xd5, 0x14, 0xb7, 0x05, 0x6b, 0x08, 0xcd, 0x57, 0x91, 0x09, 0x74 } },
            "dummy_profile_20",
            nn::nifm::NetworkProfileType_User,
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ethernet,
            true,
            true,
            { nn::nifm::InvalidWirelessSettingData },
            { { true,{},{},{} },{ true,{},{} },{ false, 0, "",{ false, "", "" } }, 1400 },
        },
        {
            { { 0x93, 0x18, 0x84, 0x40, 0xb6, 0xd5, 0x14, 0xb7, 0x05, 0x6b, 0x08, 0xcd, 0x57, 0x91, 0x09, 0x75 } },
            "dummy_profile_21",
            nn::nifm::NetworkProfileType_User,
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ethernet,
            true,
            true,
            { nn::nifm::InvalidWirelessSettingData },
            { { true,{},{},{} },{ true,{},{} },{ false, 0, "",{ false, "", "" } }, 1400 },
        },
        {
            { { 0x93, 0x18, 0x84, 0x40, 0xb6, 0xd5, 0x14, 0xb7, 0x05, 0x6b, 0x08, 0xcd, 0x57, 0x91, 0x09, 0x76 } },
            "dummy_profile_22",
            nn::nifm::NetworkProfileType_User,
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ethernet,
            true,
            true,
            { nn::nifm::InvalidWirelessSettingData },
            { { true,{},{},{} },{ true,{},{} },{ false, 0, "",{ false, "", "" } }, 1400 },
        },
        {
            { { 0x93, 0x18, 0x84, 0x40, 0xb6, 0xd5, 0x14, 0xb7, 0x05, 0x6b, 0x08, 0xcd, 0x57, 0x91, 0x09, 0x77 } },
            "dummy_profile_23",
            nn::nifm::NetworkProfileType_User,
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ethernet,
            true,
            true,
            { nn::nifm::InvalidWirelessSettingData },
            { { true,{},{},{} },{ true,{},{} },{ false, 0, "",{ false, "", "" } }, 1400 },
        },
        {
            { { 0x93, 0x18, 0x84, 0x40, 0xb6, 0xd5, 0x14, 0xb7, 0x05, 0x6b, 0x08, 0xcd, 0x57, 0x91, 0x09, 0x78 } },
            "dummy_profile_24",
            nn::nifm::NetworkProfileType_User,
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ethernet,
            true,
            true,
            { nn::nifm::InvalidWirelessSettingData },
            { { true,{},{},{} },{ true,{},{} },{ false, 0, "",{ false, "", "" } }, 1400 },
        },
        {
            { { 0x93, 0x18, 0x84, 0x40, 0xb6, 0xd5, 0x14, 0xb7, 0x05, 0x6b, 0x08, 0xcd, 0x57, 0x91, 0x09, 0x79 } },
            "dummy_profile_25",
            nn::nifm::NetworkProfileType_User,
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ethernet,
            true,
            true,
            { nn::nifm::InvalidWirelessSettingData },
            { { true,{},{},{} },{ true,{},{} },{ false, 0, "",{ false, "", "" } }, 1400 },
        },
        {
            { { 0x93, 0x18, 0x84, 0x40, 0xb6, 0xd5, 0x14, 0xb7, 0x05, 0x6b, 0x08, 0xcd, 0x57, 0x91, 0x09, 0x7a } },
            "dummy_profile_26",
            nn::nifm::NetworkProfileType_User,
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ethernet,
            true,
            true,
            { nn::nifm::InvalidWirelessSettingData },
            { { true,{},{},{} },{ true,{},{} },{ false, 0, "",{ false, "", "" } }, 1400 },
        },
        {
            { { 0x93, 0x18, 0x84, 0x40, 0xb6, 0xd5, 0x14, 0xb7, 0x05, 0x6b, 0x08, 0xcd, 0x57, 0x91, 0x09, 0x7b } },
            "dummy_profile_27",
            nn::nifm::NetworkProfileType_User,
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ethernet,
            true,
            true,
            { nn::nifm::InvalidWirelessSettingData },
            { { true,{},{},{} },{ true,{},{} },{ false, 0, "",{ false, "", "" } }, 1400 },
        },
        {
            { { 0x93, 0x18, 0x84, 0x40, 0xb6, 0xd5, 0x14, 0xb7, 0x05, 0x6b, 0x08, 0xcd, 0x57, 0x91, 0x09, 0x7c } },
            "dummy_profile_28",
            nn::nifm::NetworkProfileType_User,
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ethernet,
            true,
            true,
            { nn::nifm::InvalidWirelessSettingData },
            { { true,{},{},{} },{ true,{},{} },{ false, 0, "",{ false, "", "" } }, 1400 },
        },
        {
            { { 0x93, 0x18, 0x84, 0x40, 0xb6, 0xd5, 0x14, 0xb7, 0x05, 0x6b, 0x08, 0xcd, 0x57, 0x91, 0x09, 0x7d } },
            "dummy_profile_29",
            nn::nifm::NetworkProfileType_User,
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ethernet,
            true,
            true,
            { nn::nifm::InvalidWirelessSettingData },
            { { true,{},{},{} },{ true,{},{} },{ false, 0, "",{ false, "", "" } }, 1400 },
        },
        {
            { { 0x93, 0x18, 0x84, 0x40, 0xb6, 0xd5, 0x14, 0xb7, 0x05, 0x6b, 0x08, 0xcd, 0x57, 0x91, 0x09, 0x7e } },
            "dummy_profile_30",
            nn::nifm::NetworkProfileType_User,
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ethernet,
            true,
            true,
            { nn::nifm::InvalidWirelessSettingData },
            { { true,{},{},{} },{ true,{},{} },{ false, 0, "",{ false, "", "" } }, 1400 },
        },
        {
            { { 0x93, 0x18, 0x84, 0x40, 0xb6, 0xd5, 0x14, 0xb7, 0x05, 0x6b, 0x08, 0xcd, 0x57, 0x91, 0x09, 0x7f } },
            "dummy_profile_31",
            nn::nifm::NetworkProfileType_User,
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ethernet,
            true,
            true,
            { nn::nifm::InvalidWirelessSettingData },
            { { true,{},{},{} },{ true,{},{} },{ false, 0, "",{ false, "", "" } }, 1400 },
        },
    };

    void PrintNetworkProfile(const nn::nifm::NetworkProfileData& networkProfile)
    {
        char uuidString[nn::util::Uuid::StringSize];
        NN_UNUSED(uuidString);

        NN_LOG("- %s : (ID:%s)\n", networkProfile.name, nn::util::Uuid(networkProfile.id).ToString(uuidString, nn::util::Uuid::StringSize));

        NN_LOG("    isAutoConnect:%s\n", networkProfile.isAutoConnect ? "true" : "false");
        NN_LOG("    isLargeCapacity:%s\n", networkProfile.isLargeCapacity ? "true" : "false");

        if (networkProfile.networkInterfaceType == nn::nifm::NetworkInterfaceType_Ieee80211)
        {
            const auto& w = networkProfile.wirelessSetting;
            NN_LOG("    ssidconfig.ssid.length:%u\n", w.ssidConfig.ssid.length);
            NN_LOG("    ssidconfig.ssid.hex:");
            for (int j = 0; j < w.ssidConfig.ssid.length; ++j)
            {
                NN_LOG("%02x", w.ssidConfig.ssid.hex[j]);
            }
            NN_LOG("\n");
            NN_LOG("    ssidconfig.nonBroadcast:%s\n", w.ssidConfig.nonBroadcast ? "true" : "false");
            NN_LOG("    security.authencryption.auth:");
            switch (w.security.authEncryption.authentication)
            {
            case nn::nifm::Authentication::Authentication_Open:
                NN_LOG("Authentication_Open\n");
                break;
            case nn::nifm::Authentication::Authentication_Shared:
                NN_LOG("Authentication_Shared\n");
                break;
            case nn::nifm::Authentication::Authentication_Wpa:
                NN_LOG("Authentication_Wpa\n");
                break;
            case nn::nifm::Authentication::Authentication_WpaPsk:
                NN_LOG("Authentication_WpaPsk\n");
                break;
            case nn::nifm::Authentication::Authentication_Wpa2:
                NN_LOG("Authentication_Wpa2\n");
                break;
            case nn::nifm::Authentication::Authentication_Wpa2Psk:
                NN_LOG("Authentication_Wpa2Psk\n");
                break;
            default:
                NN_LOG("Authentication_Invalid\n");
                break;
            }
            NN_LOG("    security.authencryption.encrypt:");
            switch (w.security.authEncryption.encryption)
            {
            case nn::nifm::Encryption::Encryption_None:
                NN_LOG("Encryption_None\n");
                break;
            case nn::nifm::Encryption::Encryption_Wep:
                NN_LOG("Encryption_Wep\n");
                break;
            case nn::nifm::Encryption::Encryption_Tkip:
                NN_LOG("Encryption_Tkip\n");
                break;
            case nn::nifm::Encryption::Encryption_Aes:
                NN_LOG("Encryption_Aes\n");
                break;
            default:
                NN_LOG("Encryption_Invalid\n");
                break;
            }
            NN_LOG("    security.SharedKey.length:%u\n", w.security.sharedKey.length);
            NN_LOG("    security.SharedKey.keyMaterial:%.64s\n", w.security.sharedKey.keyMaterial);
        }

        const auto& ip = networkProfile.ipSetting;
        NN_LOG("    ip.isAuto:%s\n", ip.ip.isAuto ? "true" : "false");
        NN_LOG("    ip.address:%u.%u.%u.%u\n", ip.ip.ipAddress.data[0], ip.ip.ipAddress.data[1], ip.ip.ipAddress.data[2], ip.ip.ipAddress.data[3]);
        NN_LOG("    ip.subnetMask:%u.%u.%u.%u\n", ip.ip.subnetMask.data[0], ip.ip.subnetMask.data[1], ip.ip.subnetMask.data[2], ip.ip.subnetMask.data[3]);
        NN_LOG("    ip.defaultGateway:%u.%u.%u.%u\n", ip.ip.defaultGateway.data[0], ip.ip.defaultGateway.data[1], ip.ip.defaultGateway.data[2], ip.ip.defaultGateway.data[3]);
        NN_LOG("    dns.isAuto:%s\n", ip.dns.isAuto ? "true" : "false");
        NN_LOG("    dns.preferredDns:%u.%u.%u.%u\n", ip.dns.preferredDns.data[0], ip.dns.preferredDns.data[1], ip.dns.preferredDns.data[2], ip.dns.preferredDns.data[3]);
        NN_LOG("    dns.alternateDns:%u.%u.%u.%u\n", ip.dns.alternateDns.data[0], ip.dns.alternateDns.data[1], ip.dns.alternateDns.data[2], ip.dns.alternateDns.data[3]);
        NN_LOG("    proxy.isEnabled:%s\n", ip.proxy.isEnabled ? "true" : "false");
        NN_LOG("    proxy.port:%u\n", ip.proxy.port);
        NN_LOG("    proxy.proxy:%s\n", ip.proxy.proxy);
        NN_LOG("    proxy.authentication.isEnabled:%s\n", ip.proxy.authentication.isEnabled ? "true" : "false");
        NN_LOG("    proxy.authentication.username:%s\n", ip.proxy.authentication.username);
    }

    void PrintNetworkProfileFromBasicInfo(const nn::nifm::NetworkProfileBasicInfo& profileBasicInfo)
    {
        nn::nifm::NetworkProfileData networkProfile;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetNetworkProfile(&networkProfile, profileBasicInfo.id));

        PrintNetworkProfile(networkProfile);
    }
}

class ProfileCaseTest : public ::testing::Test
{
protected:
    virtual void TearDown() { }
};

// このテストは，プロファイルの追加，削除系のテストよりも前に実行する必要がある
// 実機では，別のテストでプロファイルの順序が更新されるため，二回目以降のテストで失敗する
TEST_F(ProfileCaseTest, GetNetworkProfile)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::InitializeAdmin());

    // 全件取得
    {
        const int bufferSize = nn::nifm::UserNetworkProfileCountMax + g_SsidListSize;
        nn::nifm::NetworkProfileBasicInfo networkProfileBasicInfo[bufferSize];
        int outCount = 0;

        // 取得できない例
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultInvalidArgument, nn::nifm::EnumerateNetworkProfiles(networkProfileBasicInfo, &outCount, bufferSize, nn::nifm::NetworkProfileType_User + 100));

        // 存在しない設定の取得
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::EnumerateNetworkProfiles(networkProfileBasicInfo, &outCount, bufferSize, nn::nifm::NetworkProfileType_Temporary));
        ASSERT_EQ(outCount, 0);

        // ユーザ設定と SSID リストの同時取得
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::EnumerateNetworkProfiles(networkProfileBasicInfo, &outCount, bufferSize,
            nn::nifm::NetworkProfileType_User | nn::nifm::NetworkProfileType_SsidList));
        ASSERT_EQ(bufferSize, outCount);
        NN_LOG("Profile count(User,SsidList): %d\n", outCount);

        int userProfileCount = 0;
        int ssidListProfileCount = 0;

        BuildSsidListnetworkProfileDataForTest();

        nn::nifm::SsidListVersion version;
        nn::nifm::GetSsidListVersion(&version);

        ASSERT_EQ(0, strncmp(nn::nifm::detail::g_SsidListVersionStr, version.data, nn::nifm::SsidListVersion::Size));

        for (auto&& basicInfo : networkProfileBasicInfo)
        {
            if (basicInfo.networkProfileType == nn::nifm::NetworkProfileType_User)
            {
                nn::nifm::NetworkProfileData networkProfileData;
                NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetNetworkProfile(&networkProfileData, basicInfo.id));
                EXPECT_EQ(g_UserNetworkProfile[userProfileCount], networkProfileData);

                ++userProfileCount;
            }
            else if (basicInfo.networkProfileType == nn::nifm::NetworkProfileType_SsidList)
            {
                nn::nifm::NetworkProfileData networkProfileData;
                NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetNetworkProfile(&networkProfileData, basicInfo.id));
                EXPECT_EQ(g_SsidListNetworkProfile[ssidListProfileCount], networkProfileData);

                ++ssidListProfileCount;
            }
            else
            {
                ASSERT_EQ(nn::nifm::NetworkProfileType_User, basicInfo.networkProfileType); // 必ず失敗
            }
            PrintNetworkProfileFromBasicInfo(basicInfo);
        }

        NN_LOG("Profile count(SsidList): %d\n", ssidListProfileCount);
        ASSERT_EQ(g_SsidListSize, ssidListProfileCount);

        // スリープ復帰後の SSID リストの確認
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::PutToSleep());
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(10));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::WakeUp()); // MinimumAwake
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::WakeUp()); // FullAwake
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));

        // SSID リストの取得
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::EnumerateNetworkProfiles(networkProfileBasicInfo, &outCount, bufferSize,
            nn::nifm::NetworkProfileType_SsidList));
        NN_LOG("Profile count(SsidList): %d\n", outCount);
        ASSERT_EQ(g_SsidListSize, outCount);

        for (int i = 0; i < outCount; ++i)
        {
            ASSERT_EQ(nn::nifm::NetworkProfileType_SsidList, networkProfileBasicInfo[i].networkProfileType);
            nn::nifm::NetworkProfileData networkProfileData;
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetNetworkProfile(&networkProfileData, networkProfileBasicInfo[i].id));
            EXPECT_EQ(g_SsidListNetworkProfile[i], networkProfileData);
        }
    }

    // ユーザ設定 1件取得
    {
        const int bufferSize = 1;
        nn::nifm::NetworkProfileBasicInfo networkProfileBasicInfo[bufferSize];
        int outCount = 0;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::EnumerateNetworkProfiles(networkProfileBasicInfo, &outCount, bufferSize, nn::nifm::NetworkProfileType_User));
        ASSERT_EQ(nn::nifm::UserNetworkProfileCountMax, outCount);
        NN_LOG("Profile count(User): %d\n", outCount);
    }

    // SSID リスト設定 0件取得
    {
        const int bufferSize = 1;
        nn::nifm::NetworkProfileBasicInfo networkProfileBasicInfo[bufferSize];
        int outCount = 0;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::EnumerateNetworkProfiles(networkProfileBasicInfo, &outCount, 0, nn::nifm::NetworkProfileType_SsidList));
        ASSERT_EQ(g_SsidListSize, outCount);
        NN_LOG("Profile count(SsidList): %d\n", outCount);
    }

    // テンポラリ設定取得
    {
        const nn::nifm::WirelessSettingData wirelessSetting = nn::nifm::InvalidWirelessSettingData;
        const nn::nifm::IpSettingData ipSetting = { { true,{},{},{} },{ true,{},{} },{ false, 0, "",{ false, "", "" } }, 1400 };
        nn::nifm::NetworkProfileData networkProfile = { nn::util::InvalidUuid,{}, nn::nifm::NetworkProfileType_User, nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ethernet, true, true,{ wirelessSetting }, ipSetting };

        nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(networkProfile);
        EXPECT_NE(nn::util::InvalidUuid, temporaryNetworkProfile.GetId());

        const int bufferSize = 1;
        nn::nifm::NetworkProfileBasicInfo networkProfileBasicInfo[bufferSize];
        int outCount = 0;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::EnumerateNetworkProfiles(networkProfileBasicInfo, &outCount, bufferSize, nn::nifm::NetworkProfileType_Temporary));
        ASSERT_EQ(1, outCount);
        NN_LOG("Profile count(Temporary): %d\n", outCount);
    }
}

TEST_F(ProfileCaseTest, PersistentTemporaryNetworkProfileOutOfResource)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::InitializeAdmin());

    const int bufferSize = nn::nifm::UserNetworkProfileCountMax;
    nn::nifm::NetworkProfileBasicInfo networkProfileBasicInfo[bufferSize];
    int outCount;
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::EnumerateNetworkProfiles(networkProfileBasicInfo, &outCount, bufferSize, nn::nifm::NetworkProfileType_User));
    EXPECT_EQ(32, outCount);

    // imatake-none
    const nn::nifm::WirelessSettingData wirelessSetting0 = {
        {  // ssidConfig
            {  // ssid
                12,  // length
                { 0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x6e,0x6f,0x6e,0x65 }  // hex
            },
        false  // nonBroadcast
        },
        {  // security
            {  //authEncryption
                nn::nifm::Authentication_Open,  // authentication
                nn::nifm::Encryption_None  // encryption
            },
            {  // sharedKey
                7,  // length
                "NotUsed"  // keyMaterial
            }
        }
    };
    const nn::nifm::IpSettingData ipSetting0 = {
        {  // ip
            true,  // isAuto
            {},  // ipAddress
            {},  // subnetMask
            {}  // defaultGateway
        },
        {  // dns
            true,  // isAuto
            {},  // preferredDns
            {}  // alternateDns
        },
        {  // proxy
            true,  // isEnabled
            8080,  // port
            "dummy.proxy.jp",  // proxy
            {  // authentication
                true,  // isEnabled
                "dummy_user",  // username
                "dummy_password"  // password
            }
        },
        1400  //mtu
    };
    const nn::nifm::NetworkProfileData networkProfile0 = {
        nn::util::InvalidUuid,  // id
        {},  // name
        nn::nifm::NetworkProfileType_Temporary, // networkProfileType
        nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,  // networkInterfaceType
        true, // isAutoConnect
        true, // isLargeCapacity
        {
            wirelessSetting0
        },
        ipSetting0
    };
    nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(networkProfile0);
    EXPECT_NE(nn::util::InvalidUuid, temporaryNetworkProfile.GetId());

    // ユーザ接続設定が既に最大数の場合は失敗
    NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultOutOfUserNetworkProfileBuffer, nn::nifm::PersistTemporaryNetworkProfile(temporaryNetworkProfile.GetHandle()));
}

TEST_F(ProfileCaseTest, SetAndRemoveUserNetworkProfile)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::InitializeAdmin());

    // 更新
    {
        nn::util::Uuid id;
        nn::nifm::NetworkProfileBasicInfo networkProfileBasicInfo[1];
        int outCount;
        NNT_ASSERT_RESULT_SUCCESS( nn::nifm::EnumerateNetworkProfiles( networkProfileBasicInfo, &outCount, 1, nn::nifm::NetworkProfileType_User) );
        EXPECT_EQ( 32, outCount );

        nn::nifm::NetworkProfileData networkProfile;
        NNT_ASSERT_RESULT_SUCCESS( nn::nifm::GetNetworkProfile( &networkProfile, networkProfileBasicInfo[0].id ) );

        nn::nifm::IpV4Address modifiedIpAddress = {{ 100, 100, 100, 100 }};
        EXPECT_NE( 0, memcmp( modifiedIpAddress.data, networkProfile.ipSetting.ip.ipAddress.data, nn::nifm::IpV4Address::Size ) );
        nn::nifm::IpV4Address originalIpAddress = networkProfile.ipSetting.ip.ipAddress;
        networkProfile.ipSetting.ip.ipAddress = modifiedIpAddress;

        NNT_ASSERT_RESULT_SUCCESS( nn::nifm::SetNetworkProfile( &id, networkProfile ) );
        EXPECT_NE( 0, memcmp( networkProfileBasicInfo[0].id.data, id.data, nn::util::Uuid::Size ) );
        EXPECT_EQ(networkProfileBasicInfo[0].ssid.length, networkProfile.wirelessSetting.ssidConfig.ssid.length);
        EXPECT_EQ(0, memcmp(networkProfileBasicInfo[0].ssid.hex, networkProfile.wirelessSetting.ssidConfig.ssid.hex, networkProfileBasicInfo[0].ssid.length));

        NNT_ASSERT_RESULT_SUCCESS( nn::nifm::GetNetworkProfile( &networkProfile, id ) );
        EXPECT_EQ( 0, memcmp( modifiedIpAddress.data, networkProfile.ipSetting.ip.ipAddress.data, nn::nifm::IpV4Address::Size ) );

        // 戻す
        networkProfile.ipSetting.ip.ipAddress = originalIpAddress;
        NNT_ASSERT_RESULT_SUCCESS( nn::nifm::SetNetworkProfile( &id, networkProfile ) );
        NNT_ASSERT_RESULT_SUCCESS( nn::nifm::GetNetworkProfile( &networkProfile, id ) );
        EXPECT_EQ( 0, memcmp( originalIpAddress.data, networkProfile.ipSetting.ip.ipAddress.data, nn::nifm::IpV4Address::Size ) );
    }

    // 存在しない ID 指定
    {
        nn::util::Uuid id = nn::util::InvalidUuid;
        nn::nifm::NetworkProfileBasicInfo networkProfileBasicInfo[1];
        int outCount;
        NNT_ASSERT_RESULT_SUCCESS( nn::nifm::EnumerateNetworkProfiles( networkProfileBasicInfo, &outCount, 1, nn::nifm::NetworkProfileType_User) );
        EXPECT_EQ( 32, outCount );

        nn::nifm::NetworkProfileData networkProfile;
        NNT_ASSERT_RESULT_SUCCESS( nn::nifm::GetNetworkProfile( &networkProfile, networkProfileBasicInfo[0].id ) );

        bool originalNonBroadCast = networkProfile.wirelessSetting.ssidConfig.nonBroadcast;
        bool modifiedNonBroadCast = !originalNonBroadCast;
        networkProfile.wirelessSetting.ssidConfig.nonBroadcast = modifiedNonBroadCast;

        networkProfile.id.data[0] = ~networkProfile.id.data[0];

        NNT_ASSERT_RESULT_FAILURE( nn::nifm::ResultUserNetworkProfileNotFound, nn::nifm::SetNetworkProfile( &id, networkProfile ) );
        NNT_ASSERT_RESULT_FAILURE( nn::nifm::ResultResourceNotFound, nn::nifm::GetNetworkProfile( &networkProfile, id ) );
        EXPECT_EQ( modifiedNonBroadCast, networkProfile.wirelessSetting.ssidConfig.nonBroadcast );
    }

    // 新規作成 + 削除 nbd-ocn1wep
    {
        nn::util::Uuid id;
        const nn::nifm::NetworkProfileType profileType = nn::nifm::NetworkProfileType_User;
        const nn::nifm::NetworkInterfaceType interfaceType = nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211;
        const nn::nifm::WirelessSettingData wirelessSetting = {{{11, {0x6E,0x62,0x64,0x2D,0x6F,0x63,0x6E,0x31,0x77,0x65,0x70}}, false}, {{nn::nifm::Authentication_Open, nn::nifm::Encryption_Wep}, {11, "keymaterial"}}};
        const nn::nifm::IpSettingData ipSetting = { {false, {{0x0,0x1,0x2,0x3}}, {{0x4,0x5,0x6,0x7}}, {{0x8,0x9,0xa,0xb}}}, {false, {{0xc,0xd,0xe,0xf}}, {{0x10,0x11,0x12,0x13}}}, {false, 0, "proxy.net", {false, "user", "pass"}}, 1400 };

        nn::nifm::NetworkProfileData userNetworkProfile = { nn::util::InvalidUuid, "setting1", profileType, interfaceType, true, true, { wirelessSetting }, ipSetting };

        const int bufferSize = nn::nifm::UserNetworkProfileCountMax;
        nn::nifm::NetworkProfileBasicInfo networkProfileBasicInfo[bufferSize];
        int outCount;
        NNT_ASSERT_RESULT_SUCCESS( nn::nifm::EnumerateNetworkProfiles( networkProfileBasicInfo, &outCount, bufferSize, nn::nifm::NetworkProfileType_User) );
        EXPECT_EQ( 32, outCount );

        // 設定が最大数なので追加できないので事前に削除
        NNT_ASSERT_RESULT_FAILURE( nn::nifm::ResultOutOfUserNetworkProfileBuffer, nn::nifm::SetNetworkProfile( &id, userNetworkProfile ) );
        NNT_ASSERT_RESULT_SUCCESS( nn::nifm::RemoveNetworkProfile( networkProfileBasicInfo[outCount - 1].id ) );

        NNT_ASSERT_RESULT_SUCCESS( nn::nifm::SetNetworkProfile( &id, userNetworkProfile ) );
        EXPECT_NE( id, nn::util::InvalidUuid );

        nn::nifm::NetworkProfileData networkProfile;
        NNT_ASSERT_RESULT_SUCCESS( nn::nifm::GetNetworkProfile( &networkProfile, id ) );

        // 接続設定作成時に発行された ID だけコピーして，元データと比較
        userNetworkProfile.id = networkProfile.id;
        EXPECT_EQ(networkProfile, userNetworkProfile);

        // 削除
        int outCount1;
        NNT_ASSERT_RESULT_SUCCESS( nn::nifm::EnumerateNetworkProfiles( networkProfileBasicInfo, &outCount1, bufferSize, nn::nifm::NetworkProfileType_User) );

        NNT_ASSERT_RESULT_SUCCESS( nn::nifm::RemoveNetworkProfile( id ) );

        int outCount2;
        NNT_ASSERT_RESULT_SUCCESS( nn::nifm::EnumerateNetworkProfiles( networkProfileBasicInfo, &outCount2, bufferSize, nn::nifm::NetworkProfileType_User ) );
        EXPECT_EQ( outCount1 - 1, outCount2 );

        // 削除（存在しないID）
        NNT_ASSERT_RESULT_FAILURE( nn::nifm::ResultResourceNotFound, nn::nifm::RemoveNetworkProfile( id ) );
        NNT_ASSERT_RESULT_FAILURE( nn::nifm::ResultResourceNotFound, nn::nifm::GetNetworkProfile( &networkProfile, id ) );

        // システム登録時にユニークな ID が振られるので、事前に削除していた設定を戻すことはできない
    }
}

TEST_F(ProfileCaseTest, SetAndRemoveSsidListNetworkProfile)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::InitializeAdmin());

    // 更新
    {
        nn::util::Uuid id;
        nn::nifm::NetworkProfileBasicInfo networkProfileBasicInfo[1];
        int outCount;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::EnumerateNetworkProfiles(networkProfileBasicInfo, &outCount, 1, nn::nifm::NetworkProfileType_SsidList));
        EXPECT_EQ(g_SsidListSize, outCount);

        nn::nifm::NetworkProfileData networkProfile;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetNetworkProfile(&networkProfile, networkProfileBasicInfo[0].id));

        nn::nifm::IpV4Address modifiedIpAddress = { { 100, 100, 100, 100 } };
        EXPECT_NE(0, memcmp(modifiedIpAddress.data, networkProfile.ipSetting.ip.ipAddress.data, nn::nifm::IpV4Address::Size));
        networkProfile.ipSetting.ip.ipAddress = modifiedIpAddress;

        // SSID リストは更新できない
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultUserNetworkProfileNotFound, nn::nifm::SetNetworkProfile(&id, networkProfile));
    }

    // 存在しない ID 指定
    {
        nn::util::Uuid id = nn::util::InvalidUuid;
        nn::nifm::NetworkProfileBasicInfo networkProfileBasicInfo[1];
        int outCount;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::EnumerateNetworkProfiles(networkProfileBasicInfo, &outCount, 1, nn::nifm::NetworkProfileType_SsidList));
        EXPECT_EQ(g_SsidListSize, outCount);

        nn::nifm::NetworkProfileData networkProfile;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetNetworkProfile(&networkProfile, networkProfileBasicInfo[0].id));

        bool originalNonBroadCast = networkProfile.wirelessSetting.ssidConfig.nonBroadcast;
        bool modifiedNonBroadCast = !originalNonBroadCast;
        networkProfile.wirelessSetting.ssidConfig.nonBroadcast = modifiedNonBroadCast;

        networkProfile.id.data[0] = ~networkProfile.id.data[0];

        // SSID リストは更新できない
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultUserNetworkProfileNotFound, nn::nifm::SetNetworkProfile(&id, networkProfile));
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultResourceNotFound, nn::nifm::GetNetworkProfile(&networkProfile, id));
    }

    // 削除
    {
        nn::nifm::NetworkProfileBasicInfo networkProfileBasicInfo[1];
        int outCount;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::EnumerateNetworkProfiles(networkProfileBasicInfo, &outCount, 1, nn::nifm::NetworkProfileType_SsidList));
        EXPECT_EQ(g_SsidListSize, outCount);

        nn::util::Uuid id = networkProfileBasicInfo[0].id;

        // SSID リストは削除できない
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultResourceNotFound, nn::nifm::RemoveNetworkProfile(id));

        nn::nifm::NetworkProfileData networkProfile;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetNetworkProfile(&networkProfile, id));
    }
}

TEST_F(ProfileCaseTest, ContinuousModification)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::InitializeAdmin());

    auto startTick = nn::os::GetSystemTick().GetInt64Value();

    const int bufferSize = nn::nifm::UserNetworkProfileCountMax;

    // 削除
    {
        nn::nifm::NetworkProfileBasicInfo networkProfileBasicInfo[bufferSize];
        int outCount;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::EnumerateNetworkProfiles(networkProfileBasicInfo, &outCount, bufferSize, nn::nifm::NetworkProfileType_User));
        EXPECT_EQ(31, outCount);
        EXPECT_LE(outCount, bufferSize);

        for (int i = outCount; i > 0; --i)
        {
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::RemoveNetworkProfile(networkProfileBasicInfo[i - 1].id));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::EnumerateNetworkProfiles(networkProfileBasicInfo, &outCount, bufferSize, nn::nifm::NetworkProfileType_User));
            EXPECT_EQ(i - 1, outCount);
        }
    }

    // 作成
    {
        nn::util::Uuid id;
        nn::nifm::NetworkProfileBasicInfo networkProfileBasicInfo[bufferSize];
        int outCount;
        nn::nifm::NetworkProfileData networkProfileData = {};
        networkProfileData.id = nn::util::InvalidUuid;

        for (int i = 0; i < bufferSize; ++i)
        {
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetNetworkProfile(&id, networkProfileData));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::EnumerateNetworkProfiles(networkProfileBasicInfo, &outCount, bufferSize, nn::nifm::NetworkProfileType_User));
            EXPECT_EQ(i + 1, outCount);
        }
    }

    // 更新
    {
        nn::util::Uuid id;
        nn::nifm::NetworkProfileBasicInfo networkProfileBasicInfo[bufferSize];
        int outCount;

        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::EnumerateNetworkProfiles(networkProfileBasicInfo, &outCount, bufferSize, nn::nifm::NetworkProfileType_User));
        EXPECT_EQ(32, outCount);

        nn::nifm::NetworkProfileData networkProfileData[bufferSize];
        for (int i = 0; i < bufferSize; ++i)
        {
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetNetworkProfile(&networkProfileData[i], networkProfileBasicInfo[i].id));
        }

        for (int i = 0; i < bufferSize; ++i)
        {
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetNetworkProfile(&id, networkProfileData[i]));
        }
    }

    // 上記の処理は、実際には遅延書き込みされる
    {
        int64_t t = (nn::os::GetSystemTick().GetInt64Value() - startTick) / nn::os::GetSystemTickFrequency();
        int s = static_cast<int>(t % 60);
        int m = static_cast<int>((t / 60) % 60);
        int h = static_cast<int>(t / (60 * 60));
        NN_LOG("%02d:%02d:%02d\n", h, m, s);
        ASSERT_LE(t, 1);
    }
}

TEST_F(ProfileCaseTest, TrivialProfileModification)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::InitializeAdmin());

    // 1件削除
    nn::nifm::NetworkProfileBasicInfo networkProfileBasicInfo;
    int outCount;
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::EnumerateNetworkProfiles(&networkProfileBasicInfo, &outCount, 1, nn::nifm::NetworkProfileType_User));
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::RemoveNetworkProfile(networkProfileBasicInfo.id));
    EXPECT_GT(outCount, 0);

    // 1件追加
    // imatake-wpa2aes
    const nn::nifm::WirelessSettingData wirelessSetting = {
        {  // ssidConfig
            {  // ssid
                15,  // length
                { 0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x77,0x70,0x61,0x32,0x61,0x65,0x73 }  // hex
            },
        false  // nonBroadcast
        },
        {  // security
            {  //authEncryption
                nn::nifm::Authentication_Wpa2Psk,  // authentication
                nn::nifm::Encryption_Aes  // encryption
            },
            {  // sharedKey
                11,  // length
                "Shi2iTaiZen"  // keyMaterial
            }
        }
    };
    const nn::nifm::IpSettingData ipSetting = {
        {  // ip
            true,  // isAuto
            {},  // ipAddress
            {},  // subnetMask
            {}  // defaultGateway
        },
        {  // dns
            true,  // isAuto
            {},  // preferredDns
            {}  // alternateDns
        },
        {  // proxy
            false,  // isEnabled
            0,  // port
            "",  // proxy
            {  // authentication
                false,  // isEnabled
                "",  // username
                ""  // password
            }
        },
        1400  //mtu
    };
    nn::nifm::NetworkProfileData networkProfileData = {
        nn::util::InvalidUuid,  // id
        {},  // name
        nn::nifm::NetworkProfileType_User, // networkProfileType
        nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,  // networkInterfaceType
        true,  // isAutoConnect
        true, // isLargeCapacity
        {
            wirelessSetting
        },
        ipSetting
    };

    nn::util::Uuid id = nn::util::InvalidUuid;
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetNetworkProfile(&id, networkProfileData));
    ASSERT_NE(nn::util::InvalidUuid, id);

    nn::nifm::NetworkConnection networkConnection;
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection.GetRequestHandle(), id));

    EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

    NNT_ASSERT_RESULT_SUCCESS(networkConnection.GetResult());

    // 接続設定名，isAutoConnect, nonBroadcast の書き換え
    networkProfileData.id = id;
    nn::util::Strlcpy(networkProfileData.name, "EthernetProfile2", sizeof("EthernetProfile2"));
    networkProfileData.isAutoConnect = !networkProfileData.isAutoConnect;
    networkProfileData.wirelessSetting.ssidConfig.nonBroadcast = !networkProfileData.wirelessSetting.ssidConfig.nonBroadcast;
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetNetworkProfile(&id, networkProfileData));

    nn::os::SleepThread(nn::TimeSpan::FromSeconds(10));

    // 接続設定 ID も変わらず，利用要求も受理されたまま
    ASSERT_EQ(id, networkProfileData.id);
    NNT_ASSERT_RESULT_SUCCESS(networkConnection.GetResult());

    // GetCurrentNetworkProfile で得られる接続設定にも変更が反映される
    nn::nifm::NetworkProfileData networkProfileData2;
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetCurrentNetworkProfile(&networkProfileData2));
    ASSERT_EQ(0, nn::util::Strncmp(networkProfileData.name, networkProfileData2.name, nn::nifm::NetworkProfileBasicInfo::NameSize));

    networkConnection.GetSystemEvent().Clear();

    // 認証タイプの書き換え
    networkProfileData.wirelessSetting.security.authEncryption.authentication = nn::nifm::Authentication_Open;
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetNetworkProfile(&id, networkProfileData));

    ASSERT_TRUE(nn::os::TimedWaitSystemEvent(networkConnection.GetSystemEvent().GetBase(), nn::TimeSpan::FromSeconds(30)));

    // 接続設定 ID は変更され，利用要求も却下される
    ASSERT_NE(id, networkProfileData.id);
    NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultNetworkProfileNoLongerAvailable, networkConnection.GetResult());
}

TEST_F(ProfileCaseTest, TruncatedKeyMaterial)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::InitializeAdmin());

    // imatake-wpa2aes
    const nn::nifm::WirelessSettingData wirelessSetting1 = {
        {  // ssidConfig
            {  // ssid
                15,  // length
                { 0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x77,0x70,0x61,0x32,0x61,0x65,0x73 }  // hex
            },
        false  // nonBroadcast
        },
        {  // security
            {  //authEncryption
                nn::nifm::Authentication_Wpa2Psk,  // authentication
                nn::nifm::Encryption_Aes  // encryption
            },
            {  // sharedKey
                100,  // length: 64 に切り詰められる
                {
                    0x62,0x61,0x64,0x38,0x38,0x34,0x30,0x62,0x61,0x31,
                    0x38,0x37,0x31,0x61,0x65,0x37,0x66,0x37,0x36,0x66,
                    0x32,0x65,0x66,0x39,0x33,0x33,0x35,0x31,0x61,0x62,
                    0x36,0x66,0x35,0x30,0x65,0x64,0x65,0x33,0x35,0x63,
                    0x66,0x62,0x63,0x39,0x39,0x35,0x64,0x39,0x63,0x33,
                    0x37,0x37,0x32,0x64,0x61,0x32,0x31,0x63,0x61,0x63,
                    0x65,0x65,0x39,0x35
                }  // keyMaterial bad8840ba1871ae7f76f2ef93351ab6f50ede35cfbc995d9c3772da21cacee95
            }
        }
    };
    const nn::nifm::IpSettingData ipSetting1 = {
        {  // ip
            true,  // isAuto
            {},  // ipAddress
            {},  // subnetMask
            {}  // defaultGateway
        },
        {  // dns
            true,  // isAuto
            {},  // preferredDns
            {}  // alternateDns
        },
        {  // proxy
            false,  // isEnabled
            0,  // port
            "",  // proxy
            {  // authentication
                false,  // isEnabled
                "",  // username
                ""  // password
            }
        },
        1400  //mtu
    };
    const nn::nifm::NetworkProfileData networkProfile1 = {
        nn::util::InvalidUuid,  // id
        {},  // name
        nn::nifm::NetworkProfileType_Temporary, // networkProfileType
        nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,  // networkInterfaceType
        true,  // isAutoConnect
        true, // isLargeCapacity
        {
            wirelessSetting1
        },
        ipSetting1
    };

    nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(networkProfile1);
    nn::util::Uuid id1 = temporaryNetworkProfile.GetId();
    ASSERT_NE(nn::util::InvalidUuid, id1);

    nn::nifm::NetworkConnection networkConnection;
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection.GetRequestHandle(), id1));

    EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_SUCCESS));
    NNT_ASSERT_RESULT_SUCCESS(networkConnection.GetResult());
}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    auto ret = RUN_ALL_TESTS();

    nnt::Exit(ret);
}
