﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <nnt.h>
#include <nnt/result/testResult_Assert.h>

#include <nn/os.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/init.h>

#include <nn/nifm/nifm_ApiForSystem.h>
#include <nn/nifm/nifm_ApiForTest.h>

#include <nn/nifm/nifm_ApiClientManagement.h>
#include <nn/nifm/nifm_ApiRequest.h>
#include <nn/nifm/nifm_ApiRequestPrivate.h>
#include <nn/nifm/nifm_ApiScan.h>
#include <nn/nifm/nifm_NetworkConnection.h>
#include <nn/nifm/nifm_TemporaryNetworkProfile.h>
#include <nn/nifm/nifm_TypesWirelessSetting.h>
#include <nn/nifm/nifm_ResultPrivate.h>

#include <nn/ae.h>
#include <nn/err.h>

#include "../Common/nifm_TestUtility.h"


namespace
{
    const int64_t TIME_OUT_IN_SECONDS_FOR_SUCCESS = 60;
    const int64_t TIME_OUT_IN_SECONDS_FOR_FAILURE = 180;
}

class ErrorReportTest : public ::testing::Test
{
protected:
    static void SetUpTestCase()
    {
    }

    static void TearDownTestCase()
    {
    }
};

TEST_F(ErrorReportTest, ErrorReportWireless)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::InitializeSystem());

    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Scan());

    NN_ASSERT(nn::nifm::test::Disconnect());

    // 一度エラーレポートを表示 (0)
    NN_LOG("nn::err::ShowError(nn::nifm::ResultInternalError()) (0)\n");
    nn::err::ShowError(nn::nifm::ResultInternalError());

    {
        // imatake-wpa2aes
        const nn::nifm::WirelessSettingData wirelessSetting = {
            {  // ssidConfig
                {  // ssid
                    15,  // length
                    { 0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x77,0x70,0x61,0x32,0x61,0x65,0x73 }  // hex
                },
            true  // nonBroadcast
            },
            {  // security
                {  //authEncryption
                    nn::nifm::Authentication_Wpa2Psk,  // authentication
                    nn::nifm::Encryption_Aes  // encryption
                },
                {  // sharedKey
                    11,  // length
                    "Shi2iTaiZen"  // keyMaterial
                }
            }
        };
        const nn::nifm::IpSettingData ipSetting = {
            {  // ip
                true,  // isAuto
                {},  // ipAddress
                {},  // subnetMask
                {}  // defaultGateway
            },
            {  // dns
                true,  // isAuto
                {},  // preferredDns
                {}  // alternateDns
            },
            {  // proxy
                false,  // isEnabled
                0,  // port
                "",  // proxy
                {  // authentication
                    false,  // isEnabled
                    "",  // username
                    ""  // password
                }
            },
            1300  //mtu
        };
        const nn::nifm::NetworkProfileData networkProfile = {
            nn::util::InvalidUuid,  // id
            {},  // name
            nn::nifm::NetworkProfileType_Temporary, // networkProfileType
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,  // networkInterfaceType
            true,  // isAutoConnect
            true, // isLargeCapacity
            {
                wirelessSetting
            },
            ipSetting
        };

        nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(networkProfile);
        ASSERT_NE(nn::util::InvalidUuid, temporaryNetworkProfile.GetId());

        {
            nn::nifm::NetworkConnection networkConnection;
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnection.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForSystemProcess));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestPriority(networkConnection.GetRequestHandle(), 0));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection.GetRequestHandle(), temporaryNetworkProfile.GetId()));

            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

            ASSERT_FALSE(nn::nifm::IsAnyForegroundRequestAccepted());

            // BG 要求のみのエラーレポート（nifm レイヤからは発生しない）(1)
            NN_LOG("nn::err::ShowError(nn::nifm::ResultInternalError()) (1)\n");
            nn::err::ShowError(nn::nifm::ResultInternalError());
        }

        {
            nn::nifm::NetworkConnection networkConnection;
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnection.GetRequestHandle(), nn::nifm::RequirementPreset_InternetGeneric));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection.GetRequestHandle(), temporaryNetworkProfile.GetId()));

            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

            nn::Result result = networkConnection.GetResult();
            NNT_ASSERT_RESULT_SUCCESS(result);

            // 無線接続状態エラーレポート（nifm レイヤからは発生しない） (2)
            NN_LOG("nn::err::ShowError(nn::nifm::ResultInternalError()) (2)\n");
            nn::err::ShowError(nn::nifm::ResultInternalError());

            nn::nifm::NetworkConnection networkConnectionLocal;
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnectionLocal.GetRequestHandle(), nn::nifm::RequirementPreset_LocalGeneric));

            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionLocal, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

            NNT_NIFM_WAIT_FOR_PROCESSING;

            result = networkConnection.GetResult();
            NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultLowPriority, result);

            // ローカル通信ではエラーレポートは更新されず，直近のエラーレポートが再度表示される (3)
            NN_LOG("nn::err::ShowError(result) (3)\n");
            nn::err::ShowError(result);

            networkConnectionLocal.CancelRequest();

            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

            nn::nifm::NetworkConnection networkConnectionBestEffort;
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnectionBestEffort.GetRequestHandle(), nn::nifm::RequirementPreset_InternetBestEffort));

            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnectionBestEffort, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

            NNT_NIFM_WAIT_FOR_PROCESSING;

            result = networkConnection.GetResult();
            NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultLowPriority, result);

            result = networkConnectionBestEffort.GetResult();
            NNT_ASSERT_RESULT_SUCCESS(result);

            // InternetGeneric の接続は切断されるが，次の BestEffort も FG 要求なので，全体としては接続状態のレポート (4)
            NN_LOG("nn::err::ShowError(nn::nifm::ResultInternalError()) (4)\n");
            nn::err::ShowError(nn::nifm::ResultInternalError());
        }
    }
} // NOLINT(impl/function_size)

TEST_F(ErrorReportTest, WifiWebAuth)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::InitializeSystem());

    {
        // imatake-wpa2aes
        const nn::nifm::WirelessSettingData wirelessSetting = {
            {  // ssidConfig
                {  // ssid
                    15,  // length
                    { 0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x77,0x70,0x61,0x32,0x61,0x65,0x73 }  // hex
                },
            false  // nonBroadcast
            },
            {  // security
                {  //authEncryption
                    nn::nifm::Authentication_Wpa2Psk,  // authentication
                    nn::nifm::Encryption_Aes  // encryption
                },
                {  // sharedKey
                    11,  // length
                    "Shi2iTaiZen"  // keyMaterial
                }
            }
        };
        const nn::nifm::IpSettingData ipSetting = {
            {  // ip
                true,  // isAuto
                {},  // ipAddress
                {},  // subnetMask
                {}  // defaultGateway
            },
            {  // dns
                false,  // isAuto
                { { 52, 68, 203, 240 } },  // preferredDns
                {}  // alternateDns
            },
            {  // proxy
                false,  // isEnabled
                0,  // port
                "",  // proxy
                {  // authentication
                    false,  // isEnabled
                    "",  // username
                    ""  // password
                }
            },
            1400  // mtu
        };
        const nn::nifm::NetworkProfileData networkProfile = {
            nn::util::InvalidUuid,  // id
            {},  // name
            nn::nifm::NetworkProfileType_Temporary, // networkProfileType
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,  // networkInterfaceType
            true,  // isAutoConnect
            true, // isLargeCapacity
            {
                wirelessSetting
            },
            ipSetting
        };

        nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(networkProfile);
        ASSERT_NE(nn::util::InvalidUuid, temporaryNetworkProfile.GetId());
        {
            nn::nifm::NetworkConnection networkConnection;
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(networkConnection.GetRequestHandle(), nn::nifm::RequirementPreset_InternetGeneric));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection.GetRequestHandle(), temporaryNetworkProfile.GetId()));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_Forced));

            // 他の実機テストと WISPr 認証のタイミングが重なった（1分以内）場合、3回までは再試行
            for (int i = 0; i < 3; ++i)
            {
                EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

                if (networkConnection.GetResult().IsSuccess())
                {
                    networkConnection.CancelRequest();
                    nn::os::SleepThread(nn::TimeSpan::FromSeconds(60));
                    continue;
                }

                break;
            }

            NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultHotspotAuthenticationViaWebAuthAppletNeeded, networkConnection.GetResult());

            // WifiWebAuth 内で接続してもエラーコンテキストは更新されない
            NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultErrorHandlingCompleted, networkConnection.HandleNetworkRequestResult());

            // リダイレクト応答を受けた時点で（認証していなくても）リダイレクトURLがコンテキストに保存される (5)
            NN_LOG("nn::err::ShowError(nn::nifm::ResultInternalError()) (5)\n");
            nn::err::ShowError(nn::nifm::ResultInternalError());
        }

        {
            // 相乗りで Hotspot 認証
            nn::nifm::NetworkConnection networkConnection1;
            ASSERT_NE(nullptr, nn::nifm::GetRequestClientPointer(networkConnection1.GetRequestHandle()));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnection1.GetRequestHandle(), 100));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection1.GetRequestHandle(), temporaryNetworkProfile.GetId()));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection1.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));

            EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection1, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

            NNT_ASSERT_RESULT_SUCCESS(networkConnection1.GetResult());

            nn::nifm::NetworkConnection networkConnection2;
            ASSERT_NE(nullptr, nn::nifm::GetRequestClientPointer(networkConnection2.GetRequestHandle()));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestRawPriority(networkConnection2.GetRequestHandle(), 100 - 10));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection2.GetRequestHandle(), temporaryNetworkProfile.GetId()));
            NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection2.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_Forced));

            // 他の実機テストと WISPr 認証のタイミングが重なった（1分以内）場合、3回までは再試行
            for (int i = 0; i < 3; ++i)
            {
                EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection2, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

                if (networkConnection2.GetResult().IsSuccess())
                {
                    networkConnection2.CancelRequest();
                    nn::os::SleepThread(nn::TimeSpan::FromSeconds(60));
                    continue;
                }

                break;
            }

            NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultHotspotAuthenticationViaWebAuthAppletNeeded, networkConnection2.GetResult());

            {
                auto pRequestClient = nn::nifm::GetRequestClientPointer(networkConnection2.GetRequestHandle());
                ASSERT_NE(nullptr, pRequestClient);

                uint32_t submitId0;
                pRequestClient->GetSubmitId(&submitId0);

                nn::nifm::AdditionalInfo additionalInfo;
                uint32_t submitId;
                NNT_ASSERT_RESULT_SUCCESS(nn::nifm::GetRequestAdditionalInfo(&additionalInfo, &submitId, networkConnection2.GetRequestHandle()));

                char idStr[nn::util::Uuid::StringSize];
                NN_LOG("AdditionalInfo: { %s, %s }, Revision: %d\n", additionalInfo.redirectUrl, additionalInfo.profileId.ToString(idStr, nn::util::Uuid::StringSize), submitId);
                ASSERT_EQ(0, std::memcmp(additionalInfo.redirectUrl, "https://ssltest3.app.nintendowifi.net/login", sizeof("https://ssltest3.app.nintendowifi.net/login")));
                ASSERT_EQ(temporaryNetworkProfile.GetId(), additionalInfo.profileId);
                ASSERT_EQ(submitId0, submitId);
            }

            // WifiWebAuth 内で接続してもエラーコンテキストは更新されない
            NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultErrorHandlingCompleted, networkConnection2.HandleNetworkRequestResult());

            // リダイレクト応答を受けた時点で（認証していなくても）リダイレクトURLがコンテキストに保存される (6)
            NN_LOG("nn::err::ShowError(nn::nifm::ResultInternalError()) (6)\n");
            nn::err::ShowError(nn::nifm::ResultInternalError());
        }
    }
} // NOLINT(impl/function_size)

TEST_F(ErrorReportTest, ErrorReportEthernet)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::InitializeSystem());

    {
        // Ethernet
        const nn::nifm::IpSettingData ipSetting = {
            {  // ip
                true,  // isAuto
                {},  // ipAddress
                {},  // subnetMask
                {}  // defaultGateway
            },
            {  // dns
                true,  // isAuto
                {},  // preferredDns
                {}  // alternateDns
            },
            {  // proxy
                false,  // isEnabled
                0,  // port
                "",  // proxy
                {  // authentication
                    false,  // isEnabled
                    "",  // username
                    ""  // password
                }
            },
            1500  //mtu
        };
        const nn::nifm::NetworkProfileData networkProfile = {
            nn::util::InvalidUuid,  // id
            {},  // name
            nn::nifm::NetworkProfileType_Temporary, // networkProfileType
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ethernet,  // networkInterfaceType
            true,  // isAutoConnect
            true, // isLargeCapacity
            {
                {}
            },
            ipSetting
        };

        nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(networkProfile);
        ASSERT_NE(nn::util::InvalidUuid, temporaryNetworkProfile.GetId());

        nn::nifm::NetworkConnection networkConnection;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_NotRequired));
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection.GetRequestHandle(), temporaryNetworkProfile.GetId()));

        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_SUCCESS));

        // 有線接続状態エラーレポート（nifm レイヤからは発生しない） (7)
        NN_LOG("nn::err::ShowError(nn::nifm::ResultInternalError()) (7)\n");
        nn::err::ShowError(nn::nifm::ResultInternalError());
    }
}

TEST_F(ErrorReportTest, ErrorReportWirelessConnectionFailed)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::InitializeSystem());

    {
        // imatake-wpa2aes(wrong security)
        const nn::nifm::WirelessSettingData wirelessSetting = {
            {  // ssidConfig
                {  // ssid
                    15,  // length
                    { 0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x77,0x70,0x61,0x32,0x61,0x65,0x73 }  // hex
                },
            false  // nonBroadcast
            },
            {  // security
                {  //authEncryption
                    nn::nifm::Authentication_Wpa2Psk,  // authentication
                    nn::nifm::Encryption_Wep  // encryption
                },
                {  // sharedKey
                    11,  // length
                    "Shi2iTaiZen"  // keyMaterial
                }
            }
        };
        const nn::nifm::IpSettingData ipSetting = {
            {  // ip
                true,  // isAuto
                {},  // ipAddress
                {},  // subnetMask
                {}  // defaultGateway
            },
            {  // dns
                true,  // isAuto
                {},  // preferredDns
                {}  // alternateDns
            },
            {  // proxy
                false,  // isEnabled
                0,  // port
                "",  // proxy
                {  // authentication
                    false,  // isEnabled
                    "",  // username
                    ""  // password
                }
            },
            1310  //mtu
        };
        const nn::nifm::NetworkProfileData networkProfile = {
            nn::util::InvalidUuid,  // id
            {},  // name
            nn::nifm::NetworkProfileType_Temporary, // networkProfileType
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,  // networkInterfaceType
            true,  // isAutoConnect
            true, // isLargeCapacity
            {
                wirelessSetting
            },
            ipSetting
        };

        nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(networkProfile);
        ASSERT_NE(nn::util::InvalidUuid, temporaryNetworkProfile.GetId());

        nn::nifm::NetworkConnection networkConnection;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection.GetRequestHandle(), temporaryNetworkProfile.GetId()));

        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_FAILURE));

        nn::Result result = networkConnection.GetResult();
        //NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultConnectionFailedWithReasonCodeAuthInvalid, result);

        // ステルスではない無線接続失敗　セキュリティ間違い (8) AP が見つからないエラーと同等
        NN_LOG("nn::err::ShowError(result) (8)\n");
        nn::err::ShowError(result);
    }

    {
        // imatake-wpa2aes(wrong password)
        const nn::nifm::WirelessSettingData wirelessSetting = {
            {  // ssidConfig
                {  // ssid
                    15,  // length
                    { 0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x77,0x70,0x61,0x32,0x61,0x65,0x73 }  // hex
                },
            false  // nonBroadcast
            },
            {  // security
                {  //authEncryption
                    nn::nifm::Authentication_Wpa2Psk,  // authentication
                    nn::nifm::Encryption_Aes  // encryption
                },
                {  // sharedKey
                    13,  // length
                    "WrongPassword"  // keyMaterial
                }
            }
        };
        const nn::nifm::IpSettingData ipSetting = {
            {  // ip
                true,  // isAuto
                {},  // ipAddress
                {},  // subnetMask
                {}  // defaultGateway
            },
            {  // dns
                true,  // isAuto
                {},  // preferredDns
                {}  // alternateDns
            },
            {  // proxy
                false,  // isEnabled
                0,  // port
                "",  // proxy
                {  // authentication
                    false,  // isEnabled
                    "",  // username
                    ""  // password
                }
            },
            1320  //mtu
        };
        const nn::nifm::NetworkProfileData networkProfile = {
            nn::util::InvalidUuid,  // id
            {},  // name
            nn::nifm::NetworkProfileType_Temporary, // networkProfileType
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,  // networkInterfaceType
            true,  // isAutoConnect
            true, // isLargeCapacity
            {
                wirelessSetting
            },
            ipSetting
        };

        nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(networkProfile);
        ASSERT_NE(nn::util::InvalidUuid, temporaryNetworkProfile.GetId());

        nn::nifm::NetworkConnection networkConnection;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection.GetRequestHandle(), temporaryNetworkProfile.GetId()));

        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_FAILURE));

        nn::Result result = networkConnection.GetResult();
        EXPECT_TRUE(result <= nn::nifm::ResultConnectionFailedWithReasonCode());
        NN_LOG("result=%d-%d\n", result.GetModule(), result.GetDescription());

        // ステルスではない無線接続失敗　パスワード間違い (9)
        // 2110-2202(ResultConnectionFailedWithReasonCodeAuthInvalid) に統一
        NN_LOG("nn::err::ShowError(nn::nifm::ResultConnectionFailedWithReasonCodeAuthInvalid()) (9)\n");
        nn::err::ShowError(nn::nifm::ResultConnectionFailedWithReasonCodeAuthInvalid());
    }

    {
        // not-exist-ap
        const nn::nifm::WirelessSettingData wirelessSetting = {
            {  // ssidConfig
                {  // ssid
                    12,  // length
                    { 0x6e,0x6f,0x74,0x2d,0x65,0x78,0x69,0x73,0x74,0x2d,0x61,0x70 } // hex
                },
            false  // nonBroadcast
            },
            {  // security
                {  //authEncryption
                    nn::nifm::Authentication_Wpa2Psk,  // authentication
                    nn::nifm::Encryption_Aes  // encryption
                },
                {  // sharedKey
                    8,  // length
                    "password"  // keyMaterial
                }
            }
        };
        const nn::nifm::IpSettingData ipSetting = {
            {  // ip
                true,  // isAuto
                {},  // ipAddress
                {},  // subnetMask
                {}  // defaultGateway
            },
            {  // dns
                true,  // isAuto
                {},  // preferredDns
                {}  // alternateDns
            },
            {  // proxy
                false,  // isEnabled
                0,  // port
                "",  // proxy
                {  // authentication
                    false,  // isEnabled
                    "",  // username
                    ""  // password
                }
            },
            1330  //mtu
        };
        const nn::nifm::NetworkProfileData networkProfile = {
            nn::util::InvalidUuid,  // id
            {},  // name
            nn::nifm::NetworkProfileType_Temporary, // networkProfileType
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,  // networkInterfaceType
            true,  // isAutoConnect
            true, // isLargeCapacity
            {
                wirelessSetting
            },
            ipSetting
        };

        nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(networkProfile);
        ASSERT_NE(nn::util::InvalidUuid, temporaryNetworkProfile.GetId());

        nn::nifm::NetworkConnection networkConnection;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection.GetRequestHandle(), temporaryNetworkProfile.GetId()));

        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_FAILURE));

        nn::Result result = networkConnection.GetResult();
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultNetworkNotFound, result);

        // ステルスではない無線接続失敗 存在しない AP (10)
        NN_LOG("nn::err::ShowError(result) (10)\n");
        nn::err::ShowError(result);
    }

    {
        // imatake-wpa2aes(wrong dns)
        const nn::nifm::WirelessSettingData wirelessSetting = {
            {  // ssidConfig
                {  // ssid
                    15,  // length
                    { 0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x77,0x70,0x61,0x32,0x61,0x65,0x73 }  // hex
                },
            false  // nonBroadcast
            },
            {  // security
                {  //authEncryption
                    nn::nifm::Authentication_Wpa2Psk,  // authentication
                    nn::nifm::Encryption_Aes  // encryption
                },
                {  // sharedKey
                    11,  // length
                    "Shi2iTaiZen"  // keyMaterial
                }
            }
        };
        const nn::nifm::IpSettingData ipSetting = {
            {  // ip
                true,  // isAuto
                {},  // ipAddress
                {},  // subnetMask
                {}  // defaultGateway
            },
            {  // dns
                false,  // isAuto
                { { 168, 254, 123, 123 } },  // preferredDns
                {}  // alternateDns
            },
            {  // proxy
                false,  // isEnabled
                0,  // port
                "",  // proxy
                {  // authentication
                    false,  // isEnabled
                    "",  // username
                    ""  // password
                }
            },
            1210  //mtu
        };
        const nn::nifm::NetworkProfileData networkProfile = {
            nn::util::InvalidUuid,  // id
            {},  // name
            nn::nifm::NetworkProfileType_Temporary, // networkProfileType
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,  // networkInterfaceType
            true,  // isAutoConnect
            true, // isLargeCapacity
            {
                wirelessSetting
            },
            ipSetting
        };

        nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(networkProfile);
        ASSERT_NE(nn::util::InvalidUuid, temporaryNetworkProfile.GetId());

        nn::nifm::NetworkConnection networkConnection;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection.GetRequestHandle(), temporaryNetworkProfile.GetId()));

        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_FAILURE));

        nn::Result result = networkConnection.GetResult();
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultConnectionTestHttpCouldntResolveHost, result);

        // DNS 間違い (11)
        NN_LOG("nn::err::ShowError(result) (11)\n");
        nn::err::ShowError(result);
    }

    {
        // imatake-wpa2aes(wrong proxy)
        const nn::nifm::WirelessSettingData wirelessSetting = {
            {  // ssidConfig
                {  // ssid
                    15,  // length
                    { 0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x77,0x70,0x61,0x32,0x61,0x65,0x73 }  // hex
                },
            false  // nonBroadcast
            },
            {  // security
                {  //authEncryption
                    nn::nifm::Authentication_Wpa2Psk,  // authentication
                    nn::nifm::Encryption_Aes  // encryption
                },
                {  // sharedKey
                    11,  // length
                    "Shi2iTaiZen"  // keyMaterial
                }
            }
        };
        const nn::nifm::IpSettingData ipSetting = {
            {  // ip
                true,  // isAuto
                {},  // ipAddress
                {},  // subnetMask
                {}  // defaultGateway
            },
            {  // dns
                true,  // isAuto
                {},  // preferredDns
                {}  // alternateDns
            },
            {  // proxy
                true,  // isEnabled
                10001,  // port
                "proxy55.nintendowifi.net",  // proxy
                {  // authentication
                    false,  // isEnabled
                    "",  // username
                    ""  // password
                }
            },
            1220  //mtu
        };
        const nn::nifm::NetworkProfileData networkProfile = {
            nn::util::InvalidUuid,  // id
            {},  // name
            nn::nifm::NetworkProfileType_Temporary, // networkProfileType
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,  // networkInterfaceType
            true,  // isAutoConnect
            true, // isLargeCapacity
            {
                wirelessSetting
            },
            ipSetting
        };

        nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(networkProfile);
        ASSERT_NE(nn::util::InvalidUuid, temporaryNetworkProfile.GetId());

        nn::nifm::NetworkConnection networkConnection;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection.GetRequestHandle(), temporaryNetworkProfile.GetId()));

        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_FAILURE));

        nn::Result result = networkConnection.GetResult();
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultConnectionTestHttpCouldntResolveProxy, result);

        // Proxy 間違い (12)
        NN_LOG("nn::err::ShowError(result) (12)\n");
        nn::err::ShowError(result);
    }

    {
        // imatake-openwep40(wrong password)
        const nn::nifm::WirelessSettingData wirelessSetting = {
            {  // ssidConfig
                {  // ssid
                    17,  // length
                    { 0x69,0x6d,0x61,0x74,0x61,0x6b,0x65,0x2d,0x6f,0x70,0x65,0x6e,0x77,0x65,0x70,0x34,0x30 }  // hex
                },
            false  // nonBroadcast
            },
            {  // security
                {  //authEncryption
                    nn::nifm::Authentication_Open,  // authentication
                    nn::nifm::Encryption_Wep  // encryption
                },
                {  // sharedKey
                    5,  // length
                    "Wrong"  // keyMaterial
                }
            }
        };
        const nn::nifm::IpSettingData ipSetting = {
            {  // ip
                true,  // isAuto
                {},  // ipAddress
                {},  // subnetMask
                {}  // defaultGateway
            },
            {  // dns
                true,  // isAuto
                {},  // preferredDns
                {}  // alternateDns
            },
            {  // proxy
                false,  // isEnabled
                0,  // port
                "",  // proxy
                {  // authentication
                    false,  // isEnabled
                    "",  // username
                    ""  // password
                }
            },
            1410  //mtu
        };
        const nn::nifm::NetworkProfileData networkProfile = {
            nn::util::InvalidUuid,  // id
            {},  // name
            nn::nifm::NetworkProfileType_Temporary, // networkProfileType
            nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211,  // networkInterfaceType
            true,  // isAutoConnect
            true, // isLargeCapacity
            {
                wirelessSetting
            },
            ipSetting
        };

        nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(networkProfile);
        ASSERT_NE(nn::util::InvalidUuid, temporaryNetworkProfile.GetId());

        nn::nifm::NetworkConnection networkConnection;
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(networkConnection.GetRequestHandle(), temporaryNetworkProfile.GetId()));

        EXPECT_TRUE(nn::nifm::test::SubmitRequestAndWait(&networkConnection, TIME_OUT_IN_SECONDS_FOR_FAILURE));

        nn::Result result = networkConnection.GetResult();
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultDhcpFailedOrPassphraseWrong, result);

        // Password 間違い (13) ただし，Wep なので Open/Shared が不明の状態
        NN_LOG("nn::err::ShowError(result) (13)\n");
        nn::err::ShowError(result);
    }

} // NOLINT(impl/function_size)

TEST_F(ErrorReportTest, ErrorReportMultiTrial)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::InitializeSystem());

    // 接続設定の imatake-wpa2aes-wrong1(DNSエラー)と imatake-wpa2aes-wrong2(PassPhrase間違い)を試して
    // 前者のエラーおよび関連情報が保存される
    {
        nn::nifm::NetworkConnection networkConnection;
        networkConnection.SubmitRequestAndWait();

        nn::Result result = networkConnection.GetResult();
        NNT_ASSERT_RESULT_FAILURE(nn::nifm::ResultConnectionTestHttpCouldntResolveHost, result);

        // DNSエラー (14)
        NN_LOG("nn::err::ShowError(result) (14)\n");
        nn::err::ShowError(result);
    }
}

// nn::ae のメッセージ処理
bool ExpectMessage(nn::ae::Message expectMessage, nn::os::SystemEventType& e) NN_NOEXCEPT
{
    auto message = nn::ae::WaitForNotificationMessage(&e);
    if (message != expectMessage)
    {
        NN_LOG("Unexpected message 0x%08x (expect=0x%08x)\n", message, expectMessage);
        return false;
    }
    return true;
}

nn::ncm::ApplicationId g_ApplicationId;
void RunApplication(nn::ae::SystemAppletParameters* param) NN_NOEXCEPT
{
    // SA の起動
    nn::os::SystemEventType e;
    nn::ae::InitializeNotificationMessageEvent(&e);
    NN_ABORT_UNLESS(ExpectMessage(nn::ae::Message_ChangeIntoForeground, e));

    auto ret = RUN_ALL_TESTS();

    nnt::Exit(ret);
}

// アロケータ用のバッファ
NN_ALIGNAS(4096) uint8_t  g_MallocBuffer[1 * 1024 * 1024];
extern "C" void nninitStartup()
{
    nn::init::InitializeAllocator(g_MallocBuffer, sizeof(g_MallocBuffer));
}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    // SystemApplet としてアプリを起動する
    nn::ae::InvokeSystemAppletMain(nn::ae::AppletId_SystemAppletMenu, RunApplication);
}
