﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// テスト用のヘッダ
#include <utility>

#include <nn/os.h>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/nn_ErrorResult.h>

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>

// テスト対象のヘッダ
#include <../../../../../Programs/Eris/Sources/Libraries/ngc/succinct/ngc_Sbv.h>

namespace {

/**
 * @brief   テストに与えるパラメータです。
 */
enum ArrayTestParam
{
    ArrayTestParam_UseMalloc = 0,           //!< デフォルトのアロケータ(std::malloc)を利用する
    ArrayTestParam_UseWorkBufAllocator      //!< nn::ngc::detail::WorkBufAllocator を利用する
};

/**
 * @brief   テストで利用するテストフィクスチャです。
 */
class SuccinctArrayTest : public ::testing::TestWithParam<int>
{
protected:

    SuccinctArrayTest() NN_NOEXCEPT : m_HeapSize(2 * 1024 * 1024), m_HeapAddr(NULL), m_CompressedArray(),
        m_Map(), m_Allocator() {}

    /**
     * @brief   テスト開始時に毎回呼び出される関数です。
     */
    virtual void SetUp()
    {
        if (GetParam() == ArrayTestParam_UseWorkBufAllocator)
        {
            m_HeapAddr = std::malloc(m_HeapSize);
            NN_ABORT_UNLESS(m_HeapAddr);
            m_Allocator.Initialize(m_HeapAddr, m_HeapSize);
            m_CompressedArray.SetAllocator(&m_Allocator);
            m_Map.SetAllocator(&m_Allocator);
        }
    }

    /**
     * @brief   テスト終了時に毎回呼び出される関数です。
     */
    virtual void TearDown()
    {
        if (GetParam() == ArrayTestParam_UseWorkBufAllocator)
        {
            m_CompressedArray.ReleaseAllocator();
            m_Map.ReleaseAllocator();
            m_Allocator.Finalize();
            std::free(m_HeapAddr);
        }
    }

protected:
    const size_t m_HeapSize;
    void* m_HeapAddr;                                   // ワークバッファ用メモリ
    nn::ngc::detail::CompressedArray m_CompressedArray;
    nn::ngc::detail::Map m_Map;
    nn::ngc::detail::WorkBufAllocator m_Allocator;      // ワークバッファ用アロケータ
};

}   // namespace

/**
 * @brief   CompressedArray の API 機能を確認します
 */
TEST_P(SuccinctArrayTest, CompressedArraySimple)
{
    const size_t arraySize = 64 * 1024;
    uint32_t pArray[arraySize];

    // pArray の作成
    for (int i = 0; i < arraySize; ++i)
    {
        pArray[i] = i;
        m_CompressedArray.PushBack(i);
    }
    ASSERT_EQ(m_CompressedArray.Size(), arraySize);
    NN_LOG("compressedArray.MemSize() + sizeof(compressedArray) : %zu sizeof(pArray) : %zu\n", m_CompressedArray.MemSize() + sizeof(m_CompressedArray), sizeof(pArray));
    EXPECT_LT(m_CompressedArray.MemSize() + sizeof(m_CompressedArray), sizeof(pArray));

    // Array の値の確認
    auto DoTest = [&](nn::ngc::detail::CompressedArray& ar)
    {
        for (size_t i = 0; i < arraySize; ++i)
        {
            EXPECT_EQ(pArray[i], ar[i]);
        }
    };

    DoTest(m_CompressedArray);

    // ムーブの確認
    nn::ngc::detail::CompressedArray compressedArray(std::move(m_CompressedArray));
    DoTest(compressedArray);
    m_CompressedArray = std::move(compressedArray);
    DoTest(m_CompressedArray);

    // Reset() の確認
    m_CompressedArray.Reset();
    ASSERT_EQ(m_CompressedArray.Size(), 0);
}

/**
 * @brief   Map の API 機能を確認します
 */
TEST_P(SuccinctArrayTest, MapSimple)
{
    const size_t valueSize = 1024;

    ASSERT_TRUE(m_Map.Init(4096));
    for (int i = 0; i < valueSize; ++i)
    {
        m_Map.TurnOn(i * 2, i * 10);
    }
    ASSERT_TRUE(m_Map.Build());

    auto DoTest = [&](nn::ngc::detail::Map& m)
    {
        for (size_t i = 0; i < valueSize * 2; ++i)
        {
            auto findType = m.Find(i);
            if (i % 2 == 0)
            {
                EXPECT_TRUE(findType.first);
                EXPECT_EQ(findType.second, i * 5);
            }
            else
            {
                EXPECT_FALSE(findType.first);
            }
        }
    };

    // ムーブの確認
    nn::ngc::detail::Map map(std::move(m_Map));
    DoTest(map);
    m_Map = std::move(map);
    DoTest(m_Map);

    // Reset() の確認
    m_Map.Reset();
    ASSERT_TRUE(m_Map.Init(4096));
    ASSERT_TRUE(m_Map.Build());
    EXPECT_FALSE(m_Map.Find(0).first);
}

INSTANTIATE_TEST_CASE_P(SwitchAllocator,
                        SuccinctArrayTest,
                        testing::Values(ArrayTestParam_UseMalloc,
                                        ArrayTestParam_UseWorkBufAllocator));
