﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// テスト用のヘッダ

#include <nn/os.h>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/nn_ErrorResult.h>
#include <nn/util/util_StringUtil.h>

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>

// テスト対象のヘッダ
#include <../../../../../Programs/Eris/Sources/Libraries/ngc/ngc_Private.h>

#if defined( NN_BUILD_CONFIG_OS_WIN32 )
#include <nn/nn_Windows.h>
#define GET_CHAR16LITERAL( string ) \
            reinterpret_cast< const char16_t* >( NN_CHAR16LITERAL( string ) )
#else
#define GET_CHAR16LITERAL( string ) NN_CHAR16LITERAL( string )
#endif

namespace {
    int wcslen( const char16_t* pString ) NN_NOEXCEPT
    {
        // TODO:g3d_ResUtility.cppから引用
        int len = 0;
        const char16_t* pCheckStr = pString;
        while ( NN_STATIC_CONDITION( 1 ) )
        {
            if ( *pCheckStr == 0x0 )
            {
                break;
            }
            ++pCheckStr;
            ++len;
        }
        return len;
    }

    int wcscmp( const char16_t* pString1, const char16_t* pString2 ) NN_NOEXCEPT
    {
        if(wcslen(pString1) != wcslen(pString2))
        {
            return -1;
        }
        return nn::util::Strncmp( pString1, pString2, wcslen(pString1) );
    }
}

class ProfanityFilterBaseTest : public ::testing::Test
{
protected:
    ProfanityFilterBaseTest() NN_NOEXCEPT {}

    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE {}

    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE {}
};

// protected関数へバイパスを行うためのクラスです。
class ProfanityFilterDummy : public nn::ngc::ProfanityFilterBase
{
public:
    static void ConvertUserInputForWord( char16_t* pConvertedWord,
                                         int nLength,
                                         const char16_t* pWord ) NN_NOEXCEPT
    {
        nn::ngc::ProfanityFilterBase::ConvertUserInputForWord( pConvertedWord, nLength, pWord );
    }
    static void ConvertUserInputForText( char* pConvertedWord,
                                         int8_t* pSourceLengthMapping,
                                         size_t nLength,
                                         const char* pWord ) NN_NOEXCEPT
    {
        nn::ngc::ProfanityFilterBase::ConvertUserInputForText( pConvertedWord,
                                                               pSourceLengthMapping,
                                                               nLength,
                                                               pWord );
    }
};

/*!--------------------------------------------------------------------------*
  Name:         ConvertUserInputForWordTest1

  @brief        ユーザーが入力した単語が正しく変換出来るかどうかを確認します。
 *---------------------------------------------------------------------------*/
TEST_F(ProfanityFilterBaseTest, ConvertUserInputForWordTest1)
{
    const size_t BufferSize = 64;
    char16_t szConverted[ BufferSize ];
    ProfanityFilterDummy::ConvertUserInputForWord( szConverted, BufferSize, GET_CHAR16LITERAL("ａｂｃ") );
    ASSERT_TRUE( wcscmp( szConverted, GET_CHAR16LITERAL("abc") ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         ConvertUserInputForWordTest2

  @brief        ユーザーが入力した単語が正しく変換出来るかどうかを確認します。
 *---------------------------------------------------------------------------*/
TEST_F(ProfanityFilterBaseTest, ConvertUserInputForWordTest2)
{
    const size_t BufferSize = 64;
    char16_t szConverted[ BufferSize ];
    ProfanityFilterDummy::ConvertUserInputForWord( szConverted, BufferSize, GET_CHAR16LITERAL(" あ い う ") );
    ASSERT_TRUE( wcscmp( szConverted, GET_CHAR16LITERAL("アイウ") ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         ConvertUserInputForWordTest3

  @brief        ユーザーが入力した単語が正しく変換出来るかどうかを確認します。
 *---------------------------------------------------------------------------*/
TEST_F(ProfanityFilterBaseTest, ConvertUserInputForWordTest3)
{
    const size_t BufferSize = 64;
    char16_t szConverted[ BufferSize ];
    ProfanityFilterDummy::ConvertUserInputForWord( szConverted, BufferSize, GET_CHAR16LITERAL("ﾊﾞﾋﾞﾌﾞﾍﾞﾎﾞ") );
    ASSERT_TRUE( wcscmp( szConverted, GET_CHAR16LITERAL("バビブベボ") ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         ConvertUserInputForWordBufferTest

  @brief        バッファーが溢れないように足りない文字は切り捨てられるか確認します。
 *---------------------------------------------------------------------------*/
TEST_F(ProfanityFilterBaseTest, ConvertUserInputForWordBufferTest)
{
    char16_t szConverted[ 16 ];
    ProfanityFilterDummy::ConvertUserInputForWord( szConverted, 4, GET_CHAR16LITERAL("ＡＢＣＤＥＦＧ") );
    ASSERT_TRUE( wcscmp( szConverted, GET_CHAR16LITERAL("abc") ) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         ConvertUserInputForWordAlphabetTest

  @brief        英語アルファベット以外のアルファベットの大文字／小文字変換テストです。
 *---------------------------------------------------------------------------*/
TEST_F(ProfanityFilterBaseTest, ConvertUserInputForWordAlphabetTest)
{
    char16_t szConverted[ 64 ];

    char16_t pWord[] = { 0x00c1, 0x00c9, 0x00cd, 0x00d3, 0x00da,                // ÁÉÍÓÚ
                         0x00c0, 0x00c8, 0x00cc, 0x00d2, 0x00d9,                // ÀÈÌÒÙ
                         0x00c2, 0x00ca, 0x00ce, 0x00d4, 0x00db,                // ÂÊÎÔÛ
                         0x00c3, 0x00d5,                                        // ÃÕ
                         0x00c4, 0x00cb, 0x00cf, 0x00d6, 0x00dc, 0x0178,        // ÄËÏÖÜŸ
                         0x00c7, 0x0152, 0x00c6, 0x00d1, 0x1e9e, 0x0132,        // ÇŒÆÑẞIJ
                         0x0410, 0x0411, 0x0412, 0x0413, 0x0414, 0x0415,        // АБВГДЕ
                         0x0401, 0x0416, 0x0417, 0x0418, 0x0419, 0x041a,        // ЁЖЗИЙК
                         0x041b, 0x041c, 0x041d, 0x041e, 0x041f, 0x0420,        // ЛМНОПР
                         0x0421, 0x0422, 0x0423, 0x0424, 0x0425, 0x0426,        // СТУФХЦ
                         0x0427, 0x0428, 0x0429, 0x042a, 0x042b, 0x042c,        // ЧШЩЪЫЬ
                         0x042d, 0x042e, 0x042f, 0                              // ЭЮЯ
                       };
    char16_t pExpected[] = { 0x00e1, 0x00e9, 0x00ed, 0x00f3, 0x00fa,            // áéíóú
                             0x00e0, 0x00e8, 0x00ec, 0x00f2, 0x00f9,            // àèìòù
                             0x00e2, 0x00ea, 0x00ee, 0x00f4, 0x00fb,            // âêîôû
                             0x00e3, 0x00f5,                                    // ãõ
                             0x00e4, 0x00eb, 0x00ef, 0x00f6, 0x00fc, 0x00ff,    // äëïöüÿ
                             0x00e7, 0x0153, 0x00e6, 0x00f1, 0x00df, 0x0133,    // çœæñßij
                             0x0430, 0x0431, 0x0432, 0x0433, 0x0434, 0x0435,    // абвгде
                             0x0451, 0x0436, 0x0437, 0x0438, 0x0439, 0x043a,    // ёжзийк
                             0x043b, 0x043c, 0x043d, 0x043e, 0x043f, 0x0440,    // лмнопр
                             0x0441, 0x0442, 0x0443, 0x0444, 0x0445, 0x0446,    // стуфхц
                             0x0447, 0x0448, 0x0449, 0x044a, 0x044b, 0x044c,    // чшщъыь
                             0x044d, 0x044e, 0x044f, 0                          // эюя
                           };
    ProfanityFilterDummy::ConvertUserInputForWord( szConverted, 64, pWord);
    ASSERT_TRUE( wcscmp( szConverted, pExpected) == 0 );
}

/*!--------------------------------------------------------------------------*
  Name:         ConvertUserInputForTextTest1

  @brief        ユーザーが入力した長文が正しく変換出来るかどうかを確認します。
 *---------------------------------------------------------------------------*/
TEST_F(ProfanityFilterBaseTest, ConvertUserInputForTextTest1)
{
    const size_t BufferSize = 64;
    char szConverted[ BufferSize ];
    int8_t mapping[ BufferSize ];

    ProfanityFilterDummy::ConvertUserInputForText( szConverted, mapping, BufferSize, "\xef\xbd\x81\xef\xbd\x82\xef\xbd\x83" );   // ａｂｃ
    ASSERT_TRUE( strcmp( szConverted, "abc" ) == 0 );
    ASSERT_TRUE( mapping[ 0 ] == 3 );
    ASSERT_TRUE( mapping[ 1 ] == 3 );
    ASSERT_TRUE( mapping[ 2 ] == 3 );
}

/*!--------------------------------------------------------------------------*
  Name:         ConvertUserInputForTextTest2

  @brief        ユーザーが入力した長文が正しく変換出来るかどうかを確認します。
 *---------------------------------------------------------------------------*/
TEST_F(ProfanityFilterBaseTest, ConvertUserInputForTextTest2)
{
    const size_t BufferSize = 64;
    char szConverted[ BufferSize ];
    int8_t mapping[ BufferSize ];

    char pWord[]     = "\xE3\x82\xA1\x20\xE3\x82\xA5\x20\xE3\x83\xB5";  // "ァ ゥ ヵ"
    char pExpected[] = "\xE3\x82\xA2\x20\xE3\x82\xA6\x20\xE3\x82\xAB";  // ア ウ カ (半角スペースは削除されない)

    ProfanityFilterDummy::ConvertUserInputForText( szConverted, mapping, BufferSize, pWord);
    ASSERT_TRUE( strcmp( szConverted, pExpected ) == 0 );
    for (size_t i = 0; i < strlen(pExpected); ++i)
    {
        EXPECT_EQ(mapping[i], 1);
    }
}

/*!--------------------------------------------------------------------------*
  Name:         ConvertUserInputForTextTest3

  @brief        ユーザーが入力した長文が正しく変換出来るかどうかを確認します。
 *---------------------------------------------------------------------------*/
TEST_F(ProfanityFilterBaseTest, ConvertUserInputForTextTest3)
{
    const size_t BufferSize = 64;
    char szConverted[ BufferSize ];
    int8_t mapping[ BufferSize ];

    // ﾊﾟﾋﾟﾌﾟﾍﾟﾎﾟ
    char pWord[] = "\xEF\xBE\x8A\xEF\xBE\x9F\xEF\xBE\x8B\xEF\xBE\x9F\xEF\xBE\x8C\xEF\xBE\x9F\xEF\xBE\x8D\xEF\xBE\x9F\xEF\xBE\x8E\xEF\xBE\x9F";
    // パピプペポ
    char pExpected[] = "\xE3\x83\x91\xE3\x83\x94\xE3\x83\x97\xE3\x83\x9A\xE3\x83\x9D";

    ProfanityFilterDummy::ConvertUserInputForText( szConverted, mapping, BufferSize, pWord );
    ASSERT_TRUE( strcmp( szConverted, pExpected ) == 0 );
    for (size_t i = 0; i < strlen(pExpected); ++i)
    {
        EXPECT_EQ(mapping[i], (i % 3 == 0) ? -2 : 2);
    }
}

/*!--------------------------------------------------------------------------*
  Name:         ConvertUserInputForTextBufferTest

  @brief        指定されたバッファサイズで打ち切られるかどうかを確認するテストです。
 *---------------------------------------------------------------------------*/
TEST_F(ProfanityFilterBaseTest, ConvertUserInputForTextBufferTest)
{
    const size_t BufferSize = 16;
    char szConverted[ BufferSize ];
    int8_t mapping[ BufferSize ];

    // ﾊﾞﾋﾞﾌﾞﾍﾞﾎﾞ
    char pWord[] = "\xEF\xBE\x8A\xEF\xBE\x9E\xEF\xBE\x8B\xEF\xBE\x9E\xEF\xBE\x8C\xEF\xBE\x9E\xEF\xBE\x8D\xEF\xBE\x9E\xEF\xBE\x8E\xEF\xBE\x9E";
    char pExpected[] = "\xE3\x83\x90\xE3\x83\x93\xE3\x83\x96";  // バビブ
    const int stopLength = 9;

    ProfanityFilterDummy::ConvertUserInputForText( szConverted, mapping, stopLength + 1, pWord );
    ASSERT_TRUE( strcmp( szConverted, pExpected ) == 0 );
    for (int i = 0; i < stopLength; ++i)
    {
        EXPECT_EQ(mapping[i], (i % 3 == 0) ? -2 : 2);
    }
}

/*!--------------------------------------------------------------------------*
  Name:         ConvertUserInputForTextAlphabetTest

  @brief        英語アルファベット以外のアルファベットの大文字／小文字変換テストです。
 *---------------------------------------------------------------------------*/
TEST_F(ProfanityFilterBaseTest, ConvertUserInputForTextAlphabetTest)
{
    const size_t BufferSize = 256;
    char szConverted[ BufferSize ];
    int8_t mapping[ BufferSize ];

    char pWord[] = "\xc3\x81\xc3\x89\xc3\x8d\xc3\x93\xc3\x9a"               // ÁÉÍÓÚ
                   "\xc3\x80\xc3\x88\xc3\x8c\xc3\x92\xc3\x99"               // ÀÈÌÒÙ
                   "\xc3\x82\xc3\x8a\xc3\x8e\xc3\x94\xc3\x9b"               // ÂÊÎÔÛ
                   "\xc3\x83\xc3\x95"                                       // ÃÕ
                   "\xc3\x84\xc3\x8b\xc3\x8f\xc3\x96\xc3\x9c\xc5\xb8"       // ÄËÏÖÜŸ
                   "\xc3\x87\xc5\x92\xc3\x86\xc3\x91\xe1\xba\x9e\xc4\xb2"   // ÇŒÆÑẞIJ
                   "\xd0\x90\xd0\x91\xd0\x92\xd0\x93\xd0\x94\xd0\x95"       // АБВГДЕ
                   "\xd0\x81\xd0\x96\xd0\x97\xd0\x98\xd0\x99\xd0\x9a"       // ЁЖЗИЙК
                   "\xd0\x9b\xd0\x9c\xd0\x9d\xd0\x9e\xd0\x9f\xd0\xa0"       // ЛМНОПР
                   "\xd0\xa1\xd0\xa2\xd0\xa3\xd0\xa4\xd0\xa5\xd0\xa6"       // СТУФХЦ
                   "\xd0\xa7\xd0\xa8\xd0\xa9\xd0\xaa\xd0\xab\xd0\xac"       // ЧШЩЪЫЬ
                   "\xd0\xad\xd0\xae\xd0\xaf";                              // ЭЮЯ
    char pExpected[] = "\xc3\xa1\xc3\xa9\xc3\xad\xc3\xb3\xc3\xba"           // áéíóú
                       "\xc3\xa0\xc3\xa8\xc3\xac\xc3\xb2\xc3\xb9"           // àèìòù
                       "\xc3\xa2\xc3\xaa\xc3\xae\xc3\xb4\xc3\xbb"           // âêîôû
                       "\xc3\xa3\xc3\xb5"                                   // ãõ
                       "\xc3\xa4\xc3\xab\xc3\xaf\xc3\xb6\xc3\xbc\xc3\xbf"   // äëïöüÿ
                       "\xc3\xa7\xc5\x93\xc3\xa6\xc3\xb1\xc3\x9f\xc4\xb3"   // çœæñßij
                       "\xd0\xb0\xd0\xb1\xd0\xb2\xd0\xb3\xd0\xb4\xd0\xb5"   // абвгде
                       "\xd1\x91\xd0\xb6\xd0\xb7\xd0\xb8\xd0\xb9\xd0\xba"   // ёжзийк
                       "\xd0\xbb\xd0\xbc\xd0\xbd\xd0\xbe\xd0\xbf\xd1\x80"   // лмнопр
                       "\xd1\x81\xd1\x82\xd1\x83\xd1\x84\xd1\x85\xd1\x86"   // стуфхц
                       "\xd1\x87\xd1\x88\xd1\x89\xd1\x8a\xd1\x8b\xd1\x8c"   // чшщъыь
                       "\xd1\x8d\xd1\x8e\xd1\x8f";                          // эюя

    ProfanityFilterDummy::ConvertUserInputForText( szConverted, mapping, BufferSize, pWord );
    ASSERT_TRUE( strcmp( szConverted, pExpected ) == 0 );
}
