﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 *  プログラムの全体像は以下の通りです。
 *
 *  - NFP ライブラリの初期化
 *  - NFC デバイスリストの取得
 *  - タグ発見、喪失イベントの設定
 *  - タグ検知の開始
 *  - タグ発見イベント待ち
 *  - タグ情報の取得
 *  - タグのマウント
 *  - タグの ROM 領域情報の取得
 *  - タグの共用領域情報の取得
 *  - タグのオーナー登録情報の取得
 *  - タグのアプリケーション専用領域の取得、設定
 *  - タグのアンマウント
 *  - タグ喪失イベント待ち
 *  - タグ検知の終了
 *  - NFP ライブラリの終了
 *
 *  本プログラムは、NFC 機能を有効にした状態で動作させてください。
 *  また、本プログラムでは下記いずれかのタグを使用してください。
 *  - フォーマットされた NFP タグ
 *  - アクセス ID 0x0FF41E00 でアクセス可能なアプリケーション専用領域を持つ NFP タグ
 *
 */


#include <nn/os.h>
#include <nn/init.h>

#include "testNfp_Writer.h"

//-----------------------------------------------------------------------------

namespace {

// Graphics.cpp で malloc しているヒープの内訳
//  VisiblePoolMemory       16 MB
//  InvisiblePoolMemory     20 MB
//  for CommandBuffer       32 MB
//  nv::InitializeGraphics   8 MB
//  for DebugFont            2 MB
const size_t MemoryHeapSize         = 128 * 1024 * 1024;
const size_t MallocHeapSize         =  96 * 1024 * 1024;

const nn::Bit32 AccessId = 0x0FF41E00; // このアプリケーションのアクセス ID
const size_t ApplicationAreaSize = 216;// このアプリケーションのアプリケーション専用領域のサイズ
const int InitialValue = 0x55;         // このアプリケーションはアプリケーション専用領域を作成するときに領域をこの値で埋めます

nn::nfp::TagInfo g_TagInfo;
nn::nfp::ModelInfo g_ModelInfo;
nn::nfp::CommonInfo g_CommonInfo;
bool g_HasRegisterInfo;
nn::nfp::RegisterInfo g_RegisterInfo;
size_t g_ApplicationAreaSize;
nn::Bit8 g_ApplicationArea[ApplicationAreaSize];

}   // namespace

//-----------------------------------------------------------------------------
// アプリケーションのメモリ管理機構を初期化
extern "C" void nninitStartup()
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::os::SetMemoryHeapSize(MemoryHeapSize));

    uintptr_t address = uintptr_t();
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::os::AllocateMemoryBlock(&address, MallocHeapSize));

    nn::init::InitializeAllocator(
        reinterpret_cast<void*>(address), MallocHeapSize);
}

//
//  メイン関数です。
//
extern "C" void nnMain()
{
    InitializeWriter();

    nn::Result result;
    nn::nfp::DeviceHandle deviceHandle;
    int deviceCount;
    nn::os::SystemEventType activateEvent;
    nn::os::SystemEventType deactivateEvent;

    // NFP ライブラリの初期化
    WriteProcess("Initialize ... ");
    nn::nfp::Initialize();
    WriteProcess(nn::ResultSuccess());

    // NFC デバイスリストの取得
    WriteProcess("ListDevices ... ");
    while(NN_STATIC_CONDITION(true))
    {
        result = nn::nfp::ListDevices(&deviceHandle, &deviceCount, 1);
        if(result.IsSuccess())
        {
            break;
        }
        else if(result <= nn::nfp::ResultNfcDeviceNotFound())
        {
            WriteProcess();
            continue;
        }
        else
        {
            WriteProcess(result);
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        }
    }
    WriteProcess(result);

    // タグ発見、喪失イベントの設定
    WriteProcess("AttachActivateEvent ... ");
    result = nn::nfp::AttachActivateEvent(&activateEvent, deviceHandle);
    WriteProcess(result);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    WriteProcess("AttachDeactivateEvent ... ");
    result = nn::nfp::AttachDeactivateEvent(&deactivateEvent, deviceHandle);
    WriteProcess(result);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // タグ検知の開始
    WriteProcess("StartDetection ... ");
    result = nn::nfp::StartDetection(deviceHandle);
    WriteProcess(result);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // タグ発見イベント待ち
    WriteProcess("Please touch a NFP tag ... ");
    while(!nn::os::TryWaitSystemEvent(&activateEvent))
    {
        nn::nfp::DeviceState deviceState = nn::nfp::GetDeviceState(deviceHandle);
        if(deviceState == nn::nfp::DeviceState_Search || deviceState == nn::nfp::DeviceState_Active)
        {
            WriteProcess();
        }
        else
        {
            WriteProcess(deviceState);
            NN_ABORT_UNLESS(deviceState == nn::nfp::DeviceState_Search || deviceState == nn::nfp::DeviceState_Active);
        }
    }
    WriteProcess(nn::ResultSuccess());

    // タグ情報の取得
    WriteProcess("GetTagInfo ... ");
    result = nn::nfp::GetTagInfo(&g_TagInfo, deviceHandle);
    WriteProcess(result);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // タグのマウント
    WriteProcess("Mount ... ");
    result = nn::nfp::Mount(deviceHandle, nn::nfp::ModelType_Amiibo);
    WriteProcess(result);
    if( result <= nn::nfp::ResultNeedRestore() )
    {
        WriteProcess("Restore ... ");
        result = nn::nfp::Restore(deviceHandle);
        WriteProcess(result);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        WriteProcess("Mount ... ");
        result = nn::nfp::Mount(deviceHandle, nn::nfp::ModelType_Amiibo);
        WriteProcess(result);
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // タグの ROM 領域情報の取得
    WriteProcess("GetModelInfo ... ");
    result = nn::nfp::GetModelInfo(&g_ModelInfo, deviceHandle);
    WriteProcess(result);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // タグの共用領域情報の取得
    WriteProcess("GetCommonInfo ... ");
    result = nn::nfp::GetCommonInfo(&g_CommonInfo, deviceHandle);
    WriteProcess(result);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // タグのオーナー登録情報の取得
    WriteProcess("GetRegisterInfo ... ");
    result = nn::nfp::GetRegisterInfo(&g_RegisterInfo, deviceHandle);
    WriteProcess(result);
    if(result.IsSuccess())
    {
        g_HasRegisterInfo = true;
    }
    else if( result <= nn::nfp::ResultNeedRegister() )
    {
        g_HasRegisterInfo = false;
    }
    else
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }

    // タグのアプリケーション専用領域の取得、設定
    WriteProcess("OpenApplicationArea ... ");
    result = nn::nfp::OpenApplicationArea(deviceHandle, AccessId);
    WriteProcess(result);
    if( result <= nn::nfp::ResultNeedCreate() )
    {
        nn::nfp::ApplicationAreaCreateInfo createInfo;
        std::memset(g_ApplicationArea, InitialValue, sizeof(g_ApplicationArea));
        createInfo.pInitialData = g_ApplicationArea;
        createInfo.initialDataSize = sizeof(g_ApplicationArea);
        createInfo.accessId = AccessId;

        WriteProcess("CreateApplicationArea ... ");
        result = nn::nfp::CreateApplicationArea(deviceHandle, createInfo);
        WriteProcess(result);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        WriteProcess("OpenApplicationArea ... ");
        result = nn::nfp::OpenApplicationArea(deviceHandle, AccessId);
        WriteProcess(result);
    }

    if( result.IsSuccess() )
    {
        WriteProcess("GetApplicationArea ... ");
        result = nn::nfp::GetApplicationArea(g_ApplicationArea, &g_ApplicationAreaSize, deviceHandle, sizeof(g_ApplicationArea));
        WriteProcess(result);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        // このアプリケーションはアプリケーション専用領域の奇数バイト目のみデータを更新(元の値を + 1 )する
        for(size_t i = 0; i < g_ApplicationAreaSize; i++)
        {
            if(i % 2 == 0)
            {
                g_ApplicationArea[i] += 1;
            }
        }

        WriteProcess("SetApplicationArea ... ");
        result = nn::nfp::SetApplicationArea(deviceHandle, g_ApplicationArea, sizeof(g_ApplicationArea));
        WriteProcess(result);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        nn::nfp::ApplicationAreaCreateInfo createInfo;
        // アプリケーション専用領域の奇数バイト目のみデータをさらに更新(元の値を + 1 )する
        for(size_t i = 0; i < g_ApplicationAreaSize; i++)
        {
            if(i % 2 == 0)
            {
                g_ApplicationArea[i] += 1;
            }
        }
        createInfo.pInitialData = g_ApplicationArea;
        createInfo.initialDataSize = sizeof(g_ApplicationArea);
        createInfo.accessId = AccessId + 1;
        WriteProcess("RecreateApplicationArea ... ");
        result = nn::nfp::RecreateApplicationArea(deviceHandle, createInfo);
        WriteProcess(result);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        WriteProcess("OpenApplicationArea ... ");
        result = nn::nfp::OpenApplicationArea(deviceHandle, AccessId + 1);
        WriteProcess(result);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        createInfo.pInitialData = g_ApplicationArea;
        createInfo.initialDataSize = sizeof(g_ApplicationArea);
        createInfo.accessId = AccessId;
        WriteProcess("RecreateApplicationArea ... ");
        result = nn::nfp::RecreateApplicationArea(deviceHandle, createInfo);
        WriteProcess(result);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        WriteProcess("Flush ... ");
        result = nn::nfp::Flush(deviceHandle);
        WriteProcess(result);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }
    else
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }

    // タグのアンマウント
    WriteProcess("Unmount ... ");
    result = nn::nfp::Unmount(deviceHandle);
    WriteProcess(result);

    // タグ喪失イベント待ち
    WriteProcess("Please remove the tag ... ");
    while(!nn::os::TryWaitSystemEvent(&deactivateEvent))
    {
        nn::nfp::DeviceState deviceState = nn::nfp::GetDeviceState(deviceHandle);
        if(deviceState != nn::nfp::DeviceState_Active)
        {
            break;
        }
        WriteProcess();
    }
    WriteProcess(nn::ResultSuccess());

    // タグ検知の終了
    WriteProcess("StopDetection ... ");
    result = nn::nfp::StopDetection(deviceHandle);
    WriteProcess(result);

    // NFP ライブラリの終了
    WriteProcess("Finalize ... ");
    nn::nfp::Finalize();
    WriteProcess(nn::ResultSuccess());

    while( NN_STATIC_CONDITION(true) )
    {
        WriteProcess(g_TagInfo,
                     g_ModelInfo,
                     g_CommonInfo,
                     g_HasRegisterInfo, g_RegisterInfo,
                     g_ApplicationArea, g_ApplicationAreaSize);
    }
} // NOLINT(impl/function_size)
