﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nfp.h>
#include <nn/oe.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/os/os_SdkThread.h>
#include <nn/os/os_Event.h>

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>

#include <nnt/nfp/testNfp_Common.h>

//================================================================================
// このテストで使用する定義です。
//================================================================================

namespace
{
    // バックアップデータを読み込むためのバッファです。
    nn::Bit8  g_BackupBuffer[4 * 1024 * 1024];
    nn::Bit8* g_pBackupData;
    const size_t BackupBufferSize = sizeof(g_BackupBuffer) - nnt::nfp::IpcAlignment;

    enum WaitTime
    {
        WaitTime_Deactivate             = 5000,
        WaitTime_CountDown              = 1000,
        WaitTime_BluetoothSignalCut     = 10000,
        WaitTime_BluetoothSignalRestore = 10000,
        WaitTime_WifiOn                 = 500,
        WaitTime_SleepStart             = 10000,
        WaitTime_SleepRestore           = 10000,
        WaitTime_RemoveTag              = 10000,
        WaitTime_AttachTag              = 10000,
        WaitTime_NextStep               = 1000,
        WaitTime_InProgressSleepTime    = 1000,
        WaitTime_MountInProgressSleepTime = 10,
    };

    const size_t StackSize = 4096;
    nn::os::ThreadType g_RunApiThread;
    NN_ALIGNAS( 4096 ) char g_Stack[ StackSize ];
    nn::os::EventType g_BootUpEvent;
    nn::os::EventType g_RunApiEvent;
    nn::Result g_RunApiThreadResult;

    // TestCaseStateTransWithEventSignalWhenFlushテスト状態管理変数
    bool g_MountSuccesFlg       = false;
    bool g_MountExpectFailFlg   = false;

    // 要求API設定
    enum RunNfpApi
    {
        RunNfpApi_Mount = 0,
        RunNfpApi_Format,
        RunNfpApi_CreateApplicationArea,
        RunNfpApi_DeleteApplicationArea,
        RunNfpApi_DeleteRegisterInfo,
        RunNfpApi_Flush,
        RunNfpApi_Restore,
    };
    // 初期値はFlushの試験に設定する
    RunNfpApi g_RunNfpApi = RunNfpApi_Flush;

    // マルチデバイス関連定義
    const int DeviceCountMax = 4;
    const int MaxCore = 3;
    struct TagData
    {
        nn::Bit8 *NormalTagData;
        unsigned char *NormalTagNickName;
    };
    struct threadinfo
    {
        int deviceNum;
    };
    const nn::Bit8 *g_NormalTagData[DeviceCountMax] = { nnt::nfp::NormalTagData,
                                                        nnt::nfp::NormalTagData1,
                                                        nnt::nfp::NormalTagData2,
                                                        nnt::nfp::NormalTagData3 };

    const unsigned char *g_NormalTagNickName[DeviceCountMax] = { nnt::nfp::NormalTagNickName,
                                                               nnt::nfp::NormalTagNickName2,
                                                               nnt::nfp::NormalTagNickName3,
                                                               nnt::nfp::NormalTagNickName4 };
    threadinfo g_threadParam[DeviceCountMax] = {{0},{1},{2},{3}};
    void DeviceTestThread(void* pArg) NN_NOEXCEPT;
    const size_t DeviceThreadStackSize = 32 * 1024;
    nn::os::ThreadType g_DeviceThread[DeviceCountMax];
    NN_OS_ALIGNAS_THREAD_STACK char g_DeviceThreadStack[DeviceCountMax][DeviceThreadStackSize];
    nn::nfp::DeviceHandle g_CurrentDeviceHandleMulti[DeviceCountMax];
    const int WaitTimeout = 5000;
    const int DeactivateWaitTimeout = 15000;
    const int TestLoop = 10;
    const nn::Bit8 NormalTagFontRegion = nn::nfp::FontRegion_Taiwan;
    const int ExceptDataSize = 20;
    int g_ErrorCount = 0;
    int g_SuccessCount = 0;
    const int ErrorCount = 3;
    const int SuccessCount =1;

    // マルチデバイス複数アクション用の定義
    struct eventthreadinfo
    {
        bool wifiOff;
    };
    eventthreadinfo g_EventThreadinfo = {};
    nn::os::EventType g_MultiDevicePrepareEvent[DeviceCountMax];
    nn::os::EventType g_MultiDeviceStartEvent;
    const size_t MultiDeviceEventPrepareThreadStackSize = 32 * 1024;
    nn::os::ThreadType g_MultiDevicePrepareEventThread;
    NN_OS_ALIGNAS_THREAD_STACK char g_MultiDevicePrepareEventThreadStack[MultiDeviceEventPrepareThreadStackSize];
    int g_SuccessDeviceCount = 0;
    const int WaitMilliSeconds = 1000;
    bool g_NfcState = true;
    nn::os::Mutex g_Mutex(false);

    // システムイベント確認スレッド用定義
    const size_t SystemEventThreadStackSize = 32 * 1024;
    nn::os::ThreadType g_SystemEventThread;
    NN_OS_ALIGNAS_THREAD_STACK char g_SystemEventThreadStack[SystemEventThreadStackSize];
    nn::os::SystemEventType g_MultiActivateEvent[DeviceCountMax];
    nn::os::SystemEventType g_MultiDeactivateEvent[DeviceCountMax];
    bool g_IsRunning = true;

    inline void CheckDeactiveAndDisconnect(bool isDeactive, bool isDisconnect)
    {
#if !defined(NNT_NFP_PLATFORM_NX) // NXではConnectは実装されない
        if(nn::os::IsRunOnSnake())
        {
            // SNAKE検証時
            // DisconnectEventのチェックは行わない
            EXPECT_TRUE(nnt::nfp::WaitForDeactivate() == isDeactive);
        }
        else
        {
            // CTR検証時
            EXPECT_TRUE(nnt::nfp::WaitForDisconnect() == isDisconnect);
            EXPECT_TRUE(nnt::nfp::WaitForDeactivate() == isDeactive);
        }
#else
        NN_UNUSED(isDisconnect);
        EXPECT_EQ(nnt::nfp::WaitForDeactivate(),isDeactive);
#endif // !defined(NNT_NFP_PLATFORM_NX)
    }

    void UnmountNfp()
    {
        nnt::nfp::FinalizeSystem();
        NN_LOG("==================================================\n");
        NN_LOG(" PLEASE ATTACH a TAG!!\n");
        NN_LOG("==================================================\n");
        nnt::nfp::DoActivate();
        NN_LOG(" TAG Attached !!\n");
        NN_LOG("==================================================\n");
        NN_LOG(" PLEASE REMOVE a TAG!!\n");
        NN_LOG("==================================================\n");
        nnt::nfp::WaitForDeactivate();
        NN_LOG(" TAG Removed!!\n");
    }

#if !defined(NNT_NFP_PLATFORM_NX) // NXではFWアップデートは実装されない
    void UpdateFangateFirmware()
    {
        nn::os::Event updateEvt;

        // 再度ファーム更新が開始できること
        WifiOn();
        NNT_NFP_EXPECT_RESULT_SUCCESS( nnt::nfp::InitializeSystemWithRetry() );
        nnt::nfp::DoConnectForFwUpdate();
        NNT_NFP_EXPECT_RESULT_SUCCESS( nnt::nfp::wrapper::UpdateTargetFirmware(&updateEvt) );

        NN_LOG("### Target Firmware Update Start!! \n");

        uint32_t prevSentSize = 0;
        for(int retryCnt=0; retryCnt<10; retryCnt++)
        {
            uint32_t sentSize=0, totalSize=0;

            NNT_NFP_EXPECT_RESULT_SUCCESS( nnt::nfp::wrapper::GetUpdateProgress(&sentSize, &totalSize) );
            NN_LOG("### Target Firmware Updating ... [ %d / %d ]  \n", sentSize, totalSize);

            if( updateEvt.Wait(nn::fnd::TimeSpan::FromMilliSeconds(1000)) )
            {
                NN_LOG("### Target Firmware Update Success!! \n");
                NNT_NFP_EXPECT_RESULT_SUCCESS( nnt::nfp::FinalizeSystem() );
                // FW 更新直後と電源投入直後は１秒間接続することができない
                nnt::nfp::Sleep(1000);
                return;
            }
            else
            {
                if(sentSize != prevSentSize)
                {
                    retryCnt = 0;
                }

                if(totalSize != 0 && sentSize <= totalSize)
                {
                    prevSentSize = sentSize;
                }
            }
        }
        NN_LOG("### Target Firmware Update Failure!! \n");
        EXPECT_TRUE(false);
    }
#endif // !defined(NNT_NFP_PLATFORM_NX)

    void ThreadRunNfpApi(void* pPtr)
    {
        NN_UNUSED(pPtr);

        g_RunApiThreadResult = nn::nfp::ResultNfcDeviceError();

        NN_LOG("### %s Activated !!\n", __FUNCTION__);

        if(nn::os::TryWaitEvent(&g_BootUpEvent) == true)
        {
            NN_LOG("### Illeagal Event State !!\n");
            nn::os::SignalEvent(&g_BootUpEvent);
            nn::os::ClearEvent(&g_BootUpEvent);
            EXPECT_TRUE(false);
            return;
        }

        NN_LOG("### Indicate Thread Boot!!\n");
        nn::os::SignalEvent(&g_BootUpEvent);

        NN_LOG("### Waiting for RunNfpApi Signal...\n");
        nn::os::WaitEvent(&g_RunApiEvent);

        switch (g_RunNfpApi)
        {
        case RunNfpApi_Mount:
            {
                NN_LOG("### Run Mount\n");
                g_RunApiThreadResult = nnt::nfp::MountRomWithRetry();
            }
            break;
        case RunNfpApi_Format:
            {
                NN_LOG("### Run Format\n");
                g_RunApiThreadResult = nnt::nfp::FormatWithRetry();
            }
            break;
        case RunNfpApi_CreateApplicationArea:
            {
                nn::nfp::ApplicationAreaCreateInfo createInfo;
                nn::Bit8 tagData[nn::nfp::ApplicationAreaSizeV2];
                std::memcpy(tagData,nnt::nfp::NormalTagData,sizeof(nnt::nfp::NormalTagData));
                createInfo.accessId        = nnt::nfp::NormalTagId;
                createInfo.pInitialData    = tagData;
                createInfo.initialDataSize = sizeof(nnt::nfp::NormalTagData);
                NN_LOG("### Run CreateApplicationArea\n");
                g_RunApiThreadResult = nnt::nfp::CreateApplicationAreaWithRetry(createInfo);
            }
            break;
        case RunNfpApi_DeleteApplicationArea:
            {
                NN_LOG("### Run DeleteApplicationArea\n");
                g_RunApiThreadResult = nnt::nfp::DeleteApplicationAreaWithRetry();
            }
            break;
        case RunNfpApi_DeleteRegisterInfo:
            {
                NN_LOG("### Run DeleteRegisterInfo\n");
                g_RunApiThreadResult = nnt::nfp::DeleteNfpRegisterInfoWithRetry();
            }
            break;
        case RunNfpApi_Flush:
            {
                NN_LOG("### Run Flush\n");
                g_RunApiThreadResult = nnt::nfp::FlushWithRetry();
            }
            break;
        case RunNfpApi_Restore:
            {
                NN_LOG("### Run Restore\n");
                g_RunApiThreadResult = nnt::nfp::RestoreWithRetry();
            }
            break;
        default:
            {
                NN_LOG("### Nothing\n");
            }
            break;
        }
        NN_LOG("### Indicate End RunApi Signal RunApi: %d, Result: %s !!\n", g_RunNfpApi, nnt::nfp::GetNfpResultTypeString(g_RunApiThreadResult));
        nn::os::ClearEvent(&g_RunApiEvent);

        if(nn::os::TryWaitEvent(&g_BootUpEvent) == true)
        {
            nn::os::SignalEvent(&g_BootUpEvent);
        }

        nn::os::ClearEvent(&g_BootUpEvent);
        return;
    }

#if !defined(NNT_NFP_PLATFORM_NX) // NXではConnectは実装されない
    void TestInitTransInitConnect()
    {
        NNT_NFP_EXPECT_RESULT_SUCCESS( nnt::nfp::InitializeSystemWithRetry() );
        nnt::nfp::DoConnect();

    }

    void TestInitTransInitDisconnect()
    {
        NNT_NFP_EXPECT_RESULT_SUCCESS( nnt::nfp::InitializeSystemWithRetry() );
        NNT_NFP_EXPECT_RESULT_SUCCESS( nnt::nfp::SetDefaultActivateEvent() );
        NNT_NFP_EXPECT_RESULT_SUCCESS( nnt::nfp::SetDefaultDeactivateEvent() );

    }

    void TestInitTransInitConnectUp()
    {
        NNT_NFP_EXPECT_RESULT_SUCCESS( nnt::nfp::InitializeSystemWithRetry() );
        nnt::nfp::DoConnectForFwUpdate();
    }

    void TestInitTransInitConnecting()
    {
        nn::Result connectionResult;
        nn::nfp::TargetConnectionStatus targetConnectionStatus;

        NNT_NFP_EXPECT_RESULT_SUCCESS( nnt::nfp::InitializeSystemWithRetry() );

        NNT_NFP_EXPECT_RESULT_SUCCESS(nnt::nfp::SetDefaultActivateEvent());
        NNT_NFP_EXPECT_RESULT_SUCCESS(nnt::nfp::SetDefaultDeactivateEvent());

        NNT_NFP_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetTargetConnectionStatus(&targetConnectionStatus));
        EXPECT_TRUE( targetConnectionStatus == nn::nfp::TARGET_DISCONNECTED );

        NNT_NFP_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Connect());

        for(int i=0; i<nnt::nfp::RETRY_COUNT_MAX; i++)
        {
            NNT_NFP_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetTargetConnectionStatus(&targetConnectionStatus));
            if (targetConnectionStatus == nn::nfp::TARGET_CONNECTING)
            {
                return;
            }
            nnt::nfp::Sleep(nnt::nfp::RetryInterval);
        }
        EXPECT_TRUE(false);
    }
#endif // !defined(NNT_NFP_PLATFORM_NX)

    void TestInitTransSearch()
    {
        NNT_NFP_EXPECT_RESULT_SUCCESS( nnt::nfp::InitializeSystemWithRetry() );
        nnt::nfp::DoSearch();

#if !defined(NNT_NFP_PLATFORM_NX)
        EXPECT_EQ( nn::nfp::RW_SEARCH, nnt::nfp::wrapper::GetDeviceState() );
#else
        EXPECT_EQ(nn::nfp::DeviceState_Search, nnt::nfp::wrapper::GetDeviceState() );
        EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState() );
#endif
    }

    void TestInitTransActive()
    {
        NNT_NFP_EXPECT_RESULT_SUCCESS( nnt::nfp::InitializeSystemWithRetry() );

        NN_LOG("==================================================\n");
        NN_LOG(" Please Attach a TAG !! \n");
        NN_LOG("==================================================\n");
        nnt::nfp::DoActivate();
        NN_LOG(" TAG Attached !!\n");

#if !defined(NNT_NFP_PLATFORM_NX)
        EXPECT_EQ( nn::nfp::RW_ACTIVE, nnt::nfp::wrapper::GetDeviceState() );
#else
        EXPECT_EQ(nn::nfp::DeviceState_Active, nnt::nfp::wrapper::GetDeviceState() );
        EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState() );
#endif
    }

    void TestInitRunApiThreadWait()
    {
        // スタック準備
        nn::os::InitializeEvent(&g_BootUpEvent, false,  nn::os::EventClearMode_ManualClear);
        nn::os::InitializeEvent(&g_RunApiEvent, false,  nn::os::EventClearMode_ManualClear);

        nn::os::ClearEvent(&g_BootUpEvent);
        nn::os::ClearEvent(&g_RunApiEvent);

        // RunNfpApi thread Start
        NNT_NFP_EXPECT_RESULT_SUCCESS(
                nn::os::CreateThread(&g_RunApiThread,
                                     ThreadRunNfpApi,
                                     nullptr,
                                     g_Stack,
                                     sizeof(g_Stack),
                                     nn::os::DefaultThreadPriority));
        nn::os::StartThread(&g_RunApiThread);

        NN_LOG("### Wainting for Thraed Start \n");
        nn::os::WaitEvent(&g_BootUpEvent);
        NN_LOG("### Thread Start Signal Detected !! \n");
        return;
    }

    void CreateTagAndMount()
    {
        nn::Bit8 writeData[nn::nfp::ApplicationAreaSizeV2]   = {};    // Write DataBuf
        nn::Bit8 readData[nn::nfp::ApplicationAreaSizeV2]    = {};    // Read DataBuf

        nnt::nfp::DoCreateNormalTag();

        /* 1. NFP StartDetection */
        nnt::nfp::DoActivate();

        g_RunApiThreadResult = nnt::nfp::MountWithRetry();
        NNT_NFP_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::NormalTagId));
        nnt::nfp::RegisterNfp();

        // NFPタグ読込
        NNT_NFP_EXPECT_RESULT_SUCCESS(
                nnt::nfp::wrapper::GetApplicationArea(readData,
                                                      nn::nfp::ApplicationAreaSizeV2) );

        // Create random data
        nnt::nfp::CreateRandomNumberSequence(writeData, nn::nfp::ApplicationAreaSizeV2);

        // 最大容量 nn::nfp::ApplicationAreaSizeV2 Byte の書き込み
        NNT_NFP_EXPECT_RESULT_SUCCESS(
                nnt::nfp::wrapper::SetApplicationArea(writeData,
                                                      nn::nfp::ApplicationAreaSizeV2,
                                                      nnt::nfp::GetTagInfoNfp().tagId) );
    }

    void TestInitTransPreMount()
    {
        NN_LOG("TestInitTransPreMount\n");
        // TestAPI設定
        g_RunNfpApi = RunNfpApi_Mount;
        // Active状態に遷移
        nnt::nfp::DoActivate();
        // TestAPI準備
        TestInitRunApiThreadWait();
    }

    void TestInitTransPreFormat()
    {
        NN_LOG("TestInitTransPreFormat\n");
        // TestAPI設定
        g_RunNfpApi = RunNfpApi_Format;
        // Active状態に遷移
        nnt::nfp::DoActivate();
        // TestAPI実行準備
        TestInitRunApiThreadWait();
    }

    void TestInitTransPreCreateApplicationArea()
    {
        NN_LOG("TestInitTransPreCreateApplicationArea\n");
        // TestAPI設定
        g_RunNfpApi = RunNfpApi_CreateApplicationArea;
        // Active状態に遷移
        nnt::nfp::DoActivate();
        // Formatでデータ消去
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FormatWithRetry());
        // Mount状態に遷移
        nnt::nfp::DoMount();
        // TestAPI実行準備
        TestInitRunApiThreadWait();
    }

    void TestInitTransPreDeleteApplicationArea()
    {
        NN_LOG("TestInitTransPreDeleteApplicationArea\n");
        // TestAPI設定
        g_RunNfpApi = RunNfpApi_DeleteApplicationArea;
        // タグを準備してMountに遷移
        CreateTagAndMount();
        // TestAPI実行準備
        TestInitRunApiThreadWait();
    }

    void TestInitTransPreDeleteRegisterInfo()
    {
        NN_LOG("TestInitTransPreDeleteRegisterInfo\n");
        // TestAPI設定
        g_RunNfpApi = RunNfpApi_DeleteRegisterInfo;
        // タグを準備してMountに遷移
        CreateTagAndMount();
        // TestAPI実行準備
        TestInitRunApiThreadWait();
    }

    void TestInitTransPreFlush()
    {
        NN_LOG("TestInitTransPreFlush\n");
        // TestAPI設定
        g_RunNfpApi = RunNfpApi_Flush;
        // タグを準備してMountに遷移
        CreateTagAndMount();
        // TestAPI実行準備
        TestInitRunApiThreadWait();
        return;
    }

    void TestInitTransPreRestore()
    {
        NN_LOG("TestInitTransPreRestore\n");
        // TestAPI設定
        g_RunNfpApi = RunNfpApi_Restore;
        // Mount状態に遷移
        nnt::nfp::DoMount();
        // Restoreを実行可能にするためにタグを破壊
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::BreakTag(nn::nfp::BreakType_Hmac));
        // Unmount状態に遷移
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Unmount());
        // Active状態に遷移
        nnt::nfp::DoActivate();
        // TestAPI実行準備
        TestInitRunApiThreadWait();
        return;
    }

    void TestInitTransDeactive()
    {
        NNT_NFP_EXPECT_RESULT_SUCCESS( nnt::nfp::InitializeSystemWithRetry() );

        NN_LOG("==================================================\n");
        NN_LOG(" Please Attach a TAG !! \n");
        NN_LOG("==================================================\n");
        nnt::nfp::DoActivate();
        NN_LOG(" TAG Attached !!\n");

        NN_LOG("==================================================\n");
        NN_LOG(" Please Remove a TAG !! \n");
        NN_LOG("==================================================\n");

        EXPECT_TRUE( nnt::nfp::WaitForDeactivate() == true );
        NN_LOG(" TAG Removed !!\n");

#if !defined(NNT_NFP_PLATFORM_NX)
        EXPECT_EQ( nn::nfp::RW_DEACTIVE, nnt::nfp::wrapper::GetDeviceState() );
#else
        EXPECT_EQ(nn::nfp::DeviceState_Deactive,
                                   nnt::nfp::wrapper::GetDeviceState() );
        EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState() );
#endif
    }

    void TestInitTransMount()
    {

        NNT_NFP_EXPECT_RESULT_SUCCESS( nnt::nfp::InitializeSystemWithRetry() );

        NN_LOG("==================================================\n");
        NN_LOG(" Please Attach a TAG !! \n");
        NN_LOG("==================================================\n");
        nnt::nfp::DoMount();
        NN_LOG(" TAG Attached !!\n");

#if !defined(NNT_NFP_PLATFORM_NX)
        EXPECT_EQ( nn::nfp::RW_MOUNT, nnt::nfp::wrapper::GetDeviceState() );
#else
        EXPECT_EQ(nn::nfp::DeviceState_Mount, nnt::nfp::wrapper::GetDeviceState());
        EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState() );
#endif
    }

    void TestInitTransMountRom()
    {

        NNT_NFP_EXPECT_RESULT_SUCCESS( nnt::nfp::InitializeSystemWithRetry() );

        NN_LOG("==================================================\n");
        NN_LOG(" Please Attach a TAG !! \n");
        NN_LOG("==================================================\n");
        nnt::nfp::DoMountRom();
        NN_LOG(" TAG Attached !!\n");

#if !defined(NNT_NFP_PLATFORM_NX)
        EXPECT_EQ( nn::nfp::RW_MOUNT_ROM, nnt::nfp::wrapper::GetDeviceState() );
#else
        EXPECT_EQ(nn::nfp::DeviceState_Mount, nnt::nfp::wrapper::GetDeviceState());
        EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState() );
#endif
    }

#if defined(NNT_NFP_LIB_VERSION_BETA)
    void TestMainWifiOff1()
    {
        nnt::nfp::WifiOff();

        // DeactivateEventが発生しない && DisconnectEventが発生する
        CheckDeactiveAndDisconnect(false, true);
#if !defined(NNT_NFP_PLATFORM_NX)
        EXPECT_EQ( nn::nfp::INIT, nnt::nfp::wrapper::GetDeviceState() );
#else
        EXPECT_EQ(nn::nfp::DeviceState_Init, nnt::nfp::wrapper::GetDeviceState() );
        EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState() );
#endif
    }

    void TestMainWifiOff2()
    {
        nnt::nfp::WifiOff();

        // DeactivateEventが発生する && DisconnectEventが発生する
        CheckDeactiveAndDisconnect(true, true);
#if !defined(NNT_NFP_PLATFORM_NX)
        EXPECT_EQ( nn::nfp::INIT, nnt::nfp::wrapper::GetDeviceState() );
#else
        EXPECT_EQ(nn::nfp::DeviceState_Init, nnt::nfp::wrapper::GetDeviceState() );
        EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState() );
#endif
    }

    void TestMainWifiOff3()
    {
        g_RunNfpApi = RunNfpApi_Flush;
#if !defined(NNT_NFP_PLATFORM_NX)
        nn::nwm::InitializeExtControl();

        g_FlushEvent.Signal();
        nn::nwm::Ext::SetWifiOff();
        NN_LOG("# WifiOFF");
        nn::nwm::FinalizeExtControl();
#else
        // Flush要求
        nn::os::SignalEvent(&g_RunApiEvent);
        // 即座にNfcを無効にする
        nnt::nfp::WifiOff();
        NN_LOG("# WifiOFF\n");
#endif

        // DeactivateEventが発生する && DisconnectEventが発生する
        CheckDeactiveAndDisconnect(true, true);
#if !defined(NNT_NFP_PLATFORM_NX)
        EXPECT_EQ( nn::nfp::INIT, nnt::nfp::wrapper::GetDeviceState() );
#else
        EXPECT_EQ(nn::nfp::DeviceState_Init, nnt::nfp::wrapper::GetDeviceState() );
        EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState() );
#endif

#if !defined(NNT_NFP_PLATFORM_NX)
        // Flushスレッドの終了まで待つ
        g_FlushThread.Join();
        g_FlushThread.Finalize();
#else
        nn::os::WaitThread(&g_RunApiThread);
        nn::os::DestroyThread(&g_RunApiThread);
#endif

#if !defined(NNT_NFP_PLATFORM_NX)
        if(nn::os::IsRunOnSnake() == false)
        {
            NNT_NFP_EXPECT_RESULT_EQUAL(nn::nfp::ResultOperationFailed(), g_FlushThreadResult);
        }
        else
        {
            NNT_NFP_EXPECT_RESULT_SUCCESS(g_FlushThreadResult);
        }
#else
        NNT_NFP_EXPECT_RESULT_EQUAL(nn::nfp::ResultNfcDisabled(), g_RunApiThreadResult);
#endif

        // タグのデータが破損していないことを確認する
        NN_LOG("### WIFI ON \n");
#if !defined(NNT_NFP_PLATFORM_NX)
        nn::srv::PublishToSubscriber(nn::srv::NOTIFICATION_WIFI_BUTTON);
#else
        nnt::nfp::WifiOn();
#endif
        nnt::nfp::Sleep(WaitTime_WifiOn);
        nnt::nfp::DoMount();
    }
#endif // defined(NNT_NFP_LIB_VERSION_BETA)

    void TestMainBluetoothCut1()
    {
        NN_LOG("==================================================\n");
        NN_LOG(" PLEASE CUT BT SIGNAL!! (Within %dms) ...         \n",WaitTime_BluetoothSignalCut);
        NN_LOG("==================================================\n");
        nnt::nfp::Sleep(WaitTime_BluetoothSignalCut);

        // DeactivateEventが発生しない && DisconnectEventが発生する
        CheckDeactiveAndDisconnect(false, true);
#if !defined(NNT_NFP_PLATFORM_NX)
        EXPECT_EQ( nn::nfp::INIT, nnt::nfp::wrapper::GetDeviceState() );
#else
        EXPECT_EQ(nn::nfp::DeviceState_Unexpected, nnt::nfp::wrapper::GetDeviceState() );
        EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState() );
#endif
    }

    void TestMainWifiOffInProgress()
    {
        NN_LOG("# TestMainWifiOffInProgress \n");
        // NfpApi要求
        nn::os::SignalEvent(&g_RunApiEvent);

        // Nfpライブラリがタグアクセスするまで少し待つ
        // タイミングは変わる可能性があるので、試験実施の際に調整してください
        // Mountの時のみ短めです。
        int sleepTime;
        if(g_RunNfpApi == RunNfpApi_Mount)
        {
            sleepTime = WaitTime_MountInProgressSleepTime;
        }
        else
        {
            sleepTime = WaitTime_InProgressSleepTime;
        }
        nnt::nfp::Sleep(sleepTime);

        // Nfcを無効にする
        nnt::nfp::WifiOff();
        NN_LOG("# WifiOFF\n");

        // DeactivateEventが発生する && DisconnectEventが発生する
        CheckDeactiveAndDisconnect(true, true);
        EXPECT_EQ(nn::nfp::DeviceState_Init, nnt::nfp::wrapper::GetDeviceState() );
        EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState() );

        // NfpApiの処理終了を待つ
        nn::os::WaitThread(&g_RunApiThread);
        nn::os::DestroyThread(&g_RunApiThread);

        // Nfc Off なのでResultNfcDisabledを期待
        NNT_NFP_EXPECT_RESULT_EQUAL(nn::nfp::ResultNfcDisabled(), g_RunApiThreadResult);

        // Nfc Onにする
        NN_LOG("# WIFI ON \n");
        nnt::nfp::WifiOn();
        nnt::nfp::Sleep(WaitTime_WifiOn);

        // Activate状態まで遷移
        nnt::nfp::DoActivate();
        // Mountを実行しタグの状態を確認する
        if (g_RunNfpApi == RunNfpApi_Mount)
        {
            // Mountは成功することを確認して終了
            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Mount());
            return;
        }
        else
        {
            // Mount以外はタグが壊れていることを期待する
            NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNeedRestore, nnt::nfp::wrapper::Mount());
        }
        // Restorerでタグを復旧
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Restore());
        // 復旧後Mountが成功すること
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Mount());
        return;
    }

    void TestMainBluetoothCut2()
    {
        NN_LOG("==================================================\n");
        NN_LOG(" PLEASE CUT BT SIGNAL!! (Within %dms) ...         \n",WaitTime_BluetoothSignalCut);
        NN_LOG("==================================================\n");
        nnt::nfp::Sleep(WaitTime_BluetoothSignalCut);

        // DeactivateEventが発生する && DisconnectEventが発生する
        CheckDeactiveAndDisconnect(true, true);
#if !defined(NNT_NFP_PLATFORM_NX)
        EXPECT_EQ( nn::nfp::INIT, nnt::nfp::wrapper::GetDeviceState() );
#else
        EXPECT_EQ(nn::nfp::DeviceState_Unexpected, nnt::nfp::wrapper::GetDeviceState() );
        EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState() );
#endif
    }

    void TestMainBluetoothCut3()
    {
        NN_LOG("==================================================\n");
        NN_LOG(" PLEASE CUT BT SIGNAL Ready... 5 ");
        nnt::nfp::Sleep(WaitTime_CountDown);
        NN_LOG(" 4 ");
        nnt::nfp::Sleep(WaitTime_CountDown);
        NN_LOG(" 3 ");
        nnt::nfp::Sleep(WaitTime_CountDown);
        NN_LOG(" 2 ");
        nnt::nfp::Sleep(WaitTime_CountDown);
        NN_LOG(" 1 ");
        nnt::nfp::Sleep(WaitTime_CountDown);
        NN_LOG(" NOW!! \n");
        NN_LOG("==================================================\n");

        nn::os::SignalEvent(&g_RunApiEvent);

        // DeactivateEventが発生する && DisconnectEventが発生する
        CheckDeactiveAndDisconnect(true, true);
#if !defined(NNT_NFP_PLATFORM_NX)
        EXPECT_EQ( nn::nfp::INIT, nnt::nfp::wrapper::GetDeviceState() );
#else
        EXPECT_EQ(nn::nfp::DeviceState_Unexpected, nnt::nfp::wrapper::GetDeviceState() );
        EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState() );
#endif

        // Flushスレッドの終了まで待つ
        nn::os::WaitThread(&g_RunApiThread);
        nn::os::DestroyThread(&g_RunApiThread);

        NNT_NFP_EXPECT_RESULT_EQUAL(nn::nfp::ResultNfcDeviceNotFound(), g_RunApiThreadResult);

        NN_LOG("==================================================\n");
        NN_LOG(" PLEASE RESTORE BT MODULE (Within %dms) ...       \n",WaitTime_BluetoothSignalRestore);
        NN_LOG("==================================================\n");
        nnt::nfp::Sleep(WaitTime_BluetoothSignalRestore);

#if !defined(NNT_NFP_PLATFORM_NX)
        // タグのデータが破損していないことを確認する
        nnt::nfp::DoMount();
#else
        // NXではFlush中に切断されると不正タグとなりMount時に
        // ResultNeedRestore()が返却されること。
        // タイミングによっては不正タグにならずMoutが成功すること。
        // タグの状態をActiveにする
        nnt::nfp::DoActivate();
        // Mountを実行して 成功 or ResultNeedRestore を確認する
        nn::Result mountResult = nnt::nfp::wrapper::Mount();
        if(mountResult.IsSuccess())
        {
            // mount成功フラグを立てる
            g_MountSuccesFlg = true;
            NN_LOG("Mount Success!!\n");
        }
        else
        {
            // ResultNeedRestore()かどうかを判定
            if(NNT_NFP_RESULT_EQUAL(nn::nfp::ResultNeedRestore(), mountResult))
            {
                // ResultNeedRestore確認フラグを立てる
                g_MountExpectFailFlg = true;
                NN_LOG("Mount ResultNeedRestore!!\n");
                // Restoreが成功すること
                NNT_NFP_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Restore());
                // Restore後にMountが成功すること
                NNT_NFP_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Mount());
            }
            else
            {
                EXPECT_TRUE(false);
                NN_LOG("Mount Not Expect err %s\n", nnt::nfp::GetNfpResultTypeString(mountResult));
            }
        }
#endif
    }

    void TestMainFinalize1()
    {
        NNT_NFP_EXPECT_RESULT_SUCCESS( nnt::nfp::FinalizeSystem() );
        EXPECT_TRUE( nnt::nfp::WaitForDeactivate( WaitTime_Deactivate ) == false );
#if !defined(NNT_NFP_PLATFORM_NX)
        EXPECT_EQ( nn::nfp::NONE, nnt::nfp::wrapper::GetDeviceState() );
#else
        EXPECT_EQ(nn::nfp::DeviceState_Unexpected, nnt::nfp::wrapper::GetDeviceState() );
        EXPECT_EQ(nn::nfp::State_None, nnt::nfp::wrapper::GetState() );
#endif
    }

    void TestMainFinalize2()
    {
        NNT_NFP_EXPECT_RESULT_SUCCESS( nnt::nfp::FinalizeSystem() );
        EXPECT_TRUE( nnt::nfp::WaitForDeactivate( WaitTime_Deactivate ) );
#if !defined(NNT_NFP_PLATFORM_NX)
        EXPECT_EQ( nn::nfp::NONE, nnt::nfp::wrapper::GetDeviceState() );
#else
        EXPECT_EQ(nn::nfp::DeviceState_Unexpected, nnt::nfp::wrapper::GetDeviceState() );
        EXPECT_EQ(nn::nfp::State_None, nnt::nfp::wrapper::GetState() );
#endif
    }

#if !defined(NNT_NFP_PLATFORM_NX) // NXではConnectは実装されない
    void TestMainDisconnect1()
    {
        nn::nfp::TargetConnectionStatus status;
        NNT_NFP_EXPECT_RESULT_SUCCESS( nnt::nfp::wrapper::GetTargetConnectionStatus(&status) );

        if( status == nn::nfp::TARGET_DISCONNECTED )
        {
            // DISCONNECT状態でのDISCONNECT()は呼出し不可となること
            EXPECT_TRUE( nnt::nfp::wrapper::Disconnect().IsFailure() );
        }
        else
        {
            NNT_NFP_EXPECT_RESULT_SUCCESS( nnt::nfp::wrapper::Disconnect() );
        }
        // DeactivateEventが発生しない && DisconnectEventが発生する
        CheckDeactiveAndDisconnect(false, true);
#if !defined(NNT_NFP_PLATFORM_NX)
        EXPECT_EQ( nn::nfp::INIT, nnt::nfp::wrapper::GetDeviceState() );
#else
        EXPECT_EQ(nn::nfp::DeviceState_Init, nnt::nfp::wrapper::GetDeviceState() );
        EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState() );
#endif
    }

    void TestMainDisconnect2()
    {
        NNT_NFP_EXPECT_RESULT_SUCCESS( nnt::nfp::wrapper::Disconnect() );

        // DeactivateEventが発生する && DisconnectEventが発生する
        CheckDeactiveAndDisconnect(true, true);
#if !defined(NNT_NFP_PLATFORM_NX)
        EXPECT_EQ( nn::nfp::INIT, nnt::nfp::wrapper::GetDeviceState() );
#else
        EXPECT_EQ(nn::nfp::DeviceState_Init, nnt::nfp::wrapper::GetDeviceState() );
        EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState() );
#endif
    }
#endif // !defined(NNT_NFP_PLATFORM_NX)

    void TestMainCloseScreen1()
    {
        NN_LOG("==================================================\n");
        NN_LOG(" PLEASE CLOSE SCREEN (Within %dms) ...           \n",WaitTime_SleepStart);
        NN_LOG("==================================================\n");
        nnt::nfp::Sleep(WaitTime_SleepStart);

        // DeactivateEventが発生しない && DisconnectEventが発生する
        CheckDeactiveAndDisconnect(false, true);
#if !defined(NNT_NFP_PLATFORM_NX)
        EXPECT_EQ( nn::nfp::INIT, nnt::nfp::wrapper::GetDeviceState() );
#else
        EXPECT_EQ(nn::nfp::DeviceState_Init, nnt::nfp::wrapper::GetDeviceState() );
        EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState() );
#endif
    }

    void TestMainCloseScreen2()
    {
        NN_LOG("==================================================\n");
        NN_LOG(" PLEASE CLOSE SCREEN (Within %dms) ...           \n",WaitTime_SleepStart);
        NN_LOG("==================================================\n");
        nnt::nfp::Sleep(WaitTime_SleepStart);

        // DeactivateEventが発生しない && DisconnectEventが発生する
        CheckDeactiveAndDisconnect(true, true);
#if !defined(NNT_NFP_PLATFORM_NX)
        EXPECT_EQ( nn::nfp::INIT, nnt::nfp::wrapper::GetDeviceState() );
#else
        EXPECT_EQ(nn::nfp::DeviceState_Init, nnt::nfp::wrapper::GetDeviceState() );
        EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState() );
#endif
    }

    void TestMainCloseScreen3()
    {
        NN_LOG("==================================================\n");
        NN_LOG(" PLEASE CLOSE SCREEN Ready... 5 ");
        nnt::nfp::Sleep(WaitTime_CountDown);
        NN_LOG(" 4 ");
        nnt::nfp::Sleep(WaitTime_CountDown);
        NN_LOG(" 3 ");
        nnt::nfp::Sleep(WaitTime_CountDown);
        NN_LOG(" 2 ");
        nnt::nfp::Sleep(WaitTime_CountDown);
        NN_LOG(" 1 ");
        nnt::nfp::Sleep(WaitTime_CountDown);
        NN_LOG(" NOW!! \n");
        NN_LOG("==================================================\n");

#if !defined(NNT_NFP_PLATFORM_NX)
        g_FlushEvent.Signal();
#else
        // Flush要求
        nn::os::SignalEvent(&g_RunApiEvent);
#endif

        // DeactivateEventが発生しない && DisconnectEventが発生する
        CheckDeactiveAndDisconnect(true, true);
#if !defined(NNT_NFP_PLATFORM_NX)
        EXPECT_EQ( nn::nfp::INIT, nnt::nfp::wrapper::GetDeviceState() );
#else
        EXPECT_EQ(nn::nfp::DeviceState_Init, nnt::nfp::wrapper::GetDeviceState() );
        EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState() );
#endif

#if !defined(NNT_NFP_PLATFORM_NX)
        // Flushスレッドの終了まで待つ
        g_FlushThread.Join();
        g_FlushThread.Finalize();
#else
        nn::os::WaitThread(&g_RunApiThread);
        nn::os::DestroyThread(&g_RunApiThread);
#endif

#if !defined(NNT_NFP_PLATFORM_NX)
        if(nn::os::IsRunOnSnake() == false)
        {
            NNT_NFP_EXPECT_RESULT_EQUAL(nn::nfp::ResultOperationFailed(), g_FlushThreadResult);

            NN_LOG("==================================================\n");
            NN_LOG(" PLEASE OPEN SCREEN (Within %dms) ...           \n",WaitTime_SleepRestore);
            NN_LOG("==================================================\n");
            nnt::nfp::Sleep(WaitTime_SleepRestore);

            // タグのデータが破損していないことを確認する
            nnt::nfp::DoMount();
        }
        else
        {
            NNT_NFP_EXPECT_RESULT_SUCCESS(g_FlushThreadResult);
        }
#else
        //Flush 中にスリープに入れたら nn::nfp::ResultSleep だがタイミングが非常に難しい
        //スリープ入るタイミングが早ければ nn::nfp::ResultInvalidDeviceState
        //遅ければ nn::nfp::ResultSuccess になる
        NNT_NFP_EXPECT_RESULT_EQUAL(nn::nfp::ResultSleep(), g_RunApiThreadResult);
#endif
    }

    void TestMainStopDetection1()
    {
#if !defined(NNT_NFP_PLATFORM_NX) // NXではConnectは実装されない
        nn::nfp::TargetConnectionStatus prevStatus, curStatus;
        NNT_NFP_EXPECT_RESULT_SUCCESS( nnt::nfp::wrapper::GetTargetConnectionStatus(&prevStatus) );
#endif // !defined(NNT_NFP_PLATFORM_NX)

        NNT_NFP_EXPECT_RESULT_SUCCESS( nnt::nfp::wrapper::StopDetection() );

#if !defined(NNT_NFP_PLATFORM_NX) // NXではConnectは実装されない
        NNT_NFP_EXPECT_RESULT_SUCCESS( nnt::nfp::wrapper::GetTargetConnectionStatus(&curStatus) );
#endif // !defined(NNT_NFP_PLATFORM_NX)

        // 接続状態が変化しない && DeactivateEventが発生しない
#if !defined(NNT_NFP_PLATFORM_NX) // NXではConnectは実装されない
        if(nn::os::IsRunOnSnake() == false)
        {
            EXPECT_EQ(prevStatus, curStatus);
        }
#endif // !defined(NNT_NFP_PLATFORM_NX)
        EXPECT_TRUE(nnt::nfp::WaitForDeactivate() == false);
#if !defined(NNT_NFP_PLATFORM_NX)
        EXPECT_EQ( nn::nfp::INIT, nnt::nfp::wrapper::GetDeviceState() );
#else
        EXPECT_EQ(nn::nfp::DeviceState_Init, nnt::nfp::wrapper::GetDeviceState() );
        EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState() );
#endif
    }

    void TestMainStopDetection2()
    {
#if !defined(NNT_NFP_PLATFORM_NX) // NXではConnectは実装されない
        nn::nfp::TargetConnectionStatus prevStatus, curStatus;
        NNT_NFP_EXPECT_RESULT_SUCCESS( nnt::nfp::wrapper::GetTargetConnectionStatus(&prevStatus) );
#endif // !defined(NNT_NFP_PLATFORM_NX)

        NNT_NFP_EXPECT_RESULT_SUCCESS( nnt::nfp::wrapper::StopDetection() );

#if !defined(NNT_NFP_PLATFORM_NX) // NXではConnectは実装されない
        NNT_NFP_EXPECT_RESULT_SUCCESS( nnt::nfp::wrapper::GetTargetConnectionStatus(&curStatus) );
#endif // !defined(NNT_NFP_PLATFORM_NX)

#if !defined(NNT_NFP_PLATFORM_NX) // NXではConnectは実装されない
        // 接続状態が変化しない && DeactivateEventが発生する
        if(nn::os::IsRunOnSnake() == false)
        {
            EXPECT_EQ(prevStatus, curStatus);
        }
#endif // !defined(NNT_NFP_PLATFORM_NX)
        EXPECT_TRUE(nnt::nfp::WaitForDeactivate());
#if !defined(NNT_NFP_PLATFORM_NX)
        EXPECT_EQ( nn::nfp::INIT, nnt::nfp::wrapper::GetDeviceState() );
#else
        EXPECT_EQ(nn::nfp::DeviceState_Init, nnt::nfp::wrapper::GetDeviceState() );
        EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState() );
#endif
    }

#if defined(NNT_NFP_LIB_VERSION_BETA) //NXではWifiのON/OFF機能が実装されるまで除外
    void TestFinalizeWifiOff()
    {
        nnt::nfp::WifiOn();
        NNT_NFP_EXPECT_RESULT_SUCCESS( nnt::nfp::FinalizeSystem() );
    }
#endif // defined(NNT_NFP_LIB_VERSION_BETA)

    void TestFinalizeBluetoothCut()
    {
        NN_LOG("==================================================\n");
        NN_LOG(" PLEASE RESTORE BT MODULE (Within %dms) ...       \n",WaitTime_BluetoothSignalRestore);
        NN_LOG("==================================================\n");
        nnt::nfp::Sleep(WaitTime_BluetoothSignalRestore);
        NNT_NFP_EXPECT_RESULT_SUCCESS( nnt::nfp::FinalizeSystem() );
    }

    void TestFinalizeCloseScreen()
    {
        NN_LOG("==================================================\n");
        NN_LOG(" PLEASE OPEN SCREEN (Within %dms) ...            \n",WaitTime_SleepRestore);
        NN_LOG("==================================================\n");
        nnt::nfp::Sleep(WaitTime_SleepRestore);
        NNT_NFP_EXPECT_RESULT_SUCCESS( nnt::nfp::FinalizeSystem() );
    }

    void TestFinalizeRunNfpApiThread()
    {
        if(g_RunApiThread._state == nn::os::ThreadType::State_Started)
        {
            nn::os::DestroyThread(&g_RunApiThread);
        }

        NNT_NFP_EXPECT_RESULT_SUCCESS( nnt::nfp::FinalizeSystem() );
        nn::os::FinalizeEvent(&g_BootUpEvent);
        nn::os::FinalizeEvent(&g_RunApiEvent);
    }

    void TestFinalizeDefault()
    {
        NNT_NFP_EXPECT_RESULT_SUCCESS( nnt::nfp::FinalizeSystem() );
    }


    void TestFinalizeNop()
    {
        /* Nop */
    }

#if !defined(NNT_NFP_PLATFORM_NX) //NX では必要としない
    //================================================================================
    //! @brief      スリープ通知を受け取るコールバックです。
    //! @param[in]  arg          コールバックの指定時に設定した引数です。このツールでは利用しません。
    //! @return     スリープ要求に対する応答です。
    //================================================================================
    nn::applet::AppletQueryReply SleepQueryCallback(uptr arg)
    {
        NN_UNUSED(arg);
        // 他のアプレットやメニューに制御が移っている場合は即座にスリープを受け入れます。
        return nn::applet::REPLY_ACCEPT;
    }

    //================================================================================
    //! @brief      スリープからの復帰通知を受け取るコールバックです。
    //! @param[in]  arg          コールバックの指定時に設定した引数です。このツールでは利用しません。
    //================================================================================
    void AwakeCallback(uptr arg)
    {
        NN_UNUSED(arg);
        NN_LOG(" AWAKE\n");
    }
#endif //!defined(NNT_NFP_PLATFORM_NX)

    bool WaitForActivate(int ms, nn::os::SystemEventType activateEvent) NN_NOEXCEPT
    {
        if((activateEvent._state != nn::os::SystemEventType::State_NotInitialized) &&
                (activateEvent._state != nn::os::SystemEventType::State_NotInitialized))
        {
            return nn::os::TimedWaitSystemEvent(&activateEvent, nn::TimeSpan::FromMilliSeconds(ms));
        }
        else
        {
            NN_LOG("WaitForActivate() ActiveEvent DeactivateEvent is NotInitialized \n");
            return false;
        }
    }

    bool WaitForDeactivate(int ms, nn::os::SystemEventType deactivateEvent) NN_NOEXCEPT
    {
        return nn::os::TimedWaitSystemEvent(&deactivateEvent, nn::TimeSpan::FromMilliSeconds(ms));
    }

    void DestroyDefaultActivateEvent(nn::os::SystemEventType activateEvent) NN_NOEXCEPT
    {
        nn::os::DestroySystemEvent(&activateEvent);
        return;
    }

    void DestroyDefaultDeactivateEvent(nn::os::SystemEventType deactivateEvent) NN_NOEXCEPT
    {
        nn::os::DestroySystemEvent(&deactivateEvent);
        return;
    }

    nnt::nfp::MiiDataPattern SetMiiPattern(int deviceNum) NN_NOEXCEPT
    {
        nnt::nfp::MiiDataPattern ret;
        switch(deviceNum)
        {
        case 0:
            ret = nnt::nfp::MiiDataPattern_0;
            break;
        case 1:
            ret = nnt::nfp::MiiDataPattern_1;
            break;
        case 2:
            ret = nnt::nfp::MiiDataPattern_2;
            break;
        case 3:
            ret = nnt::nfp::MiiDataPattern_3;
            break;
        default:
            ret = nnt::nfp::MiiDataPattern_0;
            break;
        }
        return ret;
    }

    void DeviceTestThread(void* pArg) NN_NOEXCEPT
    {
        threadinfo* threaditem = static_cast<threadinfo*>(pArg);
        int deviceNum = threaditem->deviceNum;
        // ランダムタイミングでAPIを要求する試験
        for(int i = 0; i < TestLoop; i++)
        {
            nn::os::SystemEventType activateEvent[DeviceCountMax] = {};
            nn::os::SystemEventType deactivateEvent[DeviceCountMax] = {};
            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::AttachActivateEvent(
                    &activateEvent[deviceNum], g_CurrentDeviceHandleMulti[deviceNum]));
            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::AttachDeactivateEvent(
                    &deactivateEvent[deviceNum], g_CurrentDeviceHandleMulti[deviceNum]));

            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::StartDetection(g_CurrentDeviceHandleMulti[deviceNum]));
            if (!WaitForActivate(WaitTimeout, activateEvent[deviceNum]))
            {
                NN_LOG("StartDetection err deviceNum=%d Loop=%d\n", deviceNum, i);
                EXPECT_TRUE(false);
                break;
            }

            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Format(g_CurrentDeviceHandleMulti[deviceNum]));

            EXPECT_EQ(nn::nfp::DeviceState_Active,nnt::nfp::wrapper::GetDeviceState(g_CurrentDeviceHandleMulti[deviceNum]));
            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Mount(g_CurrentDeviceHandleMulti[deviceNum], nn::nfp::ModelType_Amiibo));

            nn::nfp::RegisterInfoPrivate regInfoSet;
#if defined(NNT_NFP_LIB_MII_ENABLE)
            nnt::nfp::BuildMiiData(regInfoSet.miiData, SetMiiPattern(deviceNum));
#endif // defined(NNT_NFP_LIB_MII_ENABLE)
            std::memcpy(&regInfoSet.nickname, g_NormalTagNickName[deviceNum], sizeof(regInfoSet.nickname));
            regInfoSet.fontRegion = NormalTagFontRegion;
            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::SetRegisterInfo(g_CurrentDeviceHandleMulti[deviceNum] ,regInfoSet));

            nn::nfp::ApplicationAreaCreateInfo appInfoSet;
            nn::Bit8 tagDataSet[nn::nfp::ApplicationAreaSizeV2];
            std::memcpy(tagDataSet, g_NormalTagData[deviceNum], nn::nfp::ApplicationAreaSizeV2);
            appInfoSet.accessId        = nnt::nfp::NormalTagId;
            appInfoSet.pInitialData    = tagDataSet;
            appInfoSet.initialDataSize = nn::nfp::ApplicationAreaSizeV2;
            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::CreateApplicationArea(g_CurrentDeviceHandleMulti[deviceNum], appInfoSet));

            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Flush(g_CurrentDeviceHandleMulti[deviceNum]));

            nn::nfp::RegisterInfoPrivate afterRegInfo;
            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetRegisterInfo(&afterRegInfo, g_CurrentDeviceHandleMulti[deviceNum]));
#if defined(NNT_NFP_LIB_MII_ENABLE)
            EXPECT_TRUE(std::memcmp(&afterRegInfo.miiData, &regInfoSet.miiData,
                    regInfoSet.miiData.StructureSize - ExceptDataSize) == 0);
#endif // defined(NNT_NFP_LIB_MII_ENABLE)
            EXPECT_TRUE(std::memcmp(regInfoSet.nickname, afterRegInfo.nickname, sizeof(afterRegInfo.nickname)) == 0);
            EXPECT_TRUE(std::memcmp(&regInfoSet.fontRegion, &afterRegInfo.fontRegion, sizeof(afterRegInfo.fontRegion)) == 0);

            nn::Bit8 afterAppArea[nn::nfp::ApplicationAreaSizeV2];
            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::OpenApplicationArea(g_CurrentDeviceHandleMulti[deviceNum], nnt::nfp::NormalTagId));
            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetApplicationArea(afterAppArea, g_CurrentDeviceHandleMulti[deviceNum],
                                                                            sizeof(afterAppArea)));
            EXPECT_TRUE(std::memcmp(tagDataSet, afterAppArea, sizeof(afterAppArea)) == 0);

            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Unmount(g_CurrentDeviceHandleMulti[deviceNum]));
            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::StopDetection(g_CurrentDeviceHandleMulti[deviceNum]));
            if (!WaitForDeactivate(WaitTimeout, deactivateEvent[deviceNum]))
            {
                NN_LOG("StopDetection err deviceNum=%d Loop=%d\n", deviceNum, i);
                EXPECT_TRUE(false);
                break;
            }
            DestroyDefaultActivateEvent(activateEvent[deviceNum]);
            DestroyDefaultDeactivateEvent(deactivateEvent[deviceNum]);
        }
    }

    void DeviceDeactivateTestThread(void* pArg) NN_NOEXCEPT
    {
        threadinfo* threaditem = static_cast<threadinfo*>(pArg);
        int deviceNum = threaditem->deviceNum;

        nn::os::SystemEventType activateEvent[DeviceCountMax] = {};
        nn::os::SystemEventType deactivateEvent[DeviceCountMax] = {};
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::AttachActivateEvent(
                &activateEvent[deviceNum], g_CurrentDeviceHandleMulti[deviceNum]));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::AttachDeactivateEvent(
                &deactivateEvent[deviceNum], g_CurrentDeviceHandleMulti[deviceNum]));

        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::StartDetection(g_CurrentDeviceHandleMulti[deviceNum]));
        if (!WaitForActivate(WaitTimeout, activateEvent[deviceNum]))
        {
            EXPECT_TRUE(false);
        }
        EXPECT_EQ(nn::nfp::DeviceState_Active,nnt::nfp::wrapper::GetDeviceState(g_CurrentDeviceHandleMulti[deviceNum]));
        NN_LOG("========================================================================\n");
        NN_LOG(" Active! handle = %d\n", g_CurrentDeviceHandleMulti[deviceNum]);
        NN_LOG("========================================================================\n");
        // Deactiveが来ないことを確認するテスト
        // DeactivateWaitTimeout(15秒)間でタイムアウトさせる。
        // その間に手動で５台目のJoy-Conの接続を試みる。期待動作は接続できない。
        if (!WaitForDeactivate(DeactivateWaitTimeout, deactivateEvent[deviceNum]))
        {
            // Mountができることを確認する
            EXPECT_EQ(nn::nfp::DeviceState_Active,nnt::nfp::wrapper::GetDeviceState(g_CurrentDeviceHandleMulti[deviceNum]));
            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Mount(g_CurrentDeviceHandleMulti[deviceNum], nn::nfp::ModelType_Amiibo));
        }
        DestroyDefaultActivateEvent(activateEvent[deviceNum]);
        DestroyDefaultDeactivateEvent(deactivateEvent[deviceNum]);

    }

    void CheckOnlyOneDeviceActiveTestThread(void* pArg) NN_NOEXCEPT
    {
        threadinfo* threaditem = static_cast<threadinfo*>(pArg);
        int deviceNum = threaditem->deviceNum;

        nn::os::SystemEventType activateEvent[DeviceCountMax] = {};
        nn::os::SystemEventType deactivateEvent[DeviceCountMax] = {};
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::AttachActivateEvent(
                &activateEvent[deviceNum], g_CurrentDeviceHandleMulti[deviceNum]));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::AttachDeactivateEvent(
                &deactivateEvent[deviceNum], g_CurrentDeviceHandleMulti[deviceNum]));

        nn::Result result = nnt::nfp::wrapper::StartDetection(g_CurrentDeviceHandleMulti[deviceNum]);
        // 4台以上のJoy-Conが接続されているときは、Nfp機能は１つしか使用できないことを確認する
        if (std::strcmp(nnt::nfp::GetNfpResultTypeString(result), "ResultMaxNfcDeviceActivated") == 0)
        {
            g_ErrorCount++;
            return;
        }
        g_SuccessCount++;
        if (!WaitForActivate(WaitTimeout, activateEvent[deviceNum]))
        {
            EXPECT_TRUE(false);
        }
        EXPECT_EQ(nn::nfp::DeviceState_Active,nnt::nfp::wrapper::GetDeviceState(g_CurrentDeviceHandleMulti[deviceNum]));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Mount(g_CurrentDeviceHandleMulti[deviceNum], nn::nfp::ModelType_Amiibo));
        DestroyDefaultActivateEvent(activateEvent[deviceNum]);
        DestroyDefaultDeactivateEvent(deactivateEvent[deviceNum]);
        return;
    }

    void MultiDeviceDisconnectTagTestThread(void* pArg) NN_NOEXCEPT
    {
        threadinfo* threaditem = static_cast<threadinfo*>(pArg);
        int deviceNum = threaditem->deviceNum;

        nn::os::SystemEventType activateEvent[DeviceCountMax] = {};
        nn::os::SystemEventType deactivateEvent[DeviceCountMax] = {};
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::AttachActivateEvent(
                &activateEvent[deviceNum], g_CurrentDeviceHandleMulti[deviceNum]));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::AttachDeactivateEvent(
                &deactivateEvent[deviceNum], g_CurrentDeviceHandleMulti[deviceNum]));

        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::StartDetection(g_CurrentDeviceHandleMulti[deviceNum]));
        if (!WaitForActivate(WaitTimeout, activateEvent[deviceNum]))
        {
            NN_LOG("StartDetection err deviceNum=%d \n", deviceNum);
            EXPECT_TRUE(false);
            return;
        }

        // 他のスレッドのStartDetectionと処理が重なることによってエラーが出るので
        // 本試験では回避コードを入れておきます。
        nnt::nfp::Sleep(WaitTimeout * deviceNum);

        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Format(g_CurrentDeviceHandleMulti[deviceNum]));

        EXPECT_EQ(nn::nfp::DeviceState_Active,nnt::nfp::wrapper::GetDeviceState(g_CurrentDeviceHandleMulti[deviceNum]));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Mount(g_CurrentDeviceHandleMulti[deviceNum], nn::nfp::ModelType_Amiibo));

        nn::nfp::RegisterInfoPrivate regInfoSet;
#if defined(NNT_NFP_LIB_MII_ENABLE)
        nnt::nfp::BuildMiiData(regInfoSet.miiData, SetMiiPattern(deviceNum));
#endif // defined(NNT_NFP_LIB_MII_ENABLE)
        std::memcpy(&regInfoSet.nickname, g_NormalTagNickName[deviceNum], sizeof(regInfoSet.nickname));
        regInfoSet.fontRegion = NormalTagFontRegion;
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::SetRegisterInfo(g_CurrentDeviceHandleMulti[deviceNum] ,regInfoSet));

        nn::nfp::ApplicationAreaCreateInfo appInfoSet;
        nn::Bit8 tagDataSet[nn::nfp::ApplicationAreaSizeV2];
        std::memcpy(tagDataSet, g_NormalTagData[deviceNum], nn::nfp::ApplicationAreaSizeV2);
        appInfoSet.accessId        = nnt::nfp::NormalTagId;
        appInfoSet.pInitialData    = tagDataSet;
        appInfoSet.initialDataSize = nn::nfp::ApplicationAreaSizeV2;
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::CreateApplicationArea(g_CurrentDeviceHandleMulti[deviceNum], appInfoSet));

        //準備完了シグナルを出す
        nn::os::SignalEvent(&g_MultiDevicePrepareEvent[deviceNum]);
        //一斉に開始できるようにここで一旦止める
        nn::os::WaitEvent(&g_MultiDeviceStartEvent);
        nn::os::ClearEvent(&g_MultiDeviceStartEvent);
        NN_LOG("Flush !! handle=%d\n", g_CurrentDeviceHandleMulti[deviceNum]);
        nn::Result result = nnt::nfp::wrapper::Flush(g_CurrentDeviceHandleMulti[deviceNum]);
        // タグを離すため期待値は、ResultTagNotFoundです
        if (std::strcmp(nnt::nfp::GetNfpResultTypeString(result), "ResultTagNotFound") == 0)
        {
            NN_LOG("Expect ERROR = %s handle = %d\n", nnt::nfp::GetNfpResultTypeString(result), g_CurrentDeviceHandleMulti[deviceNum]);
            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::StartDetection(g_CurrentDeviceHandleMulti[deviceNum]));
            NN_LOG("========================================================================\n");
            NN_LOG(" Please Attach Tag! \n");
            NN_LOG("========================================================================\n");
            nnt::nfp::Sleep(WaitTimeout);
            WaitForActivate(WaitTimeout, activateEvent[deviceNum]);
            // Flush中にタグを離すとタグ破壊状態でRestore()に成功するが、
            // ResultNotBrokenになるとFlush中の試験条件を満たしていないので試験をやり直してください
            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Restore(g_CurrentDeviceHandleMulti[deviceNum]));
            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Mount(g_CurrentDeviceHandleMulti[deviceNum], nn::nfp::ModelType_Amiibo));
        }
        else if (std::strcmp(nnt::nfp::GetNfpResultTypeString(result), "ResultSuccess") == 0)
        {
            // タグを離さなかったものは成功を期待
            g_SuccessDeviceCount++;
            NN_LOG("Expect ERROR = %s handle = %d SuccessCount = %d\n", nnt::nfp::GetNfpResultTypeString(result), g_CurrentDeviceHandleMulti[deviceNum], g_SuccessDeviceCount);
        }
        else
        {
            NN_LOG("Unexpect error!! %s\n", nnt::nfp::GetNfpResultTypeString(result));
            EXPECT_TRUE(false);
        }

        nn::nfp::RegisterInfoPrivate afterRegInfo;
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetRegisterInfo(&afterRegInfo, g_CurrentDeviceHandleMulti[deviceNum]));
#if defined(NNT_NFP_LIB_MII_ENABLE)
        EXPECT_TRUE(std::memcmp(&afterRegInfo.miiData, &regInfoSet.miiData,
                regInfoSet.miiData.StructureSize - ExceptDataSize) == 0);
#endif // defined(NNT_NFP_LIB_MII_ENABLE)
        EXPECT_TRUE(std::memcmp(regInfoSet.nickname, afterRegInfo.nickname, sizeof(afterRegInfo.nickname)) == 0);
        EXPECT_TRUE(std::memcmp(&regInfoSet.fontRegion, &afterRegInfo.fontRegion, sizeof(afterRegInfo.fontRegion)) == 0);

        nn::Bit8 afterAppArea[nn::nfp::ApplicationAreaSizeV2];
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::OpenApplicationArea(g_CurrentDeviceHandleMulti[deviceNum], nnt::nfp::NormalTagId));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetApplicationArea(afterAppArea, g_CurrentDeviceHandleMulti[deviceNum],
                                                                        sizeof(afterAppArea)));
        EXPECT_TRUE(std::memcmp(tagDataSet, afterAppArea, sizeof(afterAppArea)) == 0);

        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Unmount(g_CurrentDeviceHandleMulti[deviceNum]));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::StopDetection(g_CurrentDeviceHandleMulti[deviceNum]));
        if (!WaitForDeactivate(WaitTimeout, deactivateEvent[deviceNum]))
        {
            NN_LOG("StopDetection err deviceNum=%d\n", deviceNum);
            EXPECT_TRUE(false);
            return;
        }
        DestroyDefaultActivateEvent(activateEvent[deviceNum]);
        DestroyDefaultDeactivateEvent(deactivateEvent[deviceNum]);
    }

    void MultiDeviceChangeTagTestThread(void* pArg) NN_NOEXCEPT
    {
        threadinfo* threaditem = static_cast<threadinfo*>(pArg);
        int deviceNum = threaditem->deviceNum;

        nn::os::SystemEventType activateEvent[DeviceCountMax] = {};
        nn::os::SystemEventType deactivateEvent[DeviceCountMax] = {};
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::AttachActivateEvent(
                &activateEvent[deviceNum], g_CurrentDeviceHandleMulti[deviceNum]));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::AttachDeactivateEvent(
                &deactivateEvent[deviceNum], g_CurrentDeviceHandleMulti[deviceNum]));

        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::StartDetection(g_CurrentDeviceHandleMulti[deviceNum]));
        if (!WaitForActivate(WaitTimeout, activateEvent[deviceNum]))
        {
            NN_LOG("StartDetection err deviceNum=%d \n", deviceNum);
            EXPECT_TRUE(false);
            return;
        }

        // 他のスレッドのStartDetectionと処理が重なることによってエラーが出るので
        // 本試験では回避コードを入れておきます。
        nnt::nfp::Sleep(WaitTimeout * deviceNum);

        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Format(g_CurrentDeviceHandleMulti[deviceNum]));

        EXPECT_EQ(nn::nfp::DeviceState_Active,nnt::nfp::wrapper::GetDeviceState(g_CurrentDeviceHandleMulti[deviceNum]));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Mount(g_CurrentDeviceHandleMulti[deviceNum], nn::nfp::ModelType_Amiibo));

        nn::nfp::RegisterInfoPrivate regInfoSet;
#if defined(NNT_NFP_LIB_MII_ENABLE)
        nnt::nfp::BuildMiiData(regInfoSet.miiData, SetMiiPattern(deviceNum));
#endif // defined(NNT_NFP_LIB_MII_ENABLE)
        std::memcpy(&regInfoSet.nickname, g_NormalTagNickName[deviceNum], sizeof(regInfoSet.nickname));
        regInfoSet.fontRegion = NormalTagFontRegion;
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::SetRegisterInfo(g_CurrentDeviceHandleMulti[deviceNum] ,regInfoSet));

        nn::nfp::ApplicationAreaCreateInfo appInfoSet;
        nn::Bit8 tagDataSet[nn::nfp::ApplicationAreaSizeV2];
        std::memcpy(tagDataSet, g_NormalTagData[deviceNum], nn::nfp::ApplicationAreaSizeV2);
        appInfoSet.accessId        = nnt::nfp::NormalTagId;
        appInfoSet.pInitialData    = tagDataSet;
        appInfoSet.initialDataSize = nn::nfp::ApplicationAreaSizeV2;
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::CreateApplicationArea(g_CurrentDeviceHandleMulti[deviceNum], appInfoSet));

        //準備完了シグナルを出す
        nn::os::SignalEvent(&g_MultiDevicePrepareEvent[deviceNum]);
        //一斉に開始できるようにここで一旦止める
        nn::os::WaitEvent(&g_MultiDeviceStartEvent);
        nn::os::ClearEvent(&g_MultiDeviceStartEvent);
        NN_LOG("Flush !! handle=%d\n", g_CurrentDeviceHandleMulti[deviceNum]);
        nn::Result result = nnt::nfp::wrapper::Flush(g_CurrentDeviceHandleMulti[deviceNum]);
        // タグを離すため期待値は、ResultTagNotFoundです
        if (std::strcmp(nnt::nfp::GetNfpResultTypeString(result), "ResultTagNotFound") == 0)
        {
            NN_LOG("Expect ERROR = %s handle = %d\n", nnt::nfp::GetNfpResultTypeString(result), g_CurrentDeviceHandleMulti[deviceNum]);
            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::StartDetection(g_CurrentDeviceHandleMulti[deviceNum]));
            NN_LOG("========================================================================\n");
            NN_LOG(" Please Change Tag! \n");
            NN_LOG("========================================================================\n");
            nnt::nfp::Sleep(WaitTimeout);
            WaitForActivate(WaitTimeout, activateEvent[deviceNum]);
            // Flush中にタグを離すとタグ破壊状態でRestore()に成功するが、
            // ResultNotBrokenになるとFlush中の試験条件を満たしていないので試験をやり直してください
            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Restore(g_CurrentDeviceHandleMulti[deviceNum]));
            // タグを入れ替えるためMountのみ確認
            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Mount(g_CurrentDeviceHandleMulti[deviceNum], nn::nfp::ModelType_Amiibo));
        }
        else if (std::strcmp(nnt::nfp::GetNfpResultTypeString(result), "ResultSuccess") == 0)
        {
            // タグを離さなかったものは成功を期待
            g_SuccessDeviceCount++;
            NN_LOG("Expect ERROR = %s handle = %d SuccessCount = %d\n", nnt::nfp::GetNfpResultTypeString(result), g_CurrentDeviceHandleMulti[deviceNum], g_SuccessDeviceCount);
        }
        else
        {
            NN_LOG("Unexpect error!! %s\n", nnt::nfp::GetNfpResultTypeString(result));
            EXPECT_TRUE(false);
        }
    }

    void MultiDeviceNfcOffTagTestThread(void* pArg) NN_NOEXCEPT
    {
        threadinfo* threaditem = static_cast<threadinfo*>(pArg);
        int deviceNum = threaditem->deviceNum;

        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::StartDetection(g_CurrentDeviceHandleMulti[deviceNum]));

        if (!WaitForActivate(WaitTimeout, g_MultiActivateEvent[deviceNum]))
        {
            NN_LOG("StartDetection err deviceNum=%d \n", deviceNum);
            EXPECT_TRUE(false);
            return;
        }

        // 他のスレッドのStartDetectionと処理が重なることによってエラーが出るので
        // 本試験では回避コードを入れておきます。
        //nnt::nfp::Sleep(WaitTimeout * deviceNum);
        //NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Format(g_CurrentDeviceHandleMulti[deviceNum]));
        for(int i = 0; i < 3; i++)
        {
            nn::Result formatResult = nnt::nfp::wrapper::Format(g_CurrentDeviceHandleMulti[deviceNum]);
            if((std::strcmp(nnt::nfp::GetNfpResultTypeString(formatResult), "ResultNeedRetry") == 0) ||
                    (std::strcmp(nnt::nfp::GetNfpResultTypeString(formatResult), "ResultOperationFailed") == 0) )
            {
                NN_LOG("handle=%d is %s\n", g_CurrentDeviceHandleMulti[deviceNum], nnt::nfp::GetNfpResultTypeString(formatResult));
                continue;
            }
            else if(std::strcmp(nnt::nfp::GetNfpResultTypeString(formatResult), "ResultSuccess") == 0)
            {
                break;
            }
            else
            {
                NN_LOG("handle=%d is %s\n", g_CurrentDeviceHandleMulti[deviceNum], nnt::nfp::GetNfpResultTypeString(formatResult));
                continue;
            }
        }

        EXPECT_EQ(nn::nfp::DeviceState_Active,nnt::nfp::wrapper::GetDeviceState(g_CurrentDeviceHandleMulti[deviceNum]));
        // 他のスレッドのStartDetectionと処理が重なることによってエラーが出るので
        // 本試験では回避コードを入れておきます。
        //NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Mount(g_CurrentDeviceHandleMulti[deviceNum], nn::nfp::ModelType_Amiibo));
        for(int i = 0; i < 3; i++)
        {
            nn::Result mountResult = nnt::nfp::wrapper::Mount(g_CurrentDeviceHandleMulti[deviceNum], nn::nfp::ModelType_Amiibo);
            if((std::strcmp(nnt::nfp::GetNfpResultTypeString(mountResult), "ResultNeedRetry") == 0) ||
                    (std::strcmp(nnt::nfp::GetNfpResultTypeString(mountResult), "ResultOperationFailed") == 0) )
            {
                NN_LOG("handle=%d is %s", g_CurrentDeviceHandleMulti[deviceNum], nnt::nfp::GetNfpResultTypeString(mountResult));
                continue;
            }
            else if(std::strcmp(nnt::nfp::GetNfpResultTypeString(mountResult), "ResultSuccess") == 0)
            {
                break;
            }
            else
            {
                NN_LOG("handle=%d is %s", g_CurrentDeviceHandleMulti[deviceNum], nnt::nfp::GetNfpResultTypeString(mountResult));
                EXPECT_TRUE(false);
            }
        }

        nn::nfp::RegisterInfoPrivate regInfoSet;
#if defined(NNT_NFP_LIB_MII_ENABLE)
        nnt::nfp::BuildMiiData(regInfoSet.miiData, SetMiiPattern(deviceNum));
#endif // defined(NNT_NFP_LIB_MII_ENABLE)
        std::memcpy(&regInfoSet.nickname, g_NormalTagNickName[deviceNum], sizeof(regInfoSet.nickname));
        regInfoSet.fontRegion = NormalTagFontRegion;
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::SetRegisterInfo(g_CurrentDeviceHandleMulti[deviceNum] ,regInfoSet));

        nn::nfp::ApplicationAreaCreateInfo appInfoSet;
        nn::Bit8 tagDataSet[nn::nfp::ApplicationAreaSizeV2];
        std::memcpy(tagDataSet, g_NormalTagData[deviceNum], nn::nfp::ApplicationAreaSizeV2);
        appInfoSet.accessId        = nnt::nfp::NormalTagId;
        appInfoSet.pInitialData    = tagDataSet;
        appInfoSet.initialDataSize = nn::nfp::ApplicationAreaSizeV2;
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::CreateApplicationArea(g_CurrentDeviceHandleMulti[deviceNum], appInfoSet));

        //準備完了シグナルを出す
        nn::os::SignalEvent(&g_MultiDevicePrepareEvent[deviceNum]);
        //一斉に開始できるようにここで一旦止める
        nn::os::WaitEvent(&g_MultiDeviceStartEvent);
        nn::os::ClearEvent(&g_MultiDeviceStartEvent);
        NN_LOG("Flush !! handle=%d\n", g_CurrentDeviceHandleMulti[deviceNum]);
        nn::Result result = nnt::nfp::wrapper::Flush(g_CurrentDeviceHandleMulti[deviceNum]);

        // 別スレッドでFlush中に Nfc offに変更しているので、期待値はResultNfcDisabled
        if (std::strcmp(nnt::nfp::GetNfpResultTypeString(result), "ResultNfcDisabled") == 0)
        {
            g_Mutex.Lock();
            NN_LOG("Expect ERROR = %s handle = %d\n", nnt::nfp::GetNfpResultTypeString(result), g_CurrentDeviceHandleMulti[deviceNum]);
            if (!g_NfcState)
            {
                g_NfcState = true;
                NN_LOG("# WifiON\n");
                nnt::nfp::Sleep(WaitTime_CountDown);
                nnt::nfp::WifiOn();
                NN_LOG("# WifiON\n");
                int outCount = 0;
                NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ListDevices(
                        g_CurrentDeviceHandleMulti, &outCount, DeviceCountMax));
            }
            g_Mutex.Unlock();
            EXPECT_EQ(nn::nfp::DeviceState_Init,nnt::nfp::wrapper::GetDeviceState(g_CurrentDeviceHandleMulti[deviceNum]));
            nn::nfp::DeviceState deviceState;
            deviceState = nnt::nfp::wrapper::GetDeviceState(g_CurrentDeviceHandleMulti[deviceNum]);
            NN_LOG("DeviceState = %d handle = %d\n", deviceState, g_CurrentDeviceHandleMulti[deviceNum]);
            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::StartDetection(g_CurrentDeviceHandleMulti[deviceNum]));
            deviceState = nnt::nfp::wrapper::GetDeviceState(g_CurrentDeviceHandleMulti[deviceNum]);
            NN_LOG("DeviceState = %d handle = %d\n", deviceState, g_CurrentDeviceHandleMulti[deviceNum]);
            WaitForActivate(WaitTimeout, g_MultiActivateEvent[deviceNum]);
            // Flush中に Nfc off にするとタグ破壊状態でRestore()に成功するが、
            // ResultNotBrokenになるとFlush中の試験条件を満たしていないので
            // WaitTime_InProgressSleepTimeのタイミングを見直して再試験してください。
            // SIGLO-49011 の問題があったためOK動作についての確認はできていません。
            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Restore(g_CurrentDeviceHandleMulti[deviceNum]));
            NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Mount(g_CurrentDeviceHandleMulti[deviceNum], nn::nfp::ModelType_Amiibo));
        }
        else if (std::strcmp(nnt::nfp::GetNfpResultTypeString(result), "ResultSuccess") == 0)
        {
            NN_LOG("Unexpect ERROR = %s handle = %d SuccessCount = %d\n", nnt::nfp::GetNfpResultTypeString(result), g_CurrentDeviceHandleMulti[deviceNum], g_SuccessDeviceCount);
        }
        else
        {
            NN_LOG("Unexpect error!! %s\n", nnt::nfp::GetNfpResultTypeString(result));
            EXPECT_TRUE(false);
        }
        nn::nfp::RegisterInfoPrivate afterRegInfo;
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetRegisterInfo(&afterRegInfo, g_CurrentDeviceHandleMulti[deviceNum]));
#if defined(NNT_NFP_LIB_MII_ENABLE)
        EXPECT_TRUE(std::memcmp(&afterRegInfo.miiData, &regInfoSet.miiData,
                regInfoSet.miiData.StructureSize - ExceptDataSize) == 0);
#endif // defined(NNT_NFP_LIB_MII_ENABLE)
        EXPECT_TRUE(std::memcmp(regInfoSet.nickname, afterRegInfo.nickname, sizeof(afterRegInfo.nickname)) == 0);
        EXPECT_TRUE(std::memcmp(&regInfoSet.fontRegion, &afterRegInfo.fontRegion, sizeof(afterRegInfo.fontRegion)) == 0);

        nn::Bit8 afterAppArea[nn::nfp::ApplicationAreaSizeV2];
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::OpenApplicationArea(g_CurrentDeviceHandleMulti[deviceNum], nnt::nfp::NormalTagId));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetApplicationArea(afterAppArea, g_CurrentDeviceHandleMulti[deviceNum],
                                                                        sizeof(afterAppArea)));
        EXPECT_TRUE(std::memcmp(tagDataSet, afterAppArea, sizeof(afterAppArea)) == 0);

        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Unmount(g_CurrentDeviceHandleMulti[deviceNum]));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::StopDetection(g_CurrentDeviceHandleMulti[deviceNum]));
        if (!WaitForDeactivate(WaitTimeout, g_MultiDeactivateEvent[deviceNum]))
        {
            NN_LOG("StopDetection err deviceNum=%d\n", deviceNum);
            EXPECT_TRUE(false);
            return;
        }
        DestroyDefaultActivateEvent(g_MultiActivateEvent[deviceNum]);
        DestroyDefaultDeactivateEvent(g_MultiDeactivateEvent[deviceNum]);
    }// NOLINT(impl/function_size)

    void MultiDevicePrepareEventThread(void* pArg) NN_NOEXCEPT
    {
        NN_LOG("MultiDevicePrepareEventThread start\n");
        eventthreadinfo* threaditem = static_cast<eventthreadinfo*>(pArg);
        nn::os::MultiWaitType multiWait;
        nn::os::MultiWaitHolderType multiDevicePrepareHolder[DeviceCountMax];

        // 多重待ちオブジェクトリストを構築
        nn::os::InitializeMultiWait( &multiWait );
        for(int i = 0; i < DeviceCountMax; i++)
        {
            nn::os::InitializeMultiWaitHolder(&multiDevicePrepareHolder[i], &g_MultiDevicePrepareEvent[i]);
            nn::os::LinkMultiWaitHolder(&multiWait, &multiDevicePrepareHolder[i]);
        }

        int prepareDeviceCount = 0;
        int prepareCount = 0;
        while(prepareDeviceCount != DeviceCountMax)
        {
            nn::os::MultiWaitHolderType* pSignaledHolder;
            pSignaledHolder = nn::os::TimedWaitAny(&multiWait,
                                                   nn::TimeSpan::FromMilliSeconds(WaitMilliSeconds));

            if (pSignaledHolder == &multiDevicePrepareHolder[0])
            {
                NN_LOG("****** Device 0 OK ******\n");
                nn::os::ClearEvent(&g_MultiDevicePrepareEvent[0]);
                prepareDeviceCount++;
            }
            else if (pSignaledHolder == &multiDevicePrepareHolder[1])
            {
                NN_LOG("****** Device 1 OK ******\n");
                nn::os::ClearEvent(&g_MultiDevicePrepareEvent[1]);
                prepareDeviceCount++;
            }
            else if (pSignaledHolder == &multiDevicePrepareHolder[2])
            {
                NN_LOG("****** Device 2 OK ******\n");
                nn::os::ClearEvent(&g_MultiDevicePrepareEvent[2]);
                prepareDeviceCount++;
            }
            else if (pSignaledHolder == &multiDevicePrepareHolder[3])
            {
                NN_LOG("****** Device 3 OK ******\n");
                nn::os::ClearEvent(&g_MultiDevicePrepareEvent[3]);
                prepareDeviceCount++;
            }
            else
            {
                NN_LOG("*** Now Preparing *** %d\r", prepareCount);
                prepareCount++;
            }
        }

        // リストから外す
        for(int i = 0; i < DeviceCountMax; i++)
        {
            nn::os::UnlinkMultiWaitHolder(&multiDevicePrepareHolder[i]);
        }
        // ファイナライズ
        for(int i = 0; i < DeviceCountMax; i++)
        {
            nn::os::FinalizeMultiWaitHolder(&multiDevicePrepareHolder[i]);
        }
        nn::os::FinalizeMultiWait( &multiWait );

        // タグ切断試験用
        if(!threaditem->wifiOff)
        {
            NN_LOG("==================================================\n");
            NN_LOG(" PLEASE NFP Tag Drop Ready... 5 ");
            nnt::nfp::Sleep(WaitTime_CountDown);
            NN_LOG(" 4 ");
            nnt::nfp::Sleep(WaitTime_CountDown);
            NN_LOG(" 3 ");
            nnt::nfp::Sleep(WaitTime_CountDown);
            NN_LOG(" 2 ");
            nnt::nfp::Sleep(WaitTime_CountDown);
            NN_LOG(" 1 ");
            nnt::nfp::Sleep(WaitTime_CountDown);
            NN_LOG(" NOW!! \n");
            NN_LOG("==================================================\n");
        }

        // すべてのデバイスで準備がそろったのでシグナルを発行する
        nn::os::SignalEvent(&g_MultiDeviceStartEvent);

        // nfc off試験用
        if(threaditem->wifiOff)
        {
            nnt::nfp::Sleep(WaitTime_InProgressSleepTime);

            // Nfcを無効にする
            NN_LOG("# WifiOFF\n");
            nnt::nfp::WifiOff();
            g_NfcState = false;
            NN_LOG("# WifiOFF\n");
        }

        NN_LOG("MultiDevicePrepareEventThread end\n");
    }

    void SystemEventThread(void* pArg) NN_NOEXCEPT
    {
        NN_UNUSED(pArg);
        NN_LOG("SystemEventThread start\n");

        nn::os::MultiWaitType multiWait;
        nn::os::MultiWaitHolderType activeHolder[DeviceCountMax];
        nn::os::MultiWaitHolderType deactiveHolder[DeviceCountMax];

        // 多重待ちオブジェクトリストを構築
        nn::os::InitializeMultiWait( &multiWait );
        for (int i = 0; i < DeviceCountMax; i++)
        {
            nn::os::InitializeMultiWaitHolder(&activeHolder[i], &g_MultiActivateEvent[i]);
            nn::os::LinkMultiWaitHolder(&multiWait, &activeHolder[i]);
            nn::os::InitializeMultiWaitHolder(&deactiveHolder[i], &g_MultiDeactivateEvent[i]);
            nn::os::LinkMultiWaitHolder(&multiWait, &deactiveHolder[i]);
        }

        while(g_IsRunning)
        {
            nn::os::MultiWaitHolderType* pSignaledHolder;
            pSignaledHolder = nn::os::TimedWaitAny(&multiWait,
                                                   nn::TimeSpan::FromMilliSeconds(WaitMilliSeconds));
            for(int i = 0; i < DeviceCountMax; i++)
            {
                if (pSignaledHolder == &deactiveHolder[i])
                {
                    NN_LOG("*** Deactivate Event [%d] \n", i);
                    nn::os::ClearSystemEvent(&g_MultiDeactivateEvent[i]);
                }
                else if (pSignaledHolder == &activeHolder[i])
                {
                    NN_LOG("*** Activate Event [%d] \n", i);
                    nn::os::ClearSystemEvent(&g_MultiActivateEvent[i]);
                }
            }
        }

        for (int i = 0; i < DeviceCountMax; i++)
        {
            // リストから外す
            nn::os::UnlinkMultiWaitHolder( &activeHolder[i] );
            nn::os::UnlinkMultiWaitHolder( &deactiveHolder[i] );

            // ファイナライズ
            nn::os::FinalizeMultiWaitHolder( &activeHolder[i] );
            nn::os::FinalizeMultiWaitHolder( &deactiveHolder[i] );
        }
        nn::os::FinalizeMultiWait( &multiWait );
        for (int i = 0; i < DeviceCountMax; i++)
        {
            nn::os::DestroySystemEvent(&g_MultiDeactivateEvent[i]);
            nn::os::DestroySystemEvent(&g_MultiActivateEvent[i]);
        }
        NN_LOG("SystemEventThread end\n");
    }

    void StartTestThread(nn::os::ThreadFunction threadFunction)
    {
        // NFP初期化
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::InitializeSystemWithRetry());
        // 4つのデバイスを取得
        int outCount;
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ListDevices(
                g_CurrentDeviceHandleMulti, &outCount, DeviceCountMax));
        EXPECT_TRUE( outCount == DeviceCountMax);

        if(threadFunction == MultiDeviceNfcOffTagTestThread)
        {
            // イベント監視
            for(int i = 0; i < DeviceCountMax; i++)
            {
                NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::AttachActivateEvent(
                        &g_MultiActivateEvent[i], g_CurrentDeviceHandleMulti[i]));
                NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::AttachDeactivateEvent(
                        &g_MultiDeactivateEvent[i], g_CurrentDeviceHandleMulti[i]));
            }
            NNT_EXPECT_RESULT_SUCCESS(nn::os::CreateThread(
                    &g_SystemEventThread, SystemEventThread, nullptr, g_SystemEventThreadStack,
                    sizeof(g_SystemEventThreadStack), nn::os::DefaultThreadPriority));
            nn::os::StartThread(&g_SystemEventThread);
        }

        if(outCount != DeviceCountMax)
        {
            NN_LOG("========================================================================\n");
            NN_LOG(" Please Check to Connect 4 Devices\n");
            NN_LOG("========================================================================\n");
            // 4台のデバイスが取得できなかったのでアサートさせる
            ASSERT_TRUE(false);
        }
        for(int i = 0; i < outCount; i++)
        {
            NNT_EXPECT_RESULT_SUCCESS(nn::os::CreateThread(
                    &g_DeviceThread[i], threadFunction, &g_threadParam[i], g_DeviceThreadStack[i],
                    sizeof(g_DeviceThreadStack[i]), nn::os::DefaultThreadPriority));
            // コアを指定する
            // コアは0～2までの3つが指定できるので4つ目のスレッドには0を指定する
            int idealCore = i;
            if (idealCore >= MaxCore)
            {
                idealCore = 0;
            }
            nn::os::SetThreadCoreMask(&g_DeviceThread[i], idealCore, 0x1LL << idealCore);
        }
        for(int i = 0; i < outCount; i++)
        {
            nn::os::StartThread(&g_DeviceThread[i]);
        }
    }

    void CleanUpTestThread()
    {
        for(int i = 0; i < DeviceCountMax; i++)
        {
            nn::os::WaitThread(&g_DeviceThread[i]);
            nn::os::DestroyThread(&g_DeviceThread[i]);
        }
    }
} // end of anonymous namespace

//================================================================================
// 全プラットフォームで共通のテストスイートです。
// 必ずタグを設置してからテストを開始してください。
//================================================================================

class NfpManual : public nnt::nfp::TestFramework
{
protected:

    NfpManual() NN_NOEXCEPT
    {
        // oeライブラリを初期化します。実機のみ
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        nn::oe::Initialize();
#endif //defined(NN_BUILD_CONFIG_OS_HORIZON)
        // コントローラの初期化
        nnt::nfp::InitializeHidController();
        // アライメントを調整します。
        g_pBackupData = static_cast<nn::Bit8*>(nnt::nfp::AlignBuffer(g_BackupBuffer, nnt::nfp::IpcAlignment));
#if !defined(NNT_NFP_PLATFORM_NX) //NX では必要としない
        nn::applet::SetSleepQueryCallback(SleepQueryCallback, 0);
        nn::applet::SetAwakeCallback(AwakeCallback, 0);
        nn::applet::Enable();
#endif //!defined(NNT_NFP_PLATFORM_NX)
    }

    ~NfpManual() NN_NOEXCEPT
    {
#if !defined(NNT_NFP_PLATFORM_NX) //NX では必要としない
        nn::applet::DisableSleep();
        nn::applet::SetSleepQueryCallback(nullptr, 0);
        nn::applet::SetAwakeCallback(nullptr, 0);
#endif //!defined(NNT_NFP_PLATFORM_NX)
    }

    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
        nnt::nfp::WifiOn();
    }

    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
    {
        // 次のテストに影響が出ないようにライブラリを一旦終了しておきます。
        nnt::nfp::FinalizeSystem();
    }

};

//================================================================================
// テストケースの実装です。
//================================================================================
#if !defined(NNT_NFP_PLATFORM_NX)
TEST_F(NfpManual,TestCaseResultInvalidOperationWhenDeactive)
#else
TEST_F(NfpManual,TestCaseResultInvalidDeviceStateWhenDeactive)
#endif // !defined(NNT_NFP_PLATFORM_NX)
{
    nn::Bit8  randNumSeq[nn::nfp::ApplicationAreaSizeV2] = {};
    nn::Bit8  readNumSeq[nn::nfp::ApplicationAreaSizeV2] = {};

    nn::nfp::AdminInfo                  adminInfo;
    nn::nfp::ApplicationAreaCreateInfo  createInfo;
#if !defined(NNT_NFP_PLATFORM_NX)
    nn::nfp::BackupEntryInfo            entryInfo;
    nn::nfp::BackupHeaderInfo           headerInfo;
#else
    nn::nfp::BackupDataHeader           headerInfo;
#endif // !defined(NNT_NFP_PLATFORM_NX)
#if !defined(NNT_NFP_PLATFORM_NX) // NXではConnectは実装されない
    nn::nfp::ConnectionStatus           connectStatus;
#endif // !defined(NNT_NFP_PLATFORM_NX)
    nn::nfp::RegisterInfoPrivate        regInfoSet;
    nn::nfp::RegisterInfo               regInfo;
    nn::nfp::RegisterInfoPrivate        regInfoPrivate;
    nn::nfp::ModelInfo                  modelInfo;
    nn::nfp::CommonInfo                 commonInfo;
    nn::nfp::TagInfo                    tagInfo;
    nn::os::SystemEventType             activateEvent;
    nn::os::SystemEventType             deactivateEvent;
#if !defined(NNT_NFP_PLATFORM_NX) // NXではConnectは実装されないので使用しない
    nn::os::Event                       fakeEvent;
#endif // !defined(NNT_NFP_PLATFORM_NX)
    nn::Result                          result;

    nnt::nfp::CreateRandomNumberSequence(randNumSeq, nn::nfp::ApplicationAreaSizeV2);

    // テスト前処理
    NN_LOG("==================================================\n");
    NN_LOG(" PLEASE ATTACH a TAG!!\n");
    NN_LOG("==================================================\n");
    nnt::nfp::DoActivate();
    NN_LOG(" TAG Attached !!\n");

    nnt::nfp::DoCreateZeroTag();

#if !defined(NNT_NFP_PLATFORM_NX) // NXではAPIが削除されている
    NNT_NFP_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::InitializeRegisterInfoSet(&regInfoSet));
#else
    std::memcpy(regInfoSet.nickname, nnt::nfp::NormalTagNickName, sizeof(nnt::nfp::NormalTagNickName));
    regInfoSet.fontRegion  = 0x01;
#if defined(NNT_NFP_LIB_VERSION_BETA)
    #if defined(NNT_NFP_LIB_MII_ENABLE)
    nnt::nfp::BuildMiiData(regInfoSet.miiData);
    #endif // defined(NNT_NFP_LIB_MII_ENABLE)
#endif // !defined(NNT_NFP_LIB_VERSION_BETA)
#endif // !defined(NNT_NFP_PLATFORM_NX)
    NNT_NFP_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::SetRegisterInfo(regInfoSet));
    NNT_NFP_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetRegisterInfo(&regInfo));
    NNT_NFP_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetRegisterInfo(&regInfoPrivate));
    NNT_NFP_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetTagInfo(&tagInfo));
    NNT_NFP_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetCommonInfo(&commonInfo));
    NNT_NFP_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAdminInfo(&adminInfo));
    UnmountNfp();

    /***** RW_DEACTIVE 状態 *****/   //TODO:RW_DEACTIVE 状態に遷移するにはタグをFANGATEから離す必要がある。
#if !defined(NNT_NFP_PLATFORM_NX)
    EXPECT_EQ(nn::nfp::RW_DEACTIVE, nnt::nfp::wrapper::GetDeviceState());
#else
    EXPECT_EQ(nn::nfp::DeviceState_Deactive, nnt::nfp::wrapper::GetDeviceState() );
    EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState() );
#endif
#if !defined(NNT_NFP_PLATFORM_NX) // NXでは返り値を持たない
    NNT_NFP_EXPECT_RESULT_EQUAL(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE(), nnt::nfp::InitializeSystem());
#else
    nnt::nfp::InitializeSystem();
#endif // !defined(NNT_NFP_PLATFORM_NX)
    NNT_NFP_EXPECT_RESULT_SUCCESS(nnt::nfp::FinalizeSystem());
    UnmountNfp();

#if !defined(NNT_NFP_PLATFORM_NX)
    EXPECT_EQ(nn::nfp::RW_DEACTIVE, nnt::nfp::wrapper::GetDeviceState());
#else
    EXPECT_EQ(nn::nfp::DeviceState_Deactive, nnt::nfp::wrapper::GetDeviceState() );
    EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState() );
#endif

#if !defined(NNT_NFP_PLATFORM_NX) // NXではConnectは実装されない
    if(nn::os::IsRunOnSnake() == false)
    {
        uint32_t sent=0, total=0;
        NNT_NFP_EXPECT_RESULT_EQUAL(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE(), nnt::nfp::wrapper::Connect() );
        NNT_NFP_EXPECT_RESULT_SUCCESS( nnt::nfp::wrapper::GetConnectResult(&result) );
        NNT_NFP_EXPECT_RESULT_SUCCESS( nnt::nfp::wrapper::Disconnect() );

        UnmountNfp();
        EXPECT_EQ_FATAL(nn::nfp::RW_DEACTIVE, nnt::nfp::wrapper::GetDeviceState());

        NNT_NFP_EXPECT_RESULT_EQUAL(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE(), nnt::nfp::wrapper::ConnectForFwUpdate() );
        NNT_NFP_EXPECT_RESULT_EQUAL(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE(), nnt::nfp::wrapper::UpdateTargetFirmware(&fakeEvent) );
        NNT_NFP_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetUpdateResult(&result) );
        NNT_NFP_EXPECT_RESULT_EQUAL(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE(), nnt::nfp::wrapper::GetUpdateProgress(&sent, &total) );
    }
#endif // !defined(NNT_NFP_PLATFORM_NX)

    NNT_NFP_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::StartDetection() );
    UnmountNfp();
#if !defined(NNT_NFP_PLATFORM_NX)
    EXPECT_EQ(nn::nfp::RW_DEACTIVE, nnt::nfp::wrapper::GetDeviceState());
#else
    EXPECT_EQ(nn::nfp::DeviceState_Deactive, nnt::nfp::wrapper::GetDeviceState() );
    EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState() );
#endif
    NNT_NFP_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::StopDetection());
    UnmountNfp();
#if !defined(NNT_NFP_PLATFORM_NX)
    EXPECT_EQ(nn::nfp::RW_DEACTIVE, nnt::nfp::wrapper::GetDeviceState());
#else
    EXPECT_EQ(nn::nfp::DeviceState_Deactive, nnt::nfp::wrapper::GetDeviceState() );
    EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState() );
#endif

#if !defined(NNT_NFP_PLATFORM_NX) // NXではAPIが削除されている
    NNT_NFP_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::InitializeCreateInfo(&createInfo));
    NNT_NFP_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::InitializeRegisterInfoSet(&regInfoSet));
#endif // !defined(NNT_NFP_PLATFORM_NX)

    nn::Bit8 tagData[nn::nfp::ApplicationAreaSizeV2] = {};
    std::memcpy(tagData,nnt::nfp::ZeroTagData,sizeof(nnt::nfp::ZeroTagData));
    createInfo.accessId        = nnt::nfp::ZeroTagId;
    createInfo.initialDataSize = sizeof(nnt::nfp::ZeroTagData);
    createInfo.pInitialData    = tagData;

    NNT_NFP_EXPECT_RESULT_EQUAL(nn::nfp::ResultTagNotFound(), nnt::nfp::wrapper::Mount());
    NNT_NFP_EXPECT_RESULT_EQUAL(nn::nfp::ResultTagNotFound(), nnt::nfp::wrapper::MountRom());
    NNT_NFP_EXPECT_RESULT_EQUAL(nn::nfp::ResultTagNotFound(), nnt::nfp::wrapper::Unmount());
    NNT_NFP_EXPECT_RESULT_EQUAL(nn::nfp::ResultTagNotFound(),
                                nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::ZeroTagId));
    NNT_NFP_EXPECT_RESULT_EQUAL(nn::nfp::ResultTagNotFound(),
                                nnt::nfp::wrapper::GetApplicationArea(readNumSeq,
                                                                      nn::nfp::ApplicationAreaSizeV2));
    NNT_NFP_EXPECT_RESULT_EQUAL(nn::nfp::ResultTagNotFound(),
                                nnt::nfp::wrapper::SetApplicationArea(randNumSeq,
                                                                      nn::nfp::ApplicationAreaSizeV2,
                                                                      tagInfo.tagId));
#if defined(NNT_NFP_PLATFORM_NX)
    NNT_NFP_EXPECT_RESULT_EQUAL(nn::nfp::ResultTagNotFound(),
                                nnt::nfp::wrapper::RecreateApplicationArea(createInfo));
#endif // defined(NNT_NFP_PLATFORM_NX)
    NNT_NFP_EXPECT_RESULT_EQUAL(nn::nfp::ResultTagNotFound(), nnt::nfp::wrapper::Flush());
#if defined(NNT_NFP_LIB_VERSION_ALPHA)
    NNT_NFP_EXPECT_RESULT_EQUAL(nn::nfp::ResultTagNotFound(), nnt::nfp::wrapper::Restore());
#endif // defined(NNT_NFP_LIB_VERSION_ALPHA)
    NNT_NFP_EXPECT_RESULT_EQUAL(nn::nfp::ResultTagNotFound(),
                                nnt::nfp::wrapper::CreateApplicationArea(createInfo));
    NNT_NFP_EXPECT_RESULT_EQUAL(nn::nfp::ResultTagNotFound(),
                                nnt::nfp::wrapper::GetTagInfo(&tagInfo));
    NNT_NFP_EXPECT_RESULT_EQUAL(nn::nfp::ResultTagNotFound(),
                                nnt::nfp::wrapper::GetRegisterInfo(&regInfo));
    NNT_NFP_EXPECT_RESULT_EQUAL(nn::nfp::ResultTagNotFound(),
                                nnt::nfp::wrapper::GetRegisterInfo(&regInfoPrivate));
    NNT_NFP_EXPECT_RESULT_EQUAL(nn::nfp::ResultTagNotFound(),
                                nnt::nfp::wrapper::GetCommonInfo(&commonInfo));
    NNT_NFP_EXPECT_RESULT_EQUAL(nn::nfp::ResultTagNotFound(),
                                nnt::nfp::wrapper::GetModelInfo(&modelInfo));
#if !defined(NNT_NFP_PLATFORM_NX)
    EXPECT_EQ(nn::nfp::RW_DEACTIVE, nnt::nfp::wrapper::GetDeviceState());
#else
    EXPECT_EQ(nn::nfp::DeviceState_Deactive, nnt::nfp::wrapper::GetDeviceState() );
    EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState() );
#endif
#if !defined(NNT_NFP_PLATFORM_NX) // NXではDeactive状態でも成功する
    NNT_NFP_EXPECT_RESULT_EQUAL(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE(),
                                nnt::nfp::wrapper::AttachActivateEvent(&activateEvent));
    NNT_NFP_EXPECT_RESULT_EQUAL(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE(),
                                nnt::nfp::wrapper::AttachDeactivateEvent(&deactivateEvent));
#else
    NNT_NFP_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::AttachActivateEvent(&activateEvent));
    NNT_NFP_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::AttachDeactivateEvent(&deactivateEvent));
#endif // !defined(NNT_NFP_PLATFORM_NX)

#if !defined(NNT_NFP_PLATFORM_NX) // NXではConnectは実装されない
    if(nn::os::IsRunOnSnake() == false)
    {
        NNT_NFP_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetConnectionStatus(&connectStatus));
    }
#endif // !defined(NNT_NFP_PLATFORM_NX)

    NNT_NFP_EXPECT_RESULT_EQUAL(nn::nfp::ResultTagNotFound(),
                                nnt::nfp::wrapper::Format(nnt::nfp::ZeroTagData,
                                                          nn::nfp::ApplicationAreaSizeV2));
    NNT_NFP_EXPECT_RESULT_EQUAL(nn::nfp::ResultTagNotFound(),
                                nnt::nfp::wrapper::GetAdminInfo(&adminInfo));
    NNT_NFP_EXPECT_RESULT_EQUAL(nn::nfp::ResultTagNotFound(),
                                nnt::nfp::wrapper::SetRegisterInfo(regInfoSet));
    NNT_NFP_EXPECT_RESULT_EQUAL(nn::nfp::ResultTagNotFound(),
                                nnt::nfp::wrapper::DeleteRegisterInfo());
    NNT_NFP_EXPECT_RESULT_EQUAL(nn::nfp::ResultTagNotFound(),
                                nnt::nfp::wrapper::DeleteApplicationArea());
    bool outValue = false;
    NNT_NFP_EXPECT_RESULT_EQUAL(nn::nfp::ResultTagNotFound(),
                                nnt::nfp::wrapper::ExistsApplicationArea(&outValue));
#if !defined(NNT_NFP_PLATFORM_NX)
    EXPECT_TRUE(nnt::nfp::wrapper::GetBackupSaveDataSize() != 0);
    NNT_NFP_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAllBackupSaveData(g_pBackupData, BackupBufferSize));
    NNT_NFP_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetBackupEntryFromMemory(&entryInfo, 0, g_pBackupData, BackupBufferSize));
    NNT_NFP_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetBackupHeaderFromMemory(&headerInfo, g_pBackupData, BackupBufferSize));
#else
    EXPECT_TRUE(0 < sizeof(nn::nfp::BackupData));
    EXPECT_TRUE(sizeof(nn::nfp::BackupData) < BackupBufferSize);
    size_t dataSize;
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ReadBackupData(g_pBackupData, &dataSize, BackupBufferSize));
    EXPECT_EQ(sizeof(nn::nfp::BackupData), dataSize);
    std::memcpy(&headerInfo, &(reinterpret_cast<nn::nfp::BackupData*>(g_pBackupData)->header), sizeof(headerInfo));
    EXPECT_TRUE(0 < headerInfo.entryNum);
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::WriteBackupData(g_pBackupData, sizeof(nn::nfp::BackupData)));
#endif // !defined(NNT_NFP_PLATFORM_NX)

    NNT_NFP_EXPECT_RESULT_SUCCESS(nnt::nfp::FinalizeSystem());
} // NOLINT(impl/function_size)

TEST_F(NfpManual,TestCaseStateTransWithEventSignal)
{
    struct TestCaseProc
    {
        const char* pStateName;
        const char* pEventName;
        void (*pFuncTestInitial)();
        void (*pFuncTestMain)();
        void (*pFuncTestFinal)();

    };

    enum StateKind
    {
#if !defined(NNT_NFP_PLATFORM_NX) // NXではConnectは実装されない
        StateKind_InitDisconnect  = 0,
        StateKind_InitConnecting,
        StateKind_InitConnect,
        StateKind_InitConnectUp,
        StateKind_Search,
#else
        StateKind_Search = 0,
#endif // !defined(NNT_NFP_PLATFORM_NX)
        StateKind_Deactive,
        StateKind_Active,
        StateKind_Mount,
        StateKind_MountRom,
        StateKind_Max
    };


    enum EventKind
    {
#if defined(NNT_NFP_LIB_VERSION_BETA)
        EventKind_WifiOff  = 0,
        EventKind_BluetoothCut,
#else
        EventKind_BluetoothCut = 0,
#endif // defined(NNT_NFP_LIB_VERSION_BETA)
        EventKind_Finalize,
#if !defined(NNT_NFP_PLATFORM_NX) // NXではConnectは実装されない
        EventKind_Disconnect,
#endif // !defined(NNT_NFP_PLATFORM_NX)
        EventKind_CloseScreen,
        EventKind_StopDetection,
        EventKind_Max
    };

    static const TestCaseProc testCaseTbl[StateKind_Max][EventKind_Max] =
    {
#if !defined(NNT_NFP_PLATFORM_NX) // NXではConnectは実装されない
        {
            //  StateName               EventName           Before Test                     Main                        After
            {   "INIT(DISCONNECTED)",   "Wifi OFF",         TestInitTransInitDisconnect,   TestMainWifiOff1,          TestFinalizeWifiOff         },
            {   "INIT(DISCONNECTED)",   "BT Off",           TestInitTransInitDisconnect,   TestMainBluetoothCut1,     TestFinalizeBluetoothCut    },
            {   "INIT(DISCONNECTED)",   "Finalize",         TestInitTransInitDisconnect,   TestMainFinalize1,         TestFinalizeNop             },
            {   "INIT(DISCONNECTED)",   "Disconnect",       TestInitTransInitDisconnect,   TestMainDisconnect1,       TestFinalizeDefault         },
            {   "INIT(DISCONNECTED)",   "Close Screen",     TestInitTransInitDisconnect,   TestMainCloseScreen1,      TestFinalizeCloseScreen     },
            {   "INIT(DISCONNECTED)",   "Stop Detection",   nullptr,                       nullptr,                   nullptr                     },
        },

        /************************************************** INIT(CONNECTING) ****************************************************/
        {
            //  StateName               EventName           Before Test                     Main                        After
            {   "INIT(CONNECTING)",     "Wifi OFF",         TestInitTransInitConnecting,   TestMainWifiOff1,          TestFinalizeWifiOff         },
            {   "INIT(CONNECTING)",     "BT Off",           TestInitTransInitConnecting,   TestMainBluetoothCut1,     TestFinalizeBluetoothCut    },
            {   "INIT(CONNECTING)",     "Finalize",         TestInitTransInitConnecting,   TestMainFinalize1,         TestFinalizeNop             },
            {   "INIT(CONNECTING)",     "Disconnect",       TestInitTransInitConnecting,   TestMainDisconnect1,       TestFinalizeDefault         },
            {   "INIT(CONNECTING)",     "Close Screen",     TestInitTransInitConnecting,   TestMainCloseScreen1,      TestFinalizeCloseScreen     },
            {   "INIT(CONNECTING)",     "Stop Detection",   nullptr,                       nullptr,                   nullptr                     },
        },

        /************************************************** INIT(CONNECTED) *****************************************************/
        {
            //  StateName               EventName           Before Test                     Main                        After
            {   "INIT(CONNECTED)",      "Wifi OFF",         TestInitTransInitConnect,      TestMainWifiOff1,          TestFinalizeWifiOff         },
            {   "INIT(CONNECTED)",      "BT Off",           TestInitTransInitConnect,      TestMainBluetoothCut1,     TestFinalizeBluetoothCut    },
            {   "INIT(CONNECTED)",      "Finalize",         TestInitTransInitConnect,      TestMainFinalize1,         TestFinalizeNop             },
            {   "INIT(CONNECTED)",      "Disconnect",       TestInitTransInitConnect,      TestMainDisconnect1,       TestFinalizeDefault         },
            {   "INIT(CONNECTED)",      "Close Screen",     TestInitTransInitConnect,      TestMainCloseScreen1,      TestFinalizeCloseScreen     },
            {   "INIT(CONNECTED)",      "Stop Detection",   nullptr,                       nullptr,                   nullptr                     },
        },

        /************************************************** INIT(CONNECTED_UP) **************************************************/
        {
            //  StateName               EventName           Before Test                     Main                        After
            {   "INIT(CONNECTED_UP)",   "Wifi OFF",         TestInitTransInitConnectUp,    TestMainWifiOff1,          TestFinalizeWifiOff         },
            {   "INIT(CONNECTED_UP)",   "BT Off",           TestInitTransInitConnectUp,    TestMainBluetoothCut1,     TestFinalizeBluetoothCut    },
            {   "INIT(CONNECTED_UP)",   "Finalize",         TestInitTransInitConnectUp,    TestMainFinalize1,         TestFinalizeNop             },
            {   "INIT(CONNECTED_UP)",   "Disconnect",       TestInitTransInitConnectUp,    TestMainDisconnect1,       TestFinalizeDefault         },
            {   "INIT(CONNECTED_UP)",   "Close Screen",     TestInitTransInitConnectUp,    TestMainCloseScreen1,      TestFinalizeCloseScreen     },
            {   "INIT(CONNECTED_UP)"    "Stop Detection",   nullptr,                       nullptr,                   nullptr                     },
        },
#endif // !defined(NNT_NFP_PLATFORM_NX)

        /************************************************** SEARCH **************************************************************/
        {
            //  StateName               EventName           Before Test                     Main                        After
#if defined(NNT_NFP_LIB_VERSION_BETA)
            {   "SEARCH",               "Wifi OFF",         TestInitTransSearch,           TestMainWifiOff1,          TestFinalizeWifiOff         },
#endif // defined(NNT_NFP_LIB_VERSION_BETA)
            {   "SEARCH",               "BT Off",           TestInitTransSearch,           TestMainBluetoothCut1,     TestFinalizeBluetoothCut    },
            {   "SEARCH",               "Finalize",         TestInitTransSearch,           TestMainFinalize1,         TestFinalizeNop             },
#if !defined(NNT_NFP_PLATFORM_NX) // NXではConnectは実装されない
            {   "SEARCH",               "Disconnect",       TestInitTransSearch,           TestMainDisconnect1,       TestFinalizeDefault         },
#endif // !defined(NNT_NFP_PLATFORM_NX)
            {   "SEARCH",               "Close Screen",     TestInitTransSearch,           TestMainCloseScreen1,      TestFinalizeCloseScreen     },
            {   "SEARCH",               "Stop Detection",   TestInitTransSearch,           TestMainStopDetection1,    TestFinalizeDefault         },
        },

        /************************************************** DEACTIVE ************************************************************/
        {
            //  StateName               EventName           Before Test                     Main                        After
#if defined(NNT_NFP_LIB_VERSION_BETA)
            {   "DEACTIVE",             "Wifi OFF",         TestInitTransDeactive,         TestMainWifiOff1,          TestFinalizeWifiOff         },
#endif // defined(NNT_NFP_LIB_VERSION_BETA)
            {   "DEACTIVE",             "BT Off",           TestInitTransDeactive,         TestMainBluetoothCut1,     TestFinalizeBluetoothCut    },
            {   "DEACTIVE",             "Finalize",         TestInitTransDeactive,         TestMainFinalize1,         TestFinalizeNop             },
#if !defined(NNT_NFP_PLATFORM_NX) // NXではConnectは実装されない
            {   "DEACTIVE",             "Disconnect",       TestInitTransDeactive,         TestMainDisconnect1,       TestFinalizeDefault         },
#endif // !defined(NNT_NFP_PLATFORM_NX)
            {   "DEACTIVE",             "Close Screen",     TestInitTransDeactive,         TestMainCloseScreen1,      TestFinalizeCloseScreen     },
            {   "DEACTIVE",             "Stop Detection",   TestInitTransDeactive,         TestMainStopDetection1,    TestFinalizeDefault         },
        },

        /************************************************** ACTIVE **************************************************************/
        {
            //  StateName               EventName           Before Test                     Main                        After
#if defined(NNT_NFP_LIB_VERSION_BETA)
            {   "ACTIVE",               "Wifi OFF",         TestInitTransActive,           TestMainWifiOff2,          TestFinalizeWifiOff         },
#endif // defined(NNT_NFP_LIB_VERSION_BETA)
            {   "ACTIVE",               "BT Off",           TestInitTransActive,           TestMainBluetoothCut2,     TestFinalizeBluetoothCut    },
            {   "ACTIVE",               "Finalize",         TestInitTransActive,           TestMainFinalize2,         TestFinalizeNop             },
#if !defined(NNT_NFP_PLATFORM_NX) // NXではConnectは実装されない
            {   "ACTIVE",               "Disconnect",       TestInitTransActive,           TestMainDisconnect2,       TestFinalizeDefault         },
#endif // !defined(NNT_NFP_PLATFORM_NX)
            {   "ACTIVE",               "Close Screen",     TestInitTransActive,           TestMainCloseScreen2,      TestFinalizeCloseScreen     },
            {   "ACTIVE",               "Stop Detection",   TestInitTransActive,           TestMainStopDetection2,    TestFinalizeDefault         },
        },

        /************************************************** MOUNT ***************************************************************/
        {
            //  StateName               EventName           Before Test                     Main                        After
#if defined(NNT_NFP_LIB_VERSION_BETA)
            {   "MOUNT",                "Wifi OFF",         TestInitTransMount,            TestMainWifiOff2,          TestFinalizeWifiOff         },
#endif // defined(NNT_NFP_LIB_VERSION_BETA)
            {   "MOUNT",                "BT Off",           TestInitTransMount,            TestMainBluetoothCut2,     TestFinalizeBluetoothCut    },
            {   "MOUNT",                "Finalize",         TestInitTransMount,            TestMainFinalize2,         TestFinalizeNop             },
#if !defined(NNT_NFP_PLATFORM_NX) // NXではConnectは実装されない
            {   "MOUNT",                "Disconnect",       TestInitTransMount,            TestMainDisconnect2,       TestFinalizeDefault         },
#endif // !defined(NNT_NFP_PLATFORM_NX)
            {   "MOUNT",                "Close Screen",     TestInitTransMount,            TestMainCloseScreen2,      TestFinalizeCloseScreen     },
            {   "MOUNT",                "Stop Detection",   TestInitTransMount,            TestMainStopDetection2,    TestFinalizeDefault         },
        },

        /************************************************** MOUNT_ROM ***********************************************************/
        {
            //  StateName               EventName           Before Test                     Main                        After
#if defined(NNT_NFP_LIB_VERSION_BETA)
            {   "MOUNT_ROM",            "Wifi OFF",         TestInitTransMountRom,         TestMainWifiOff2,          TestFinalizeWifiOff         },
#endif // defined(NNT_NFP_LIB_VERSION_BETA)
            {   "MOUNT_ROM",            "BT Off",           TestInitTransMountRom,         TestMainBluetoothCut2,     TestFinalizeBluetoothCut    },
            {   "MOUNT_ROM",            "Finalize",         TestInitTransMountRom,         TestMainFinalize2,         TestFinalizeNop             },
#if !defined(NNT_NFP_PLATFORM_NX) // NXではConnectは実装されない
            {   "MOUNT_ROM",            "Disconnect",       TestInitTransMountRom,         TestMainDisconnect2,       TestFinalizeDefault         },
#endif // !defined(NNT_NFP_PLATFORM_NX)
            {   "MOUNT_ROM",            "Close Screen",     TestInitTransMountRom,         TestMainCloseScreen2,      TestFinalizeCloseScreen     },
            {   "MOUNT_ROM",            "Stop Detection",   TestInitTransMountRom,         TestMainStopDetection2,    TestFinalizeDefault         },
        },
    };

    NN_LOG("==================================================\n");
    NN_LOG(" PLEASE REMOVE a TAG (Within %dms) !!             \n",WaitTime_RemoveTag);
    NN_LOG("==================================================\n");

    nnt::nfp::Sleep(WaitTime_RemoveTag);

    for(uint8_t stateCount = 0; stateCount < StateKind_Max; stateCount++)
    {
#if !defined(NNT_NFP_PLATFORM_NX) // NXではConnectは実装されない
        // SNAKE対象外の状態をスキップ
        if(nn::os::IsRunOnSnake() == true)
        {
            if( (i == StateKind_InitDisconnect) ||
                (i == StateKind_InitConnecting) ||
                (i == StateKind_InitConnectUp) )
            {   continue;   }
        }

        if(stateCount == StateKind_InitConnecting)
        {
            // CONNECTING状態確認時はBTモジュールとの通信不可前提で実施する
            NN_LOG("==================================================\n");
            NN_LOG(" PLEASE CUT BT SIGNAL (Within 10sec) !! \n");
            NN_LOG("==================================================\n");
            nnt::nfp::Sleep(10000);
        }
#endif // !defined(NNT_NFP_PLATFORM_NX)

        for(uint8_t eventCount = 0; eventCount < EventKind_Max; eventCount++)
        {
            const TestCaseProc *pTestCaseTbl =
                    &testCaseTbl[stateCount][eventCount];

            if( pTestCaseTbl->pFuncTestInitial == nullptr )
            {
                continue;
            }

#if !defined(NNT_NFP_PLATFORM_NX) // NXではConnectは実装されない
            // SNAKE対象外の試験をスキップ
            if(eventCount == EventKind_BluetoothCut || eventCount == EventKind_Disconnect)
            {
                if(nn::os::IsRunOnSnake() == true) {continue;}
            }
#endif // !defined(NNT_NFP_PLATFORM_NX)

            NN_LOG("**************************************************\n");
            NN_LOG("*   TestState : %s \n", pTestCaseTbl->pStateName );
            NN_LOG("*   TestEvent : %s \n", pTestCaseTbl->pEventName );
            NN_LOG("**************************************************\n");


            // 事前処理
            pTestCaseTbl->pFuncTestInitial();
            nnt::nfp::Sleep(WaitTime_NextStep);

            // テストケース
            pTestCaseTbl->pFuncTestMain();
            nnt::nfp::Sleep(WaitTime_NextStep);

            // あとしまつ
            pTestCaseTbl->pFuncTestFinal();
            nnt::nfp::Sleep(WaitTime_NextStep);

            NN_LOG("\n");
        }

#if !defined(NNT_NFP_PLATFORM_NX) // NXではConnectは実装されない
        if(stateCount == StateKind_InitConnecting)
        {
            // INIT_OCNNECTING 状態終了時は通信できるようにする
            NN_LOG("==================================================\n");
            NN_LOG(" PLEASE RESTORE BT MODULE (Within 10sec) !! \n");
            NN_LOG("==================================================\n");
            nnt::nfp::Sleep(10000);
        }
        else if(stateCount == StateKind_InitConnectUp)
        {
            // CONNECTED_UP状態テスト終了時はFangateFirmwareを修復する
            UpdateFangateFirmware();
        }
#endif // !defined(NNT_NFP_PLATFORM_NX)
    }
} // NOLINT(impl/function_size)

TEST_F(NfpManual,TestCaseStateTransWithEventSignalWhenFlush)
{

    struct TestCaseProc
    {
        const char* pStateName;
        const char* pEventName;
        void  (*pFuncTestInitial)();
        void  (*pFuncTestMain)();
        void  (*pFuncTestFinal)();
        int   TestRepeatCnt;
        bool  TestRepeatCheckFlg;

    };

    enum StateKind
    {
        StateKind_Flush  = 0,
        StateKind_Max
    };

    enum EventKind
    {
#if defined(NNT_NFP_LIB_VERSION_BETA)
        EventKind_WifiOff  = 0,
        EventKind_MountInProgressWifiOff,
        EventKind_FormatInProgressWifiOff,
        EventKind_CreateAppInfoInProgressWifiOff,
        EventKind_DeleteAppInfoInProgressWifiOff,
        EventKind_DeteleReginfoInProgressWifiOff,
        EventKind_FlushInProgressWifiOff,
        EventKind_RestoreInProgressWifiOff,
        EventKind_BluetoothCut,
#else
        EventKind_BluetoothCut = 0,
#endif // defined(NNT_NFP_LIB_VERSION_BETA)
        EventKind_CloseScreen,
        EventKind_Max
    };

    const int RetryCntMax = 100;

    static const TestCaseProc testCaseTbl[StateKind_Max][EventKind_Max] =
    {
        /************************************************** FLUSH ***********************************************************/
        {
            //  StateName           EventName               Before Test                            Main                       After                    TestRepeatCnt  TestRepeatCheckFlg
#if defined(NNT_NFP_LIB_VERSION_BETA)
            {   "FLUSHING",         "Wifi OFF",             TestInitTransPreFlush,                 TestMainWifiOff3,          TestFinalizeRunNfpApiThread, 1,             false },
            {   "MOUNTING",         "In Progress Wifi OFF", TestInitTransPreMount,                 TestMainWifiOffInProgress, TestFinalizeRunNfpApiThread, 1,             false },
            {   "FORMATTING",       "In Progress Wifi OFF", TestInitTransPreFormat,                TestMainWifiOffInProgress, TestFinalizeRunNfpApiThread, 1,             false },
            {   "CREATINGAPPINFO",  "In Progress Wifi OFF", TestInitTransPreCreateApplicationArea, TestMainWifiOffInProgress, TestFinalizeRunNfpApiThread, 1,             false },
            {   "DELETINGAPPINFO",  "In Progress Wifi OFF", TestInitTransPreDeleteApplicationArea, TestMainWifiOffInProgress, TestFinalizeRunNfpApiThread, 1,             false },
            {   "DELETINGREGIINFO", "In Progress Wifi OFF", TestInitTransPreDeleteRegisterInfo,    TestMainWifiOffInProgress, TestFinalizeRunNfpApiThread, 1,             false },
            {   "FLUSHING",         "In Progress Wifi OFF", TestInitTransPreFlush,                 TestMainWifiOffInProgress, TestFinalizeRunNfpApiThread, 1,             false },
            {   "RESTORING",        "In Progress Wifi OFF", TestInitTransPreRestore,               TestMainWifiOffInProgress, TestFinalizeRunNfpApiThread, 1,             false },
#endif // defined(NNT_NFP_LIB_VERSION_BETA)
            {   "FLUSHING",         "BT Off",               TestInitTransPreFlush,                 TestMainBluetoothCut3,     TestFinalizeRunNfpApiThread, RetryCntMax,   true  },
            {   "FLUSHING",   "Close Screen"  ,   TestInitTransPreFlush,   TestMainCloseScreen3,      TestFinalizeRunNfpApiThread, 1,             false },
        }
    };

    NN_LOG("==================================================\n");
    NN_LOG(" PLEASE ATTACH a TAG (Within %dms) !! \n",WaitTime_AttachTag);
    NN_LOG("==================================================\n");

    nnt::nfp::Sleep(WaitTime_AttachTag);

    for(uint8_t stateCount = 0; stateCount < StateKind_Max; stateCount++)
    {
        for(uint8_t eventCount = 0; eventCount < EventKind_Max; eventCount++)
        {
            const TestCaseProc* pTestCaseTbl =
                    &testCaseTbl[stateCount][eventCount];

            if( pTestCaseTbl->pFuncTestInitial == nullptr )
            {
                continue;
            }

#if !defined(NNT_NFP_PLATFORM_NX) // NXではスキップしない
            // SNAKE対象外の試験をスキップ
            if(eventCount == EventKind_BluetoothCut)
            {
                if(nn::os::IsRunOnSnake() == true) {continue;}
            }
#endif // !defined(NNT_NFP_PLATFORM_NX)

            NN_LOG("**************************************************\n");
            NN_LOG("*   TestState : %s \n", pTestCaseTbl->pStateName );
            NN_LOG("*   TestEvent : %s \n", pTestCaseTbl->pEventName );
            NN_LOG("**************************************************\n");

            uint8_t runCount;
            for(runCount = 0; runCount < pTestCaseTbl->TestRepeatCnt; runCount++)
            {
                // 事前処理
                pTestCaseTbl->pFuncTestInitial();
                nnt::nfp::Sleep(WaitTime_NextStep);

                // テストケース
                pTestCaseTbl->pFuncTestMain();
                nnt::nfp::Sleep(WaitTime_NextStep);

                // あとしまつ
                pTestCaseTbl->pFuncTestFinal();
                nnt::nfp::Sleep(WaitTime_NextStep);

                // BT Offの試験完了判定
                if((std::strcmp(pTestCaseTbl->pEventName, "BT Off") == 0) &&
                        (g_MountSuccesFlg == true) &&
                        (g_MountExpectFailFlg == true))
                {
                    g_MountSuccesFlg = false;
                    g_MountExpectFailFlg = false;
                    // 試験終了
                    break;
                }
            }
            // テスト実施回数が設定回数となった場合NGとなる試験の確認
            if(pTestCaseTbl->TestRepeatCheckFlg)
            {
                NN_LOG("Test %d count\n", runCount);
                EXPECT_TRUE(runCount <= pTestCaseTbl->TestRepeatCnt);
            }
            NN_LOG("\n");
        }
    }
}

#if !defined(NNT_NFP_PLATFORM_NX) // NXではFWアップデートは実装されない
TEST_F(NfpManual,TestCaseFangateFwUpdateWhenError)
{
    nn::os::Event updateEvt;
    nn::Result updateResult;

    // NFPライブラリ初期化 & WIFI ON
    NNT_NFP_EXPECT_RESULT_SUCCESS( InitializeSystemWithRetry() );
    WifiOn();

    // FANGATEファーム破壊時はGetUpdateResult結果を参照しない
    updateResult = nn::nfp::ResultFailed();
    NNT_NFP_EXPECT_RESULT_SUCCESS( nnt::nfp::wrapper::BreakTargetFirmware() );

    // ファーム更新接続
    NNT_NFP_EXPECT_RESULT_SUCCESS( nnt::nfp::wrapper::ConnectForFwUpdate() );
    NN_LOG("==================================================\n");
    NN_LOG(" Connecting to FANGATE for Update ...             \n");
    NN_LOG("==================================================\n");
    EXPECT_TRUE( WaitForConnect() == true);

    // ファーム更新要求
    NNT_NFP_EXPECT_RESULT_SUCCESS( nnt::nfp::wrapper::UpdateTargetFirmware(&updateEvt) );

    // ファーム更新中はGetUpdateResult結果は不定

    NN_LOG("==================================================\n");
    NN_LOG(" PLEASE CUT BT SIGNAL (within 10sec)          \n");
    NN_LOG("==================================================\n");

    for(int i=0; i<nnt::nfp::RETRY_COUNT_MAX; i++)
    {
        bool ret1 = updateEvt.Wait( nn::fnd::TimeSpan::FromMilliSeconds(5000) );
        if( ret1 )
        {
            bool ret2 = nnt::nfp::WaitForDisconnect();
            EXPECT_TRUE( ret2 && ret1 );
        }
    }
    NN_LOG("Firmware update aborted!! \n" );

    NN_LOG("==================================================\n");
    NN_LOG(" PLEASE RESTORE BT MODULE (within 10Sec) !!       \n");
    NN_LOG("==================================================\n");
    nnt::nfp::Sleep(10000);

    updateEvt.ClearSignal();    // 念のためSignalを消しておく。

    NN_LOG("==================================================\n");
    NN_LOG(" Connecting to FANGATE for Update ...             \n");
    NN_LOG("==================================================\n");

    // 再度ファーム更新が開始できること
    NNT_NFP_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ConnectForFwUpdate());
    EXPECT_TRUE( WaitForConnect() == true);
    NNT_NFP_EXPECT_RESULT_SUCCESS( nnt::nfp::wrapper::UpdateTargetFirmware(&updateEvt) );

    uint32_t prevSentSize = 0;
    while(1)
    {
        uint32_t sentSize=0, totalSize=0;
        NNT_NFP_EXPECT_RESULT_SUCCESS( nnt::nfp::wrapper::GetUpdateProgress(&sentSize, &totalSize) );

        // 実際にアップデート処理が始まるまでは、sentSize, totalSize ともに 0 になっている

        //  pSentSize が TotalSize 未満まで徐々に増えていくこと
        if(sentSize <= totalSize)
        {
            if( sentSize > prevSentSize )
            {
                NN_LOG("### Firmware Updating ... [ %d / %d ]  \n", sentSize, totalSize);
                prevSentSize = sentSize;

                bool ret = updateEvt.Wait( nn::fnd::TimeSpan::FromMilliSeconds(1000) );
                if(ret == true) { break; }
            }
        }
        else
        {
            EXPECT_TRUE( false );
            return;
        }
    }

    // ファーム更新完了するとResultSuccessが返されること
    updateResult = nn::nfp::ResultFailed();
    NNT_NFP_EXPECT_RESULT_SUCCESS( nnt::nfp::wrapper::GetUpdateResult(&updateResult) );
    NNT_NFP_EXPECT_RESULT_SUCCESS( updateResult );

    // ファーム正常時はResultUpdateNotRequiredを返すこと
    NNT_NFP_EXPECT_RESULT_SUCCESS( nnt::nfp::wrapper::ConnectForFwUpdate() );
    NN_LOG("==================================================\n");
    NN_LOG(" Connecting to FANGATE for Update ...             \n");
    NN_LOG("==================================================\n");
    EXPECT_TRUE( WaitForConnect(5000) == false);
    nn::Result connectResult;
    NNT_NFP_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetConnectResult(&connectResult));
    NNT_NFP_EXPECT_RESULT_EQUAL( nn::nfp::ResultUpdateNotRequired(), connectResult );
}
#endif // !defined(NNT_NFP_PLATFORM_NX)

TEST_F(NfpManual, TestCaseInvalidSizeSetApplicationAreaTest)
{
// -------------------------------------------------------------------------
// 本試験はAbortすることが期待動作の試験です。
// Mount状態で不正なサイズを指定したSetApplicationArea()を要求します。
// -------------------------------------------------------------------------
    NN_LOG("========================================================================\n");
    NN_LOG(" THIS TEST IS EXPECTED TO ABORT \n");
    NN_LOG("========================================================================\n");

    // Mount状態にする
    nnt::nfp::DoMount();
    // タグ情報を取得する
    nn::nfp::TagInfo tagInfo;
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetTagInfo(&tagInfo));
    // ----------------------------------------
    // テスト対象 : nn::nfp::SetApplicationArea()
    //              サイズオーバー指定
    // ----------------------------------------
    nn::Result result;
    result = nnt::nfp::wrapper::SetApplicationArea(nnt::nfp::ZeroTagData,
                                                           nn::nfp::ApplicationAreaSizeV2 + 1,
                                                           tagInfo.tagId);
    EXPECT_TRUE(false);
    NN_LOG("Not Expect ! %s\n", nnt::nfp::GetNfpResultTypeString(result));
}

TEST_F(NfpManual, TestCaseInvalidSizeCreateApplicationAreaTest)
{
// -------------------------------------------------------------------------
// 本試験はAbortすることが期待動作の試験です。
// Mount状態で不正なサイズを指定したCreateApplicationArea()を要求します。
// -------------------------------------------------------------------------
    NN_LOG("========================================================================\n");
    NN_LOG(" THIS TEST IS EXPECTED TO ABORT \n");
    NN_LOG("========================================================================\n");

    // Mount状態にする
    nnt::nfp::DoMount();
    // ----------------------------------------
    // テスト対象 : nn::nfp::CreateApplicationArea()
    //              サイズオーバー指定
    // ----------------------------------------
    nn::nfp::ApplicationAreaCreateInfo appAreaInfo;
    nn::Bit8 tagData[nn::nfp::ApplicationAreaSizeV2 + 1] = {};
    std::memcpy(tagData,nnt::nfp::NormalTagData,nn::nfp::ApplicationAreaSizeV2);
    appAreaInfo.pInitialData = tagData;
    appAreaInfo.initialDataSize = nn::nfp::ApplicationAreaSizeV2 + 1;
    nn::Result result;
    result = nnt::nfp::wrapper::CreateApplicationArea(appAreaInfo);
    EXPECT_TRUE(false);
    NN_LOG("Not Expect ! %s\n", nnt::nfp::GetNfpResultTypeString(result));
}

TEST_F(NfpManual, TestCaseInvalidSizeRecreateApplicationAreaTest)
{
// -------------------------------------------------------------------------
// 本試験はAbortすることが期待動作の試験です。
// Mount状態で不正なサイズを指定したRecreateApplicationArea()を要求します。
// -------------------------------------------------------------------------
    NN_LOG("========================================================================\n");
    NN_LOG(" THIS TEST IS EXPECTED TO ABORT \n");
    NN_LOG("========================================================================\n");

    // Mount状態にする
    nnt::nfp::DoMount();
    // ----------------------------------------
    // テスト対象 : nn::nfp::RecreateApplicationArea()
    //              サイズオーバー指定
    // ----------------------------------------
    nn::nfp::ApplicationAreaCreateInfo appAreaInfo;
    nn::Bit8 tagData[nn::nfp::ApplicationAreaSizeV2 + 1] = {};
    std::memcpy(tagData,nnt::nfp::NormalTagData,nn::nfp::ApplicationAreaSizeV2);
    appAreaInfo.pInitialData = tagData;
    appAreaInfo.initialDataSize = nn::nfp::ApplicationAreaSizeV2 + 1;
    nn::Result result;
    result = nnt::nfp::wrapper::RecreateApplicationArea(appAreaInfo);
    EXPECT_TRUE(false);
    NN_LOG("Not Expect ! %s\n", nnt::nfp::GetNfpResultTypeString(result));
}

TEST_F(NfpManual, TestCaseMultiDeviceAccess)
{
    NN_LOG("========================================================================\n");
    NN_LOG(" Please Pairing 4 Devices and Setting NFP Tags on All Device \n");
    NN_LOG("========================================================================\n");
    nnt::nfp::Sleep(WaitTimeout);

    StartTestThread(DeviceTestThread);
    CleanUpTestThread();
}

TEST_F(NfpManual, TestCaseOverMultiDeviceAccess)
{
    NN_LOG("========================================================================\n");
    NN_LOG(" Please Pairing 4 Devices and Setting NFP Tags on All Device \n");
    NN_LOG("========================================================================\n");
    nnt::nfp::Sleep(WaitTimeout);

    StartTestThread(DeviceDeactivateTestThread);
    NN_LOG("========================================================================\n");
    NN_LOG(" If Complete to Attach 4 Devices, Please Check to Connect 5th Device\n");
    NN_LOG("========================================================================\n");
    CleanUpTestThread();
}

TEST_F(NfpManual, TestCaseBeforeUsingOverMultiDeviceAccess)
{
    NN_LOG("========================================================================\n");
    NN_LOG(" Please Pairing 4 Right Joy-Con and Setting NFP Tags on All Device And Pairing 1 Left Joy-Con\n");
    NN_LOG("========================================================================\n");
    nnt::nfp::Sleep(WaitTimeout);
    g_ErrorCount = 0;
    g_SuccessCount = 0;

    StartTestThread(CheckOnlyOneDeviceActiveTestThread);
    CleanUpTestThread();

    EXPECT_TRUE((ErrorCount == g_ErrorCount) && (SuccessCount == g_SuccessCount));
    g_ErrorCount = 0;
    g_SuccessCount = 0;
}


TEST_F(NfpManual,TestCaseMultiDeviceStateTransWithEventSignalWhenFlush)
{
    NN_LOG("========================================================================\n");
    NN_LOG(" Please Pairing 4 Devices and Setting NFP Tags on All Device \n");
    NN_LOG("========================================================================\n");
    nnt::nfp::Sleep(WaitTimeout);

    // イベント初期化
    for(int i = 0; i < DeviceCountMax; i++)
    {
        nn::os::InitializeEvent(&g_MultiDevicePrepareEvent[i], false,  nn::os::EventClearMode_ManualClear);
        nn::os::ClearEvent(&g_MultiDevicePrepareEvent[i]);
    }
    nn::os::InitializeEvent(&g_MultiDeviceStartEvent, false,  nn::os::EventClearMode_ManualClear);
    nn::os::ClearEvent(&g_MultiDeviceStartEvent);
    // イベントハンドラ開始
    g_EventThreadinfo.wifiOff = false;
    NNT_EXPECT_RESULT_SUCCESS(nn::os::CreateThread(
            &g_MultiDevicePrepareEventThread, MultiDevicePrepareEventThread, &g_EventThreadinfo, g_MultiDevicePrepareEventThreadStack,
            sizeof(g_MultiDevicePrepareEventThreadStack), nn::os::DefaultThreadPriority));
    nn::os::StartThread(&g_MultiDevicePrepareEventThread);

    StartTestThread(MultiDeviceDisconnectTagTestThread);
    CleanUpTestThread();

    nn::os::WaitThread(&g_MultiDevicePrepareEventThread);
    nn::os::DestroyThread(&g_MultiDevicePrepareEventThread);

    // イベント破棄
    for(int i = 0; i < DeviceCountMax; i++)
    {
        nn::os::FinalizeEvent(&g_MultiDevicePrepareEvent[i]);
    }
    nn::os::FinalizeEvent(&g_MultiDeviceStartEvent);
}

TEST_F(NfpManual,TestCaseMultiDeviceStateTransWithChangeTagWhenFlush)
{
    NN_LOG("========================================================================\n");
    NN_LOG(" Please Pairing 4 Devices and Setting NFP Tags on All Device \n");
    NN_LOG("========================================================================\n");
    nnt::nfp::Sleep(WaitTimeout);

    // イベント初期化
    for(int i = 0; i < DeviceCountMax; i++)
    {
        nn::os::InitializeEvent(&g_MultiDevicePrepareEvent[i], false,  nn::os::EventClearMode_ManualClear);
        nn::os::ClearEvent(&g_MultiDevicePrepareEvent[i]);
    }
    nn::os::InitializeEvent(&g_MultiDeviceStartEvent, false,  nn::os::EventClearMode_ManualClear);
    nn::os::ClearEvent(&g_MultiDeviceStartEvent);
    // イベントハンドラ開始
    g_EventThreadinfo.wifiOff = false;
    NNT_EXPECT_RESULT_SUCCESS(nn::os::CreateThread(
            &g_MultiDevicePrepareEventThread, MultiDevicePrepareEventThread, &g_EventThreadinfo, g_MultiDevicePrepareEventThreadStack,
            sizeof(g_MultiDevicePrepareEventThreadStack), nn::os::DefaultThreadPriority));
    nn::os::StartThread(&g_MultiDevicePrepareEventThread);

    StartTestThread(MultiDeviceChangeTagTestThread);
    CleanUpTestThread();

    nn::os::WaitThread(&g_MultiDevicePrepareEventThread);
    nn::os::DestroyThread(&g_MultiDevicePrepareEventThread);

    // イベント破棄
    for(int i = 0; i < DeviceCountMax; i++)
    {
        nn::os::FinalizeEvent(&g_MultiDevicePrepareEvent[i]);
    }
    nn::os::FinalizeEvent(&g_MultiDeviceStartEvent);
}


TEST_F(NfpManual,TestCaseMultiDeviceNfcStateTransWithEventSignalWhenFlush)
{
    NN_LOG("========================================================================\n");
    NN_LOG(" Please Pairing 4 Devices and Setting NFP Tags on All Device \n");
    NN_LOG("========================================================================\n");
    nnt::nfp::Sleep(WaitTimeout);

    // イベント初期化
    for(int i = 0; i < DeviceCountMax; i++)
    {
        nn::os::InitializeEvent(&g_MultiDevicePrepareEvent[i], false,  nn::os::EventClearMode_ManualClear);
        nn::os::ClearEvent(&g_MultiDevicePrepareEvent[i]);
    }
    nn::os::InitializeEvent(&g_MultiDeviceStartEvent, false,  nn::os::EventClearMode_ManualClear);
    nn::os::ClearEvent(&g_MultiDeviceStartEvent);
    // イベントハンドラ開始
    g_EventThreadinfo.wifiOff = true;
    NNT_EXPECT_RESULT_SUCCESS(nn::os::CreateThread(
            &g_MultiDevicePrepareEventThread, MultiDevicePrepareEventThread, &g_EventThreadinfo, g_MultiDevicePrepareEventThreadStack,
            sizeof(g_MultiDevicePrepareEventThreadStack), nn::os::DefaultThreadPriority));
    nn::os::StartThread(&g_MultiDevicePrepareEventThread);

    StartTestThread(MultiDeviceNfcOffTagTestThread);
    CleanUpTestThread();

    g_IsRunning = false;
    nn::os::WaitThread(&g_SystemEventThread);
    nn::os::DestroyThread(&g_SystemEventThread);

    // イベント破棄
    for(int i = 0; i < DeviceCountMax; i++)
    {
        nn::os::FinalizeEvent(&g_MultiDevicePrepareEvent[i]);
    }
    nn::os::FinalizeEvent(&g_MultiDeviceStartEvent);
}
