﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nfp.h>
#include <nn/oe.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/os/os_Event.h>

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>

#include <nnt/nfp/testNfp_Common.h>

//================================================================================
// このテストで使用する定義です。
//================================================================================

namespace
{
    const uint32_t   AgingRepeat = 1000000;
    const uint32_t   RandomSleepTimeMax = 300;
    const char* NfpDeviceState[] =
#if !defined(NNT_NFP_PLATFORM_NX)
    {
        "NONE",         //!< NFC ライブラリが初期化されていない状態です。起動時はこの状態です。
        "INIT",         //!< NFC ライブラリが初期化されている状態です。
        "RW_SEARCH",    //!< タグを探している状態です。
        "RW_ACTIVE",    //!< RW_SEARCH でタグを検知するとこの状態になります。
        "RW_DEACTIVE",  //!< タグを検知した後タグが離れるとこの状態になります。
        "RW_MOUNT",     //!< RW_ACTIVE で近くにあるタグをマウントするとこの状態になります。
        "UNEXPECTED"    //!< 想定していない状態です。
    };
#else
    {
        "Init",          //!< タグを探していない状態です。
        "Search",        //!< タグを探している状態です。
        "Active",        //!< DeviceState_Search でタグを検知するとこの状態になります。
        "Deactive",      //!< タグを検知した後、タグが離れるとこの状態になります。
        "Mount",         //!< DeviceState_Active で近くにあるタグをマウントするとこの状態になります。
        "Unexpected"     //!< 想定していない状態です。
    };


    const char* NfpLibraryState[] =
    {
        "None",          //!< NFP ライブラリが初期化されていない状態です。起動時はこの状態です。
        "Init",          //!< NFP ライブラリが初期化されている状態です。
        "Unexpected"     //!< 想定していない状態です。
    };
#endif // !defined(NNT_NFP_PLATFORM_NX)

    nn::Bit8 g_TagData[nn::nfp::ApplicationAreaSizeV2] = {0};
    nn::Bit8 g_RandNumSeq[nn::nfp::ApplicationAreaSizeV2] = {0};
    nn::Bit8 g_ReadNumSeq[nn::nfp::ApplicationAreaSizeV2] = {0};

} // end of anonymous namespace

//------------------------------------------------------------------
// Macro Defnition
//------------------------------------------------------------------

#define NFP_LOG( ... ) \
    do { \
        NN_LOG("[NFP_AGING] LIB_STATE = %s DEV_STATE = %s LINE = %04d ",\
               NfpLibraryState[nnt::nfp::wrapper::GetState()],          \
               NfpDeviceState[nnt::nfp::wrapper::GetDeviceState()],     \
               __LINE__);                                               \
        NN_LOG(__VA_ARGS__);                                            \
    } while(NN_STATIC_CONDITION(false))

#define NNT_NFP_SWITCH_NFC_ENABLE // NfcのON／OFF切り替え機能スイッチ

//================================================================================
// テストで使用する共通関数です。
//================================================================================
namespace {
    int RandomSleep(int max)
    {
        nn::Bit16 randNum;
        nnt::nfp::CreateRandomNumberSequence(&randNum, 1);
        int ms = randNum % max;
        nnt::nfp::Sleep(ms);
        return ms;
    }

    bool IsAllowResultFlush(nn::Result targetResult)
    {
        if(targetResult <= nn::nfp::ResultNfcDeviceNotFound())
        {
            return false;
        }
        else if(targetResult <= nn::nfp::ResultNfcDisabled())
        {
            return false;
        }
        else if(targetResult <= nn::nfp::ResultNeedRetry())
        {
            return false;
        }
        else if(targetResult <= nn::nfp::ResultNeedRestart())
        {
            return false;
        }
        NNT_EXPECT_RESULT_SUCCESS(targetResult);
        return true;
    }

    bool IsAllowResultMount(nn::Result targetResult)
    {
        // ResultInvalidTag を返すが一般アプリには ResultNotSupported を返す。
        if(targetResult <= nn::nfp::ResultNeedRestore())
        {
#if defined(NNT_NFP_LIB_VERSION_ALPHA)
            nnt::nfp::wrapper::Restore();
#endif // defined(NNT_NFP_LIB_VERSION_ALPHA)
            return false;
        }
        else if(targetResult <= nn::nfp::ResultNfcDeviceNotFound())
        {
            return false;
        }
        else if(targetResult <= nn::nfp::ResultNfcDisabled())
        {
            return false;
        }
        else if(targetResult <= nn::nfp::ResultNeedRetry())
        {
            return false;
        }
        else if(targetResult <= nn::nfp::ResultNeedRestart())
        {
            return false;
        }
        else if(targetResult <= nn::nfp::ResultNeedFormat())
        {
            return false;
        }
        else if(targetResult <= nn::nfp::ResultNotSupported())
        {
            return false;
        }
        else if(targetResult <= nn::nfp::ResultInvalidFormatVersion())
        {
            return false;
        }
#if !defined(NNT_NFP_PLATFORM_NX) //NXではResultInvalidTagを返さない
        else if(targetResult <= nn::nfp::ResultInvalidTag())
        {
            return false;
        }
#endif // !defined(NNT_NFP_PLATFORM_NX)
        NNT_EXPECT_RESULT_SUCCESS(targetResult);
        return true;
    }

    bool IsAllowResultFormat(nn::Result targetResult)
    {
        // ResultInvalidTag を返すが一般アプリには ResultNotSupported を返す。
        if(targetResult <= nn::nfp::ResultNeedRestore() ||
           targetResult <= nn::nfp::ResultNeedRetry())
        {
#if defined(NNT_NFP_LIB_VERSION_ALPHA)
            nnt::nfp::wrapper::Restore();
#endif // defined(NNT_NFP_LIB_VERSION_ALPHA)
            return false;
        }
        else if(targetResult <= nn::nfp::ResultNfcDeviceNotFound())
        {
            return false;
        }
#if !defined(NNT_NFP_PLATFORM_NX) //NXではResultInvalidTagを返さない
        else if(targetResult <= nn::nfp::ResultInvalidTag())
        {
            return false;
        }
#endif // !defined(NNT_NFP_PLATFORM_NX)
        else if(targetResult <= nn::nfp::ResultNfcDisabled())
        {
            return false;
        }
        else if(targetResult <= nn::nfp::ResultNeedRestart())
        {
            return false;
        }
        else if(targetResult <= nn::nfp::ResultNotSupported())
        {
            return false;
        }
        NNT_EXPECT_RESULT_SUCCESS(targetResult);
        return true;
    }

    bool IsAllowResultDeleteApplicationArea(nn::Result targetResult)
    {
        if(targetResult <= nn::nfp::ResultTagNotFound())
        {
            return false;
        }
        else if(targetResult <= nn::nfp::ResultNfcDisabled())
        {
            return false;
        }
        else if(targetResult <= nn::nfp::ResultNeedRetry())
        {
            return false;
        }
        else if(targetResult <= nn::nfp::ResultNeedCreate())
        {
            return false;
        }
        NNT_EXPECT_RESULT_SUCCESS(targetResult);
        return true;
    }

    bool IsAllowResultCreateApplicationArea(nn::Result targetResult)
    {
        if(targetResult <= nn::nfp::ResultNfcDeviceNotFound())
        {
            return false;
        }
        else if(targetResult <= nn::nfp::ResultNfcDisabled())
        {
            return false;
        }
        else if(targetResult <= nn::nfp::ResultNeedRetry())
        {
            return false;
        }
        else if(targetResult <= nn::nfp::ResultNeedRestart())
        {
            return false;
        }
        else if(targetResult <= nn::nfp::ResultAlreadyCreated())
        {
            return false;
        }
        NNT_EXPECT_RESULT_SUCCESS(targetResult);
        return true;
    }

    bool IsAllowResultDeleteRegisterInfo(nn::Result targetResult)
    {
        if(targetResult <= nn::nfp::ResultNfcDeviceNotFound())
        {
            return false;
        }
        else if(targetResult <= nn::nfp::ResultNfcDisabled())
        {
            return false;
        }
        else if(targetResult <= nn::nfp::ResultNeedRetry())
        {
            return false;
        }
        else if(targetResult <= nn::nfp::ResultNeedRestart())
        {
            return false;
        }
        else if(targetResult <= nn::nfp::ResultNeedCreate())
        {
            return false;
        }
        NNT_EXPECT_RESULT_SUCCESS(targetResult);
        return true;
    }

    bool IsAllowResultAccessInfo(nn::Result targetResult)
    {
        if(targetResult <= nn::nfp::ResultNfcDeviceNotFound())
        {
            return false;
        }
        else if(targetResult <= nn::nfp::ResultNfcDisabled())
        {
            return false;
        }
        else if(targetResult <= nn::nfp::ResultNeedRestart())
        {
            return false;
        }
        NNT_EXPECT_RESULT_SUCCESS(targetResult);
        return true;
    }

    bool IsAllowResultAccessApplicationArea(nn::Result targetResult)
    {
        if(targetResult <= nn::nfp::ResultNfcDeviceNotFound())
        {
            return false;
        }
        else if(targetResult <= nn::nfp::ResultNfcDisabled())
        {
            return false;
        }
        else if(targetResult <= nn::nfp::ResultNeedRestart())
        {
            return false;
        }
        NNT_EXPECT_RESULT_SUCCESS(targetResult);
        return true;
    }

    bool IsAllowResultStopDetect(nn::Result targetResult)
    {
        if(targetResult <= nn::nfp::ResultNfcDeviceNotFound())
        {
            return false;
        }
        else if(targetResult <= nn::nfp::ResultNfcDisabled())
        {
            return false;
        }
        NNT_EXPECT_RESULT_SUCCESS(targetResult);
        return true;
    }

    bool IsAllowResultUnmount(nn::Result targetResult)
    {
        if(targetResult <= nn::nfp::ResultNfcDeviceNotFound())
        {
            return false;
        }
        else if(targetResult <= nn::nfp::ResultNfcDisabled())
        {
            return false;
        }
        else if(targetResult <= nn::nfp::ResultNeedRestart())
        {
            return false;
        }
        NNT_EXPECT_RESULT_SUCCESS(targetResult);
        return true;
    }

    bool CheckForActivateEvent(bool isActivate)
    {
        EXPECT_TRUE(isActivate);
        return isActivate;
    }

    bool ReadWriteTag()
    {
        nn::Result result;
#if defined(NNT_NFP_LIB_VERSION_BETA)
        nn::Bit8 miiData[nnt::nfp::TestMiiDataSize] = {};
#endif // defined(NNT_NFP_LIB_VERSION_BETA)
        nn::Bit8 nickNameRandNumSeq[nn::nfp::NicknameLengthMax + 1] = {};
        nn::nfp::RegisterInfoPrivate regInfoSet = {};
        nn::nfp::RegisterInfo        regInfoGet = {};
        nn::nfp::TagInfo             tagInfo    = {};
        nn::nfp::CommonInfo          commonInfo = {};
        nn::nfp::AdminInfo           adminInfo  = {};

        std::memcpy(g_TagData,nnt::nfp::ZeroTagData,sizeof(g_TagData));
        std::memset(g_RandNumSeq,0x00,sizeof(g_RandNumSeq));
        std::memset(g_ReadNumSeq,0x00,sizeof(g_ReadNumSeq));

        NFP_LOG("nn::nfp::OpenApplicationArea\n");
        result = nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::ZeroTagId);
        if(result.IsFailure())
        {

            nn::nfp::ApplicationAreaCreateInfo info;
#if !defined(NNT_NFP_PLATFORM_NX) // NXではInitializeCreateInfoは実装されない
            NNT_NFP_ASSERT_RESULT_SUCCESS_FATAL(nnt::nfp::wrapper::InitializeCreateInfo(&info));
#endif // !defined(NNT_NFP_PLATFORM_NX)

            info.accessId        = nnt::nfp::ZeroTagId;
            info.pInitialData    = g_TagData;
            info.initialDataSize = nn::nfp::ApplicationAreaSizeV2;
            NFP_LOG("nn::nfp::CreateApplicationArea\n");
            if(false == IsAllowResultCreateApplicationArea(
                            nnt::nfp::CreateApplicationAreaWithRetry(info)))
            {
                return false;
            }
        }
#if defined(NNT_NFP_LIB_VERSION_BETA)
        nnt::nfp::CreateRandomNumberSequence(miiData, nnt::nfp::TestMiiDataSize);
#endif // !defined(NNT_NFP_LIB_VERSION_BETA)
        nnt::nfp::CreateRandomAllowAsciiCodeSequence(nickNameRandNumSeq, nn::nfp::NicknameLengthMax + 1);
        nickNameRandNumSeq[nn::nfp::NicknameLengthMax] = 0x00; // 終端文字

#if !defined(NNT_NFP_PLATFORM_NX) // NXではConnectは実装されない
        if(false ==
                IsAllowResultAccessInfo( nnt::nfp::wrapper::InitializeRegisterInfoSet(&regInfoSet)))
        {
            return false;
        }
#endif // !defined(NNT_NFP_PLATFORM_NX)
        regInfoSet.fontRegion  = 0x01;
#if !defined(NNT_NFP_PLATFORM_NX) // NXでは不定値を設定するとアボートする
        std::memcpy(regInfoSet.miiData.data, miiData, 96);
#else
    #if defined(NNT_NFP_LIB_VERSION_BETA) // miiデータの使用が決定するまで除外
        #if defined(NNT_NFP_LIB_MII_ENABLE)
        nnt::nfp::BuildMiiData(regInfoSet.miiData);
        #endif // defined(NNT_NFP_LIB_MII_ENABLE)
    #endif // !defined(NNT_NFP_LIB_VERSION_BETA)
#endif // !defined(NNT_NFP_PLATFORM_NX)
        std::memcpy(regInfoSet.nickname, nickNameRandNumSeq, sizeof(nickNameRandNumSeq));

        NFP_LOG("nn::nfp::SetRegisterInfo\n");
        if(false == IsAllowResultAccessInfo(nnt::nfp::wrapper::SetRegisterInfo(regInfoSet)))
        {
            return false;
        }

        NFP_LOG("nn::nfp::GetRegisterInfo\n");
        if(false == IsAllowResultAccessInfo(nnt::nfp::wrapper::GetRegisterInfo(&regInfoGet)))
        {
            return false;
        }

        NFP_LOG("nn::nfp::GetTagInfo\n");
        if(false == IsAllowResultAccessInfo(nnt::nfp::wrapper::GetTagInfo(&tagInfo)))
        {
            return false;
        }

        NFP_LOG("nn::nfp::GetCommonInfo\n");
        if(false == IsAllowResultAccessInfo(nnt::nfp::wrapper::GetCommonInfo(&commonInfo)))
        {
            return false;
        }

        NFP_LOG("nn::nfp::GetAdminInfo\n");
        if(false == IsAllowResultAccessInfo(nnt::nfp::wrapper::GetAdminInfo(&adminInfo)))
        {
            return false;
        }

        nnt::nfp::CreateRandomNumberSequence(g_RandNumSeq, nn::nfp::ApplicationAreaSizeV2);

        // 最大容量 nn::nfp::ApplicationAreaSizeV2 Byte の書き込みと読み出し
        NFP_LOG("nn::nfp::SetApplicationArea\n");
        if(false == IsAllowResultAccessApplicationArea(
                        nnt::nfp::wrapper::SetApplicationArea(g_RandNumSeq,
                                                              nn::nfp::ApplicationAreaSizeV2,
                                                              tagInfo.tagId)))
        {
            return false;
        }

        NFP_LOG("nnt::nfp::FlushWithRetry\n");
        if(false == IsAllowResultFlush(nnt::nfp::FlushWithRetry()))
        {
            return false;
        }

        NFP_LOG("nn::nfp::GetApplicationArea\n");
        if(false == IsAllowResultAccessApplicationArea(
                        nnt::nfp::wrapper::GetApplicationArea(g_ReadNumSeq,
                                                              nn::nfp::ApplicationAreaSizeV2)))
        {
            return false;
        }

        NNT_NFP_ASSERT_FATAL(
                std::memcmp(g_RandNumSeq, g_ReadNumSeq, nn::nfp::ApplicationAreaSizeV2) == 0);

        // 最小容量 1 Byte の書き込みと読み出し
        NFP_LOG("nn::nfp::SetApplicationArea\n");
        if(false == IsAllowResultAccessApplicationArea(
                        nnt::nfp::wrapper::SetApplicationArea(g_RandNumSeq, 1, tagInfo.tagId)))
        {
            return false;
        }

        // 連続した Flush の前に Wait
        RandomSleep(RandomSleepTimeMax);

        NFP_LOG("nnt::nfp::FlushWithRetry\n");
        if(false == IsAllowResultFlush(nnt::nfp::FlushWithRetry()))
        {
            return false;
        }

        NFP_LOG("nn::nfp::GetApplicationArea\n");
        if(false == IsAllowResultAccessApplicationArea(
                        nnt::nfp::wrapper::GetApplicationArea(g_ReadNumSeq, nn::nfp::ApplicationAreaSizeV2)))
        {
            return false;
        }

        NNT_NFP_ASSERT_FATAL(std::memcmp(g_RandNumSeq, g_ReadNumSeq, 1) == 0);
        NNT_NFP_ASSERT_FATAL(
                std::memcmp(g_RandNumSeq, g_ReadNumSeq, nn::nfp::ApplicationAreaSizeV2) != 0);   //書込みサイズ以降は乱数で埋められるので一致しない

        // 連続した Flush の前に Wait
        RandomSleep(RandomSleepTimeMax);

#if defined(NNT_NFP_PLATFORM_NX)
        nnt::nfp::CreateRandomNumberSequence(g_RandNumSeq, nn::nfp::ApplicationAreaSizeV2);

        // 最大容量 nn::nfp::ApplicationAreaSizeV2 Byte の書き込みと読み出し
        NFP_LOG("nn::nfp::RecreateApplicationArea\n");
        {
            nn::nfp::ApplicationAreaCreateInfo info;
            info.accessId        = nnt::nfp::ZeroTagId + 1;
            info.pInitialData    = g_RandNumSeq;
            info.initialDataSize = nn::nfp::ApplicationAreaSizeV2;

            if(false == IsAllowResultAccessApplicationArea(
                            nnt::nfp::wrapper::RecreateApplicationArea(info)))
            {
                return false;
            }
        }

        NFP_LOG("nnt::nfp::FlushWithRetry\n");
        if(false == IsAllowResultFlush(nnt::nfp::FlushWithRetry()))
        {
            return false;
        }

        NFP_LOG("nn::nfp::GetApplicationArea\n");
        if(false == IsAllowResultAccessApplicationArea(
                        nnt::nfp::wrapper::GetApplicationArea(g_ReadNumSeq,
                                                              nn::nfp::ApplicationAreaSizeV2)))
        {
            return false;
        }

        NNT_NFP_ASSERT_FATAL(
                std::memcmp(g_RandNumSeq, g_ReadNumSeq, nn::nfp::ApplicationAreaSizeV2) == 0);

        NFP_LOG("nn::nfp::OpenApplicationArea\n");
        result = nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::ZeroTagId);
        if(!nn::nfp::ResultAccessIdMisMatch::Includes(result))
        {
            return false;
        }

        result = nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::ZeroTagId + 1);
        if(result.IsFailure())
        {
            return false;
        }

        NFP_LOG("nn::nfp::GetApplicationArea\n");
        if(false == IsAllowResultAccessApplicationArea(
                        nnt::nfp::wrapper::GetApplicationArea(g_ReadNumSeq,
                                                              nn::nfp::ApplicationAreaSizeV2)))
        {
            return false;
        }

        NNT_NFP_ASSERT_FATAL(
                std::memcmp(g_RandNumSeq, g_ReadNumSeq, nn::nfp::ApplicationAreaSizeV2) == 0);
#endif // defined(NNT_NFP_PLATFORM_NX)

        NFP_LOG("nnt::nfp::DelAppWithRetry\n");
        if(false == IsAllowResultDeleteApplicationArea(nnt::nfp::DeleteApplicationAreaWithRetry()))
        {
            return false;
        }

        // 連続した Flush の前に Wait
        RandomSleep(RandomSleepTimeMax);

        NFP_LOG("nnt::nfp::DelRegWithRetry\n");
        if(false == IsAllowResultDeleteRegisterInfo(nnt::nfp::DeleteNfpRegisterInfoWithRetry()))
        {
            return false;
        }

        return true;
    } // NOLINT(impl/function_size)

#if defined(NNT_NFP_SWITCH_NFC_ENABLE)
    // スレッドスタックサイズ
    const size_t ThreadStackSize = 4 * 1024;
    // スレッドタイプ
    nn::os::ThreadType  g_Thread;
    // スレッド継続停止のスイッチ
    bool g_ThreadSwitch = true;
    // スレッドのスタック
    NN_OS_ALIGNAS_THREAD_STACK char g_ThreadStack[ ThreadStackSize ];

    // ランダムでSetNfcEnabled()を切り替えるスレッド
    void SwitchNfcEnabledThread(void* pArg) NN_NOEXCEPT
    {
        NN_UNUSED(pArg);
        NFP_LOG("%s start \n", NN_CURRENT_FUNCTION_NAME);
        // Nfc On 時間の最大値と最小値(ms)
        const int NfcOnTimeMax = 30000;
        const int NfcOnTimeMin = 10000;
        // Nfc Off 時間の最大値と最小値(ms)
        const int NfcOffTimeMax = 5000;
        const int NfcOffTimeMin = 1000;
        // 試験が終了するまで繰り返す
        g_ThreadSwitch = true;
        while(g_ThreadSwitch)
        {
            std::srand(nn::os::GetSystemTick().GetInt64Value() & 0xFFFFFFFF);
            // スリープ終了でNfc有効無効を切り替える
            if(nnt::nfp::IsNfcEnable())
            {
                // スリープ時間を設定
                int intervalTime = (std::rand() % (NfcOnTimeMax - NfcOnTimeMin)) + NfcOnTimeMin;
                NFP_LOG("%s sleep IsNfcEnable = %d intervalTime = %d\n",
                        NN_CURRENT_FUNCTION_NAME, nnt::nfp::IsNfcEnable(), intervalTime);
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(intervalTime));
                nnt::nfp::WifiOff();
                NFP_LOG("%s <<Nfc Off>> IsNfcEnable = %d \n",
                        NN_CURRENT_FUNCTION_NAME, nnt::nfp::IsNfcEnable());
            }
            else
            {
                // スリープ時間を設定
                int intervalTime = (std::rand() % (NfcOffTimeMax - NfcOffTimeMin)) + NfcOffTimeMin;
                NFP_LOG("%s sleep IsNfcEnable = %d intervalTime = %d\n",
                        NN_CURRENT_FUNCTION_NAME, nnt::nfp::IsNfcEnable(), intervalTime);
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(intervalTime));
                nnt::nfp::WifiOn();
                NFP_LOG("%s <<Nfc On>> IsNfcEnable = %d \n",
                        NN_CURRENT_FUNCTION_NAME, nnt::nfp::IsNfcEnable());
            }
        }
        NFP_LOG("%s end \n", NN_CURRENT_FUNCTION_NAME);
    }
#endif // defined(NNT_NFP_SWITCH_NFC_ENABLE)
}

//================================================================================
// NFPライブラリの状態に応じて書込みとフォーマットを繰り返すエイジングです。
//================================================================================

class NfpAgingRemoveTag : public nnt::nfp::TestFramework
{
protected:

    NfpAgingRemoveTag() NN_NOEXCEPT
    {
        // oeライブラリを初期化します。実機のみ
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        nn::oe::Initialize();
#endif //defined(NN_BUILD_CONFIG_OS_HORIZON)
        // コントローラの初期化
        nnt::nfp::InitializeHidController();
        nnt::nfp::wrapper::SetApiCallLoggingMode(nnt::nfp::wrapper::LogMode_Aging);
    }
};

//================================================================================
// エイジングの実装です。
//================================================================================

TEST_F(NfpAgingRemoveTag,TestCaseAgingRemoveTag)
{
    // テスト前処理
    nnt::nfp::WifiOn();

    nnt::nfp::DoCreateZeroTagAndEnd();
    bool doNextStep = true;

    for(int agingCount = 0; agingCount < AgingRepeat; agingCount++)
    {
        if(true == doNextStep)
        {
            NFP_LOG("RepeatCount : %0d \n", agingCount);
        }
        doNextStep = true;

        int ms = RandomSleep(RandomSleepTimeMax);

        nn::Bit16 command;
        nnt::nfp::CreateRandomNumberSequence(&command, 1);

        nn::nfp::DeviceState deviceState = nnt::nfp::wrapper::GetDeviceState();
        nn::nfp::State libState = nnt::nfp::wrapper::GetState();
        NFP_LOG("SleepTime : %04d \n", ms);
        NFP_LOG("DeviceState  : %02d \n", deviceState);
        NFP_LOG("LibraryState : %02d \n", libState);

#if defined(NNT_NFP_PLATFORM_NX)
        if(libState != nn::nfp::State_Init)
        {
            NNT_NFP_ASSERT_EQUAL(nn::nfp::DeviceState_Unexpected,deviceState);
            NFP_LOG("nnt::nfp::Initialize\n");
            NNT_NFP_ASSERT_RESULT_SUCCESS_FATAL(nnt::nfp::InitializeSystemWithRetry());
#if defined(NNT_NFP_SWITCH_NFC_ENABLE)
            // NfcのOn／Offを切り替えるスレッドを起動する
            NFP_LOG("\nCreate SwitchNfcEnabledThread\n");
            NNT_EXPECT_RESULT_SUCCESS(nn::os::CreateThread(
                                          &g_Thread, SwitchNfcEnabledThread, nullptr, g_ThreadStack,
                                          sizeof(g_ThreadStack), nn::os::DefaultThreadPriority));
            nn::os::StartThread(&g_Thread);
#endif // defined(NNT_NFP_SWITCH_NFC_ENABLE)
            continue;
        }
#endif //  defined(NNT_NFP_PLATFORM_NX)

        switch(deviceState)
        {
#if !defined(NNT_NFP_PLATFORM_NX)
        case nn::nfp::NONE:
            {
                NFP_LOG("nnt::nfp::Initialize\n");
                NNT_NFP_ASSERT_RESULT_SUCCESS_FATAL(nnt::nfp::InitializeSystemWithRetry());
            }
            break;
#endif // !defined(NNT_NFP_PLATFORM_NX)
#if !defined(NNT_NFP_PLATFORM_NX)
        case nn::nfp::INIT:
#else
        case nn::nfp::DeviceState_Init:
#endif // !defined(NNT_NFP_PLATFORM_NX)
            {
#if !defined(NNT_NFP_PLATFORM_NX) // NXではConnectは実装されない
                nn::nfp::TargetConnectionStatus state;
                NNT_NFP_ASSERT_RESULT_SUCCESS_FATAL(nnt::nfp::wrapper::GetTargetConnectionStatus(&state));
                if (state == nn::nfp::TARGET_DISCONNECTED)
                {
                    NFP_LOG("nnt::nfp::DoConnect\n");
                    nnt::nfp::DoConnect();
                }
                else
#endif // !defined(NNT_NFP_PLATFORM_NX)
                {
                    NFP_LOG("nnt::nfp::DoSearch\n");
                    nnt::nfp::DoSearch();
                }
            }
            break;

#if !defined(NNT_NFP_PLATFORM_NX)
        case nn::nfp::RW_SEARCH:
#else
        case nn::nfp::DeviceState_Search:
#endif // !defined(NNT_NFP_PLATFORM_NX)
            {
                NFP_LOG("======= Waiting Tag Attached =======\n");

                switch (command % 5)
                {
                case 0:
                case 1:
                case 2:
                    {
                        doNextStep = CheckForActivateEvent(nnt::nfp::WaitForActivate());
                        if(false == doNextStep)
                        {
                            break;
                        }
                        NFP_LOG("Tag is detected\n");
                    }
                    break;
                case 3:
                    {
                        NFP_LOG("nn::nfp::StopDetection\n");
                        doNextStep = IsAllowResultStopDetect(nnt::nfp::wrapper::StopDetection());
                    }
                    break;
                case 4:
                    {
#if !defined(NNT_NFP_PLATFORM_NX) // NXではConnectは実装されない
                        NFP_LOG("nn::nfp::Disconnect\n");
                        doNextStep = IsAllowResultStartStopDetect(nnt::nfp::wrapper::Disconnect());
#else
                        NFP_LOG("nn::nfp::StopDetection\n");
                        doNextStep = IsAllowResultStopDetect(nnt::nfp::wrapper::StopDetection());
#endif // !defined(NNT_NFP_PLATFORM_NX)
                    }
                    break;
                default: NN_UNEXPECTED_DEFAULT;
                }
            }
            break;

#if !defined(NNT_NFP_PLATFORM_NX)
        case nn::nfp::RW_ACTIVE:
#else
        case nn::nfp::DeviceState_Active:
#endif // !defined(NNT_NFP_PLATFORM_NX)
            switch(command % 5)
            {
            case 0:
#if !defined(NNT_NFP_PLATFORM_NX) // NXではConnectは実装されない
                {
                    NFP_LOG("nn::nfp::Disconnect\n");
                    doNextStep = IsAllowResultStartStopDetect(nnt::nfp::wrapper::Disconnect());
                }
                break;
#endif // !defined(NNT_NFP_PLATFORM_NX)
            case 1:
                {
                    NFP_LOG("nn::nfp::StopDetection\n");
                    doNextStep = IsAllowResultStopDetect(nnt::nfp::wrapper::StopDetection());
                }
                break;
            case 2:
            case 3:
                {
                    NFP_LOG("nn::nfp::Mount\n");
                    doNextStep = IsAllowResultMount(nnt::nfp::MountWithRetry());
                }
                break;
            case 4:
                {
                    NFP_LOG("nn::nfp::Format\n");
                    doNextStep = IsAllowResultFormat(
                                    nnt::nfp::FormatWithRetry(
                                            nnt::nfp::ZeroTagData,
                                            nn::nfp::ApplicationAreaSizeV2));
                }
                break;
            default: NN_UNEXPECTED_DEFAULT;
            }
            break;

#if !defined(NNT_NFP_PLATFORM_NX)
        case nn::nfp::RW_MOUNT:
#else
        case nn::nfp::DeviceState_Mount:
#endif // !defined(NNT_NFP_PLATFORM_NX)
            switch(command % 12)
            {
            case 0:
                {
                    NFP_LOG("nn::nfp::StopDetection\n");
                    doNextStep = IsAllowResultStopDetect(nnt::nfp::wrapper::StopDetection());
                }
                break;
            case 1:
#if !defined(NNT_NFP_PLATFORM_NX) // NXではConnectは実装されない
                {
                    NFP_LOG("nn::nfp::Disconnect\n");
                    doNextStep = IsAllowResultStartStopDetect(nnt::nfp::wrapper::Disconnect());
                }
                break;
#endif // !defined(NNT_NFP_PLATFORM_NX)
            case 2:
                {
#if defined(NNT_NFP_SWITCH_NFC_ENABLE)
                    g_ThreadSwitch = false;
                    nn::os::WaitThread(&g_Thread);
                    nn::os::DestroyThread(&g_Thread);
#endif // defined(NNT_NFP_SWITCH_NFC_ENABLE)
                    NFP_LOG("nnt::nfp::FinalizeSystem\n");
                    NNT_NFP_ASSERT_RESULT_SUCCESS_FATAL(nnt::nfp::FinalizeSystem());
                }
                break;
            case 4:
                {
                    NFP_LOG("nn::nfp::Unmount\n");
                    doNextStep = IsAllowResultUnmount(nnt::nfp::wrapper::Unmount());
                }
                break;
            default:
                {
                    doNextStep = ReadWriteTag();
                }
                break;
            }
            break;

#if !defined(NNT_NFP_PLATFORM_NX)
        case nn::nfp::RW_DEACTIVE:
#else
        case nn::nfp::DeviceState_Deactive:
#endif // !defined(NNT_NFP_PLATFORM_NX)
            switch(command % 4)
            {
            case 0:
                {
#if defined(NNT_NFP_SWITCH_NFC_ENABLE)
                    g_ThreadSwitch = false;
                    nn::os::WaitThread(&g_Thread);
                    nn::os::DestroyThread(&g_Thread);
#endif // defined(NNT_NFP_SWITCH_NFC_ENABLE)
                    NFP_LOG("nnt::nfp::FinalizeSystem\n");
                    NNT_NFP_ASSERT_RESULT_SUCCESS_FATAL(nnt::nfp::FinalizeSystem());
                }
                break;
            case 1:
                {
                    NFP_LOG("nn::nfp::StopDetection\n");
                    doNextStep = IsAllowResultStopDetect(nnt::nfp::wrapper::StopDetection());
                }
                break;
            case 2:
                {
                    NFP_LOG("nnt::nfp::DoSearch\n");
                    nnt::nfp::DoSearch();
                }
                break;
            case 3:
                {
#if !defined(NNT_NFP_PLATFORM_NX) // NXではConnectは実装されない
                    NFP_LOG("nn::nfp::Disconnect\n");
                    doNextStep = IsAllowResultStartStopDetect(nnt::nfp::wrapper::Disconnect());
#else
                    NFP_LOG("nnt::nfp::DoSearch\n");
                    nnt::nfp::DoSearch();
#endif // !defined(NNT_NFP_PLATFORM_NX)
                }
                break;
            default: NN_UNEXPECTED_DEFAULT;
            }
            break;

#if defined(NNT_NFP_PLATFORM_NX)
        case nn::nfp::DeviceState_Unexpected:
            {
                NFP_LOG("nnt::nfp::DoSearch\n");
                nnt::nfp::DoSearch();
            }
            break;
#endif // defined(NNT_NFP_PLATFORM_NX)

        default: NN_UNEXPECTED_DEFAULT;
            break;
        }
        if(false == doNextStep)
        {
            //処理失敗時のリトライなのでループカウントがインクリメントされないようにする
            agingCount--;
        }
    }
#if defined(NNT_NFP_SWITCH_NFC_ENABLE)
    //NfcのOn／Offを切り替えるスレッドを停止する
    g_ThreadSwitch = false;
    nn::os::WaitThread(&g_Thread);
    nn::os::DestroyThread(&g_Thread);
#endif // defined(NNT_NFP_SWITCH_NFC_ENABLE)

    NNT_NFP_ASSERT_RESULT_SUCCESS_FATAL(nnt::nfp::wrapper::Unmount());
    NNT_NFP_ASSERT_RESULT_SUCCESS_FATAL(nnt::nfp::wrapper::StopDetection());
    NNT_NFP_ASSERT_RESULT_SUCCESS_FATAL(nnt::nfp::FinalizeSystem());
} // NOLINT(impl/function_size)
