﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nfp.h>
#include <nn/oe.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/os/os_Event.h>

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>

#include <nnt/nfp/testNfp_Common.h>

//================================================================================
// このテストで使用する定義です。
//================================================================================

namespace
{
    // エージング時間の定数です
    const uint32_t AgingHours = 12;

    // Deactive発生回数の閾値です
    const uint32_t DeactiveLimit = 2;

} // end of anonymous namespace

//================================================================================
// 全プラットフォームで共通のテストスイートです。
// 必ずタグを設置してからテストを開始してください。
//================================================================================

class NfpAgingDeactive : public nnt::nfp::TestFramework
{
protected:

    NfpAgingDeactive() NN_NOEXCEPT
    {
        // oeライブラリを初期化します。実機のみ
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        nn::oe::Initialize();
#endif //defined(NN_BUILD_CONFIG_OS_HORIZON)
        // コントローラの初期化
        nnt::nfp::InitializeHidController();
        nnt::nfp::wrapper::SetApiCallLoggingMode(nnt::nfp::wrapper::LogMode_Aging);
    }
};

//================================================================================
// テストケースの実装です。
//================================================================================

TEST_F(NfpAgingDeactive,TestCaseAgingDeactive)
{
    const int DeactiveCountMax = 255;
    nn::Bit8 deactiveCount = 0;
    nn::TimeSpan currentTime;
    nn::TimeSpan recordTime[DeactiveCountMax];

    // テスト前処理
    nnt::nfp::WifiOn();
    nnt::nfp::DoCreateZeroTagAndEnd();

    // テスト開始時間取得
    nn::TimeSpan startTime = nn::os::ConvertToTimeSpan(nn::os::GetSystemTick());

    NN_LOG("==================== START AGING TEST ====================\n");

    nnt::nfp::DoSearch();
    NN_LOG("### Waiting for Activate Event... ");
    NNT_NFP_ASSERT_FATAL(nnt::nfp::WaitForActivate());
    NN_LOG("done!\n");

    // AgingHours内のDeactiveEvent発生回数を記録する
    while(deactiveCount < DeactiveCountMax)
    {
        bool isDetect = false;

        // Deactive検知まで500ms待つ
        isDetect = nnt::nfp::WaitForDeactivate(500);

        if(isDetect == true)
        {
            recordTime[deactiveCount] =
                    nn::os::ConvertToTimeSpan(nn::os::GetSystemTick()) - startTime;

            NN_LOG("### DEACTIVATE EVENT DETECTED!! \n");
            NN_LOG(" Elapsed time : %lld ms  \n",recordTime[deactiveCount].GetMilliSeconds());
            deactiveCount++;

            NNT_ASSERT_RESULT_SUCCESS(nnt::nfp::wrapper::StopDetection());
#if !defined(NNT_NFP_PLATFORM_NX)
            NNT_NFP_ASSERT_EQUAL_FATAL(nn::nfp::INIT, nnt::nfp::wrapper::GetDeviceState() );
#else
            NNT_NFP_ASSERT_EQUAL_FATAL(nn::nfp::DeviceState_Init, nnt::nfp::wrapper::GetDeviceState() );
            NNT_NFP_ASSERT_EQUAL_FATAL(nn::nfp::State_Init, nnt::nfp::wrapper::GetState() );
#endif
            NN_LOG("### Waiting for Activate Event... ");
            nnt::nfp::DoActivate();
            NN_LOG("done!\n");
        }

        currentTime =
                nn::os::ConvertToTimeSpan(nn::os::GetSystemTick()) - startTime;

        if( currentTime.GetHours() >= AgingHours )
        {
            break;
        }
    }


    NN_LOG("==================== END AGING TEST ====================\n");
    NN_LOG(" Deactivate Count : %d\n", deactiveCount);
    NN_LOG(" total time : %lld ms\n",currentTime.GetMilliSeconds());
    NN_LOG("========================================================\n");


    for(int count=0; count < deactiveCount; count++)
    {
        NN_LOG("---------- EVENT HISTORY [%d] ----------\n", count);
        NN_LOG(" Elapsed time : %lld ms  \n",recordTime[count].GetMilliSeconds());
        NN_LOG("----------------------------------------\n");
    }

    NNT_NFP_ASSERT_FATAL( deactiveCount < DeactiveLimit );
}
