﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nn_Result.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_SdkText.h>
#include <nn/hid.h>
#include <nn/mii.h>
#include <nn/nfc.h>
#include <nn/nfc/nfc_PrivateApi.h>
#include <nn/time/time_Api.h>
#include <nn/time/time_TimeZoneApi.h>
#include <nn/time/time_StandardUserSystemClock.h>
#include <nn/hid/system/hid_Npad.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#include <nn/mii/mii_StoreDataAccessor.h>

#include <nnt/result/testResult_Assert.h>
#include <nnt/nfp/testNpt_Common.h>


namespace nnt { namespace npt {

    //======================================================
    // テストで使用する定数です。
    //======================================================

    //======================================================
    // ユーティリティ関数で使用する使用する変数です。
    //======================================================

    nn::nfc::DeviceHandle      g_CurrentDeviceHandle;

    nn::os::SystemEventType            g_ActivateEvent = {};
    nn::os::SystemEventType            g_DeactivateEvent = {};

    nn::TimeSpan g_Span;

    //======================================================
    // テストで使用するデータです。
    //======================================================

    // Npt Standard 1k のデータブロック数は
    // 16byte のブロックが 64 ブロック
    // そのうちデータブロックで使用できないブロックを引いた数
    // 使用できないのは 0 と 4の倍数+3 のブロック
    // Npt Standard 1k では 47ブロック を使用できる
    // ここでは、試験用に 95ブロック のアドレスを作っておく
    const int DataBlockCount = 95;
    uint8_t g_DataBlockAddr[DataBlockCount] = {};

    //======================================================
    // コントローラ初期化に関連する変数です。
    //======================================================
    const nn::hid::NpadIdType NpadIds[] = {nn::hid::NpadId::No1,
                                       nn::hid::NpadId::No2,
                                       nn::hid::NpadId::No3,
                                       nn::hid::NpadId::No4,
                                       nn::hid::NpadId::Handheld,
                                       };
    const int NpadIdCountMax = sizeof(NpadIds) / sizeof(nn::hid::NpadIdType);

    //======================================================
    // プラットフォーム毎に異なる処理をラップする関数です。
    //======================================================

    nn::Result InitializeSystem() NN_NOEXCEPT
    {

        if( nnt::npt::wrapper::GetState() == nn::nfc::State_None )
        {
            return nnt::npt::wrapper::InitializeDebug();
        }
        else
        {
            NN_LOG("* Duplicate function call : InitializeDebug() \n");
            return nn::ResultSuccess();
        }
    }

    nn::Result FinalizeSystem() NN_NOEXCEPT
    {
        if( nnt::npt::wrapper::GetState() != nn::nfc::State_None )
        {
            nnt::npt::wrapper::FinalizeDebug();
        }
        else
        {
            NN_LOG("* Duplicate function call : FinalizeDebug()");
        }
        return nn::ResultSuccess();
    }

    nn::Result SetDefaultActivateEvent() NN_NOEXCEPT
    {
        if(g_ActivateEvent._state != nn::os::SystemEventType::State_NotInitialized)
        {
            DestroyDefaultActivateEvent();
        }
        nn::Result result;
        result = nnt::npt::wrapper::AttachActivateEvent(&g_ActivateEvent);
        if(!result.IsSuccess())
        {
            std::memset(&g_ActivateEvent, 0x00, sizeof(g_ActivateEvent));
        }
        return result;
    }

    nn::Result SetDefaultDeactivateEvent() NN_NOEXCEPT
    {
        if(g_DeactivateEvent._state != nn::os::SystemEventType::State_NotInitialized)
        {
            DestroyDefaultDeactivateEvent();
        }
        nn::Result result;
        result = nnt::npt::wrapper::AttachDeactivateEvent(&g_DeactivateEvent);
        if(!result.IsSuccess())
        {
            std::memset(&g_DeactivateEvent, 0x00, sizeof(g_DeactivateEvent));
        }
        return result;
    }

    void DestroyDefaultActivateEvent() NN_NOEXCEPT
    {
        nn::os::DestroySystemEvent(&g_ActivateEvent);
        return;
    }

    void DestroyDefaultDeactivateEvent() NN_NOEXCEPT
    {
        nn::os::DestroySystemEvent(&g_DeactivateEvent);
        return;
    }

    bool WaitForActivate(int ms) NN_NOEXCEPT
    {
        if((g_ActivateEvent._state != nn::os::SystemEventType::State_NotInitialized) &&
                (g_DeactivateEvent._state != nn::os::SystemEventType::State_NotInitialized))
        {
            return nn::os::TimedWaitSystemEvent(&g_ActivateEvent, nn::TimeSpan::FromMilliSeconds(ms));
        }
        else
        {
            NN_LOG("WaitForActivate() ActiveEvent DeactivateEvent is NotInitialized \n");
            return false;
        }
    }

    bool WaitForActivate() NN_NOEXCEPT
    {
        return WaitForActivate(ActivateTimeout);
    }

    bool WaitForDeactivate(int ms) NN_NOEXCEPT
    {
        return nn::os::TimedWaitSystemEvent(&g_DeactivateEvent, nn::TimeSpan::FromMilliSeconds(ms));
    }

    bool WaitForDeactivate() NN_NOEXCEPT
    {
        return WaitForDeactivate(DeactivateTimeout);
    }

    //==================================================
    // NPT ライブラリの API をラップしてリトライに対応した関数です。
    //==================================================

    nn::Result InitializeWithRetry() NN_NOEXCEPT
    {
        nn::Result result;

        if( nnt::npt::wrapper::GetState() == nn::nfc::State_Init )
        {
            return nn::ResultSuccess();
        }

        result = nnt::npt::wrapper::Initialize();

        return result;
    }

    nn::Result InitializeSystemWithRetry() NN_NOEXCEPT
    {
        nn::Result result;

        if( nnt::npt::wrapper::GetState() == nn::nfc::State_Init )
        {
            return nn::ResultSuccess();
        }

        result = InitializeSystem();

        return result;
    }

    //==================================================
    // 頻繁に利用されるシーケンスをまとめた関数です。
    //==================================================

    nn::nfc::DeviceHandle GetCurrentDeviceHandle() NN_NOEXCEPT
    {
        return g_CurrentDeviceHandle;
    }

    void SetCurrentDeviceHandle(nn::nfc::DeviceHandle deviceHandle) NN_NOEXCEPT
    {
        g_CurrentDeviceHandle = deviceHandle;
    }

    nn::Result SearchDeviceHandle() NN_NOEXCEPT
    {
        int outCount;
        nn::nfc::State state = nnt::npt::wrapper::GetState();

        //デバイスリストが取得できない状態でこの関数が実行された場合は、テスト手順が不正
        EXPECT_NE(nn::nfc::State_None,state);

        return nnt::npt::wrapper::ListDevices(&g_CurrentDeviceHandle, &outCount, 1);
    }

    void DoSearchDeviceHandle() NN_NOEXCEPT
    {
        nn::nfc::State state;
        state = nnt::npt::wrapper::GetState();
        if (state == nn::nfc::State_None)
        {
            NNT_EXPECT_RESULT_SUCCESS(InitializeSystemWithRetry());
        }

        //NXではConnectの代わりにデバイスハンドルの取得を行う
        NNT_EXPECT_RESULT_SUCCESS(SearchDeviceHandle());

        //イベント取得準備
        NNT_EXPECT_RESULT_SUCCESS(SetDefaultActivateEvent());
        NNT_EXPECT_RESULT_SUCCESS(SetDefaultDeactivateEvent());

        return;
    }

    void DoSearch() NN_NOEXCEPT
    {
        DoSearchDeviceHandle();
        nn::nfc::DeviceState state = nnt::npt::wrapper::GetDeviceState();

        nn::nfc::State libState = nnt::npt::wrapper::GetState();
        if ((libState == nn::nfc::State_Init)
            && (state == nn::nfc::DeviceState_Init || state == nn::nfc::DeviceState_Deactive))
        {
            NNT_EXPECT_RESULT_SUCCESS(nnt::npt::wrapper::StartDetection());
        }
    }

    void DoActivate() NN_NOEXCEPT
    {
        DoSearch();
        nn::nfc::DeviceState state = nnt::npt::wrapper::GetDeviceState();
        nn::nfc::State libState = nnt::npt::wrapper::GetState();
        if ((libState == nn::nfc::State_Init)
            && (state == nn::nfc::DeviceState_Search))
        {
            EXPECT_TRUE(WaitForActivate());
            EXPECT_EQ(nn::nfc::DeviceState_Active,nnt::npt::wrapper::GetDeviceState());
        }
    }

    void DoKeep() NN_NOEXCEPT
    {
        DoActivate();
        NNT_EXPECT_RESULT_SUCCESS(nnt::npt::wrapper::KeepPassThroughSession());
        EXPECT_EQ(nn::nfc::DeviceState_Keep,nnt::npt::wrapper::GetDeviceState());
    }

    //==================================================
    // その他のユーティリティです。
    //==================================================

    void Sleep(int ms) NN_NOEXCEPT
    {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(ms));
    }

    void CreateRandomNumberSequence(nn::Bit16* pOut, int size) NN_NOEXCEPT
    {
        std::srand(nn::os::GetSystemTick().GetInt64Value() & 0xFFFFFFFF);
        for(int i = 0; i < size; i++)
        {
            pOut[i] = std::rand() & 0xFFFF;
        }
    }

    void CreateRandomNumberSequence(char* pOut, int size) NN_NOEXCEPT
    {
        std::srand(nn::os::GetSystemTick().GetInt64Value() & 0xFFFFFFFF);
        for(int i = 0; i < size; i++)
        {
            pOut[i] = std::rand() & 0xFF;
        }
    }

    void NfcOn() NN_NOEXCEPT
    {
        if( nnt::npt::wrapper::GetState() == nn::nfc::State_None )
        {
            nn::nfc::InitializeSystem();
            nn::nfc::SetNfcEnabled(true);
            nn::nfc::FinalizeSystem();
        }
        else
        {
            nn::nfc::SetNfcEnabled(true);
        }
    }

    void NfcOff() NN_NOEXCEPT
    {
        if( nnt::npt::wrapper::GetState() == nn::nfc::State_None )
        {
            nn::nfc::InitializeSystem();
            nn::nfc::SetNfcEnabled(false);
            nn::nfc::FinalizeSystem();
        }
        else
        {
            nn::nfc::SetNfcEnabled(false);
        }
    }

    const char* GetNptResultTypeString(nn::Result result) NN_NOEXCEPT
    {
        const char* pString = nullptr;
        #define NPT_GET_RESULT_STRING(resultType,result,pOut)                               \
            else if ((nn::nfc::resultType().GetModule() == (result).GetModule()) &&         \
                     (nn::nfc::resultType().GetDescription() == (result).GetDescription())) \
            { \
                pOut = #resultType;\
            }

        if (result.IsSuccess())
        {
            pString = "ResultSuccess";
        }
        //private
        NPT_GET_RESULT_STRING(ResultBadRequest,result,pString)
        NPT_GET_RESULT_STRING(ResultInvalidDeviceState,result,pString)
        NPT_GET_RESULT_STRING(ResultTagNotFound,result,pString)
        NPT_GET_RESULT_STRING(ResultUidMisMatch,result,pString)
        NPT_GET_RESULT_STRING(ResultSleep,result,pString)
        NPT_GET_RESULT_STRING(ResultNotForeground,result,pString)
        NPT_GET_RESULT_STRING(ResultNfcDeviceError,result,pString)
        NPT_GET_RESULT_STRING(ResultBackupSystemError,result,pString)
        NPT_GET_RESULT_STRING(ResultNeedUpdate,result,pString)
        NPT_GET_RESULT_STRING(ResultNeedCharge,result,pString)
        NPT_GET_RESULT_STRING(ResultInvalidTag,result,pString)
        NPT_GET_RESULT_STRING(ResultAuthenticationError,result,pString)
        NPT_GET_RESULT_STRING(ResultAccessTimeOutError,result,pString)
        NPT_GET_RESULT_STRING(ResultAccessOperationFailed,result,pString)
        //public
        NPT_GET_RESULT_STRING(ResultNfcDeviceNotFound,result,pString)
        NPT_GET_RESULT_STRING(ResultNfcDisabled,result,pString)
        NPT_GET_RESULT_STRING(ResultNeedRestart,result,pString)
        NPT_GET_RESULT_STRING(ResultNotSupported,result,pString)
        NPT_GET_RESULT_STRING(ResultMaxNfcDeviceActivated,result,pString)
        NPT_GET_RESULT_STRING(ResultConflictFunction,result,pString)
        NPT_GET_RESULT_STRING(ResultAccessError,result,pString)
        else
        {
            //不明な場合はエラーコードを表示する
            static char unknownErrorStringBuffer[64];
            std::memset(unknownErrorStringBuffer,0x00,sizeof(unknownErrorStringBuffer));
            std::sprintf(unknownErrorStringBuffer,
                         "Unknown(Module=%d,Description=%d)",result.GetModule(),
                                                             result.GetDescription());
            pString = unknownErrorStringBuffer;
        }
        #undef NPT_GET_RESULT_STRING

        return pString;
    }

    bool CheckNpadStyle(nn::hid::NpadIdType npadId) NN_NOEXCEPT
    {
        bool ret = false;

        // 取得したNpad IDを指定してGetNpadStyleSetを実行する
        nn::hid::NpadStyleSet style;
        style = nn::hid::GetNpadStyleSet(npadId);
        if ((style.Test<nn::hid::NpadStyleFullKey>() == true) |
                (style.Test<nn::hid::NpadStyleHandheld>() == true) |
                (style.Test<nn::hid::NpadStyleJoyDual>() == true) |
                (style.Test<nn::hid::NpadStyleJoyLeft>() == true) |
                (style.Test<nn::hid::NpadStyleJoyRight>() == true) |
                (style.Test<nn::hid::system::NpadStyleSystem>() == true) |
                (style.Test<nn::hid::system::NpadStyleSystemExt>() == true))
        {
            ret = true;
        }
        return ret;
    }

    bool IsNfcEnable() NN_NOEXCEPT
    {
        bool ret;

        if( nnt::npt::wrapper::GetState() == nn::nfc::State_None )
        {
            nn::nfc::InitializeSystem();
            ret = nn::nfc::IsNfcEnabled();
            nn::nfc::FinalizeSystem();
        }
        else
        {
            ret = nn::nfc::IsNfcEnabled();
        }

        return ret;
    }

    void InitializeHidController() NN_NOEXCEPT
    {
        nn::hid::InitializeNpad();

        //使用する操作形態を設定
        nn::hid::SetSupportedNpadStyleSet(nn::hid::NpadStyleFullKey::Mask |
                nn::hid::NpadStyleJoyDual::Mask | nn::hid::NpadStyleHandheld::Mask);

        // 使用する Npad を設定
        nn::hid::SetSupportedNpadIdType(NpadIds, NpadIdCountMax);
    }

    void CheckNPadId(nn::hid::NpadIdType npadId) NN_NOEXCEPT
    {
        // 取得したNpad IDを指定してGetNpadStyleSetを実行する
        nn::hid::NpadStyleSet style;
        style = nn::hid::GetNpadStyleSet(npadId);
        EXPECT_TRUE((style.Test<nn::hid::NpadStyleFullKey>() == true) |
                (style.Test<nn::hid::NpadStyleHandheld>() == true) |
                (style.Test<nn::hid::NpadStyleJoyDual>() == true) |
                (style.Test<nn::hid::NpadStyleJoyLeft>() == true) |
                (style.Test<nn::hid::NpadStyleJoyRight>() == true) |
                (style.Test<nn::hid::system::NpadStyleSystem>() == true) |
                (style.Test<nn::hid::system::NpadStyleSystemExt>() == true));
    }

    void PrintSendWriteCommandData(nn::Bit8* sendWriteCommandData, size_t dataSize) NN_NOEXCEPT
    {
        NN_LOG("-WriteData-\n");
        for(size_t i = 0; i < dataSize; i++)
        {
            NN_LOG("0x%02x ", sendWriteCommandData[2 + i]);
        }
        NN_LOG("\n");
        return;
    }

    void PrepareSendWriteCommandData(nn::Bit8* sendWriteCommandData, size_t dataSize) NN_NOEXCEPT
    {
        NNT_NPT_ASSERT_FATAL(dataSize == 6);
        sendWriteCommandData[0] = 0xA2; // WRITE コマンド
        sendWriteCommandData[1] = 0x04; // Page 4
        // ランダムのデータ設定
        CreateRandomNumberSequence(reinterpret_cast<char*>(&sendWriteCommandData[2]), 4);
    }

    void PrintSendReadCommandResponseData(nn::Bit8* sendReadCommandResponseData, size_t dataSize) NN_NOEXCEPT
    {
        NN_LOG("-ReadData-\n");
        if(dataSize != 16)
        {
            NN_LOG("-ErrorData-\n");
        }
        for(size_t i = 0; i < dataSize; i++)
        {
            NN_LOG("0x%02x ", sendReadCommandResponseData[i]);
        }
        NN_LOG("\n");
        return;
    }

    void PrepareSendReadCommandData(nn::Bit8* sendReadCommandData, size_t dataSize) NN_NOEXCEPT
    {
        NNT_NPT_ASSERT_FATAL(dataSize == 2);
        sendReadCommandData[0] = 0x30; // READ コマンド
        sendReadCommandData[1] = 0x04; // Start Page 4
    }

    void CheckReadData(nn::Bit8* sendReadCommandResponseData, size_t sendReadCommandResponseDataSize,
                       nn::Bit8* sendWriteCommandData, size_t sendWriteCommandDataSize) NN_NOEXCEPT
    {
        EXPECT_TRUE(sendReadCommandResponseDataSize == 16);
        EXPECT_TRUE(std::memcmp(sendReadCommandResponseData, &sendWriteCommandData[2], 4) == 0);
    }
}} // end of namespace nnt::npt
