﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../Common/testNews_Common.h"
#include <nn/hid.h>

namespace
{
    nn::ApplicationId g_ApplicationId = {0x0100000000001000};
    const char* g_Passphrase = "xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx";

    const char* g_TopicId = "nx_news_sp_siglo03";
}

namespace
{
    nn::os::SystemEvent g_NewlyArrivedEvent;
    nn::os::SystemEvent g_OverwriteEvent;
}

static void ShowUsage() NN_NOEXCEPT
{
    NN_LOG("--------------------------------------------------\n");
    NN_LOG("A:      RequestImmediateReception\n");
    NN_LOG("B:      ShowList\n");
    NN_LOG("X:      ClearStorage\n");
    NN_LOG("Y:      ClearSubscribeAll\n");
    NN_LOG("Select: ShowUsage\n");
    NN_LOG("Start:  Exit\n");
    NN_LOG("--------------------------------------------------\n");
}

TEST(OverwriteNews, Initialize)
{
    nn::nifm::Initialize();
    nn::hid::InitializeDebugPad();

    EXPECT_RESULT_SUCCESS(nn::news::RegisterNewlyArrivedEvent(&g_NewlyArrivedEvent));
    EXPECT_RESULT_SUCCESS(nn::news::RegisterOverwriteEvent(&g_OverwriteEvent));

    EXPECT_RESULT_SUCCESS(nn::news::SetPassphrase(g_ApplicationId, g_Passphrase));
    EXPECT_RESULT_SUCCESS(nn::news::SetSubscriptionStatus(g_TopicId, nn::news::SubscriptionStatus_Subscribed));
}

TEST(OverwriteNews, MainLoop)
{
    ShowUsage();

    nn::hid::DebugPadState prevPadState = {};

    while (NN_STATIC_CONDITION(1))
    {
        nn::hid::DebugPadState padState = {};
        nn::hid::GetDebugPadState(&padState);

        nn::hid::DebugPadButtonSet trigger = (~prevPadState.buttons) & padState.buttons;
        prevPadState = padState;

        if (trigger.Test<nn::hid::DebugPadButton::A>())
        {
            NN_LOG("RequestImmediateReception.\n");
            EXPECT_RESULT_SUCCESS(nn::news::SetSubscriptionStatus(g_TopicId, nn::news::SubscriptionStatus_Subscribed));
            EXPECT_RESULT_SUCCESS(nn::news::RequestImmediateReception(g_TopicId));
        }
        else if (trigger.Test<nn::hid::DebugPadButton::B>())
        {
            nn::news::Database db;
            nn::news::NewsRecord records[10];
            int count = 0;

            NN_LOG("ShowList.\n");
            ASSERT_RESULT_SUCCESS(db.Open());
            ASSERT_RESULT_SUCCESS(db.GetList(&count, records, "", "received_at DESC", 0, 10));

            NN_LOG("--------------------------------------------------\n");
            NN_LOG("news.db\n");
            NN_LOG("--------------------------------------------------\n");
            NN_LOG("Count = %d\n", count);

            for (int i = 0; i < count; i++)
            {
                NN_LOG("--------------------------------------------------\n");
                NN_LOG("NewsRecord[%d]\n", i);
                NN_LOG("    - NewsId: %s\n", records[i].newsId.value);
                NN_LOG("    - UserId: %s\n", records[i].userId.value);
                NN_LOG("    - ReceivedTime: %lld\n", records[i].receivedTime.value);
                NN_LOG("    - Read: %d\n", records[i].read);
                NN_LOG("    - Newly: %d\n", records[i].newly);
                NN_LOG("    - Displayed: %d\n", records[i].displayed);

                char wherePhrase[64] = {};
                nn::util::SNPrintf(wherePhrase, sizeof (wherePhrase), "news_id = '%s'", records[i].newsId.value);

                EXPECT_RESULT_SUCCESS(db.UpdateWithAddition("displayed", 1, wherePhrase));
            }

            NN_LOG("--------------------------------------------------\n");
        }
        else if (trigger.Test<nn::hid::DebugPadButton::X>())
        {
            NN_LOG("ClearStorage.\n");
            EXPECT_RESULT_SUCCESS(nn::news::ClearStorage());
        }
        else if (trigger.Test<nn::hid::DebugPadButton::Y>())
        {
            NN_LOG("ClearSubscriptionStatusAll.\n");
            EXPECT_RESULT_SUCCESS(nn::news::ClearSubscriptionStatusAll());
        }
        else if (trigger.Test<nn::hid::DebugPadButton::Select>())
        {
            ShowUsage();
        }
        else if (trigger.Test<nn::hid::DebugPadButton::Start>())
        {
            NN_LOG("Exit.\n");
            break;
        }

        if (g_NewlyArrivedEvent.TryWait())
        {
            g_NewlyArrivedEvent.Clear();
            NN_LOG("Newly arrived event signaled.\n");
        }
        if (g_OverwriteEvent.TryWait())
        {
            g_OverwriteEvent.Clear();
            NN_LOG("Overwrite event signaled.\n");
        }

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
    }
}

TEST(OverwriteNews, Finalize)
{
#if defined (NN_BUILD_CONFIG_OS_WIN)
    nn::news::service::StopServer();
#endif
}
