﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../Common/testNews_Common.h"
#include <nn/news/detail/service/core/news_NewsTaskManager.h>

using namespace nn::news::detail::service::core;

TEST(NewsTaskManager, Initialize)
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountHost("news-sys", "C:\\siglo\\news-sys"));
    FileSystem::EnableTestMode();

    nn::time::Initialize();
}

TEST(NewsTaskManager, Load)
{
    NewsTaskManager::GetInstance().NotifyNetworkConnected();

    EXPECT_RESULT_SUCCESS(NewsTaskManager::GetInstance().Load());
}

TEST(NewsTaskManager, Clear)
{
    // 複数回呼出確認。
    EXPECT_RESULT_SUCCESS(NewsTaskManager::GetInstance().Clear());
    EXPECT_RESULT_SUCCESS(NewsTaskManager::GetInstance().Clear());
}

TEST(NewsTaskManager, AddRemove)
{
    const nn::news::TopicId topicId1 = {"nx_topic_1"};
    const nn::news::TopicId topicId2 = {"nx_topic_2"};
    const nn::news::TopicId topicId3 = {"nx_topic_3"};
    const nn::news::TopicId topicId4 = {"nx_topic_4"};

    EXPECT_RESULT_SUCCESS(NewsTaskManager::GetInstance().SetSubscriptionStatus(topicId1, nn::news::SubscriptionStatus_Unsubscribed));
    EXPECT_RESULT_SUCCESS(NewsTaskManager::GetInstance().SetSubscriptionStatus(topicId2, nn::news::SubscriptionStatus_Subscribed));
    EXPECT_RESULT_SUCCESS(NewsTaskManager::GetInstance().SetSubscriptionStatus(topicId3, nn::news::SubscriptionStatus_AutoSubscribed));
    EXPECT_RESULT_SUCCESS(NewsTaskManager::GetInstance().SetSubscriptionStatus(topicId2, nn::news::SubscriptionStatus_Unconfigured));
    EXPECT_RESULT_SUCCESS(NewsTaskManager::GetInstance().SetSubscriptionStatus(topicId4, nn::news::SubscriptionStatus_Subscribed));
}

TEST(NewsTaskManager, Max)
{
    EXPECT_RESULT_SUCCESS(NewsTaskManager::GetInstance().Clear());

    for (int i = 0; i < NewsTaskManager::TaskCountMax; i++)
    {
        nn::news::TopicId topicId = {};
        nn::util::SNPrintf(topicId.value, sizeof (topicId.value), "nx_topic_%04d", i);

        if (i < NewsTaskManager::SubscriptionCountMax)
        {
            EXPECT_RESULT_SUCCESS(NewsTaskManager::GetInstance().SetSubscriptionStatus(topicId,
                nn::news::SubscriptionStatus_Subscribed));
        }
        else
        {
            EXPECT_RESULT(NewsTaskManager::GetInstance().SetSubscriptionStatus(topicId,
                nn::news::SubscriptionStatus_Subscribed), nn::news::ResultSubscriptionLimitReached);

            EXPECT_RESULT_SUCCESS(NewsTaskManager::GetInstance().SetSubscriptionStatus(topicId,
                nn::news::SubscriptionStatus_Unsubscribed));

            // 新規追加でない時の確認。
            EXPECT_RESULT(NewsTaskManager::GetInstance().SetSubscriptionStatus(topicId,
                nn::news::SubscriptionStatus_Subscribed), nn::news::ResultSubscriptionLimitReached);
        }
    }
}

TEST(NewsTaskManager, Max2)
{
    EXPECT_RESULT_SUCCESS(NewsTaskManager::GetInstance().Clear());

    for (int i = 0; i < 3; i++)
    {
        for (int j = 0; j < NewsTaskManager::SubscriptionCountMax; j++)
        {
            nn::news::TopicId topicId = {};
            nn::util::SNPrintf(topicId.value, sizeof (topicId.value), "nx_topic_%04d", i * NewsTaskManager::SubscriptionCountMax + j);

            EXPECT_RESULT_SUCCESS(NewsTaskManager::GetInstance().SetSubscriptionStatus(topicId,
                nn::news::SubscriptionStatus_Subscribed));

            EXPECT_RESULT_SUCCESS(NewsTaskManager::GetInstance().SetSubscriptionStatus(topicId,
                nn::news::SubscriptionStatus_Unsubscribed));
        }
    }
}

TEST(NewsTaskManager, AutoRemoveOldTopic)
{
    EXPECT_RESULT_SUCCESS(NewsTaskManager::GetInstance().Clear());

    for (int i = 0; i < NewsTaskManager::TaskCountMax; i++)
    {
        nn::news::TopicId topicId = {};
        nn::util::SNPrintf(topicId.value, sizeof (topicId.value), "nx_topic_0%04d", i);

        EXPECT_RESULT_SUCCESS(NewsTaskManager::GetInstance().SetSubscriptionStatus(topicId,
            nn::news::SubscriptionStatus_Unsubscribed));
    }

    for (int i = 0; i < NewsTaskManager::SubscriptionCountMax; i++)
    {
        nn::news::TopicId topicId1_1 = {};
        nn::util::SNPrintf(topicId1_1.value, sizeof (topicId1_1.value), "nx_topic_0%04d", i);

        nn::news::TopicId topicId1_2 = {};
        nn::util::SNPrintf(topicId1_2.value, sizeof (topicId1_2.value), "nx_topic_0%04d", i + 1);

        nn::news::TopicId topicId2 = {};
        nn::util::SNPrintf(topicId2.value, sizeof (topicId2.value), "nx_topic_1%04d", i);

        EXPECT_TRUE(NewsTaskManager::GetInstance().GetSubscriptionStatus(topicId1_1) == nn::news::SubscriptionStatus_Unsubscribed);

        EXPECT_RESULT_SUCCESS(NewsTaskManager::GetInstance().SetSubscriptionStatus(topicId2,
            nn::news::SubscriptionStatus_Subscribed));

        // レコードに存在しないトピックを登録する度に、古いものから削除される。
        EXPECT_TRUE(NewsTaskManager::GetInstance().GetSubscriptionStatus(topicId1_1) == nn::news::SubscriptionStatus_Unconfigured);
        EXPECT_TRUE(NewsTaskManager::GetInstance().GetSubscriptionStatus(topicId1_2) == nn::news::SubscriptionStatus_Unsubscribed);
    }

    EXPECT_RESULT_SUCCESS(NewsTaskManager::GetInstance().Clear());

    for (int i = 0; i < NewsTaskManager::TaskCountMax; i++)
    {
        nn::news::TopicId topicId = {};
        nn::util::SNPrintf(topicId.value, sizeof (topicId.value), "nx_topic_0%04d", i);

        EXPECT_RESULT_SUCCESS(NewsTaskManager::GetInstance().SetSubscriptionStatus(topicId,
            nn::news::SubscriptionStatus_Unsubscribed));
    }

    for (int i = 0; i < NewsTaskManager::TaskCountMax; i++)
    {
        nn::news::TopicId topicId = {};
        nn::util::SNPrintf(topicId.value, sizeof (topicId.value), "nx_topic_1%04d", i);

        EXPECT_RESULT_SUCCESS(NewsTaskManager::GetInstance().SetSubscriptionStatus(topicId,
            nn::news::SubscriptionStatus_Unsubscribed));
    }

    for (int i = 0; i < NewsTaskManager::TaskCountMax; i++)
    {
        nn::news::TopicId topicId = {};
        nn::util::SNPrintf(topicId.value, sizeof (topicId.value), "nx_topic_0%04d", i);

        EXPECT_TRUE(NewsTaskManager::GetInstance().GetSubscriptionStatus(topicId) == nn::news::SubscriptionStatus_Unconfigured);
    }
}

TEST(NewsTaskManager, Done)
{
    EXPECT_RESULT_SUCCESS(NewsTaskManager::GetInstance().Clear());

    const nn::news::TopicId topicId1 = {"nx_topic_1"};
    const nn::news::TopicId topicId2 = {"nx_topic_2"};
    const nn::news::detail::ETag eTag = {"1"};

    EXPECT_RESULT_SUCCESS(NewsTaskManager::GetInstance().SetSubscriptionStatus(topicId1, nn::news::SubscriptionStatus_Subscribed));
    EXPECT_RESULT_SUCCESS(NewsTaskManager::GetInstance().SetSubscriptionStatus(topicId2, nn::news::SubscriptionStatus_Subscribed));

    NewsTaskManager::Record record = {};
    EXPECT_RESULT_SUCCESS(NewsTaskManager::GetInstance().GetCurrentTask(&record));
    EXPECT_RESULT_SUCCESS(NewsTaskManager::GetInstance().UpdateETag(record.topicId, eTag));
    EXPECT_RESULT_SUCCESS(NewsTaskManager::GetInstance().NotifyDone(record, nn::ResultSuccess()));

    EXPECT_RESULT_SUCCESS(NewsTaskManager::GetInstance().GetCurrentTask(&record));
    EXPECT_RESULT_SUCCESS(NewsTaskManager::GetInstance().UpdateETag(record.topicId, eTag));
    EXPECT_RESULT_SUCCESS(NewsTaskManager::GetInstance().NotifyDone(record, nn::ResultSuccess()));

    EXPECT_RESULT(NewsTaskManager::GetInstance().GetCurrentTask(&record), nn::news::ResultNotFound);
}

TEST(NewsTaskManager, SystemUpdateRequired)
{
    EXPECT_RESULT_SUCCESS(NewsTaskManager::GetInstance().Clear());

    const nn::news::TopicId topicId = {"nx_topic"};

    EXPECT_RESULT_SUCCESS(NewsTaskManager::GetInstance().SetSubscriptionStatus(topicId, nn::news::SubscriptionStatus_Subscribed));

    NewsTaskManager::Record record = {};
    EXPECT_RESULT_SUCCESS(NewsTaskManager::GetInstance().GetCurrentTask(&record));
    EXPECT_RESULT_SUCCESS(NewsTaskManager::GetInstance().NotifyDone(record, nn::news::ResultSystemUpdateRequired()));
    EXPECT_RESULT_SUCCESS(Version::GetInstance().RequireSystemUpdate());

    EXPECT_TRUE(NewsTaskManager::GetInstance().GetSubscriptionStatus(topicId) == nn::news::SubscriptionStatus_Subscribed);

    EXPECT_RESULT_SUCCESS(Version::GetInstance().Clear());

    EXPECT_RESULT_SUCCESS(NewsTaskManager::GetInstance().Load());
    EXPECT_RESULT_SUCCESS(NewsTaskManager::GetInstance().NotifySystemUpdated());

    EXPECT_RESULT_SUCCESS(NewsTaskManager::GetInstance().GetCurrentTask(&record));
}

TEST(NewsTaskManager, NintendoAccountRequired)
{
    EXPECT_RESULT_SUCCESS(NewsTaskManager::GetInstance().Clear());

    const nn::news::TopicId topicId = {"nx_topic"};

    EXPECT_RESULT_SUCCESS(NewsTaskManager::GetInstance().SetSubscriptionStatus(topicId, nn::news::SubscriptionStatus_Subscribed));

    NewsTaskManager::Record record = {};
    EXPECT_RESULT_SUCCESS(NewsTaskManager::GetInstance().GetCurrentTask(&record));
    EXPECT_RESULT_SUCCESS(NewsTaskManager::GetInstance().NotifyDone(record, nn::news::ResultNintendoAccountNotLinked()));

    EXPECT_RESULT(NewsTaskManager::GetInstance().GetCurrentTask(&record), nn::news::ResultNotFound);

    EXPECT_TRUE(NewsTaskManager::GetInstance().GetSubscriptionStatus(topicId) == nn::news::SubscriptionStatus_Subscribed);

    EXPECT_RESULT_SUCCESS(Version::GetInstance().Clear());

    EXPECT_RESULT_SUCCESS(NewsTaskManager::GetInstance().Load());
    EXPECT_RESULT_SUCCESS(NewsTaskManager::GetInstance().NotifyNintendoAccountLinked());

    EXPECT_RESULT_SUCCESS(NewsTaskManager::GetInstance().GetCurrentTask(&record));
}

TEST(NewsTaskManager, Wait)
{
    EXPECT_RESULT_SUCCESS(NewsTaskManager::GetInstance().Clear());

    const nn::news::TopicId topicId = {"nx_topic"};

    EXPECT_RESULT_SUCCESS(NewsTaskManager::GetInstance().SetSubscriptionStatus(topicId, nn::news::SubscriptionStatus_Subscribed));

    NewsTaskManager::Record record = {};
    EXPECT_RESULT_SUCCESS(NewsTaskManager::GetInstance().GetCurrentTask(&record));
    EXPECT_RESULT_SUCCESS(NewsTaskManager::GetInstance().Wait(topicId, 1));

    NewsTaskManager::GetInstance().Schedule();

    EXPECT_RESULT(NewsTaskManager::GetInstance().GetCurrentTask(&record), nn::news::ResultNotFound);

    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1500));

    NewsTaskManager::GetInstance().Schedule();

    EXPECT_RESULT_SUCCESS(NewsTaskManager::GetInstance().GetCurrentTask(&record));
    EXPECT_TRUE(nn::util::Strncmp(record.topicId.value, "nx_topic", nn::news::TopicIdLengthMax + 1) == 0);
}

TEST(NewsTaskManager, Expired)
{
    EXPECT_RESULT_SUCCESS(NewsTaskManager::GetInstance().Clear());

    const nn::news::TopicId topicId = {"nx_topic"};

    EXPECT_RESULT_SUCCESS(NewsTaskManager::GetInstance().SetSubscriptionStatus(topicId, nn::news::SubscriptionStatus_Subscribed));

    NewsTaskManager::Record record = {};
    EXPECT_RESULT_SUCCESS(NewsTaskManager::GetInstance().GetCurrentTask(&record));
    EXPECT_RESULT_SUCCESS(NewsTaskManager::GetInstance().NotifyDone(record, nn::news::ResultServiceExpired()));

    EXPECT_RESULT(NewsTaskManager::GetInstance().GetCurrentTask(&record), nn::news::ResultNotFound);

    EXPECT_TRUE(NewsTaskManager::GetInstance().GetSubscriptionStatus(topicId) == nn::news::SubscriptionStatus_Subscribed);
}

TEST(NewsTaskManager, SleepAwaked)
{
    EXPECT_RESULT_SUCCESS(NewsTaskManager::GetInstance().Clear());

    const nn::news::TopicId topicId = {"nx_topic"};

    EXPECT_RESULT_SUCCESS(NewsTaskManager::GetInstance().SetSubscriptionStatus(topicId, nn::news::SubscriptionStatus_Subscribed));

    NewsTaskManager::Record record = {};
    EXPECT_RESULT_SUCCESS(NewsTaskManager::GetInstance().GetCurrentTask(&record));
    EXPECT_RESULT_SUCCESS(NewsTaskManager::GetInstance().NotifyDone(record, nn::news::ResultServerError404()));

    EXPECT_RESULT(NewsTaskManager::GetInstance().GetCurrentTask(&record), nn::news::ResultNotFound);

    EXPECT_TRUE(NewsTaskManager::GetInstance().GetSubscriptionStatus(topicId) == nn::news::SubscriptionStatus_Subscribed);

    NewsTaskManager::GetInstance().NotifySleepAwaked();

    EXPECT_RESULT_SUCCESS(NewsTaskManager::GetInstance().GetCurrentTask(&record));
}

TEST(NewsTaskManager, Finalize)
{
    nn::time::Finalize();
}
